/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: thing.c
 */

#include <string.h>
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "pixmap.h"
#include "widget.h"
#include "readable.h"
#include "thing.h"
#include "logging.h"

/*
 *
 *   void add_thing_to_selection_list (GtkCList *, object_handle_t, gboolean)
 *
 *   Description:
 *      This routine appends a row to the standard selection window
 *      GtkCList with the information from the given object handle.
 *      It optionally can mark the appended entry as selected.
 *
 *   Entry:
 *      widget        - address of the selection window GtkCList widget
 *      handle        - handle of object to place in selection window
 *      mark_selected - TRUE if row added should be selected
 *
 *   Exit:
 *      A new row is appended to the standard selection window list and the
 *      row index is returned. Otherwise, -1 is returned for any error.
 *
 */
gint add_thing_to_selection_list(GtkCList * clist, object_handle_t handle, gboolean mark_selected)
{
	gint rc;
	gint row = -1;
	gchar *text[] = { "", "", "", "" };
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
	} else {
		switch (object->type) {
		case VOLUME:
			text[SL_SIZE_COLUMN] =
			    make_sectors_readable_string(object->info.volume.vol_size);
			text[SL_DESC_COLUMN] = object->info.volume.name;
			break;

		case EVMS_OBJECT:
		case REGION:
		case SEGMENT:
		case DISK:
			text[SL_SIZE_COLUMN] =
			    make_sectors_readable_string(object->info.object.size);
			text[SL_DESC_COLUMN] = object->info.object.name;
			break;

		case CONTAINER:
			text[SL_SIZE_COLUMN] =
			    make_sectors_readable_string(object->info.container.size);
			text[SL_DESC_COLUMN] = object->info.container.name;
			break;

		case PLUGIN:
			text[SL_DESC_COLUMN] = object->info.plugin.long_name;
			break;

		default:
			log_debug("%s: Unknown type %d.\n", __FUNCTION__, object->type);
			break;
		}

		row = clist_append_row(clist, text);

		if (mark_selected)
			gtk_clist_select_row(clist, row, 0);

		if (text[SL_SIZE_COLUMN] != "")
			g_free(text[SL_SIZE_COLUMN]);

		/*
		 * Associate things's handle with row data so that we can
		 * use it when we invoke an operation based on the currently
		 * selected row.
		 */

		if (row != -1) {
			set_clist_row_pixmap(clist, row, object->type);
			gtk_clist_set_row_data(clist, row, GUINT_TO_POINTER(handle));
		}

		evms_free(object);
	}

	return row;
}

/*
 * The following two routines are basically getter/setter functions
 * to allow binding and retrieving an object handle from a toplevel
 * widget.
 */

inline void bind_object_handle_to_toplevel_widget(GtkWidget * widget, object_handle_t object)
{
	GtkWidget *toplevel = gtk_widget_get_toplevel(widget);

	gtk_object_set_data(GTK_OBJECT(toplevel), "object_handle", GUINT_TO_POINTER(object));
}

inline object_handle_t retrieve_object_handle_from_toplevel_widget(GtkWidget * widget)
{
	GtkWidget *toplevel = gtk_widget_get_toplevel(widget);

	return GPOINTER_TO_UINT(gtk_object_get_data(GTK_OBJECT(toplevel), "object_handle"));
}

/*
 *
 *   void add_thing_as_selected_list_item (GtkCList *, gpointer)
 *
 *   Description:
 *      This routine populates the standard selection window GtkCList
 *      with the information from one object handle. It also selects
 *      this single entry in the list so that the "Next" button is
 *      made active to complete an operation on the row.
 *
 *   Entry:
 *      widget    - address of the selection window GtkCList widget
 *      user_data - contains handle of object to place in selection window
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void add_thing_as_selected_list_item(GtkCList * clist, gpointer user_data)
{
	object_type_t type;
	object_handle_t handle = GPOINTER_TO_UINT(user_data);

	if (evms_get_handle_object_type(handle, &type) == SUCCESS)
		set_selection_window_clist_column_titles(clist, _("Size"),
							 make_object_type_readable_string(type),
							 NULL);

	add_thing_to_selection_list(clist, handle, TRUE);
}

/*
 *
 *   GtkWidget* create_thing_details_window (extended_info_array_t *)
 *
 *   Description:
 *      This routine creates a window with a close button and a notebook
 *      widget that contains plugin specific information. Each extended
 *      info structure is convereted to a widget that represents the data
 *      and the widgets are placed in a widget notebook which is placed
 *      in the window.
 *
 *   Entry:
 *      handle        - the handle for the thing so we can get additional info
 *      extended_info - address of structure containing array of extended_info_t
 *                      structures
 *      field_title    - title of field that was drilled into or NULL if toplevel
 *
 *   Exit:
 *      The address of the window containing the extended info widget notebook
 *      is returned.
 *
 */
GtkWidget *create_thing_details_window(object_handle_t handle,
				       extended_info_array_t * extended_info, gchar * field_title)
{
	gint i;
	gchar *frame_title;
	gchar *window_title;
	gchar *thing_name;
	gchar *plugin_name;
	gchar *field_drilled = NULL;
	GSList *widgets = NULL;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *frame;
	GtkWidget *window;
	GtkWidget *widget;
	GtkWidget *button;
	GtkWidget *hbuttonbox;
	GtkWidget *hseparator;
	GtkWidget *widget_notebook;
	GtkWidget *pixmap;
	GdkBitmap *mask;
	GdkPixmap *gdk_pixmap;
	GtkAccelGroup *accel_group;

	get_dialog_pixmap(INFO_PIXMAP, &gdk_pixmap, &mask);

	get_object_and_plugin_names(handle, &thing_name, &plugin_name);

	if (field_title)
		field_drilled = g_strconcat((" - "), field_title, NULL);

	frame_title = g_strconcat(thing_name, " (", plugin_name, ")", field_drilled, NULL);
	window_title = g_strconcat(_("Detailed Information"), " - ", thing_name, NULL);

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_default_size(GTK_WINDOW(window), 550, -1);

	accel_group = gtk_accel_group_new();
	vbox = gtk_vbox_new(FALSE, 0);
	hbox = gtk_hbox_new(FALSE, 5);
	frame = gtk_frame_new(frame_title);
	hbuttonbox = gtk_hbutton_box_new();
	hseparator = gtk_hseparator_new();
	button = gtk_button_new();
	pixmap = gtk_pixmap_new(gdk_pixmap, mask);

	gtk_accel_group_attach(accel_group, GTK_OBJECT(window));

	get_ok_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(button, gdk_pixmap, mask, _("_OK"), accel_group);

	for (i = 0; i < extended_info->count; i++) {
		widget = create_widget_from_extended_info(&(extended_info->info[i]));

		if (widget != NULL)
			widgets = g_slist_append(widgets, widget);
	}

	widget_notebook = create_widget_notebook(widgets, NULL, 7, 0);

	gtk_box_pack_start(GTK_BOX(hbox), pixmap, FALSE, FALSE, 0);
	gtk_misc_set_alignment(GTK_MISC(pixmap), 0.05, 0.05);

	gtk_box_pack_start(GTK_BOX(hbox), frame, TRUE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 2);

	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hseparator, FALSE, FALSE, 12);
	gtk_box_pack_start(GTK_BOX(vbox), hbuttonbox, FALSE, FALSE, 0);

	gtk_window_set_title(GTK_WINDOW(window), window_title);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
	gtk_container_set_border_width(GTK_CONTAINER(hbuttonbox), 5);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 6);
	gtk_container_set_border_width(GTK_CONTAINER(widget_notebook), 5);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbuttonbox), -1, 44);
	gtk_button_box_set_child_ipadding(GTK_BUTTON_BOX(hbuttonbox), 19, 0);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), button);
	gtk_container_add(GTK_CONTAINER(frame), widget_notebook);
	gtk_container_add(GTK_CONTAINER(window), vbox);

	gtk_widget_show(widget_notebook);
	gtk_widget_show(button);
	gtk_widget_show(hbuttonbox);
	gtk_widget_show(hseparator);
	gtk_widget_show(vbox);
	gtk_widget_show(frame);
	gtk_widget_show(pixmap);
	gtk_widget_show(hbox);

	gtk_signal_connect_object(GTK_OBJECT(button), "clicked", gtk_widget_destroy,
				  GTK_OBJECT(window));
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);

	gtk_widget_grab_default(button);

	g_free(thing_name);
	g_free(plugin_name);
	g_free(frame_title);
	g_free(window_title);
	g_free(field_drilled);

	g_slist_free(widgets);

	return window;
}

/*
 *   inline gboolean is_storage_object (object_handle_t)
 *
 *   Description:
 *      This routine checks the object handle type to see if it belongs to
 *      a disk, segment, region, or feature object.
 *
 *   Entry:
 *      handle - the object handle
 *
 *   Exit:
 *      Return TRUE if the handle is not for a volume, container or plugin
 */
inline gboolean is_storage_object(object_handle_t handle)
{
	gboolean result = FALSE;
	object_type_t type;

	if (evms_get_handle_object_type(handle, &type) == SUCCESS) {
		if (type != PLUGIN && type != VOLUME && type != CONTAINER)
			result = TRUE;
	}

	return result;
}

/*
 *   struct extra_extended_info *create_extra_extended_info (object_handle_t, extended_info_array_t *)
 *
 *   Description:
 *      This routine adds some extra common information to extended info for a storage
 *      object. It essentially creates a new extended info array with additional entries
 *      along with a copy of the original ones.
 *
 *   Entry:
 *      handle - the object handle
 *      extended_info - the original extended info array
 *
 *   Exit:
 *      Returns a pointer a extra_extended_info structure containing the address of the
 *      new and original extended info arrays.
 */
struct extra_extended_info *create_extra_extended_info(object_handle_t handle,
						       extended_info_array_t * extended_info)
{
	gint rc;
	struct extra_extended_info *extra = NULL;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);

	if (rc == SUCCESS) {
		extended_info_array_t *new_info;

		extra = g_new(struct extra_extended_info, 1);
		new_info =
		    g_malloc0(sizeof(extended_info_array_t) +
			      (sizeof(extended_info_t) * (extended_info->count + 2)));

		new_info->count = extended_info->count + 2;
		memcpy(&(new_info->info[2]), &(extended_info->info[0]),
		       sizeof(extended_info_t) * extended_info->count);

		new_info->info[0].name = _("major_no");
		new_info->info[0].title = _("Major Number");
		new_info->info[0].desc =
		    _("This field describes the object's device major number.");
		new_info->info[0].type = EVMS_Type_Unsigned_Int32;
		new_info->info[0].unit = EVMS_Unit_None;
		new_info->info[0].format = EVMS_Format_Normal;
		new_info->info[0].collection_type = EVMS_Collection_None;
		new_info->info[0].value.ui32 = object->info.object.dev_major;

		new_info->info[1].name = _("minor_no");
		new_info->info[1].title = _("Minor Number");
		new_info->info[1].desc =
		    _("This field describes the object's device minor number.");
		new_info->info[1].type = EVMS_Type_Unsigned_Int32;
		new_info->info[1].unit = EVMS_Unit_None;
		new_info->info[1].format = EVMS_Format_Normal;
		new_info->info[1].collection_type = EVMS_Collection_None;
		new_info->info[1].value.ui32 = object->info.object.dev_minor;


		extra->new = new_info;
		extra->original = extended_info;
	}

	return extra;
}

/*
 *   void free_extra_extended_info (struct extra_extended_info *)
 *
 *   Description:
 *      This routine frees the extended info array we created along
 *      with the structure that tracked it and the original.
 *
 *   Entry:
 *      info - the address of the extra_extended_info structure
 *
 *   Exit:
 *      Return nothing.
 */
void free_extra_extended_info(struct extra_extended_info *info)
{
	g_free(info->new);
	g_free(info);
}

/*
 *
 *   void display_thing_details (object_handle_t, gchar *, gchar *)
 *
 *   Description:
 *      This routine displays the plugin specific information on a thing.
 *
 *   Entry:
 *      handle      - the handle of the thing to get extended info for
 *      field_name  - an optional field name to gather specific info on
 *      field_title - the title/description of the optional field name
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void display_thing_details(object_handle_t handle, gchar * field_name, gchar * field_title)
{
	gint rc;
	extended_info_array_t *extended_info = NULL;

	rc = evms_get_extended_info(handle, field_name, &extended_info);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_extended_info() returned error code %d.\n", __FUNCTION__,
			  rc);
		display_popup_window(_("View Detailed Information"),
				     _("No detailed information is available."));
	} else if (extended_info != NULL && extended_info->count > 0) {
		GtkWidget *window;
		struct extra_extended_info *extra_info = NULL;

		if (is_storage_object(handle)) {
			extra_info = create_extra_extended_info(handle, extended_info);
			if (extra_info != NULL)
				extended_info = extra_info->new;
		}

		window = create_thing_details_window(handle, extended_info, field_title);
		bind_object_handle_to_toplevel_widget(window, handle);
		gtk_widget_show(window);

		if (extra_info) {
			extended_info = extra_info->original;
			free_extra_extended_info(extra_info);
		}

		evms_free(extended_info);
	} else {
		display_popup_window(_("View Detailed Information"),
				     _("No detailed information is available."));
		evms_free(extended_info);
	}
}

/*
 *
 *   void on_display_thing_details_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine displays the plugin specific information on the thing
 *      selected in one of the browser views. The information is retrieved
 *      through the evms_get_extended_info() API.
 *
 *   Entry:
 *      menuitem  - menu item selected
 *      user_data - contains handle of the object we want extended info on
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_display_thing_details_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	display_thing_details(GPOINTER_TO_UINT(user_data), NULL, NULL);
}

/*
 *
 *   void on_display_more_thing_details_button_clicked (GtkButton *, field_details *)
 *
 *   Description:
 *      This routine displays the plugin specific information on the field
 *      for a thing if the field had the EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE
 *      flag set.
 *
 *   Entry:
 *      button - id of the button clicked
 *      field  - contains reference info of the specific field
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_display_more_thing_details_button_clicked(GtkButton * button, field_details_t * field)
{
	gchar *name = NULL;

	object_handle_t handle = retrieve_object_handle_from_toplevel_widget(GTK_WIDGET(button));

	/*
	 * Assign field->name to name if non-NULL and non-empty string.
	 *
	 * An empty field name equates to retrieving full info for the
	 * object. This empty string trick is used by the
	 * create_volume_pseudo_extended_info() routine to allow getting
	 * toplevel extended info from an FSIM for a volume.
	 */
	if (field->name && *(field->name) != '\0')
		name = field->name;

	display_thing_details(handle, name, field->title);
}

/*
 *
 *   GtkWidget *create_multi_results_window (GtkWidget *, gchar *, GtkCList **)
 *
 *   Description:
 *      This routine creates the multiple results window used for
 *      operations like remove and destroy where multiple selections
 *      and iterations of an operation are permitted.
 *
 *   Entry:
 *      widget - a reference widget to obtain and append to the window list
 *      title  - the results window title
 *      rclist - the address to a GtkCList * to return the id of the clist
 *               in the results window
 *
 *   Exit:
 *      The results window is created and its id returned to the caller.
 *
 */
GtkWidget *create_multi_results_window(GtkWidget * widget, gchar * title, GtkCList ** rclist)
{
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *text_box_frame;
	GtkWidget *scrolledwindow;
	GtkWidget *clist;
	GtkWidget *icon_col_label;
	GtkWidget *object_col_label;
	GtkWidget *result_col_label;
	GtkWidget *hseparator;
	GtkWidget *hbuttonbox;
	GtkWidget *button;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkAccelGroup *accel_group;
	GtkTooltips *tooltips;

	tooltips = gtk_tooltips_new();
	accel_group = gtk_accel_group_new();
	window = gtk_window_new(GTK_WINDOW_DIALOG);
	vbox = gtk_vbox_new(FALSE, 0);
	text_box_frame = gtk_frame_new(NULL);
	scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
	clist = gtk_clist_new(MAX_MR_COLUMNS);
	icon_col_label = gtk_label_new("");
	object_col_label = gtk_label_new(_("Storage Object"));
	result_col_label = gtk_label_new(_("Operation Result"));
	hseparator = gtk_hseparator_new();
	hbuttonbox = gtk_hbutton_box_new();
	button = gtk_button_new();
	*rclist = (GtkCList *) clist;

	gtk_accel_group_attach(accel_group, GTK_OBJECT(window));

	get_ok_pixmap(&pixmap, &mask);
	add_pixmap_label_to_button(button, pixmap, mask, _("_OK"), accel_group);

	gtk_widget_show(vbox);
	gtk_widget_show(text_box_frame);
	gtk_widget_show(scrolledwindow);
	gtk_widget_show(clist);
	gtk_widget_show(icon_col_label);
	gtk_widget_show(object_col_label);
	gtk_widget_show(result_col_label);
	gtk_widget_show(hseparator);
	gtk_widget_show(hbuttonbox);
	gtk_widget_show(button);

	gtk_window_set_title(GTK_WINDOW(window), title);
	gtk_window_set_default_size(GTK_WINDOW(window), 600, 410);
	gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);

	gtk_container_add(GTK_CONTAINER(window), vbox);

	gtk_box_pack_start(GTK_BOX(vbox), text_box_frame, TRUE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(text_box_frame), 6);

	gtk_container_add(GTK_CONTAINER(text_box_frame), scrolledwindow);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
				       GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);

	gtk_container_add(GTK_CONTAINER(scrolledwindow), clist);
	gtk_clist_set_column_width(GTK_CLIST(clist), MR_ICON_COLUMN, 24);
	gtk_clist_set_column_width(GTK_CLIST(clist), MR_NAME_COLUMN, 160);
	gtk_clist_set_column_width(GTK_CLIST(clist), MR_RESULTS_COLUMN, 200);
	gtk_clist_column_titles_show(GTK_CLIST(clist));

	gtk_clist_set_column_widget(GTK_CLIST(clist), MR_ICON_COLUMN, icon_col_label);
	gtk_clist_set_column_widget(GTK_CLIST(clist), MR_NAME_COLUMN, object_col_label);
	gtk_clist_set_column_widget(GTK_CLIST(clist), MR_RESULTS_COLUMN, result_col_label);

	gtk_box_pack_start(GTK_BOX(vbox), hseparator, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbuttonbox, FALSE, FALSE, 0);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbuttonbox), 18);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbuttonbox), 108, 46);

	gtk_container_set_border_width(GTK_CONTAINER(hbuttonbox), 13);
	gtk_container_add(GTK_CONTAINER(hbuttonbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);

	gtk_tooltips_set_tip(tooltips, button, _("Close this window"), NULL);

	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  gtk_widget_destroy, GTK_OBJECT(window));

	gtk_signal_connect(GTK_OBJECT(clist), "select_row",
			   GTK_SIGNAL_FUNC(on_readonly_clist_select_row), NULL);

	set_window_list(window, g_list_append(NULL, gtk_widget_get_toplevel(widget)));

	return window;
}

/*
 *
 *   inline gint append_row_to_results_clist (GtkCList *, object_handle_t)
 *
 *   Description:
 *      This routine appends a new row in the clist contained in the
 *      multi-results window.
 *
 *   Entry:
 *      clist  - the multiple results window GtkCList
 *      handle - the handle of the object we want to display info on
 *
 *   Exit:
 *      A new row is added with the proper information and the row number
 *      is returned.
 *
 */
inline gint append_row_to_results_clist(GtkCList * clist, object_handle_t handle)
{
	gint rc;
	gint row = -1;
	gchar *text[MAX_MR_COLUMNS];
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
	} else {
		text[MR_ICON_COLUMN] = "";
		text[MR_RESULTS_COLUMN] = "";

		switch (object->type) {
		case VOLUME:
			text[MR_NAME_COLUMN] = object->info.volume.name;
			break;

		case EVMS_OBJECT:
		case REGION:
		case SEGMENT:
		case DISK:
			text[MR_NAME_COLUMN] = object->info.object.name;
			break;

		case CONTAINER:
			text[MR_NAME_COLUMN] = object->info.container.name;
			break;

		default:
			text[MR_NAME_COLUMN] = _("Unknown");
			log_debug("%s: Unhandled type %d.\n", __FUNCTION__, object->type);
			break;
		}

		row = clist_append_row(clist, text);

		if (row != -1)
			set_clist_row_pixmap(clist, row, object->type);

		evms_free(object);
	}

	return row;
}

/*
 *
 *  inline void set_results_clist_row (GtkCList *, gint, gint, gchar *)
 *
 *   Description:
 *      This routine updates the results column text for the given
 *      row in the multi-results window list.
 *
 *   Entry:
 *      clist - the multiple results window GtkCList
 *      row   - the row in the clist to update
 *      rc    - the return code
 *      success_msg - the message to use if rc == SUCCESS
 *
 *   Exit:
 *      The results column is updated for the given row with the proper
 *      message.
 *
 */
inline void set_results_clist_row(GtkCList * clist, gint row, gint rc, gchar * success_msg)
{
	gchar *msg;

	if (rc == SUCCESS)
		msg = success_msg;
	else if (rc == E_CANCELED)
		msg = _("The operation was cancelled by the user.");
	else
		msg = (gchar *) evms_strerror(ABS(rc));

	gtk_clist_set_text(clist, row, MR_RESULTS_COLUMN, msg);
}

/*
 *
 *  void display_multi_results_window (GtkWidget *, gchar *)
 *
 *   Description:
 *      This routine displays the results window and destroys the selection
 *      window if the operation was successful.
 *
 *   Entry:
 *      window - the multiple results window
 *      status - the status message for the status bar
 *
 *   Exit:
 *      The results window is displayed and the previous windows
 *      are destroyed.
 *
 */
void display_multi_results_window(GtkWidget * window, gchar * status)
{
	GList *window_list;

	window_list = get_window_list(window);
	complete_display_results_window_work(window, status, window_list, TRUE);
	gtk_window_set_transient_for(GTK_WINDOW(window), GTK_WINDOW(get_main_window_id()));
	gtk_widget_show(window);
}

/**
 *	duplicate_handle_array - create a dynamically allocated copy of a given handle array
 *	@old: the original/source handle array
 *
 *	This routine make a copy of given handle array. The copy should be freed
 *	with g_free() when no longer needed.
 */
handle_array_t *duplicate_handle_array(handle_array_t * old)
{
	handle_array_t *new;

	new = g_memdup(old, sizeof(handle_array_t) + (sizeof(object_handle_t) * old->count));

	return new;
}

/**
 *	merge_handle_array - combine two handles areas to produce a new one
 *	@array1: the first handle array
 *	@array2: the second handle array
 *
 *	This routine takes the address of two handle arrays, allocates space
 *	to copy the handles in both into one array and returns the new array.
 */
handle_array_t *merge_handle_array(handle_array_t * array1, handle_array_t * array2)
{
	int array1_size, array2_size, merged_size;
	handle_array_t *merged_array = NULL;

	if (array1 && array2) {
		array1_size = sizeof(object_handle_t) * array1->count;
		array2_size = sizeof(object_handle_t) * array2->count;
		merged_size = sizeof(handle_array_t) + array1_size + array2_size;

		merged_array = g_malloc(merged_size);

		if (merged_array) {
			merged_array->count = array1->count + array2->count;
			memcpy(&(merged_array->handle[0]), &(array1->handle[0]), array1_size);
			memcpy(&(merged_array->handle[array1->count]), &(array2->handle[0]),
			       array2_size);
		}
	}
	return merged_array;
}
