/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: Swap FSIM
 * File: evms2/engine/plugins/swap/swapfs.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <plugin.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <asm/page.h> /* to get PAGE_SIZE */
#include <sys/swap.h>

#include "swapfs.h"

plugin_record_t *my_plugin_record = &swap_plugin_record;
engine_functions_t *EngFncs = NULL;

/**
 * fsim_rw_diskblocks
 * @dev_ptr:	File handle of an opened device to read/write
 * @disk_offset: Byte offset from beginning of device for start of disk
 *		block read/write
 * @disk_count:	Number of bytes to read/write
 * @data_buffer: On read this will be filled in with data read from
 *		disk; on write this contains data to be written
 * @mode:	GET (read) or PUT (write)
 *
 * Read or write specific number of bytes for an opened device.
 **/
static int fsim_rw_diskblocks(logical_volume_t *volume,
			      int dev_ptr,
			      int64_t disk_offset,
			      int32_t disk_count,
			      void *data_buffer,
			      int mode)
{
	int32_t bytes = 0;
	int rc = 0;

	LOG_ENTRY();

	switch (mode) {
	case GET:
		bytes = EngFncs->read_volume(volume, dev_ptr, data_buffer,
					     disk_count, disk_offset);
		break;
	case PUT:
		bytes = EngFncs->write_volume(volume, dev_ptr, data_buffer,
					      disk_count, disk_offset);
		break;
	default:
		break;
	}

	if (bytes != disk_count) {
		rc = EIO;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * set_mkfs_options
 * @options:	options array passed from EVMS engine.
 * @argv:	mkfs options array
 * @vol_name:	volume name on which program will be executed
 *
 * Build options array (argv) for mkfs.swap
 **/
static void set_mkfs_options(option_array_t *options,
			     char **argv, char *vol_name)
{
	int i, opt_count = 1;

	LOG_ENTRY();

	argv[0] = "mkswap";

	for (i = 0; i < options->count; i++) {
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name, "checkbad")) {
				options->option[i].number = SWAP_MKFS_BADBLOCK_INDEX;
			}
		}

		switch (options->option[i].number) {
		case SWAP_MKFS_BADBLOCK_INDEX:
			/* Check-for-bad-blocks option. */
			if (options->option[i].value.b) {
				argv[opt_count++] = "-c";
			}
			break;
		default:
			break;
		}
	}

	argv[opt_count++] = vol_name;
	argv[opt_count] = NULL;

	LOG_EXIT_VOID();
	return;
}

/**
 * mkfs
 *
 * Format the volume.
 **/
static int mkfs(logical_volume_t *volume, option_array_t *options)
{
	char *argv[SWAP_MKFS_OPTIONS_COUNT + 3];
	char *buffer;
	pid_t pidm;
	int fds2[2];  /* Pipe for stderr and stdout. 0=read, 1=write */
	int status, bytes_read, rc;

	LOG_ENTRY();

	rc = pipe(fds2);
	if (rc) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN);
	if (!buffer) {
		rc = ENOMEM;
		goto out;
	}

	set_mkfs_options(options, argv, volume->dev_node);

	pidm = EngFncs->fork_and_execvp(volume, argv, NULL, fds2, fds2);
	if (pidm != -1) {
		fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
		while (! waitpid(pidm, &status, WNOHANG)) {
			bytes_read = read(fds2[0], buffer, MAX_USER_MESSAGE_LEN);
			if (bytes_read > 0) {
				LOG_DEFAULT("mkswap output: \n%s", buffer);
				memset(buffer, 0, bytes_read); /* Clear out message. */
			}
			usleep(10000);
		}
		if (WIFEXITED(status)) {
			write(fds2[1], "\n", 1); /* Flush the pipe. */
			do {
				bytes_read = read(fds2[0], buffer,
						  MAX_USER_MESSAGE_LEN);
				if (bytes_read > 0) {
					LOG_DEFAULT("mkswap output: \n%s", buffer);
				}
			} while (bytes_read == MAX_USER_MESSAGE_LEN);
			LOG_DEFAULT("mkswap completed with rc = %d \n", status);
			rc = WEXITSTATUS(status);
		} else {
			rc = EINTR;
		}
	} else {
		rc = EIO;
	}

out:
	EngFncs->engine_free(buffer);
	close(fds2[0]);
	close(fds2[1]);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * get_swap_swapfs_super_block
 * @vol_name:	Volume name from which to get the swapfs_super_block.
 * @sb_ptr:	Pointer to swapfs_super_block.
 *
 * Get and validate a Swapfs super-block.
 **/
static int get_swapfs_super_block(logical_volume_t *volume)
{
	char magic[10];
	int fd, rc;

	LOG_ENTRY();

	fd = EngFncs->open_volume(volume, O_RDONLY, 0);
	if (fd < 0) {
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	/* Get primary swapfs_super_block */
	rc = fsim_rw_diskblocks(volume, fd,
				SWAP_MAGIC_OFFSET_IN_BYTES,
				sizeof(magic), magic, GET);
	if (!rc) {
		/* See if primary swapfs_super_block is Swap */
		if (strncmp(magic, SWAPFS_MAGIC_STRING, sizeof(magic)) &&
		    strncmp(magic, SWAPFS_MAGIC_STRING2, sizeof(magic))) {
			rc = EINVAL;
		} else {
			volume->flags |= VOLFLAG_NOT_MOUNTABLE;
		}
	}

	EngFncs->close_volume(volume, fd);
	LOG_EXIT_INT(rc);
	return rc;
}


/**
 * swap_test_version
 *
 * Simply need to see if mkswap exists.
 **/
static int swap_test_version(void)
{
	char *argv[3] = {"mkswap", "-V", NULL};
	char *buffer;
	pid_t pidm;
	int fds2[2];  /* Pipe for stderr and stdout. 0=read, 1=write */
	int status, rc;

	LOG_ENTRY();

	rc = pipe(fds2);
	if (rc) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	buffer = EngFncs->engine_alloc(1000);
	if (!buffer) {
		rc = ENOMEM;
		goto out;
	}

	pidm = EngFncs->fork_and_execvp(NULL, argv, NULL, fds2, fds2);
	if (pidm != -1) {
		waitpid(pidm, &status, 0);
		if (WIFEXITED(status) && WEXITSTATUS(status) != ENOENT) {
			rc = 0;
		} else {
			rc = ENOSYS;
		}
	} else {
		rc = EIO;
	}

out:
	EngFncs->engine_free(buffer);
	close(fds2[0]);
	close(fds2[1]);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * swap_setup
 **/
static int swap_setup(engine_functions_t *engine_function_table)
{
	int rc;

	EngFncs = engine_function_table;
	LOG_ENTRY();

	rc = swap_test_version();

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * swap_probe
 *
 * Does this volume contain a swap filesystem?
 **/
static int swap_probe(logical_volume_t *volume)
{
	int rc;

	LOG_ENTRY();

	/* Get and validate swapfs super block */
	rc = get_swapfs_super_block(volume);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * swap_get_fs_size
 *
 * Get the current size of this volume
 **/
static int swap_get_fs_size(logical_volume_t *volume,
			    sector_count_t *size)
{
	LOG_ENTRY();

	*size = volume->vol_size;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * swap_get_fs_limits
 *
 * Get the size limits for this volume.
 **/
static int swap_get_fs_limits(logical_volume_t *volume,
			      sector_count_t *min_size,
			      sector_count_t *max_volume_size,
			      sector_count_t *max_object_size)
{
	LOG_ENTRY();

	*max_volume_size = (sector_count_t)-1;	/* No limit. */
	*max_object_size = (sector_count_t)-1;	/* No limit. */
	*min_size = (sector_count_t)(PAGE_SIZE * 10);	/* 10 page minimum. */

	LOG_EXTRA("Volume: %s, min: %"PRIu64", max: %"PRIu64"\n",
		  volume->name, *min_size, *max_volume_size);
	LOG_EXTRA("fssize: %"PRIu64", vol_size: %"PRIu64"\n",
		  volume->fs_size, volume->vol_size);

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * swap_can_mkfs
 *
 * Can we add a swap filesystem to this volume?
 **/
static int swap_can_mkfs(logical_volume_t *volume)
{
	int rc = 0;

	LOG_ENTRY();

	if (volume->vol_size < SWAPFS_MIN_SIZE) {
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * swap_can_unmkfs
 *
 * Can we remove the swap filesystem from this volume?
 **/
static int swap_can_unmkfs(logical_volume_t *volume)
{
	int rc = 0;

	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't unmkfs. */
		rc = EBUSY;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * swap_can_expand_by
 *
 * How much can the swap filesystem be expanded?
 **/
static int swap_can_expand_by(logical_volume_t *volume,
			      sector_count_t *delta)
{
	int rc = 0;

	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		rc = EBUSY;
	} else {
		swap_get_fs_limits(volume, &volume->min_fs_size,
				   &volume->max_vol_size,
				   &volume->max_fs_size);
		if (*delta > volume->max_fs_size - volume->fs_size) {
			*delta = volume->max_fs_size - volume->fs_size;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * swap_can_shrink_by
 *
 * How much can the swap filesystem be shrunk?
 **/
static int swap_can_shrink_by(logical_volume_t *volume,
			      sector_count_t *delta)
{
	int rc = 0;

	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		rc = EBUSY;
	} else {
		swap_get_fs_limits(volume, &volume->min_fs_size,
				   &volume->max_vol_size,
				   &volume->max_fs_size);
		if (*delta > volume->fs_size - volume->min_fs_size) {
			*delta = volume->fs_size - volume->min_fs_size;
		}
		if (volume->min_fs_size >= volume->fs_size) {
			rc = ENOSPC;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * swap_mkfs
 *
 * Make a swap filesystem on this volume.
 **/
static int swap_mkfs(logical_volume_t *volume, option_array_t *options)
{
	int rc;

	LOG_ENTRY();

	/* Don't format if mounted. */
	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		rc = EBUSY;
	} else {
		rc = mkfs(volume, options);
		if (!rc) {
			volume->flags |= VOLFLAG_NOT_MOUNTABLE;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * swap_discard
 *
 * The swap FSIM has no private data attached to the volume.
 **/
static int swap_discard(logical_volume_t *volume)
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * swap_unmkfs
 *
 * Remove the swap filesystem from this volume.
 **/
static int swap_unmkfs(logical_volume_t * volume)
{
	int rc, fd;
	char magic[11] = {0};

	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't unmkfs. */
		rc = EBUSY;
		goto out;
	}

	fd = EngFncs->open_volume(volume, O_RDWR|O_EXCL, 0);
	if (fd < 0) {
		rc = EIO;
		goto out;
	}

	/* Erase the primary swapfs_super_block. */
	rc = fsim_rw_diskblocks(volume, fd,
				SWAP_MAGIC_OFFSET_IN_BYTES,
				10, magic, PUT);

	EngFncs->close_volume(volume, fd);
	volume->private_data = NULL;
	volume->flags &= ~VOLFLAG_NOT_MOUNTABLE;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * swap_expand
 *
 * Expand the swap filesystem on this volume to new_size. If the volume is
 * not expanded exactly to new_size, set new_size to the new_size of the volume.
 **/
static int swap_expand(logical_volume_t *volume,
		       sector_count_t *new_size)
{
	option_array_t options;
	int rc;

	LOG_ENTRY();

	/* We know that the swap space is not in use, so we
	 * can simply re-mkfs the volume. Cheap, but easy.
	 */
	options.count = 0;
	rc = swap_mkfs(volume, &options);
	if (!rc) {
		swap_get_fs_size(volume, new_size);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * swap_shrink
 *
 * Shrink the swap filesystem on this volume to new_size. If the volume is not
 * shrunk exactly to new_size, set new_size to the new_size of the filesystem.
 */
static int swap_shrink(logical_volume_t *volume,
		       sector_count_t requested_size,
		       sector_count_t *new_size)
{
	int rc;

	LOG_ENTRY();

	/* Exactly the same as expanding the filesystem. */
	rc = swap_expand(volume, new_size);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * swap_get_option_count
 *
 * Return the total number of supported options for the specified task.
 **/
static int swap_get_option_count(task_context_t *context)
{
	int count = -1;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_mkfs:
		count = SWAP_MKFS_OPTIONS_COUNT;
		break;
	case EVMS_Task_swapon:
		count = SWAP_ON_OPTIONS_COUNT;
		break;
	case EVMS_Task_swapoff:
	case EVMS_Task_Expand:
	case EVMS_Task_Shrink:
		count = 0;
		break;
	default:
		break;
	}

	LOG_EXIT_INT(count);
	return count;
}

/**
 * init_mkfs_acceptable_objects
 *
 * Initialize a mkfs task's acceptable-objects list by enumerating the volumes,
 * finding those that have no FSIM and are of the proper size and adding them
 * to the acceptable-objects list.
 **/
static int init_mkfs_acceptable_objects(task_context_t *context)
{
	list_anchor_t global_volumes;
	list_element_t iter;
	logical_volume_t *volume;
	int rc;

	LOG_ENTRY();

	rc = EngFncs->get_volume_list(NULL, NULL, 0, &global_volumes);
	if (rc) {
		goto out;
	}

	LIST_FOR_EACH(global_volumes, iter, volume) {
		if (volume->file_system_manager == NULL &&
		    ! EngFncs->is_mounted(volume->dev_node, NULL) &&
		    volume->vol_size > SWAPFS_MIN_SIZE) {
			EngFncs->insert_thing(context->acceptable_objects,
					      volume, INSERT_AFTER, NULL);
		}
	}
	EngFncs->destroy_list(global_volumes);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * swap_init_task
 *
 * Fill in the initial list of acceptable objects.  Fill in the minimum and
 * maximum nuber of objects that must/can be selected.  Set up all initial
 * values in the option_descriptors in the context record for the given
 * task.  Some fields in the option_descriptor may be dependent on a
 * selected object.  Leave such fields blank for now, and fill in during the
 * set_objects call.
 **/
static int swap_init_task(task_context_t *context)
{
	option_desc_array_t *od = context->option_descriptors;
	int i, rc = 0;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_mkfs:
		rc = init_mkfs_acceptable_objects(context);
		if (rc) {
			break;
		}

		od->count = SWAP_MKFS_OPTIONS_COUNT;

		/* Bad-block-check option */
		i = SWAP_MKFS_BADBLOCK_INDEX;
		od->option[i].name = EngFncs->engine_strdup("badblocks");
		od->option[i].title = EngFncs->engine_strdup(_("Check for bad blocks"));
		od->option[i].tip = EngFncs->engine_strdup(_("Check for bad blocks when making swap space"));
		od->option[i].type = EVMS_Type_Boolean;
		od->option[i].unit = EVMS_Unit_Kilobytes;
		od->option[i].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
		od->option[i].value.b = FALSE;

		context->min_selected_objects = 1;
		context->max_selected_objects = 1;
		break;

	case EVMS_Task_swapon:
		od->count = SWAP_ON_OPTIONS_COUNT;

		/*
		 * The "priority" option is simply a switch for
		 * enabling/disabling the "level" option.  The "priority" option
		 * doesn't correspond to a parameter setting.
		 */
		i = SWAP_ON_PRIORITY_INDEX;
		od->option[i].name = EngFncs->engine_strdup(SWAP_ON_PRIORITY_NAME);
		od->option[i].title = EngFncs->engine_strdup(_("Priority"));
		od->option[i].tip = EngFncs->engine_strdup(_("Swap priority for this device"));
		od->option[i].type = EVMS_Type_String;
		od->option[i].unit = EVMS_Unit_None;
		od->option[i].min_len = 1;
		od->option[i].constraint_type = EVMS_Collection_List;
		od->option[i].constraint.list = EngFncs->engine_alloc(sizeof(value_list_t) + 2 * sizeof(value_t));
		if (od->option[i].constraint.list != NULL) {

			/* Priority constraint list index 0, "low" is like
			 * FALSE, i.e., it's not high priority, which has a
			 * level.  Similarly, Priority constraint list index 1,
			 * "High" is like TRUE.  If the priority level s High
			 * the level option is made active.  Conversely, if the
			 * priority option is Low the level option is marked
			 * inactive.
			 */
			od->option[i].constraint.list->count = 2;
			od->option[i].constraint.list->value[0].s = EngFncs->engine_strdup(_("Low"));
			od->option[i].constraint.list->value[1].s = EngFncs->engine_strdup(_("High"));

			od->option[i].max_len = max(strlen(od->option[i].constraint.list->value[0].s),
						    strlen(od->option[i].constraint.list->value[1].s))
						+ 1;
		} else {
			LOG_CRITICAL("Unable to get memory for a range constraint.\n");
			rc = ENOMEM;
			break;
		}
		od->option[i].value.s = EngFncs->engine_alloc(max(strlen(od->option[i].constraint.list->value[0].s),
								  strlen(od->option[i].constraint.list->value[1].s)) + 1);
		if (od->option[i].value.s != NULL) {
			strcpy(od->option[i].value.s, od->option[i].constraint.list->value[0].s);
		}

		i = SWAP_ON_LEVEL_INDEX;
		od->option[i].name = EngFncs->engine_strdup(SWAP_ON_LEVEL_NAME);
		od->option[i].title = EngFncs->engine_strdup(_("Level"));
		od->option[i].tip = EngFncs->engine_strdup(_("Swap priority for this device"));
		od->option[i].type = EVMS_Type_Int;
		od->option[i].unit = EVMS_Unit_None;
		od->option[i].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
				      EVMS_OPTION_FLAGS_INACTIVE;
		od->option[i].constraint_type = EVMS_Collection_Range;
		od->option[i].constraint.range = EngFncs->engine_alloc(sizeof(value_range_t));
		if (od->option[i].constraint.range != NULL) {
			od->option[i].constraint.range->min.i = SWAP_PRIORITY_MIN;
			od->option[i].constraint.range->max.i = SWAP_PRIORITY_MAX;
			od->option[i].constraint.range->increment.i = 1;
		} else {
			LOG_CRITICAL("Unable to get memory for a range constraint.\n");
			rc = ENOMEM;
			break;
		}

		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		break;

	case EVMS_Task_swapoff:
	case EVMS_Task_Expand:
	case EVMS_Task_Shrink:
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * swap_set_option
 *
 * Examine the specified value, and determine if it is valid for the task
 * and option_descriptor index. If it is acceptable, set that value in the
 * appropriate entry in the option_descriptor.
 **/
static int swap_set_option(task_context_t *context,
			   u_int32_t index,
			   value_t *value,
			   task_effect_t *effect)
{
	int rc = 0;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_mkfs:
		switch (index) {
		case SWAP_MKFS_BADBLOCK_INDEX:
			context->option_descriptors->option[index].value.b = value->b;
			break;
		default:
			LOG_ERROR("%d is not a valid option index for the EVMS_Task_mkfs task.\n",
				  index);
			rc = EINVAL;
			break;
		}
		break;
	case EVMS_Task_swapon:
		switch (index) {
		case SWAP_ON_PRIORITY_INDEX:
                        /*
			 * The "priority" option is simply a switch for
			 * enabling/disabling the "level" option.  The
			 * "priority" option doesn't correspond to a
			 * parameter setting.
			 *
			 * Priority constraint list index 0, "low" is like
			 * FALSE, i.e., it's not high priority, which has a
			 * level.  Similarly, Priority constraint list index 0,
			 * "High" is like TRUE.  If the priority level s High
			 * the level option is made active.  Conversely, if the
			 * priority option is Low the level option is marked
			 * inactive.
			 */

			/* If priority option is "Low"...*/
			if (strcasecmp(value->s, context->option_descriptors->option[index].constraint.list->value[0].s) == 0) {
				strcpy(context->option_descriptors->option[index].value.s, value->s);
				context->option_descriptors->option[SWAP_ON_LEVEL_INDEX].flags |= EVMS_OPTION_FLAGS_INACTIVE;
				*effect |= EVMS_Effect_Reload_Options;

			/* If priority option is "High"...*/
			} else if (strcasecmp(value->s, context->option_descriptors->option[index].constraint.list->value[1].s) == 0) {
				strcpy(context->option_descriptors->option[index].value.s, value->s);
				context->option_descriptors->option[SWAP_ON_LEVEL_INDEX].flags &= ~EVMS_OPTION_FLAGS_INACTIVE;
				*effect |= EVMS_Effect_Reload_Options;
			} else {
				LOG_ERROR("%s is not a valid setting for the %s option.\n",
					  value->s, SWAP_ON_LEVEL_NAME);
			}
			break;
		case SWAP_ON_LEVEL_INDEX:
			if ((value->i >= SWAP_PRIORITY_MIN) &&
			    (value->i <= SWAP_PRIORITY_MAX)) {
				context->option_descriptors->option[index].value.i = value->i;

				/* If priority option is "Low"...*/
				if (strcasecmp(context->option_descriptors->option[SWAP_ON_PRIORITY_INDEX].value.s,
					       context->option_descriptors->option[SWAP_ON_PRIORITY_INDEX].constraint.list->value[0].s)) {
					
					/* Set priority option to "High". */
					strcpy(context->option_descriptors->option[SWAP_ON_PRIORITY_INDEX].value.s,
					       context->option_descriptors->option[SWAP_ON_PRIORITY_INDEX].constraint.list->value[1].s);
					*effect |= EVMS_Effect_Reload_Options;
				}

			} else {
				LOG_ERROR("%d is out of the range of acceptable priority values.  "
					  "The priority must be between %d and %d, inclusive.\n",
					  value->i, SWAP_PRIORITY_MIN, SWAP_PRIORITY_MAX);
				rc = EINVAL;
			}
			break;
		default:
			LOG_ERROR("%d is not a valid option index for the EVMS_Task_swapon task.\n",
				  index);
			rc = EINVAL;
			break;
		}
		break;

	default:
		LOG_ERROR("Task action %d (%#x) does not have any options.\n",
			  context->action, context->action);
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * swap_set_volumes
 *
 * Validate the volumes in the selected_objects list in the task context.
 **/
static int swap_set_volumes(task_context_t *context,
			    list_anchor_t declined_volumes,
			    task_effect_t *effect)
{
	logical_volume_t *vol;
	int rc = 0;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_mkfs:
		/* Get the selected volume */
		vol = EngFncs->first_thing(context->selected_objects, NULL);
		if (!vol) {
			rc = ENODATA;
			break;
		}

		if (EngFncs->is_mounted(vol->dev_node, NULL)) {
			rc = EBUSY;
			break;
		}

		if (vol->vol_size < SWAPFS_MIN_SIZE) {
			rc = ENOSPC;
			break;
		}

		break;

	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * swap_get_plugin_info
 *
 * Display info about the swap plugin.
 **/
static int swap_get_plugin_info(char *descriptor_name,
				extended_info_array_t **info)
{
	extended_info_array_t *Info;
	char engine_api_version[64];
	char fsim_api_version[64];
	char version[64];

	LOG_ENTRY();

	*info = NULL;	  // init to no info returned

	Info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     6 * sizeof(extended_info_t));
	if (!Info) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	sprintf(version, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
	sprintf(engine_api_version, "%d.%d.%d",
		my_plugin_record->required_engine_api_version.major,
		my_plugin_record->required_engine_api_version.minor,
		my_plugin_record->required_engine_api_version.patchlevel);
	sprintf(fsim_api_version, "%d.%d.%d",
		my_plugin_record->required_plugin_api_version.fsim.major,
		my_plugin_record->required_plugin_api_version.fsim.minor,
		my_plugin_record->required_plugin_api_version.fsim.patchlevel);

	Info->info[0].name = EngFncs->engine_strdup("Short Name");
	Info->info[0].title = EngFncs->engine_strdup(_("Short Name"));
	Info->info[0].desc = EngFncs->engine_strdup(_("A short name given to this plug-in"));
	Info->info[0].value.s = EngFncs->engine_strdup(my_plugin_record->short_name);
	Info->info[0].type = EVMS_Type_String;

	Info->info[1].name = EngFncs->engine_strdup("Long Name");
	Info->info[1].title = EngFncs->engine_strdup(_("Long Name"));
	Info->info[1].desc = EngFncs->engine_strdup(_("A longer, more descriptive name for this plug-in"));
	Info->info[1].value.s = EngFncs->engine_strdup(my_plugin_record->long_name);
	Info->info[1].type = EVMS_Type_String;

	Info->info[2].name = EngFncs->engine_strdup("Type");
	Info->info[2].title = EngFncs->engine_strdup(_("Plug-in Type"));
	Info->info[2].desc = EngFncs->engine_strdup(_("There are various types of plug-ins, each responsible for some kind of storage object or logical volume."));
	Info->info[2].value.s = EngFncs->engine_strdup(_("File System Interface Module"));
	Info->info[2].type = EVMS_Type_String;

	Info->info[3].name = EngFncs->engine_strdup("Version");
	Info->info[3].title = EngFncs->engine_strdup(_("Plugin Version"));
	Info->info[3].desc = EngFncs->engine_strdup(_("This is the version number of the plug-in."));
	Info->info[3].value.s = EngFncs->engine_strdup(version);
	Info->info[3].type = EVMS_Type_String;

	Info->info[4].name = EngFncs->engine_strdup("Required Engine Services Version");
	Info->info[4].title = EngFncs->engine_strdup(_("Required Engine Services Version"));
	Info->info[4].desc = EngFncs->engine_strdup(_("This is the version of the Engine services that this plug-in requires.  "
						      "It will not run on older versions of the Engine services."));
	Info->info[4].value.s = EngFncs->engine_strdup(engine_api_version);
	Info->info[4].type = EVMS_Type_String;

	Info->info[5].name = EngFncs->engine_strdup("Required Engine FSIM API Version");
	Info->info[5].title = EngFncs->engine_strdup(_("Required Engine FSIM API Version"));
	Info->info[5].desc = EngFncs->engine_strdup(_("This is the version of the Engine FSIM API that this plug-in requires.  "
						      "It will not run on older versions of the Engine FSIM API."));
	Info->info[5].value.s = EngFncs->engine_strdup(fsim_api_version);
	Info->info[5].type = EVMS_Type_String;

	Info->count = 6;
	*info = Info;

	LOG_EXIT_INT(0);
	return 0;
}


/*
 * Return an array of plug-in functions that you support for this
 * volume.
 */
static int swap_get_plugin_functions(logical_volume_t        * volume,
				     function_info_array_t * * actions) {

	function_info_array_t * fia;

	LOG_ENTRY();

	/*
	 * "volume" is NULL when the Engine is asking for functions that
	 * apply to the plug-in rather than to a particular volume.
	 */
	if (volume == NULL) {
		LOG_DEBUG("There are no functions targeted at this plug-in.\n");
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (volume->file_system_manager != my_plugin_record) {
		/* It's not my volume. */
		LOG_DEBUG("Volume %s is not a swap volume.\n", volume->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	fia = EngFncs->engine_alloc(sizeof(function_info_array_t) + sizeof(function_info_t));
	if (fia == NULL) {
		LOG_CRITICAL("Unable to get memory for a function_info_array_t.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	if (EngFncs->is_mounted(volume->name, NULL)) {
		fia->info[0].function = EVMS_Task_swapoff;
		fia->info[0].name = EngFncs->engine_strdup("swapoff");
		fia->info[0].title = EngFncs->engine_strdup("Swap off");
		fia->info[0].verb = EngFncs->engine_strdup("Swap off");
		fia->info[0].help = EngFncs->engine_strdup(_("Turn off swapping on this volume."));

	} else {
		fia->info[0].function = EVMS_Task_swapon;
		fia->info[0].name = EngFncs->engine_strdup("swapon");
		fia->info[0].title = EngFncs->engine_strdup("Swap on");
		fia->info[0].verb = EngFncs->engine_strdup("Swap on");
		fia->info[0].help = EngFncs->engine_strdup(_("Turn on swapping on this volume."));
	}

	fia->count = 1;
	*actions = fia;

	LOG_EXIT_INT(0);
	return 0;
}


/*
 * Execute the plug-in function on the volume.
 */
static int swap_plugin_function(logical_volume_t * volume,
				task_action_t      action,
				list_anchor_t      objects,
				option_array_t   * options) {

	int status;
	int rc = 0;
	int i;
	int priority = 0;

	LOG_ENTRY();

	if (volume == NULL) {
		LOG_ERROR("No volume specified.\n");
		rc = EINVAL;
	}

	switch (action) {
	case EVMS_Task_swapon:
		for (i = 0; i < options->count; i++) {
			if (!options->option[i].is_number_based) {
				if (!strcmp(options->option[i].name, SWAP_ON_PRIORITY_NAME)) {
					options->option[i].number = SWAP_ON_PRIORITY_INDEX;
				} else if (!strcmp(options->option[i].name, SWAP_ON_LEVEL_NAME)) {
					options->option[i].number = SWAP_ON_LEVEL_INDEX;
				}
			}

			switch (options->option[i].number) {
			case SWAP_ON_PRIORITY_INDEX:
				/*
				 * The "priority" option is simply a switch for
				 * enabling/disabling the "level" option.  The
				 * "priority" option doesn't correspond to a
				 * parameter setting.
				 */
				break;

			case SWAP_ON_LEVEL_INDEX:
				if ((options->option[i].value.i >= SWAP_PRIORITY_MIN) &&
				    (options->option[i].value.i <= SWAP_PRIORITY_MAX)) {
					priority = (options->option[i].value.i << SWAP_FLAG_PRIO_SHIFT) & SWAP_FLAG_PRIO_MASK;
					priority |= SWAP_FLAG_PREFER;
				}
				break;

			default:
				if (options->option[i].is_number_based) {
					LOG_ERROR("Option index %d is not valid.  The option is ignored.\n",
						  options->option[i].number);
				} else {
					LOG_ERROR("Option name \"%s\" is not valid.  The option is ignored.\n",
						  options->option[i].name);
				}
				break;
			}
		}

		LOG_DEBUG("swapon(%s, %#x)\n", volume->dev_node, priority);
		status = swapon(volume->dev_node, priority);
		if (status != 0) {
			LOG_WARNING("swapon(%s, %#x) failed with error code %d: %s\n",
				    volume->dev_node, priority, errno, strerror(errno));
			rc = errno;
		}
		break;

	case EVMS_Task_swapoff:
		LOG_DEBUG("swapoff(%s)\n", volume->dev_node);
		status = swapoff(volume->dev_node);
		if (status != 0) {
			LOG_WARNING("swapoff(%s) failed with error code %d: %s\n",
				    volume->dev_node, errno, strerror(errno));
			rc = errno;
		}
		break;

	default:
		LOG_ERROR("Plug-in function %d (%#x) is not supported.\n",
			  action, action);
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(0);
	return 0;
}


static fsim_functions_t swap_ops = {
	.setup_evms_plugin	= swap_setup,
	.probe			= swap_probe,
	.get_fs_size		= swap_get_fs_size,
	.get_fs_limits		= swap_get_fs_limits,
	.can_mkfs		= swap_can_mkfs,
	.can_unmkfs		= swap_can_unmkfs,
	.can_expand_by		= swap_can_expand_by,
	.can_shrink_by		= swap_can_shrink_by,
	.mkfs			= swap_mkfs,
	.discard		= swap_discard,
	.unmkfs			= swap_unmkfs,
	.expand			= swap_expand,
	.shrink			= swap_shrink,
	.get_option_count	= swap_get_option_count,
	.init_task		= swap_init_task,
	.set_option		= swap_set_option,
	.set_volumes		= swap_set_volumes,
	.get_plugin_info	= swap_get_plugin_info,
	.get_plugin_functions	= swap_get_plugin_functions,
	.plugin_function	= swap_plugin_function
};

plugin_record_t swap_plugin_record = {
	.id = EVMS_SWAP_PLUGIN_ID,
	.version = {
		.major		= MAJOR_VERSION,
		.minor		= MINOR_VERSION,
		.patchlevel	= PATCH_LEVEL
	},
	.required_engine_api_version = {
		.major		= 15,
		.minor		= 0,
		.patchlevel	= 0
	},
	.required_plugin_api_version = {
		.fsim = {
			.major		= 11,
			.minor		= 0,
			.patchlevel	= 0
		}
	},
	.short_name = EVMS_SWAP_PLUGIN_SHORT_NAME,
	.long_name = EVMS_SWAP_PLUGIN_LONG_NAME,
	.oem_name = EVMS_IBM_OEM_NAME,
	.functions = {
		.fsim = &swap_ops
	},
};

plugin_record_t *evms_plugin_records[] = {
	&swap_plugin_record,
	NULL
};

