/*
 *
 *    Copyright 1999-2001 Sistina Software, Inc.
 *    Portions Copyright 2001 The OpenGFS Project
 *
 *    This is free software released under the GNU General Public License.
 *    There is no warranty for this software.  See the file COPYING for
 *    details.
 *
 *    See the file AUTHORS for a list of contributors.
 *
 */
#ifndef _STOMITH_H
#define _STOMITH_H

#include "global.h"

/*
 *	Magic numbers for specific STOMITH methods
 */
enum {
	STM_BRCD_ZONE = 0xDEAD0101,
	STM_BRCD_PORT = 0xDEAD0102,
	STM_WTI_NPS = 0xDEAD0201,
	STM_VACM_EMP = 0xDEAD0401,
	STM_GNBD = 0xDEAD0501,
	STM_MEATWARE = 0xDEAD0601,
	STM_MANUAL = 0xDEAD0602,
	STM_VXHUB_BYPASS = 0xDEAD0701,
	STM_APC_MS = 0xDEAD0801,
};

/*
 *	Match magic numbers with tags used in gfsconf param file.
 */
struct stomith_names {
	char		*name;
	uint32_t	number;
};

#if defined(WANT_STOMITH_NAMES)
const struct stomith_names stomith_names[] = {
	{"brocade_zone",	STM_BRCD_ZONE},
	{"brocade_port",	STM_BRCD_PORT},
	{"wti_nps",		STM_WTI_NPS},
	{"vacm_emp",		STM_VACM_EMP},
	{"gnbd_gclient",	STM_GNBD},
	{"meatware",		STM_MEATWARE},
	{"manual",		STM_MANUAL},
	{"vxhub_bypass",	STM_VXHUB_BYPASS},
	{"apc_ms",		STM_APC_MS},
	{"",			0x0}
};
#else
extern const struct stomith_names stomith_names[];
#endif /* WANT_STOMITH_NAMES */

/* number of seconds the stomith module will wait for stomith daemon */
#define STMD_TIMEOUT 10


/*
 *	Structures for each stomith type
 */

/* maximum length of login names, password, etc. (multiple of 4 bytes) */
#define STM_STRING_LEN 20


/*
 * Hard FC zoning via Brocade switch, model: Silkworm 2400 and 2800 with
 * zoning capability.
 *
 * Two approaches to zoning:
 *
 *  1. All machines are in one common zone and one common configuration.
 *     Each machine has its own alias.  In this case, "st_command" is 'zoneremove',
 *     "st_zonecfg" is the name assigned to the common configuration, "st_comzone"
 *     is the name given to the common zone, and "st_actname" is each machine's
 *     alias.  The zoning proceedure goes as follows:
 *       > zoneremove <comzone>, <actname>
 *       > cfgenable <zonecfg>
 *
 *  2. Each machine is placed in its own zone.  All the zones are placed into
 *     a common configuration.  Machine's aliases are not used.  In this case,
 *     "st_command" is 'cfgremove', "st_zonecfg" is the name assigned to the 
 *     common zone, "st_comzone" is not used, and "st_actname" is the name of
 *     each machine's zone.  The zoning proceedure goes as follows:
 *       > cfgremove <cfgname>, <actname>
 *	 > cfgenable <zonecfg>
 */
struct stm_brcd_zone {
	uint32_t  st_ipaddr;	/* IP address of switch */
	uint32_t  st_kernel_code;	/* Use kernel stomith code */
	char st_login[STM_STRING_LEN];	/* Switch login name */
	char st_passwd[STM_STRING_LEN];	/* Switch login password */
	char st_command[STM_STRING_LEN];	/* Command eg zoneremove */
	char st_cfgname[STM_STRING_LEN];	/* Configuration name */
	char st_comzone[STM_STRING_LEN];	/* Common zone name */

	char st_actname[STM_STRING_LEN];	/* Alias or per-node zone */
};

/* portdisable on a Brocade switch */
struct stm_brcd_port {
	uint32_t 	st_ipaddr;	/* IP address of switch */
	uint32_t	st_kernel_code;	/* Use kernel stomith code */
	char 	st_login[STM_STRING_LEN];	/* Switch login name */
	char 	st_passwd[STM_STRING_LEN];	/* Switch login password */

	int st_port;		/* Port number */
};


/* power cycle via WTI Network Power Switch (NPS), model: 115, 230 */
struct stm_wti_nps {
	uint32_t  st_ipaddr;	/* IP address of switch */
	uint32_t  st_kernel_code;	/* Use kernel stomith code */
	char st_passwd[STM_STRING_LEN];	/* Login password */

	unsigned int st_plug;		/* Plug number */
};

/* power cycle with VACM EMP */
struct stm_vacm_emp {
	char st_nexxus[STM_STRING_LEN];
	char st_login[STM_STRING_LEN];
	char st_passwd[STM_STRING_LEN];

	char st_nodeid[STM_STRING_LEN];
};


/* GNBD server ignores stomithed client */
struct stm_gnbd {
	uint32_t st_ipaddr;
	int st_pad;
};

/* humanoid resets machine and then runs program */
struct stm_meatware {
	uint32_t st_ipaddr;
	int st_pad;
};

/* turn off the port of the victim node on the Vixel hub */
struct stm_vx_hub {
	uint32_t st_ipaddr;	/* IP address of the Vixel 2100 hub */
	int st_port;		/* Port number - From 1 to 8 */
	char st_passwd[STM_STRING_LEN];	/* Login password */
};


/* power cycle with APC MasterSwitch */
struct stm_apc_ms {
	uint32_t st_ipaddr;	/* IP address of the masterswitch */
	int st_outlet;	/* Outlet number */
	char st_login[STM_STRING_LEN];	/* Login name */
	char st_passwd[STM_STRING_LEN];	/* Login password */
};

/* general stomith structure */
struct cluster_stomith {
	uint32_t cs_magic;	/* Magic number */
	char cs_name[STM_STRING_LEN];
	union {
		struct stm_brcd_zone brcd_zone_s;
		struct stm_brcd_port brcd_port_s;
		struct stm_wti_nps wti_nps_s;
		struct stm_vacm_emp vacm_emp_s;
		struct stm_gnbd gnbd_s;
		struct stm_meatware meatware_s;
		struct stm_vx_hub vx_hub_s;
		struct stm_apc_ms apc_ms_s;
	} cs_st;
};

#endif /* _STOMITH_H */
