/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *
 * Module: crc.c
 *
 * Functions:
 *
 *              Build_CRC_table()
 *              CalculateCRC()
 *
 *
 * Description: The functions in this module provide a means of calculating
 *              the 32 bit CRC for a block of data.  Build_CRC_table must
 *              be called to initialize this module.  CalculateCRC must
 *              NOT be used until after Build_CRC_table has been called.
 *              Once Build_CRC_table has been called, CalculateCRC can
 *              be used to calculate the CRC of the data residing in a
 *              user specified buffer.
 *
 */

#include <stdlib.h>
#include <common.h>

#include "fullengine.h"
#include "engine.h"
#include "crc.h"


/*--------------------------------------------------
 * Private Constants
 --------------------------------------------------*/
#define CRC_POLYNOMIAL     0xEDB88320L


/*--------------------------------------------------
 * Private Global Variables.
 --------------------------------------------------*/
static u_int32_t  CRC_table[ 256 ];

static boolean CRC_table_Built=FALSE;  /* TRUE=table is built, FALSE=table not built yet */

/*********************************************************************/
/*                                                                   */
/*   Function Name: build_CRC_table                                  */
/*                                                                   */
/*   Descriptive Name: This module implements the CRC function using */
/*                     a table driven method.  The required table    */
/*                     must be setup before the CalculateCRC function*/
/*                     can be used.  This table only needs to be set */
/*                     up once.  This function sets up the CRC table */
/*                     needed by CalculateCRC.                       */
/*                                                                   */
/*   Input: None                                                     */
/*                                                                   */
/*   Output: None                                                    */
/*                                                                   */
/*   Error Handling: N/A                                             */
/*                                                                   */
/*   Side Effects:  The internal CRC Table is initialized.           */
/*                                                                   */
/*   Notes:  None.                                                   */
/*                                                                   */
/*********************************************************************/
static void  build_CRC_table( void ) {
	u_int32_t   i, j, CRC;

	LOG_PROC_ENTRY();

	for (i = 0; i <= 255 ; i++) {

		CRC = i;

		for (j = 8 ; j > 0; j--) {

			if (CRC & 1)
				CRC = ( CRC >> 1 ) ^ CRC_POLYNOMIAL;
			else
				CRC >>= 1;
		}

		CRC_table[ i ] = CRC;

	}

	CRC_table_Built = TRUE;

	LOG_PROC_EXIT_VOID();
}


/*********************************************************************/
/*                                                                   */
/*   Function Name: evms_calculate_crc                               */
/*                                                                   */
/*   Descriptive Name: This function calculates the CRC value for    */
/*                     the data in the buffer specified by Buffer.   */
/*                                                                   */
/*   Input: CARDINAL32   CRC : This is the starting CRC.  If you are */
/*                             starting a new CRC calculation, then  */
/*                             this should be set to 0xFFFFFFFFL.  If*/
/*                             you are continuing a CRC calculation  */
/*                             (i.e. all of the data did not fit in  */
/*                             the buffer so you could not calculate */
/*                             the CRC in a single operation), then  */
/*                             this is the CRC output by the last    */
/*                             CalculateCRC call.                    */
/*                                                                   */
/*   Output: The CRC for the data in the buffer, based upon the value*/
/*           of the input parameter CRC.                             */
/*                                                                   */
/*   Error Handling: None.                                           */
/*                                                                   */
/*   Side Effects:  None.                                            */
/*                                                                   */
/*   Notes:  None.                                                   */
/*                                                                   */
/*********************************************************************/
u_int32_t evms_calculate_crc( u_int32_t CRC, void * Buffer, u_int32_t buffer_size) {
	u_char  * current_byte;
	u_int32_t temp1, temp2, I;

	LOG_PROC_ENTRY();

	current_byte = ( u_char *) Buffer;

	/* Make sure the CRC table is available */
	if (CRC_table_Built==FALSE)  build_CRC_table();

	/* Process each byte in the buffer. */
	for (I = 0; I < buffer_size; I++) {

		temp1 = (CRC >> 8) & 0x00FFFFFFL;
		temp2 = CRC_table[ ( CRC ^ (u_int32_t) *current_byte ) & (u_int32_t) 0xff ];
		current_byte++;
		CRC = temp1 ^ temp2;
	}

	LOG_PROC_EXIT_UINT(CRC);
	return( CRC );
}


