/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * 	Module: frag.c
 */
#include <ece.h>
#include "eceinternal.h"


/* used for message assembly */
typedef struct frag_s {
	unsigned char 	*seq_bmap;
	char 	*datastr;
	uint	frag_size;
	uint	tot_size;
} frag_t;
static frag_t *fragment=NULL;

/* used for sending message. The messages are kept in a linked list
 * for the mainloop thread to pick them up and send them
 * Since the IPC library which sends the messages is not thread safe
 * we are forced to send all messages in the context of a single thread
 */
typedef struct frag_msg_s {
	pthread_cond_t  fr_cond; /* condition to co-ordinate btw threads*/
	int  		fr_ret;  /* the return value */
	const ece_msg_t *fr_ecemsg; /* the 1st parameter of frag_send() */
	gboolean  	fr_mcast; /* the 1st parameter of frag_send() */
	const char	*fr_node; /* the 2nd parameter of frag_send() */
	u_int32_t 	fr_corr; /* the 3rd parameter of frag_send() */
	u_int32_t 	fr_crc; /* the 4th parameter of frag_send() */
	u_int32_t 	fr_ackval; /* the 5th parameter of frag_send() */
}  frag_msg_t;
static GSList 			*frag_list=NULL;
static gboolean 		frag_list_in_use=FALSE; /* guard it by lock */
static pthread_cond_t  		frag_cond =PTHREAD_COND_INITIALIZER;
static pthread_mutex_t 		frag_mutex = PTHREAD_MUTEX_INITIALIZER;


/* BEGIN OF FUNCTION THAT HANDLES AGGREGATION OF MESSAGE FRAGMENTS */
static unsigned char *
bmap_alloc(const unsigned int no_bits)
{
	unsigned int no_bytes = ((no_bits-1)/CHAR_BIT)+1;
	unsigned char *bitmap = (unsigned char *)
		g_malloc0(no_bytes*sizeof(char));
	LOG_ENTRY();
	LOG_EXIT_PTR(bitmap);
	return bitmap;
}

static void
bmap_set(unsigned char *bmp, const unsigned int pos)
{
	unsigned int byteno = (pos)/CHAR_BIT;
	unsigned int bitno = pos%CHAR_BIT;
	LOG_ENTRY();
	bmp[byteno] |= 1<<bitno;
	LOG_EXIT_VOID();
}

static gboolean
bmap_all_set(const unsigned char *bmp, const unsigned int no_bits)
{
	unsigned int num,shifts,i;
	unsigned int no_bytes = ((no_bits-1)/CHAR_BIT);
	gboolean ret;
	LOG_ENTRY();

	for (i=0; i< no_bytes; i++) {
		if(bmp[i]!=UCHAR_MAX) {
			LOG_EXIT_BOOL(FALSE);
			return FALSE;
		}
	}
	shifts = (no_bits-1)%CHAR_BIT+1;
	num = ((1<<shifts)-1);
	ret = ((bmp[no_bytes]&num)==num);
	LOG_EXIT_BOOL(ret);
	return ret;
}

static inline void
bmap_free(char *bmp)
{
	LOG_ENTRY();
	g_free(bmp);
	LOG_EXIT_VOID();
}


void
frag_init(int n_nodes)
{
	LOG_ENTRY();
	fragment = (frag_t *)g_malloc0(n_nodes*sizeof(frag_t));
	LOG_EXIT_VOID();
}

void
frag_clean(const char* node)
{
	int indx;
	LOG_ENTRY();
	if(!fragment) {
		LOG_EXIT_VOID();
		return;
	}
	indx = llm_get_idx(node);
	g_free(fragment[indx].datastr);
	bmap_free(fragment[indx].seq_bmap);
	fragment[indx].datastr = NULL;
	fragment[indx].seq_bmap = NULL;
	LOG_EXIT_VOID();
}

void
frag_cleanup(void)
{
	uint i, n;

	LOG_ENTRY();
	if(!fragment) {
		LOG_EXIT_VOID();
		return;
	}
	n = sizeof(fragment)/sizeof(frag_t);
	for ( i = 0 ; i < n ; i++ ) {
		g_free(fragment[i].datastr);
		bmap_free(fragment[i].seq_bmap);
	}
	g_free(fragment);
	fragment=NULL;

	frag_list=NULL;
	frag_list_in_use=FALSE;
	pthread_mutex_init(&frag_mutex,NULL);
	pthread_cond_init(&frag_cond,NULL);

	LOG_EXIT_VOID();
}
	
int
frag_assemble(const struct ha_msg *h_msg, char **data_str)
{
	const char *d_str, *cntl_str, *seqno_str, *orig;
	int indx, seqno, tlen, flen, no_frag, tmp;
	frag_t *fr=NULL;


	LOG_ENTRY();
	if(!fragment) {
		LOG_WARNING("Recevied message before initialization.\n");
		LOG_EXIT_BOOL(FALSE);
		return FALSE;
	}

	ECE_ASSERT(d_str = ha_msg_value(h_msg, ECE_DATA));
	ECE_ASSERT(cntl_str = ha_msg_value(h_msg, ECE_CNTL));
	ECE_ASSERT(seqno_str = ha_msg_value(h_msg, ECE_SEQNO));
	ECE_ASSERT(orig = ha_msg_value(h_msg, F_ORIG));

	indx = llm_get_idx(orig);
	fr = fragment+indx;

	seqno = atoi(seqno_str);
	sscanf(cntl_str,"%d,%d,%d,%d,%d,%d",
		&tmp,/*dont' care */
		&tmp,/*dont' care */
		&flen,
		&tlen,
		&tmp,/*dont' care */
		&tmp/*dont' care */);



	if(fr->datastr) {
		ECE_ASSERT(flen == fr->frag_size);
		ECE_ASSERT(tlen == fr->tot_size);
		ECE_ASSERT(fr->seq_bmap);
	} else {
		fr->tot_size = tlen;
		fr->frag_size = flen;
		fr->seq_bmap = bmap_alloc(((tlen-1)/flen)+1);
		fr->datastr = (char *)g_malloc0(tlen+1);
	}
	memcpy(fr->datastr+seqno*flen, d_str, strlen(d_str));
	bmap_set(fr->seq_bmap, seqno);

	no_frag = (tlen-1)/flen+1;
	if(bmap_all_set(fr->seq_bmap,no_frag)){
		*data_str = fr->datastr;
		memset(fr,0,sizeof(frag_t));
		bmap_free(fr->seq_bmap);
		fr->seq_bmap = NULL;
		LOG_EXIT_BOOL(TRUE);
		return TRUE;
	}
	LOG_EXIT_BOOL(FALSE);
	return FALSE;
}



static int
frag_send_msg(const ece_msg_t *ecemsg,
		const gboolean mcast,
		const char *node,
		const u_int32_t corr,
		const u_int32_t crc,
		const u_int32_t ackval)
{
	gchar *cor_str;
	gchar *cmd_str;
	gchar *data_str, *dataptr;
	gchar *seq_str;
	gchar *cntl_str;
	uint strsize, fragsize, cpsize, thisfrag;
	int  ret=0;
        struct ha_msg *h_msg;
	char tmpchar;

	LOG_ENTRY();

	cor_str = g_strdup_printf("%d", corr);
	cmd_str = g_strdup_printf("%d", ecemsg->cmd);

	/* convert message to str */
	strsize = B64_stringspace(ecemsg->size);
	data_str = (gchar *)g_malloc0(strsize+1);
	binary_to_base64(ecemsg->msg, ecemsg->size, data_str,
			strsize);

	fragsize = MAXMSG/2;
	cntl_str = g_strdup_printf("%d,%d,%d,%d,%d,%d",
			ackval,crc,fragsize,
			strsize,ece_get_mode(),mcast);

	seq_str = g_malloc(4); /* A string of three characters will hold an ASCII
	                          number up to 999.  Will there be more than
				  1000 fragments?? */

	cpsize = 0;
	while(cpsize < strsize) {

		if ((h_msg=ha_msg_new(0)) == NULL) {
			LOG_CRITICAL("Error: Internal resource allocation problem.  Try again.\n");
			ret = EAGAIN;
			goto end;
		}
		thisfrag = (cpsize+fragsize < strsize)?
			fragsize : strsize-cpsize;
		dataptr = data_str+cpsize;
		tmpchar = data_str[cpsize+thisfrag];
		data_str[cpsize+thisfrag] = '\0';
		g_snprintf(seq_str, 4, "%d", (cpsize/fragsize));
		if ((ha_msg_add(h_msg, F_TYPE, ECE_MSG) == HA_FAIL)
			||(ha_msg_add(h_msg, ECE_CNTL, cntl_str) == HA_FAIL)
			||(ha_msg_add(h_msg, ECE_SEQNO, seq_str) == HA_FAIL)
			||(ha_msg_add(h_msg, ECE_CMD, cmd_str) == HA_FAIL)
			||(ha_msg_add(h_msg, ECE_CORRELATOR, cor_str)==HA_FAIL)
			||(ha_msg_add(h_msg, ECE_DATA, dataptr) == HA_FAIL)) {
				LOG_CRITICAL("Cannot allocate to send a message.\n");
				ha_msg_del(h_msg);
				ret = EAGAIN;
				goto end;
		}

		switch(ece_get_mode()) {
			const char *str;
		case MASTER:
			if (mcast) {
				hb_send_to_cluster(h_msg);
			} else {
				if(strcmp(node, llm_getmynodeid())==0){
					/* send this message to the peer */
					if(ha_msg_add(h_msg, F_ORIG, node)
							==HA_FAIL) {
						LOG_CRITICAL("Cannot add a field to send a message.\n");
						ha_msg_del(h_msg);
						ret = EAGAIN;
						goto end;
					}
					str = msg2string(h_msg);
					msg_track(FALSE, h_msg, NULL, FALSE, DLTS);
					ret = peer_send((void *)str,
						strlen(str)+1, ece_get_peer());
					if(ret) {
						ha_msg_del(h_msg);
						LOG_SERIOUS("Cannot send the message. The slave is probably dead.\n");
						goto end;
					}
				} else {
					msg_track(FALSE, h_msg, NULL, FALSE, DLTR);
					hb_send_to_node(h_msg, node);
				}
			}
			break;
		case SLAVE:
			if (!mcast) {
				if(ha_msg_add(h_msg, ECEi_TOWHOM, node)
						== HA_FAIL) {
					ha_msg_del(h_msg);
					LOG_CRITICAL("Cannot create a message.\n");
					ret = EAGAIN;
					goto end;
				}
			}
			/* send this message to the master */
			msg_track(FALSE, h_msg, NULL, FALSE, STL);
			str = msg2string(h_msg);
			ret = peer_send((void *)str, strlen(str)+1, ece_get_peer());
			break;
		default:
			break;
		}

		data_str[cpsize+thisfrag] = tmpchar;
		cpsize += thisfrag;
		ha_msg_del(h_msg);
	}

end:
	g_free(cor_str);
	g_free(cmd_str);
	g_free(data_str);
	g_free(seq_str);
	g_free(cntl_str);

	LOG_EXIT_INT(ret);
	return ret;
}

int
frag_send_msg_wait(const ece_msg_t *ecemsg,
		const gboolean mcast,
		const char *node,
		const u_int32_t corr,
		const u_int32_t crc,
		const u_int32_t ackval)
{
	int ret;
	frag_msg_t *fr_msg;

	LOG_ENTRY();

	fr_msg = (frag_msg_t *)g_malloc(sizeof(frag_msg_t));
	fr_msg->fr_ecemsg = ecemsg;
	fr_msg->fr_mcast  = mcast;
	fr_msg->fr_node   = node;
	fr_msg->fr_corr   = corr;
	fr_msg->fr_crc    = crc;
	fr_msg->fr_ackval = ackval;
	fr_msg->fr_ret    = ETIMEDOUT;
	pthread_cond_init(&(fr_msg->fr_cond),NULL);

   	pthread_mutex_lock(&frag_mutex);
	while(frag_list_in_use) {
		pthread_cond_wait(&frag_cond, &frag_mutex);
	}
  	frag_list = g_slist_append(frag_list, (void *)fr_msg);
	pthread_cond_wait(&(fr_msg->fr_cond), &frag_mutex);
	ret = fr_msg->fr_ret;
   	pthread_mutex_unlock(&frag_mutex);

	pthread_cond_destroy(&(fr_msg->fr_cond));
	g_free(fr_msg);
	LOG_EXIT_INT(ret);
	return ret;
}


void
frag_send_check()
{
	frag_msg_t *fr_msg;

	LOG_EXTRA_ENTRY();
   	pthread_mutex_lock(&frag_mutex);
	if(!frag_list) {
   		pthread_mutex_unlock(&frag_mutex);
		LOG_EXTRA_EXIT_VOID();
		return;
	}
	frag_list_in_use = TRUE;
   	pthread_mutex_unlock(&frag_mutex);

	while(frag_list) {
		fr_msg = g_slist_nth_data(frag_list, 0);
		frag_list = g_slist_remove(frag_list, fr_msg);
		/* check if the node still exists or if the daemon
		 * on the node still exists
		 */
		if(!llm_is_present(fr_msg->fr_node) ||
		 	!llm_get_ece_status(fr_msg->fr_node)) {
        		 LOG_ERROR("Error: %s not a member\n",
				 fr_msg->fr_node);
			fr_msg->fr_ret = ENODEV;
		} else {
			fr_msg->fr_ret = frag_send_msg(
				fr_msg->fr_ecemsg,
				fr_msg->fr_mcast,
				fr_msg->fr_node,
				fr_msg->fr_corr,
				fr_msg->fr_crc,
				fr_msg->fr_ackval);
		}
		pthread_cond_signal(&(fr_msg->fr_cond));
	}

   	pthread_mutex_lock(&frag_mutex);
	frag_list_in_use = FALSE;
	pthread_cond_broadcast(&frag_cond);
   	pthread_mutex_unlock(&frag_mutex);
	LOG_EXTRA_EXIT_VOID();
}
