/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdos.so
 *
 *   File: embedded.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "ptables.h"
#include "segs.h"
#include "embedded.h"
#include "bsd.h"
#include "solarisX86.h"
#include "unixware.h"


/*
 *  Called to allocate a segment storage object
 */
DISKSEG *  alloc_diskseg_object( LOGICALDISK *ld )
{
        int      rc;
        DISKSEG *seg=NULL;


        LOG_ENTRY();

        rc = EngFncs->allocate_segment( NULL, &seg );
        if (rc==0) {

                if (rc == 0) {

                        seg->plugin      = Seg_My_PluginRecord_Ptr;
                        seg->object_type = SEGMENT;

                        memcpy(&seg->geometry, &ld->geometry, sizeof(geometry_t) );

                        seg->private_data = calloc(1, sizeof(SEG_PRIVATE_DATA));
                        if (seg->private_data) {

                                ((SEG_PRIVATE_DATA *)seg->private_data)->signature = DOS_SEG_MGR_PDATA_SIGNATURE;
                                ((SEG_PRIVATE_DATA *)seg->private_data)->logical_disk = ld;

                        }
                        else {
                                EngFncs->free_segment( seg );
                                seg = NULL;
                        }
                }
        }

        LOG_EXIT_PTR(seg);
        return seg;
}


/*
 * Builds a disk segment matching the embedded partition info
 */
DISKSEG * build_segment_for_embedded_partition( LOGICALDISK       *ld,
                                                DISKSEG           *msdos_seg,
                                                u_int32_t          start,
                                                u_int32_t          size,
                                                u_int32_t          sys_id,
                                                u_int32_t          ptable_index,
                                                u_int32_t          minor )
{
        DISKSEG           *seg        = NULL;
        SEG_PRIVATE_DATA  *pdata      = NULL;
        int                rc;
        

        LOG_ENTRY();

        seg = alloc_diskseg_object( ld );
        if (seg) {

                if (EngFncs->insert_thing(seg->child_objects, msdos_seg, INSERT_BEFORE, NULL)) {
                        rc = 0;
                }
                else {
                        rc = EPERM;
                }

                if (rc == 0) {

                        pdata = (SEG_PRIVATE_DATA *) seg->private_data;

                        seg->size              = size;
                        seg->start             = start;
                        seg->data_type         = DATA_TYPE;

                        pdata->sys_id          = sys_id;
                        pdata->part_number     = minor;
                        pdata->ptable_index    = ptable_index;

                }
                else {
                        free(seg->private_data);
                        EngFncs->free_segment( seg );
                        seg = NULL;
                }


        }

        LOG_EXIT_PTR(seg);
        return seg;
}



/*
 *  Returns DISKSEG matching query information
 */
DISKSEG * get_matching_segment( list_anchor_t seglist, lba_t start, sector_count_t size )
{
        DISKSEG          *seg;
        list_element_t    iter;

        LIST_FOR_EACH( seglist, iter, seg ){
                if ( (seg->start == start) && (seg->size == size) ) return seg;
        }

        return NULL;
}


/*
 *  Called to remove the embedded partitions, found in the recovery_list, from
 *  the specified logical disk.  This routine is called when embedded partition
 *  discovery has failed for a certain primary partition we we need to recover
 *  the state of the drive prior to when embedded partition discovery failed.
 */
int  remove_embedded_partitions_from_disk( LOGICALDISK *ld, list_anchor_t recovery_list )
{
        DISKSEG          *seg;
        int               rc=0;
        list_element_t    iter, iter2;


        LOG_ENTRY();

        LIST_FOR_EACH_SAFE( recovery_list, iter, iter2, seg ) {
                LOG_DEBUG("removing %s\n", seg->name );
                free_disk_segment( seg );
                EngFncs->delete_element(iter);
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to look for an embedded partition scheme on a drive, examine
 *  the disk label, and return the geometry we discover.
 */
int read_embedded_geometry( LOGICALDISK *ld, geometry_t *geometry )
{
        struct bsd_disklabel       *bl;
        struct unixware_disklabel  *ul;
        char                        data[EVMS_VSECTOR_SIZE];
        Master_Boot_Record          mbr[EVMS_VSECTOR_SIZE];
        int                         rc = ENOSYS;
        struct plugin_functions_s  *dft;
        int                         i;
        Partition_Record           *part;
        boolean                     success=FALSE;

        LOG_ENTRY();


        // get disk manager function table
        dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
        if (dft==NULL) {
                LOG_EXIT_INT(EINVAL);
                return EINVAL;
        }

        // read mbr sector
        rc = dft->read( ld, 0, 1, (void *) &mbr );
        if (rc) {
                LOG_EXIT_INT(rc);
                return rc;
        }

        // look for an embedded partition in the mbr
        for (i=0; (i < 4) && (success == FALSE); i++) {

                part = &mbr->Partition_Table[i];

                switch ( SYS_IND(part) ) {
                
                case BSD_PARTITION:
                case NETBSD_PARTITION:
                case OPENBSD_PARTITION:

                        // read bsd info
                        rc = dft->read( ld,
                                        DISK_TO_CPU32( START_LBA(part) ) + BSD_DISKLABEL_PART_TABLE_SECTOR_OFFSET,
                                        1,
                                        (void *) &data );

                        if (rc) {
                                LOG_EXIT_INT(rc);
                                return rc;
                        }

                        // check for BSD magic number
                        bl = (struct bsd_disklabel *) &data;

                        LOG_DEBUG("found bsd partition \n");

                        if ( DISK_TO_CPU32(bl->d_magic) == BSD_DISKMAGIC) {
                                geometry->cylinders         = DISK_TO_CPU32(bl->d_ncylinders);
                                geometry->heads             = DISK_TO_CPU32(bl->d_ntracks);
                                geometry->sectors_per_track = DISK_TO_CPU32(bl->d_nsectors);

                                success = TRUE;
                        }

                        break;


                case UNIXWARE_PARTITION:

                        // read UnixWare info
                        rc = dft->read( ld,
                                        DISK_TO_CPU32( START_LBA(part) ) + UNIXWARE_PART_TABLE_SECTOR_OFFSET,
                                        1,
                                        (void *) &data );

                        if (rc) {
                                LOG_EXIT_INT(rc);
                                return rc;
                        }

                        // check unixware magic number
                        ul = (struct unixware_disklabel *) &data;

                        LOG_DEBUG("found unixware partition\n");

                        if ( ( DISK_TO_CPU32(ul->d_magic)      == UNIXWARE_DISKMAGIC ) &&
                             ( DISK_TO_CPU32(ul->vtoc.v_magic) == UNIXWARE_DISKMAGIC2 )) {
                                geometry->cylinders         = DISK_TO_CPU32(ul->d_ncylinders);
                                geometry->heads             = DISK_TO_CPU32(ul->d_ntracks);
                                geometry->sectors_per_track = DISK_TO_CPU32(ul->d_nsectors);

                                success = TRUE;
                        }

                        break;

                case SOLARIS_X86_PARTITION:

                        // not implemented yet
                        break;


                default:
                        break;
                }

        }


        if (success == TRUE) {
                LOG_DEBUG("success:  C= %"PRIu64"  H= %d  S= %d\n", geometry->cylinders, geometry->heads, geometry->sectors_per_track);
                rc = 0;
        }
        else {
                LOG_DEBUG("failure\n");
                if (rc == 0) rc = EINVAL;
        }

        LOG_EXIT_INT(rc);
        return rc;
}
