/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: config.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>

#include "fullengine.h"
#include "config.h"
#include "engine.h"
#include "memman.h"
#include "message.h"

					
#define DEFAULT_CONFIG_FILE_NAME	"/etc/evms.conf"

static char * config_file_name = NULL;
static int line_num;

typedef enum {
	VALUE,
	ARRAY,
	SECTION
} kv_type_t;

typedef struct value_array_s {
	int    count;
	char * * strings;
	union {
		u_int32_t * u32;
		u_int64_t * u64;
	} values;
} value_array_t;


typedef struct key_value_s {
	struct key_value_s * next;
	char * key;
	union {
		char * string;
		value_array_t * array;
		struct key_value_s * section;
	} value;
	kv_type_t type;
} key_value_t;


/* Forward references */
static key_value_t * parse_value(char * * pp, char * key);
static void free_kv(key_value_t * kv);

/*
 * The hash table size should be prime number.  A prime number promotes
 * a more even distribution of entries in the hash table.
 * Other "prime" candidates that are near a power of 10 or power of 2:
 * 7, 11, 13, 17, 31, 37, 61, 67, 101, 103, 127, 257, 499, 503, 509, 521,
 * 997, 1009, 1021, 1031
 */
#define HASH_TABLE_SIZE 127

#define HASH_INDEX(string) (hash_value(string) % HASH_TABLE_SIZE)

static key_value_t * * hash_table = NULL;

/*
 * Initialize the hash_table if it hasn't been started already.
 */
static int initialize_hash_table(void) {

	int rc = 0;

	LOG_PROC_ENTRY();

	/* Has the cache already been initialized? */
	if (hash_table == NULL) {
		hash_table = (key_value_t * *) engine_alloc(HASH_TABLE_SIZE * sizeof(key_value_t *));

		if (hash_table == NULL) {
			LOG_CRITICAL("Error allocating memory for the hash table.\n");
			rc = ENOMEM;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Hash value computer.  Modeled after ElfHash().
 */
static u_int32_t hash_value(char * string) {

	u_int32_t h = 0;
	u_int32_t g;
	int i;
	u_char * pByte = string;

	LOG_PROC_EXTRA_ENTRY();

	for (i = 0; pByte[i] != '\0'; i++) {
		h = (h << 4) + pByte[i];
		g = h & 0xF0000000;

		if (g != 0) {
			h ^= g >> 24;
		}

		h &= ~g;
	}

	LOG_PROC_EXTRA_EXIT_UINT(h);
	return h;
}


#define NEXT_CHAR(p) (((*(p) == '\\') && (*((p)+1) != '\0')) ? (p)+=2 : (p)++)

#define IS_WHITE_SPACE(c) (((c) == ' ') || ((c) == '\t') || ((c) == '\n') || ((c) == '\r'))

#define IS_KEY_VALUE_DELIMITER(c) (((c) == '=') || ((c) == ':'))


static char * skip_white_space(char * p, char * extra_chars) {

	char c;

	LOG_PROC_ENTRY();

	c = *p;

	if (extra_chars != NULL) {
		while ((c != '\0') &&
		       ((strchr(extra_chars, c) != NULL) ||
			IS_WHITE_SPACE(c) ||
			(c == '#'))) {

			/* Comments are white space. */
			if (c == '#') {
				while (c != '\n') {
					NEXT_CHAR(p);
					c = *p;
				}
			}

			if (c == '\n') {
				line_num++;
			}

			NEXT_CHAR(p);
			c = *p;
		}

	} else {
		while ((c != '\0') &&
		       (IS_WHITE_SPACE(c) || (c == '#'))) {

			/* Comments are white space. */
			if (c == '#') {
				while (c != '\n') {
					NEXT_CHAR(p);
					c = *p;
				}
			}

			if (c == '\n') {
				line_num++;
			}

			NEXT_CHAR(p);
			c = *p;
		}
	}

	LOG_PROC_EXIT_PTR(p);
	return p;
}


/* Convert a two digit ASCII hex string to binary. */
static int hex_byte(char * p) {

	int i;
	int value = 0;

	LOG_PROC_ENTRY();

	for (i = 0; i < 2; i++) {
		value <<= 8;
		if ((*p >= '0') && (*p <= '9')) {
			value += *p - '0';
		} else if ((*p >= 'a') && (*p <= 'f')) {
			value += (*p - 'a') + 10;
		} else if ((*p >= 'A') && (*p <= 'F')) {
			value += (*p - 'A') + 10;
		} else {
			LOG_ERROR("Parse error on line %d in file %s.  %c is not a hexadecimal digit.\n",
				  line_num, config_file_name, *p);
			return 0;
		}
	}

	LOG_PROC_EXIT_INT(value);
	return value;
}


/* Convert a three digit ASCII octal string to binary. */
static int oct_byte(char * p) {

	int i;
	int value = 0;

	LOG_PROC_ENTRY();

	for (i = 0; i < 3; i++) {
		value <<= 8;
		if ((*p >= '0') && (*p <= '7')) {
			value += *p - '0';
		} else {
			LOG_ERROR("Parse error on line %d in file %s.  %c is not an octal digit.\n",
				  line_num, config_file_name, *p);
			return 0;
		}
	}

	LOG_PROC_EXIT_INT(value);
	return value;
}


static void compress_escapes(char * string) {

	LOG_PROC_ENTRY();

	while (*string != '\0') {
		if (*string == '\\') {
			char * p;
			char * p2;

			p = string;
			p2 = p + 2;
			/* Handle special escape characters. */
			switch (*(p+1)) {
			case 'n':
				*p = '\n';
				break;
			case 'r':
				*p = '\r';
				break;
			case 'b':
				*p = '\b';
				break;
			case 't':
				*p = '\t';
				break;
			case 'f':
				*p = '\f';
				break;
			case 'x':
				*p = hex_byte(p+1);
				p2 = p + 4;
				break;
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
				*p = oct_byte(p+1);
				p2 = p + 4;
				break;
			default:
				*p = *(p+1);
			}

			for (p++; *p2 != '\0'; p++, p2++) {
				*p = *p2;
			}
		}

		string++;
	}

	LOG_PROC_EXIT_VOID();
}


static inline void no_value_msg(char * key) {
	engine_user_message(NULL, NULL,
			    _("Parse error on line %d in file %s.  Key \"%s\" does not have a value assigned to it.\n"),
			    line_num, config_file_name, key);
}


static inline void bad_value_msg(char * key) {
	engine_user_message(NULL, NULL,
			    _("Parse error on line %d in file %s.  %s not a valid value.\n"),
			    line_num, config_file_name, key);
}


static char * find_string_end(char * p, char * delimiters) {

	LOG_PROC_ENTRY();

	while (*p != '\0') {

		if (IS_WHITE_SPACE(*p)) {
			break;

		}

		if (delimiters != NULL) {
			char * p2 = delimiters;

			while ((*p2 != '\0') && (*p != *p2)) {
				p2++;
			}

			if (*p2 != '\0') {
				/* Found a delimiter. */
				break;
			}
		}

		switch (*p) {
		case '\'':
			p++;
			while (*p != '\'') {
				/* Skip over escaped characters. */
				if (*p == '\\') {
					/* Except null. */
					if (*(p+1) != '\0') {
						p++;
					}
				}
				p++;
			}

			if (*p == '\'') {
				/* Skip over the closing '\''. */
				p++;

			} else {
				engine_user_message(NULL, NULL,
						    _("Parse error on line %d in file %s.  String does not have a closing single quote.\n"),
						    line_num, config_file_name);
			}
			break;

		case '\"':
			p++;
			while ((*p != '\0') && (*p != '\"')) {
				/* Skip over escaped characters. */
				if (*p == '\\') {
					/* Except null. */
					if (*(p+1) != '\0') {
						p++;
					}
				}
				p++;
			}

			if (*p == '\"') {
				/* Skip over the closing '\"'. */
				p++;

			} else {
				engine_user_message(NULL, NULL,
						    _("Parse error on line %d in file %s.  String does not have a closing double quote.\n"),
						    line_num, config_file_name);
			}
			break;

		case '\\':
			/* Skip over escaped character. */
			if (*p == '\\') {
				/* Except nul. */
				if (*(p+1) != '\0') {
					p++;
				}
			}
			p++;
			break;

		default:
			p++;
		}

	}

	LOG_PROC_EXIT_PTR(p);
	return p;
}


static char * parse_key(char * * pp) {

	char * p = *pp;
	char * key = NULL;

	LOG_PROC_ENTRY();

	p = skip_white_space(p, NULL);

	if (*p != '\0') {
		key = p;

		/* Skip over the key. */
		p = find_string_end(p, "=:");

		/* Terminate the key. */
		if (*p != '\0') {
			*p = '\0';
			p++;
		}

		compress_escapes(key);
	}

	*pp = p;

	LOG_PROC_EXIT_PTR(key);
	return key;
}


static char * get_string(char * p) {

	char * p2 = find_string_end(p, NULL);
	int str_len = p2 - p;
	char * string = engine_alloc(str_len + 1);
	boolean finished;

	LOG_PROC_ENTRY();

	if (string != NULL) {
		memcpy(string, p, str_len);
		string[str_len] = '\0';

		/* Strip off any outside quote marks. */
		p = string;
		finished = FALSE;
		while ((*p != '\0') && ! finished) {
			switch (*p) {
			case '\'':
				/*
				 * Remove the opening '\'' by copying the string
				 * over it.
				 */
				for (p2 = p; *p2 != '\0'; p2++) {
					*p2 = *(p2+1);
				}

				/* Search for the closing '\''. */
				while ((*p != '\0') && (*p != '\'')) {
					/*
					 * NEXT_CHAR() skips over escaped
					 * characters properly.
					 */
					NEXT_CHAR(p);
				}

				/*
				 * Remove the closing '\'' by copying the string
				 * over it.
				 */
				for (p2 = p; *p2 != '\0'; p2++) {
					*p2 = *(p2+1);
				}
				break;

			case '\"':
				/*
				 * Remove the opening '\"' by copying the string
				 * over it.
				 */
				for (p2 = p; *p2 != '\0'; p2++) {
					*p2 = *(p2+1);
				}

				/* Search for the closing '\"'. */
				while ((*p != '\0') && (*p != '\"')) {
					/*
					 * NEXT_CHAR() skips over escaped
					 * characters properly.
					 */
					NEXT_CHAR(p);
				}

				/* Remove the closing '\"' by copying the string over it. */
				for (p2 = p; *p2 != '\0'; p2++) {
					*p2 = *(p2+1);
				}
				break;

			default:
				finished = TRUE;
			}
		}

		compress_escapes(string);

	} else {
		LOG_CRITICAL("Error allocating memory for a value string.\n");
	}

	LOG_PROC_EXIT_PTR(string);
	return string;
}


static key_value_t * parse_section(char * * pp, char * key) {

	char * p = *pp;
	key_value_t * kv = NULL;
	key_value_t * kv_child = NULL;

	LOG_PROC_ENTRY();

	kv = engine_alloc(sizeof(key_value_t));

	if (kv != NULL) {
		kv->key = engine_strdup(key);
		kv->type = SECTION;

		/* Skip over the '{'.*/
		p++;
		p = skip_white_space(p, NULL);

		while ((*p != '\0') && (*p != '}')) {
			key = parse_key(&p);

			if (key != NULL) {

				if (*p != '\0') {
					p = skip_white_space(p, "=:");

					if (*p != '\0') {
						kv_child = parse_value(&p, key);

						if (kv_child != NULL) {
							kv_child->next = kv->value.section;
							kv->value.section = kv_child;
						}

					} else {
						no_value_msg(key);
					}

				} else {
					no_value_msg(key);
				}
			}

			p = skip_white_space(p, NULL);
		}

		if (*p != '\0') {
			/* Skip over the '}'.*/
			p++;

		} else {
			engine_user_message(NULL, NULL,
					    _("Parse error on line %d in file %s.  Section does not have a closing curly brace (}).\n"),
					    line_num, config_file_name);
		}

	} else {
		LOG_CRITICAL("Error allocating memory for a key_value_t structure.\n");
	}

	*pp = p;

	LOG_PROC_EXIT_PTR(kv);
	return kv;
}


static key_value_t * parse_array(char * * pp, char * key) {

	int array_count = 8;
	char * p = *pp;
	char * p2;
	char c;
	key_value_t * kv = NULL;

	LOG_PROC_ENTRY();

	kv = engine_alloc(sizeof(key_value_t));

	if (kv != NULL) {
		kv->key = engine_strdup(key);
		kv->type = ARRAY;

		kv->value.array = engine_alloc(sizeof(value_array_t));

		if (kv->value.array != NULL) {
			kv->value.array->strings = engine_alloc(sizeof(char *) * array_count);

			if (kv->value.array->strings != NULL) {

				/* Skip over the '['.*/
				p++;
				p = skip_white_space(p, NULL);

				while ((*p != '\0') && (*p != ']')) {

					/*
					 * An array entry can itself contain
					 * brackets.
					 */
					int bracket_count = 0;

					p2 = find_string_end(p, "[],");
					while ((*p2 == '[') || (bracket_count != 0)) {
						switch (*p2) {
						case '[':
							bracket_count ++;
							p2++;
							break;

						case ']':
							bracket_count--;
							p2++;
							break;

						default:
							/*
							 * Hit a ',' or white
							 * space.  We are at the
							 * end of the entry.
							 */
							bracket_count = 0;
						}

						if (bracket_count != 0) {
							p2 = find_string_end(p2, "[],");
						}
					}

					c = *p2;
					*p2 = '\0';

					if (kv->value.array->count >= array_count) {
						/* Need more space in the array. */
						array_count += 8;
						kv->value.array->strings = engine_realloc(kv->value.array->strings, (sizeof(char *) * array_count));

						if (kv->value.array->strings == NULL) {
							LOG_CRITICAL("Error allocating memory for an array of strings.\n");
							free_kv(kv);
							kv = NULL;
							break;
						}
					}

					kv->value.array->strings[kv->value.array->count] = get_string(p);

					if (kv->value.array->strings[kv->value.array->count] != NULL) {
						kv->value.array->count++;

					} else {
						LOG_CRITICAL("Error making a copy of string %s.\n", p);
						break;
					}

					*p2 = c;

					p = skip_white_space(p2, ",");
				}

				if ((kv != NULL) && (kv->value.array->count < array_count)) {
					/*
					 * NULL terminate the string array for
					 * safety.
					 */
					kv->value.array->strings[kv->value.array->count] = NULL;
				}

				if (*p == ']') {
					/* Skip over the ']'.*/
					p++;

				} else {
					engine_user_message(NULL ,NULL,
							    _("Parse error on line %d in file %s.  Section does not have a closing square bracket (]).\n"),
							    line_num, config_file_name);
				}

			} else {
				LOG_CRITICAL("Error allocating memory for a value_array_t structure.\n");
				engine_free(kv);
				kv = NULL;
			}

		} else {
			LOG_CRITICAL("Error allocating memory for a value_array_t structure.\n");
			engine_free(kv);
			kv = NULL;
		}


	} else {
		LOG_CRITICAL("Error allocating memory for a key_value_t structure.\n");
	}

	*pp = p;

	LOG_PROC_EXIT_PTR(kv);
	return kv;
}


static key_value_t * parse_string(char * * pp, char * key) {

	key_value_t * kv = NULL;

	LOG_PROC_ENTRY();

	kv = engine_alloc(sizeof(key_value_t));

	if (kv != NULL) {
		kv->key = engine_strdup(key);
		kv->type = VALUE;
		kv->value.string = get_string(*pp);

		*pp = find_string_end(*pp, NULL);

	} else {
		LOG_CRITICAL("Error allocating memory for a key_value_t structure.\n");
	}

	LOG_PROC_EXIT_PTR(kv);
	return kv;
}


static key_value_t * parse_value(char * * pp, char * key) {

	key_value_t * kv = NULL;

	LOG_PROC_ENTRY();

	/* *(*p)) is guaranteed to point to a non-nul character. */
	switch (*(*pp)) {
	case '{':
		kv = parse_section(pp, key);
		break;

	case '[':
		kv = parse_array(pp, key);
		break;

	case '}':
	case ']':
		engine_user_message(NULL, NULL,
				    _("Parse error on line %d in file %s.  Unexpected '%c'.\n"),
				    line_num, config_file_name, *(*pp));
		break;

	default:
		kv = parse_string(pp, key);
	}

	LOG_PROC_EXIT_PTR(kv);
	return kv;
}


static void parse_config(char * file) {

	char * key;
	key_value_t * kv;

	LOG_PROC_ENTRY();

	line_num = 1;

	while (*file != '\0') {

		key = parse_key(&file);

		if (key != NULL) {

			if (*file != '\0') {
				file = skip_white_space(file, "=:");

				if (*file != '\0') {
					kv = parse_value(&file, key);

					if (kv != NULL) {
						int hash_index = HASH_INDEX(key);

						kv->next = hash_table[hash_index];
						hash_table[hash_index] = kv;
					}

				} else {
					no_value_msg(key);
				}

			} else {
				no_value_msg(key);
			}
		}
	}

	LOG_PROC_EXIT_VOID();
	return;
}


static int read_config(char * * buffer) {

	int rc;
	int config_fd;
	struct stat stat_buf;
	char * buf = NULL;

	LOG_PROC_ENTRY();

	rc = stat(config_file_name, &stat_buf);
	if (rc == 0) {
		config_fd = open(config_file_name, O_RDONLY);

		if (config_fd > 0) {
			/* Close this file for all exec'd processes. */
			fcntl(config_fd, F_SETFD, FD_CLOEXEC);

			buf = engine_alloc(stat_buf.st_size + 1);
			if (buf != NULL) {
				rc = read(config_fd, buf, stat_buf.st_size);
				if (rc == stat_buf.st_size) {
					buf[stat_buf.st_size] = '\0';
					rc = 0;

				} else {
					LOG_SERIOUS("Error reading the configuration file %s.\n", config_file_name);
					engine_free(buf);
					buf = NULL;
					rc = EIO;
				}

			} else {
				LOG_CRITICAL("Error getting memory for reading in the configuration file %s.\n", config_file_name);
				rc = ENOMEM;
			}

		} else {
			rc = errno;
			LOG_DEFAULT("Error opening %s: %s\n", strerror(rc), config_file_name);
		}

	} else {
		rc = errno;
		LOG_DEFAULT("stat() of %s returned error %d: %s\n", config_file_name, rc, strerror(rc));
	}

	*buffer = buf;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Free a key:value entry.
 */
static void free_kv(key_value_t * kv) {

	int i;
	key_value_t * kv2;

	LOG_PROC_ENTRY();

	engine_free(kv->key);

	switch (kv->type) {
	case VALUE:
		engine_free(kv->value.string);
		break;

	case SECTION:
		kv2 = kv->value.section;
		while (kv2 != NULL) {
			key_value_t * next_kv = kv2->next;
			free_kv(kv2);
			kv2 = next_kv;
		}
		break;

	case ARRAY:
		for (i = 0; i < kv->value.array->count; i++) {
			engine_free(kv->value.array->strings[i]);
		}
		engine_free(kv->value.array->strings);
		if (kv->value.array->values.u32 != NULL) {
			engine_free(kv->value.array->values.u32);
		}
		engine_free(kv->value.array);
		break;
	}

	engine_free(kv);

	LOG_PROC_EXIT_VOID();
}


/*
 * Free the key:value tree.
 */
void evms_free_config(void) {

	int i;
	key_value_t * kv;

	LOG_PROC_ENTRY();

	if (hash_table != NULL) {
		for (i = 0; i < HASH_TABLE_SIZE; i++) {
			kv = hash_table[i];
			while (kv != NULL) {
				key_value_t * next_kv = kv->next;
				free_kv(kv);
				kv = next_kv;
			}
		}

		engine_free(hash_table);
		hash_table = NULL;
	}

	engine_free(config_file_name);
	config_file_name = NULL;

	LOG_PROC_EXIT_VOID();
}


/*
 * Read the configuration file and parse its contents into a key:value tree.
 */
int evms_get_config(char * file_name) {

	int rc = 0;
	char * config;

	LOG_PROC_ENTRY();

	if (file_name == NULL) {
		file_name = DEFAULT_CONFIG_FILE_NAME;
	}

	if (hash_table != NULL) {
		if (!(strcmp(config_file_name, DEFAULT_CONFIG_FILE_NAME) == 0)) {
			engine_user_message(NULL,NULL,
					    _("Unable to get the configuration from file %s.  "
					      "The configuration has already been read from file %s.\n"),
					    file_name, config_file_name);

			LOG_PROC_EXIT_INT(EINVAL);
			return EINVAL;
		}
	}

	config_file_name = engine_strdup(file_name);
	if (config_file_name == NULL) {
		LOG_CRITICAL("Error getting memory to copy the config file name.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	rc = initialize_hash_table();

	if (rc == 0) {
		rc = read_config(&config);

		if (rc == 0) {
			parse_config(config);

			engine_free(config);

		} else {
			engine_free(hash_table);
			hash_table = NULL;
		}
	}

	if (rc != 0) {
		engine_free(config_file_name);
		config_file_name = NULL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static void convert_to_array(key_value_t * kv) {


	value_array_t * va = engine_alloc(sizeof(value_array_t));

	LOG_PROC_ENTRY();

	if (va != NULL) {
		va->strings = engine_alloc(sizeof(char *));

		if (va->strings != NULL) {
			va->strings[0] = kv->value.string;
			va->count = 1;
			kv->value.array = va;
			kv->type = ARRAY;

		} else {
			LOG_CRITICAL("Error allocating memory for an array of strings.\n");
			engine_free(va);
		}

	} else {
		LOG_CRITICAL("Error allocating memory for a value_array_t structure.\n");
	}

	LOG_PROC_EXIT_VOID();
}


static int lookup_section_key(key_value_t * parent_kv, char * element, key_value_t * * p_kv) {

	int rc = 0;
	key_value_t * kv;
	char * sub_section;

	LOG_PROC_ENTRY();

	/* Check for embedded sections. */
	sub_section = strchr(element, '.');

	if (sub_section != NULL) {
		*sub_section = '\0';
	}

	kv = parent_kv->value.section;
	while ((kv != NULL) && (strcmp(kv->key, element) != 0)) {
		kv = kv->next;
	}

	if (kv != NULL) {
		if ((sub_section != NULL) && (kv->type == SECTION)) {
			rc = lookup_section_key(kv, sub_section + 1, &kv);

		} else {
			if ((sub_section != NULL) || (kv->type == SECTION)) {
				/*
				 * Either the caller specified a section which
				 * doesn't exist, or the caller did not specify
				 * a section when there is one.
				 */
				rc = EINVAL;
				kv = NULL;
			}
		}

	} else {
		rc = ENOENT;
	}

	*p_kv = kv;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int lookup_key(char * lookup_key, key_value_t * * p_kv) {

	int rc = 0;
	char * key;
	int hash_index;
	key_value_t * kv;
	char * sub_section;

	LOG_PROC_ENTRY();

	if (hash_table == NULL) {
		rc = evms_get_config(NULL);

		if (rc != 0) {
			LOG_PROC_EXIT_INT(rc);
			return rc;
		}
	}

	/* Make a copy of the key so that we can hack it up if necessary. */
	key = engine_strdup(lookup_key);
	if (key == NULL) {
		LOG_CRITICAL("Error making a copy of the key.\n");

		LOG_PROC_EXIT_INT(ENOENT);
		return ENOENT;
	}

	sub_section = strchr(key, '.');

	if (sub_section != NULL) {
		*sub_section = '\0';
	}

	hash_index = HASH_INDEX(key);

	kv = hash_table[hash_index];
	while ((kv != NULL) && (strcmp(kv->key, key) != 0)) {
		kv = kv->next;
	}

	if (kv != NULL) {
		if ((sub_section != NULL) && (kv->type == SECTION)) {
			rc = lookup_section_key(kv, sub_section + 1, &kv);

		} else {
			if ((sub_section != NULL) || (kv->type == SECTION)) {
				/*
				 * Either the caller specified a section which
				 * doesn't exist, or the caller did not specify
				 * a section when there is one.
				 */
				rc = EINVAL;
				kv = NULL;
			}
		}

	} else {
		rc = ENOENT;
	}

	*p_kv = kv;

	engine_free(key);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_get_config_bool(char * key, boolean * p_value) {

	int rc;
	key_value_t * kv;

	LOG_PROC_ENTRY();

	rc = lookup_key(key, &kv);

	if (rc == 0) {
		if (kv->type == VALUE) {
			if ((strcmp(kv->value.string, "1") == 0) ||
			    (strcasecmp(kv->value.string, "on") == 0) ||
			    (strcasecmp(kv->value.string, "yes") == 0) ||
			    (strcasecmp(kv->value.string, "true") == 0)) {

				*p_value = TRUE;

			} else if ((strcmp(kv->value.string, "0") == 0) ||
				   (strcasecmp(kv->value.string, "off") == 0) ||
				   (strcasecmp(kv->value.string, "no") == 0) ||
				   (strcasecmp(kv->value.string, "false") == 0)) {

				*p_value = FALSE;

			} else {
				/* It's none of the boolean key words. */
				rc = EINVAL;
			}

		} else {
			/* Value is a section or an array. */
			rc = EINVAL;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_get_config_uint32(char * key, u_int32_t * value) {

	int rc;
	key_value_t * kv;
	char * end_ptr;
	u_int64_t num = 0;

	LOG_PROC_ENTRY();

	rc = lookup_key(key, &kv);

	if (rc == 0) {
		if (kv->type == VALUE) {
			num = strtoull(kv->value.string, &end_ptr, 0);
		} else if ((kv->type == ARRAY) && (kv->value.array->count == 1)) {
			num = strtoull(kv->value.array->strings[0], &end_ptr, 0);
		} else {
			/*
			 * The value is either a section or an array with
			 * multiple entries.
			 */
			rc = EINVAL;
		}

		if (rc == 0) {
			if (*end_ptr == '\0') {
				/*
				 * Check for out of bounds error.
				 * errno will be ERANGE if error.
				 */
				if (num == -1) {
					rc = errno;
				}

				/*
				 * Check if the number is outside the bounds
				 * of a 32-bit value.
				 */
				if (rc == 0) {
					if (num >= 0x100000000ULL) {
						rc = ERANGE;
					}
				}

				if (rc == 0) {
					*value = num;
				}

			} else {
				/*
				 * Some invalid character was found.
				 * It's not a number.
				 */
				rc = EINVAL;
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_get_config_uint32_array(char * key, int * p_count, const u_int32_t * * p_array) {

	int rc;
	int i;
	int count;
	u_int64_t num;
	char * end_ptr;
	key_value_t * kv;

	LOG_PROC_ENTRY();

	rc = lookup_key(key, &kv);

	if (rc == 0) {

		if (kv->type == VALUE) {
			convert_to_array(kv);
		}

		if (kv->type == ARRAY) {
			count = kv->value.array->count;

			if (kv->value.array->values.u64 != NULL) {
				engine_free(kv->value.array->values.u64);
			}

			kv->value.array->values.u32 = engine_alloc(sizeof(u_int32_t) * count);

			if (kv->value.array->values.u32 != NULL) {

				for (i = 0; (rc == 0) && (i < count); i++) {

					num = strtoull(kv->value.array->strings[i], &end_ptr, 0);
					if (*end_ptr == '\0') {
						/*
						 * Check for out of bounds error.
						 * errno will be ERANGE if error.
						 */
						if (num == -1) {
							rc = errno;
						}

						/*
						 * Check if the number is
						 * outside the bounds of a
						 * 32-bit value.
						 */
						if (rc == 0) {
							if (num >= 0x100000000ULL) {
								rc = ERANGE;
							}
						}

						if (rc == 0) {
							kv->value.array->values.u32[i] = num;
						}

					} else {
						/*
						 * Some invalid character was
						 * found.  It's not a number.
						 */
						rc = EINVAL;
					}
				}

				if (rc == 0) {
					*p_count = count;
					*p_array = kv->value.array->values.u32;
				}

			} else {
				LOG_CRITICAL("Error getting memory for reading in the configuration file %s.\n", config_file_name);
				rc = ENOMEM;
			}

		} else {
			/* Value is not an array. */
			rc = EINVAL;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_get_config_uint64(char * key, u_int64_t * value) {

	int rc;
	key_value_t * kv;
	char * end_ptr;
	u_int64_t num = 0;

	LOG_PROC_ENTRY();

	rc = lookup_key(key, &kv);

	if (rc == 0) {
		if (kv->type == VALUE) {
			num = strtoull(kv->value.string, &end_ptr, 0);
		} else if ((kv->type == ARRAY) && (kv->value.array->count == 1)) {
			num = strtoull(kv->value.array->strings[0], &end_ptr, 0);
		} else {
			/*
			 * The value is either a section or an array
			 * with multiple entries.
			 */
			rc = EINVAL;
		}

		if (rc == 0) {
			if (*end_ptr == '\0') {
				/*
				 * Check for out of bounds error.
				 * errno will be ERANGE if error.
				 */
				if (num == -1) {
					rc = errno;
				}

				if (rc == 0) {
					*value = num;
				}

			} else {
				/*
				 * Some invalid character was found.
				 * It's not a number.
				 */
				rc = EINVAL;
			}

		} else {
			rc = EINVAL;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_get_config_uint64_array(char * key, int * p_count, const u_int64_t * * p_array) {

	int rc;
	int i;
	int count;
	u_int64_t num;
	char * end_ptr;
	key_value_t * kv;

	LOG_PROC_ENTRY();

	rc = lookup_key(key, &kv);

	if (rc == 0) {

		if (kv->type == VALUE) {
			convert_to_array(kv);
		}

		if (kv->type == ARRAY) {
			count = kv->value.array->count;

			if (kv->value.array->values.u64 != NULL) {
				engine_free(kv->value.array->values.u64);
			}

			kv->value.array->values.u64 = engine_alloc(sizeof(u_int64_t) * count);

			if (kv->value.array->values.u64 != NULL) {

				for (i = 0; (rc == 0) && (i < count); i++) {

					num = strtoull(kv->value.array->strings[i], &end_ptr, 0);
					if (*end_ptr == '\0') {
						/*
						 * check for out of bounds error.
						 * errno will be ERANGE if error.
						 */
						if (num == -1) {
							rc = errno;
						}

						if (rc == 0) {
							kv->value.array->values.u64[i] = num;
						}
					} else {
						/*
						 * Some invalid character was
						 * found.  It's not a number.
						 */
						rc = EINVAL;
					}
				}

				if (rc == 0) {
					*p_count = count;
					*p_array = kv->value.array->values.u64;
				}

			} else {
				LOG_CRITICAL("Error getting memory for reading in the configuration file %s.\n", config_file_name);
				rc = ENOMEM;
			}

		} else {
			/* Value is not an array. */
			rc = EINVAL;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_get_config_string(char * key, const char * * value) {

	int rc;
	key_value_t * kv;

	LOG_PROC_ENTRY();

	rc = lookup_key(key, &kv);

	if (rc == 0) {
		if (kv->type == VALUE) {
			*value = kv->value.string;
		} else if ((kv->type == ARRAY) && (kv->value.array->count == 1)) {
			*value = kv->value.array->strings[0];
		} else {
			/*
			 * The value is either a section or an array
			 * with multiple entries.
			 */
			rc = EINVAL;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int evms_get_config_string_array(char * key, int * p_count, const char * const * * p_array) {

	int rc;
	key_value_t * kv;

	LOG_PROC_ENTRY();

	rc = lookup_key(key, &kv);

	if (rc == 0) {

		if (kv->type == VALUE) {
			convert_to_array(kv);
		}

		if (kv->type == ARRAY) {
			*p_count = kv->value.array->count;
			*p_array = (const char * const *) kv->value.array->strings;
		} else {
			rc = EINVAL;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}
