/*
 *
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: evmsd.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <libgen.h>
#include <string.h>
#include <semaphore.h>

#include <frontend.h>


static char * prog_name;				
static debug_level_t debug_level = -1;
static char * log_file = NULL;
static boolean quiet = FALSE;
static boolean verbose = FALSE;


static boolean is_numeric(char * str) {

	while (*str != '\0') {
		if ((*str < '0') || (*str > '9')) {
			return FALSE;
		}
		str++;
	}

	return TRUE;
}


static void show_help(void) {
	printf("The EVMS daemon.\n\n");
	printf("Usage: %s [options]\n", prog_name);
	printf("  [-d debug_level, --debug-level debug_level]\n"
	       "     debug_level = [0-9] | \n"
	       "                   [[critical | serious | error | warning | default |\n"
	       "                     details | entry_exit | debug | extra | everything]]\n"
	       "  [-l log-file-name]\n"
	       "  [-q] quiet mode\n"
	       "  [-v] verbose mode\n"
	       "  [-h | -? | --help]\n"
	       "     display this help\n");
}


static int parse_options(int argc, char * * argv) {

	int    rc = 0;
	int    c;
	char * short_opts = "d:l:qvh?";
	struct option long_opts[] = {
		{ "debug-level",  required_argument, NULL, 'd'},
		{ "log-file",     required_argument, NULL, 'l'},
		{ "help",         no_argument,       NULL, 'h'},
		{ NULL,           0,                 NULL,   0}
	};

	while ((c = getopt_long(argc, argv,
				short_opts, long_opts, NULL)) != EOF) {
		switch (c) {
		case 'd':
			if (is_numeric(optarg)) {
				int level = atoi(optarg);

				if ((level < CRITICAL) || (level > EVERYTHING)) {
					printf("%s is not a valid debug level.\n", optarg);
					/* Display the help. */
					rc = 1;
				}
			}

			if (strcasecmp(optarg, "critical") == 0) {
				debug_level = CRITICAL;

			} else if (strcasecmp(optarg, "serious") == 0) {
				debug_level = SERIOUS;

			} else if (strcasecmp(optarg, "error") == 0) {
				debug_level = ERROR;

			} else if (strcasecmp(optarg, "warning") == 0) {
				debug_level = WARNING;

			} else if (strcasecmp(optarg, "default") == 0) {
				debug_level = DEFAULT;

			} else if (strcasecmp(optarg, "details") == 0) {
				debug_level = DETAILS;

			} else if (strcasecmp(optarg, "debug") == 0) {
				debug_level = DEBUG;

			} else if (strcasecmp(optarg, "extra") == 0) {
				debug_level = EXTRA;

			} else if (strcasecmp(optarg, "entry_exit") == 0) {
				debug_level = ENTRY_EXIT;

			} else if (strcasecmp(optarg, "everything") == 0) {
				debug_level = EVERYTHING;
			} else {
				printf("%s is not a valid debug level.\n", optarg);
				/* Display the help. */
				rc = EINVAL;
			}

			break;

		case 'l':
			log_file = strdup(optarg);
			break;

		case 'q':
			quiet = TRUE;
			break;

		case 'v':
			verbose = TRUE;
			break;

		case 'h':
		case '?':
			/* Display the help. */
			rc = EINVAL;
			break;

		default:
			printf("%s -- unrecognized option \"%c\"\n\n", prog_name, c);
			/* Display the help. */
			rc = EINVAL;
			break;
		}
	}

	return rc;
}


static int message_callback(char   * message_text,
			    int    * answer,
			    char * * choices){

	printf(message_text);

	if (choices != NULL) {
		printf("%s: Responding with default selection \"%s\".\n", prog_name, choices[*answer]);
	}

	return 0;
}


static ui_callbacks_t callbacks = {
	user_message:       message_callback,
	user_communication: NULL,
	progress:           NULL};


/*
 * Open the EVMS Engine in daemon mode.  If successful, then this process is the
 * daemon.  Sleep forever to keep the process.
 */

int main(int argc, char * * argv, char * * env){

	int     rc = 0;

	prog_name = basename(argv[0]);

	rc = parse_options(argc, argv);
	if (rc) {
		show_help();
		return rc;
	}

	if (verbose) {
		printf("%s: Opening the EVMS Engine in daemon mode...\n", prog_name);
	}
	rc = evms_open_engine(NULL, ENGINE_DAEMON, &callbacks, debug_level, log_file);

	if (rc == 0) {
		sem_t exit_sem;

		if (verbose) {
			printf("%s: The EVMS daemon is running.\n", prog_name);
		}

		/*
		 * Keep this process by waiting on a semaphore that will never
		 * get posted.
		 */

		sem_init(&exit_sem, 0, 0);
		do {
			rc = sem_wait(&exit_sem);
		} while ((rc != 0) && (errno == EINTR));

		if (verbose) {
			printf("%s: Exit semaphore posted.\n", prog_name);
		}

		evms_close_engine();
	}

	/* Error messages will have been displayed by evms_open_engine(). */
	return rc;
}

