/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <frontend.h>
#include <glib.h>
#include <ctype.h>
#include <ncurses.h>
#include <panel.h>

#include "common.h"
#include "readable.h"
#include "window.h"
#include "menu.h"
#include "clist.h"
#include "enum.h"
#include "views.h"

static PANEL *view_panel[MAX_VIEWS];			/* The panels we raise and lower for the views */
static struct clist *view_clist[MAX_VIEWS];		/* The columned lists in each view panel/window */
static populate_func populate_view_func[MAX_VIEWS];	/* The function responsible for populating a view */
static gboolean repopulate_view[MAX_VIEWS];		/* Whether the view clist needs to be populated before */
							/* displaying it */
static unsigned int curr_view;				/* The index of the current view */

/**
 *	on_view_item_activated - displays a context menu next to view item activated
 *	@clist: the choice list
 *	@item: the choice activated
 *
 *	This routine is invoked when a item in a view is activated by pressing ENTER.
 *	We determine the screen coordinates for the item chosen and place a context
 *	popup menu near the choice activated.
 */
void on_view_item_activated(struct clist *clist, struct clist_item *item)
{
	int starty, startx;

	starty = g_list_index(get_clist_top_item(clist), item) + getbegy(clist->win);
	startx = getbegx(clist->win) + 4;
	popup_context_menu(GPOINTER_TO_UINT(item->user_data), starty, startx);
}

/**
 *	format_volume_view_item - return column strings for a row in the volume view
 *	@handle: the volume handle
 *	@not_used: typically extra data but we expect none
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for a volume in the volume's
 *	view. This is a clist_format_func type function called by clist_populate().
 */
int format_volume_view_item(object_handle_t handle, void *not_used, GPtrArray *text)
{
	int rc;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);
	if (rc == 0) {
		g_ptr_array_add(text, g_strdup(object->info.volume.name));
		g_ptr_array_add(text, make_sectors_readable_string(object->info.volume.vol_size));
		g_ptr_array_add(text, checkmark_if_true(object->info.volume.flags & VOLFLAG_DIRTY));
		g_ptr_array_add(text, checkmark_if_true(object->info.volume.flags & VOLFLAG_ACTIVE));
		g_ptr_array_add(text, checkmark_if_true(object->info.volume.flags & VOLFLAG_READ_ONLY));
		g_ptr_array_add(text, get_volume_fsim_name(&(object->info.volume)));
		g_ptr_array_add(text, g_strdup(object->info.volume.mount_point ? 
					object->info.volume.mount_point : " "));
		evms_free(object);
	}
	return rc;
}

/**
 *	populate_volumes_view - populates the volumes view with volumes
 *	@void
 *
 *	This routine enumerates the volumes and populates the volumes view.
 */
void populate_volumes_view(void)
{
	clist_populate(view_clist[VOLUME_VIEW], enum_volumes, clist_allow_all,
			format_volume_view_item, NULL, NULL);
}

/**
 *	format_object_view_item - return columns strings for a row in the objects view
 *	@handle: the topmost storage object handle
 *	@not_used: typically extra data but we expect none
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for an available topmpst storage
 *	object in the available objects	view. This is a clist_format_func type function
 *	called by clist_populate().
 */
int format_object_view_item(object_handle_t handle, void *not_used, GPtrArray *text)
{
	int rc;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);
	if (rc == 0) {
		g_ptr_array_add(text, g_strdup(object->info.object.name));
		g_ptr_array_add(text, make_sectors_readable_string(object->info.object.size));
		g_ptr_array_add(text, checkmark_if_true(object->info.object.flags & SOFLAG_DIRTY));
		g_ptr_array_add(text, checkmark_if_true(object->info.object.flags & SOFLAG_ACTIVE));
		g_ptr_array_add(text, checkmark_if_true(object->info.object.flags & SOFLAG_READ_ONLY));
		g_ptr_array_add(text, get_plugin_name(object->info.object.plugin));
		evms_free(object);
	}
	return rc;
}

/**
 *	populate_objects_view - populates the available objects view with topmost objects
 *	@void
 *
 *	This routine enumerates the available topmost objects and populates the available
 *	objects view.
 */
void populate_objects_view(void)
{
	clist_populate(view_clist[OBJECT_VIEW], enum_topmost_objects, clist_allow_all,
			format_object_view_item, NULL, NULL);
}

/**
 *	format_feature_view_item - return column strings for a row in the features view
 *	@handle: the feature object handle
 *	@not_used: typically extra data but we expect none
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for a feature object in the
 *	feature objects view. This is a clist_format_func type function called by
 *	clist_populate().
 */
int format_feature_view_item(object_handle_t handle, void *not_used, GPtrArray *text)
{
	int rc;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);
	if (rc == 0) {
		g_ptr_array_add(text, g_strdup(object->info.object.name));
		g_ptr_array_add(text, make_sectors_readable_string(object->info.object.size));
		g_ptr_array_add(text, checkmark_if_true(object->info.object.flags & SOFLAG_DIRTY));
		g_ptr_array_add(text, checkmark_if_true(object->info.object.flags & SOFLAG_ACTIVE));
		g_ptr_array_add(text, checkmark_if_true(object->info.object.flags & SOFLAG_READ_ONLY));
		g_ptr_array_add(text, get_plugin_name(object->info.object.plugin));
		evms_free(object);
	}
	return rc;
}

/**
 *	populate_features_view - populates the features view with feature objects
 *	@void
 *
 *	This routine enumerates the feature objects and populates the features view.
 */
void populate_features_view(void)
{
	clist_populate(view_clist[FEATURE_VIEW], enum_feature_objects, clist_allow_all,
			format_feature_view_item, NULL, NULL);
}

/**
 *	format_region_view_item - return column strings for row in the regions view
 *	@handle: the storage region handle
 *	@not_used: typically extra data but we expect none
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for a storage region in the
 *	regions view. This is a clist_format_func type function called by clist_populate().
 */
int format_region_view_item(object_handle_t handle, void *not_used, GPtrArray *text)
{
	int rc;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);
	if (rc == 0) {
		g_ptr_array_add(text, g_strdup(object->info.object.name));
		g_ptr_array_add(text, make_sectors_readable_string(object->info.object.size));
		g_ptr_array_add(text, checkmark_if_true(object->info.object.flags & SOFLAG_DIRTY));
		g_ptr_array_add(text, checkmark_if_true(object->info.object.flags & SOFLAG_ACTIVE));
		g_ptr_array_add(text, checkmark_if_true(object->info.object.flags & SOFLAG_READ_ONLY));
		g_ptr_array_add(text, checkmark_if_true(object->info.object.flags & SOFLAG_CORRUPT));
		g_ptr_array_add(text, get_plugin_name(object->info.object.plugin));
		evms_free(object);
	}
	return rc;
}

/**
 *	populate_regions_view - populates the regions view with regions
 *	@void
 *
 *	This routine enumerates the volumes and populates the regions view.
 */
void populate_regions_view(void)
{
	clist_populate(view_clist[REGION_VIEW], enum_regions, clist_allow_all,
			format_region_view_item, NULL, NULL);
}

/**
 *	format_container_view_item - return column strings for a row in the containers view
 *	@handle: the storage container's handle
 *	@not_used: typically extra data but we expect none
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row text for a container in the containers
 *	view. This is a clist_format_func type function called by clist_populate().
 */
int format_container_view_item(object_handle_t handle, void *not_used, GPtrArray *text)
{
	int rc;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);
	if (rc == 0) {
		g_ptr_array_add(text, g_strdup(object->info.container.name));
		g_ptr_array_add(text, make_sectors_readable_string(object->info.container.size));
		g_ptr_array_add(text, checkmark_if_true(object->info.container.flags & SCFLAG_DIRTY));
		g_ptr_array_add(text, get_plugin_name(object->info.container.plugin));
		evms_free(object);
	}
	return rc;
}

/**
 *	populate_containers_view - populates the containers view with containers
 *	@void
 *
 *	This routine enumerates the containers and populates the containers view.
 */
void populate_containers_view(void)
{
	clist_populate(view_clist[CONTAINER_VIEW], enum_containers, clist_allow_all,
			format_container_view_item, NULL, NULL);
}

/**
 *	format_segment_view_item - return column strings for a row in the  segments view
 *	@handle: the segment handle
 *	@not_used: typically extra data but we expect none
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for a segment in the segment's
 *	view. This is a clist_format_func type function called by clist_populate().
 */
int format_segment_view_item(object_handle_t handle, void *not_used, GPtrArray *text)
{
	int rc;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);
	if (rc == 0) {
		g_ptr_array_add(text, g_strdup(object->info.object.name));
		g_ptr_array_add(text, make_sectors_readable_string(object->info.object.size));
		g_ptr_array_add(text, checkmark_if_true(object->info.object.flags & SOFLAG_DIRTY));
		g_ptr_array_add(text, checkmark_if_true(object->info.object.flags & SOFLAG_ACTIVE));
		g_ptr_array_add(text, checkmark_if_true(object->info.object.flags & SOFLAG_READ_ONLY));
		g_ptr_array_add(text, make_data_type_readable_string(object->info.object.data_type));
		g_ptr_array_add(text, g_strdup_printf("%"PRIu64, object->info.object.start));
		g_ptr_array_add(text, get_plugin_name(object->info.object.plugin));
		evms_free(object);
	}
	return rc;
}

/**
 *	populate_segments_view - populates the segments view with segments
 *	@void
 *
 *	This routine enumerates the segments and populates the segments
 *	view.
 */
void populate_segments_view(void)
{
	clist_populate(view_clist[SEGMENT_VIEW], enum_segments, clist_allow_all,
			format_segment_view_item, NULL, NULL);
}

/**
 *	format_disk_view_item - return column strings for a row in the disks view
 *	@handle: the disk handle
 *	@not_used: typically extra data but we expect none
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row text for a disk in the disks
 *	view. This is a clist_format_func type function called by clist_populate().
 */
int format_disk_view_item(object_handle_t handle, void *not_used, GPtrArray *text)
{
	int rc;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);
	if (rc == 0) {
		g_ptr_array_add(text, g_strdup(object->info.object.name));
		g_ptr_array_add(text, make_sectors_readable_string(object->info.object.size));
		g_ptr_array_add(text, checkmark_if_true(object->info.object.flags & SOFLAG_DIRTY));
		g_ptr_array_add(text, checkmark_if_true(object->info.object.flags & SOFLAG_ACTIVE));
		g_ptr_array_add(text, checkmark_if_true(object->info.object.flags & SOFLAG_READ_ONLY));
		g_ptr_array_add(text, g_strdup_printf("%"PRIu64",%u,%u", object->info.object.geometry.cylinders,
						object->info.object.geometry.heads,
						object->info.object.geometry.sectors_per_track));
		g_ptr_array_add(text, get_plugin_name(object->info.object.plugin));
		evms_free(object);
	}
	return rc;
}

/**
 *	populate_disks_view - populates the disks view with disks
 *	@void
 *
 *	This routine enumerates the disks and populates the disks view.
 */
void populate_disks_view(void)
{
	clist_populate(view_clist[DISK_VIEW], enum_disks, clist_allow_all,
			format_disk_view_item, NULL, NULL);
}

/**
 *	format_plugin_view_item - return column strings for a row in the plugins view
 *	@handle: the plugin handle
 *	@not_used: typically extra data but we expect none
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for a plugin in the plugins
 *	view. This is a clist_format_func type function called by clist_populate().
 */
int format_plugin_view_item(object_handle_t handle, void *not_used, GPtrArray *text)
{
	int rc;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);
	if (rc == 0) {
		g_ptr_array_add(text, g_strdup_printf("%08X", object->info.plugin.id));
		g_ptr_array_add(text, g_strdup_printf("%u.%u.%u", object->info.plugin.version.major,
								object->info.plugin.version.minor,
								object->info.plugin.version.patchlevel));
		g_ptr_array_add(text, g_strdup(object->info.plugin.short_name));
		g_ptr_array_add(text, g_strdup(object->info.plugin.oem_name));
		g_ptr_array_add(text, g_strdup(object->info.plugin.long_name));
		evms_free(object);
	}
	return rc;
}

/**
 *	populate_plugins_view - populates the plugin view with plugins
 *	@void
 *
 *	This routine enumerates the plugins and populates the disks view.
 */
void populate_plugins_view(void)
{
	clist_populate(view_clist[PLUGIN_VIEW], enum_all_plugins, clist_allow_all,
			format_plugin_view_item, NULL, NULL);
}

/**
 *	create_view_panel - creates a standard panel for a view
 *	@void
 *
 *	This routine creates the standard size view window and
 *	panel.
 */
inline PANEL *create_view_panel(void)
{
	PANEL *panel;

	panel = create_panel_window(getmaxy(stdscr) - 2, getmaxx(stdscr), 1, 0, FALSE, BLUE_BKGD);
	return panel;
}

/**
 *	create_view_clist - creates a standard choice list for a view
 *	@panel: the panel to contain the new clist
 *
 *	This routine creates the standard size choice list for a view.
 */
inline struct clist *create_view_clist(PANEL *panel, int cols)
{
	WINDOW *win;
	struct clist *clist;

	win = panel_window(panel);
	show_panel_window(panel, FALSE);
	clist = create_clist(win, cols, getmaxy(win) - 6, getmaxx(win) - 2, 5, 1, NULL);
	set_clist_activate_item_cb(clist, (clist_activate_item_cb)on_view_item_activated);
	set_clist_select_item_cb(clist, (clist_select_item_cb)disallow_select_events);

	return clist;
}

/**
 *	populate_view - populates a view clist if necessary
 *	@view - the view index to to populate
 *
 *	This routine checks the repopulate_view flag for the given view
 *	and re/populates the view if necessary.
 */
void populate_view(view_type_t view)
{
	if (repopulate_view[view] == TRUE) {
		clear_clist(view_clist[view]);
		populate_view_func[view]();
		repopulate_view[view] = FALSE;
	}
}

/**
 *	load_view - populate the given view
 *	@view: the view to load
 *
 *	This routine populates a view. If the view is empty then
 *	it returns an error code.
 */
inline int load_view(view_type_t view)
{
	int rc = 0;
	
	populate_view(view);
	if (is_clist_empty(view_clist[view]))
		rc = ENODATA;
	
	return rc;
}

/**
 *	show_non_empty_view - shows the given view if not empty
 *	@view: the view to show
 *
 *	This routine populates a view and if not empty it goes
 *	ahead and draws the view.
 */
int show_non_empty_view(view_type_t view)
{
	int rc;
	
	rc = load_view(view);
	if (rc == 0) {
		draw_clist(view_clist[view]);
		show_panel_window(view_panel[view], TRUE);
		curr_view = view;
	}
	return rc;
}

/**
 *	show_next_view - shows the next non-empty view
 *	@void
 *
 *	This routine set the curr_view index to the next non-empty view, repopulating
 *	it if necessary.
 */
void show_next_view(void)
{
	while (1) {
		curr_view = (curr_view + 1) % MAX_VIEWS;
		if (show_non_empty_view(curr_view) == 0)
			break;
	}
}

/**
 *	show_previous_view - shows the previous non-empty view
 *	@void
 *
 *	This routine set the curr_view index to the previous non-empty view, repopulating
 *	it if necessary.
 */
void show_previous_view(void)
{
	while (1) {
		curr_view = MIN((curr_view - 1), (MAX_VIEWS - 1));
		if (show_non_empty_view(curr_view) == 0)
			break;			
	}
}

/**
 *	refresh_views - repopulate the views
 *	@void
 *
 *	This routine marks all the views as needing to be repopulated
 *	and attempts to redraw the current view or the next non-empty
 *	view.
 */
void refresh_views(void)
{
	int i;

	for (i = 0; i < MAX_VIEWS; i++) {
		repopulate_view[i] = TRUE;
	}
	curr_view = curr_view - 1;
	show_next_view();
}

/**
 *	create_view - create a view
 *	@index: the index of the view to create
 *	@cols: the number of columns in the view
 *
 *	This routine creates a view with the requested number of columns.
 */
void create_view(int index, int cols)
{
	view_panel[index] = create_view_panel();
	view_clist[index] = create_view_clist(view_panel[index], cols);
	repopulate_view[index] = TRUE;
}

/**
 *	create_volumes_view - create the volumes view
 *	@void
 *
 *	This routine creates the volumes view.
 */
void create_volumes_view(void)
{
	WINDOW *win;
	struct clist *clist;

	create_view(VOLUME_VIEW, VOLUME_VIEW_COLUMNS);
	populate_view_func[VOLUME_VIEW] = populate_volumes_view;

	clist = view_clist[VOLUME_VIEW];
	win = panel_window(view_panel[VOLUME_VIEW]);

	mvwaddstr(win, 0, 1, _("0=Logical Volumes"));
	mvwhline(win, 3, 1, ACS_HLINE, getmaxx(win) - 2);

	set_clist_column_info(clist, 0, calc_clist_column_width(clist, 0.38), 0, 
					CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 1, calc_clist_column_width(clist, 0.12),
					get_clist_column_end(clist, 0),
					CLIST_TEXT_JUSTIFY_RIGHT);
	set_clist_column_info(clist, 2, calc_clist_column_width(clist, 0.08),
					get_clist_column_end(clist, 1),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 3, calc_clist_column_width(clist, 0.08),
					get_clist_column_end(clist, 2),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 4, calc_clist_column_width(clist, 0.04),
					get_clist_column_end(clist, 3),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 5, calc_clist_column_width(clist, 0.12),
					get_clist_column_end(clist, 4),
					CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 6, calc_clist_column_width(clist, 0.13),
					get_clist_column_end(clist, 5),
					CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(clist, 0, _("Name"));
	print_clist_column_title(clist, 1, _("Size"));
	print_clist_column_title(clist, 2, _("Dirty"));
	print_clist_column_title(clist, 3, _("Active"));
	print_clist_column_title(clist, 4, _("R/O"));
	print_clist_column_title(clist, 5, _("Plug-in"));
	print_clist_column_title(clist, 6, _("Mountpoint"));
}

/**
 *	create_objects_view - create the available topmost objects view
 *	@void
 *
 *	This routine creates the available topmost objects view.
 */
void create_objects_view(void)
{
	WINDOW *win;
	struct clist *clist;

	create_view(OBJECT_VIEW, OBJECT_VIEW_COLUMNS);
	populate_view_func[OBJECT_VIEW] = populate_objects_view;

	clist = view_clist[OBJECT_VIEW];
	win = panel_window(view_panel[OBJECT_VIEW]);

	mvwaddstr(win, 0, 1, _("1=Available Objects"));
	mvwhline(win, 3, 1, ACS_HLINE, getmaxx(win) - 2);

	set_clist_column_info(clist, 0, calc_clist_column_width(clist, 0.38), 0,
					CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 1, calc_clist_column_width(clist, 0.12),
					get_clist_column_end(clist, 0),
					CLIST_TEXT_JUSTIFY_RIGHT);
	set_clist_column_info(clist, 2, calc_clist_column_width(clist, 0.08),
					get_clist_column_end(clist, 1),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 3, calc_clist_column_width(clist, 0.08),
					get_clist_column_end(clist, 2),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 4, calc_clist_column_width(clist, 0.04),
					get_clist_column_end(clist, 3),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 5, calc_clist_column_width(clist, 0.26),
					get_clist_column_end(clist, 4),
					CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(clist, 0, _("Name"));
	print_clist_column_title(clist, 1, _("Size"));
	print_clist_column_title(clist, 2, _("Dirty"));
	print_clist_column_title(clist, 3, _("Active"));
	print_clist_column_title(clist, 4, _("R/O"));
	print_clist_column_title(clist, 5, _("Plug-in"));
}

/**
 *	create_features_view - create the feature objects view
 *	@void
 *
 *	This routine creates the feature objects view.
 */
void create_features_view(void)
{
	WINDOW *win;
	struct clist *clist;

	create_view(FEATURE_VIEW, FEATURE_VIEW_COLUMNS);
	populate_view_func[FEATURE_VIEW] = populate_features_view;

	clist = view_clist[FEATURE_VIEW];
	win = panel_window(view_panel[FEATURE_VIEW]);

	mvwaddstr(win, 0, 1, _("2=Feature Objects"));
	mvwhline(win, 3, 1, ACS_HLINE, getmaxx(win) - 2);

	set_clist_column_info(clist, 0, calc_clist_column_width(clist, 0.38), 0,
					CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 1, calc_clist_column_width(clist, 0.12),
					get_clist_column_end(clist, 0),
					CLIST_TEXT_JUSTIFY_RIGHT);
	set_clist_column_info(clist, 2, calc_clist_column_width(clist, 0.08),
					get_clist_column_end(clist, 1),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 3, calc_clist_column_width(clist, 0.08),
					get_clist_column_end(clist, 2),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 4, calc_clist_column_width(clist, 0.04),
					get_clist_column_end(clist, 3),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 5, calc_clist_column_width(clist, 0.26),
					get_clist_column_end(clist, 4),
					CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(clist, 0, _("Name"));
	print_clist_column_title(clist, 1, _("Size"));
	print_clist_column_title(clist, 2, _("Dirty"));
	print_clist_column_title(clist, 3, _("Active"));
	print_clist_column_title(clist, 4, _("R/O"));
	print_clist_column_title(clist, 5, _("Plug-in"));
}

/**
 *	create_regions_view - create the storage regions view
 *	@void
 *
 *	This routine creates the storage regions view.
 */
void create_regions_view(void)
{
	WINDOW *win;
	struct clist *clist;

	create_view(REGION_VIEW, REGION_VIEW_COLUMNS);
	populate_view_func[REGION_VIEW] = populate_regions_view;

	clist = view_clist[REGION_VIEW];
	win = panel_window(view_panel[REGION_VIEW]);

	mvwaddstr(win, 0, 1, _("3=Storage Regions"));
	mvwhline(win, 3, 1, ACS_HLINE, getmaxx(win) - 2);

	set_clist_column_info(clist, 0, calc_clist_column_width(clist, 0.38), 0,
					CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 1, calc_clist_column_width(clist, 0.12),
					get_clist_column_end(clist, 0),
					CLIST_TEXT_JUSTIFY_RIGHT);
	set_clist_column_info(clist, 2, calc_clist_column_width(clist, 0.08),
					get_clist_column_end(clist, 1),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 3, calc_clist_column_width(clist, 0.08),
					get_clist_column_end(clist, 2),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 4, calc_clist_column_width(clist, 0.04),
					get_clist_column_end(clist, 3),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 5, calc_clist_column_width(clist, 0.09),
					get_clist_column_end(clist, 4),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 6, calc_clist_column_width(clist, 0.15),
					get_clist_column_end(clist, 5),
					CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(clist, 0, _("Name"));
	print_clist_column_title(clist, 1, _("Size"));
	print_clist_column_title(clist, 2, _("Dirty"));
	print_clist_column_title(clist, 3, _("Active"));
	print_clist_column_title(clist, 4, _("R/O"));
	print_clist_column_title(clist, 5, _("Corrupt"));
	print_clist_column_title(clist, 6, _("Plug-in"));
}

/**
 *	create_containers_view - create the storage containers view
 *	@void
 *
 *	This routine creates the storage containers view.
 */
void create_containers_view(void)
{
	WINDOW *win;
	struct clist *clist;

	create_view(CONTAINER_VIEW, CONTAINER_VIEW_COLUMNS);
	populate_view_func[CONTAINER_VIEW] = populate_containers_view;

	clist = view_clist[CONTAINER_VIEW];
	win = panel_window(view_panel[CONTAINER_VIEW]);

	mvwaddstr(win, 0, 1, _("4=Storage Containers"));
	mvwhline(win, 3, 1, ACS_HLINE, getmaxx(win) - 2);

	set_clist_column_info(clist, 0, calc_clist_column_width(clist, 0.44), 0,
					CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 1, calc_clist_column_width(clist, 0.12),
					get_clist_column_end(clist, 0),
					CLIST_TEXT_JUSTIFY_RIGHT);
	set_clist_column_info(clist, 2, calc_clist_column_width(clist, 0.08),
					get_clist_column_end(clist, 1),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 3, calc_clist_column_width(clist, 0.53),
					get_clist_column_end(clist, 2),
					CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(clist, 0, _("Name"));
	print_clist_column_title(clist, 1, _("Size"));
	print_clist_column_title(clist, 2, _("Dirty"));
	print_clist_column_title(clist, 3, _("Plug-in"));
}

/**
 *	create_segments_view - create the disk segments view
 *	@void
 *
 *	This routine creates the disk segments view.
 */
void create_segments_view(void)
{
	WINDOW *win;
	struct clist *clist;

	create_view(SEGMENT_VIEW, SEGMENT_VIEW_COLUMNS);
	populate_view_func[SEGMENT_VIEW] = populate_segments_view;

	clist = view_clist[SEGMENT_VIEW];
	win = panel_window(view_panel[SEGMENT_VIEW]);

	mvwaddstr(win, 0, 1, _("5=Disk Segments"));
	mvwhline(win, 3, 1, ACS_HLINE, getmaxx(win) - 2);

	set_clist_column_info(clist, 0, calc_clist_column_width(clist, 0.27), 0,
					CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 1, calc_clist_column_width(clist, 0.12), 
					get_clist_column_end(clist, 0),
					CLIST_TEXT_JUSTIFY_RIGHT);
	set_clist_column_info(clist, 2, calc_clist_column_width(clist, 0.08),
					get_clist_column_end(clist, 1),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 3, calc_clist_column_width(clist, 0.08),
					get_clist_column_end(clist, 2),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 4, calc_clist_column_width(clist, 0.04),
					get_clist_column_end(clist, 3),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 5, calc_clist_column_width(clist, 0.12),
					get_clist_column_end(clist, 4),
					CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 6, calc_clist_column_width(clist, 0.12),
					get_clist_column_end(clist, 5),
					CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 7, calc_clist_column_width(clist, 0.12),
					get_clist_column_end(clist, 6),
					CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(clist, 0, _("Name"));
	print_clist_column_title(clist, 1, _("Size"));
	print_clist_column_title(clist, 2, _("Dirty"));
	print_clist_column_title(clist, 3, _("Active"));
	print_clist_column_title(clist, 4, _("R/O"));
	print_clist_column_title(clist, 5, _("Type"));
	print_clist_column_title(clist, 6, _("Offset"));
	print_clist_column_title(clist, 7, _("Plug-in"));
}

/**
 *	create_disks_view - create the logical disks view
 *	@void
 *
 *	This routine creates the logical disks view.
 */
void create_disks_view(void)
{
	WINDOW *win;
	struct clist *clist;

	create_view(DISK_VIEW, DISK_VIEW_COLUMNS);
	populate_view_func[DISK_VIEW] = populate_disks_view;

	clist = view_clist[DISK_VIEW];
	win = panel_window(view_panel[DISK_VIEW]);

	mvwaddstr(win, 0, 1, _("6=Logical Disks"));
	mvwhline(win, 3, 1, ACS_HLINE, getmaxx(win) - 2);

	set_clist_column_info(clist, 0, calc_clist_column_width(clist, 0.31), 0,
					CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 1, calc_clist_column_width(clist, 0.12),
					get_clist_column_end(clist, 0),
					CLIST_TEXT_JUSTIFY_RIGHT);
	set_clist_column_info(clist, 2, calc_clist_column_width(clist, 0.08),
					get_clist_column_end(clist, 1),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 3, calc_clist_column_width(clist, 0.08),
					get_clist_column_end(clist, 2),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 4, calc_clist_column_width(clist, 0.04),
					get_clist_column_end(clist, 3),
					CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 5, calc_clist_column_width(clist, 0.15),
					get_clist_column_end(clist, 4),
					CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 6, calc_clist_column_width(clist, 0.17),
					get_clist_column_end(clist, 5),
					CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(clist, 0, _("Name"));
	print_clist_column_title(clist, 1, _("Size"));
	print_clist_column_title(clist, 2, _("Dirty"));
	print_clist_column_title(clist, 3, _("Active"));
	print_clist_column_title(clist, 4, _("R/O"));
	print_clist_column_title(clist, 5, _("Geometry"));
	print_clist_column_title(clist, 6, _("Plug-in"));
}

/**
 *	create_plugins_view - create the plugins view
 *	@void
 *
 *	This routine creates the plugins view.
 */
void create_plugins_view(void)
{
	WINDOW *win;
	struct clist *clist;

	create_view(PLUGIN_VIEW, PLUGIN_VIEW_COLUMNS);
	populate_view_func[PLUGIN_VIEW] = populate_plugins_view;

	clist = view_clist[PLUGIN_VIEW];
	win = panel_window(view_panel[PLUGIN_VIEW]);

	mvwaddstr(win, 0, 1, _("7=Plug-ins"));
	mvwhline(win, 3, 1, ACS_HLINE, getmaxx(win) - 2);

	set_clist_column_info(clist, 0, calc_clist_column_width(clist, 0.14), 0,
					CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 1, calc_clist_column_width(clist, 0.12),
					get_clist_column_end(clist, 0),
					CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 2, calc_clist_column_width(clist, 0.17),
					get_clist_column_end(clist, 1),
					CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 3, calc_clist_column_width(clist, 0.12),
					get_clist_column_end(clist, 2),
					CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 4, calc_clist_column_width(clist, 0.43),
					get_clist_column_end(clist, 3),
					CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(clist, 0, _("Id"));
	print_clist_column_title(clist, 1, _("Version"));
	print_clist_column_title(clist, 2, _("Name"));
	print_clist_column_title(clist, 3, _("Developer"));
	print_clist_column_title(clist, 4, _("Description"));
}

/**
 *	draw_initial_view - determine which view is the initial view
 *	@void
 *
 *	This routine is nothing special. It sets the curr_view index
 *	to negative one knowing full well that the show_next_view()
 *	will cycle through the views to figure out which one to show.
 */
void draw_initial_view(void)
{
	curr_view = -1;
	show_next_view();
}

/**
 *	init_views - this routine initializes the views and draws the initial view
 *	@void
 *
 *	This routine draws the window containing the views.
 *	It draws the tabs for the available views and displays
 *	the initial view.
 **/
void init_views(void)
{
	create_volumes_view();
	create_objects_view();
	create_features_view();
	create_regions_view();
	create_containers_view();
	create_segments_view();
	create_disks_view();
	create_plugins_view();

	draw_initial_view();
}

/**
 *	process_notebook_events - the routine checks for keyboard input for switching views
 *	@key: the input character
 *
 *	This routine takes keyboard input and if recognizable causes a new view to
 *	be brought to the foreground or some other event.
 */
int process_notebook_events(int key)
{
	int event_handled = FALSE;

	if (key == KEY_TAB) {
		show_next_view();
		event_handled = TRUE;
	} else if (key == KEY_ESCAPE) {
		show_previous_view();
		event_handled = TRUE;
	} else if (isdigit(key)) {
		char tmp[2];
		int new_view;
		
		tmp[0] = key;
		tmp[1] = '\0';
		new_view = atoi(tmp);
		if (new_view < MAX_VIEWS) {
			show_non_empty_view(new_view);
			event_handled = TRUE;
		}
	}
	return event_handled;
}

/**
 *	process_current_view_events - this routine processes events for the current view
 *	@void
 *
 *	This routine takes keyboard input and translates that to some action/event.
 *	The processing order is view events are processed first. If they are not
 *	handled then notebook events are processed second and if those are not
 *	handled then menubar events are processed last. 
 */
int process_current_view_events(void)
{
	int rc = 0, event_handled, key;

	key = panel_getch(view_panel[curr_view]);
	event_handled = process_clist_events(view_clist[curr_view], key);
	if (event_handled == FALSE) {
		event_handled = process_notebook_events(key);

		if (event_handled == FALSE) { 
			rc = process_menubar_events(key);
		}
	}
	return rc;
}

/**
 *	show_view - displays a requested view with focus placed on an item with the given handle
 *	@view: the view index
 *	@handle: the handle that should be contained in a clist_item's user data
 *
 *	This routine makes the specified view the current view. It repopulates the view
 *	if necessary and sets the focus item to the item containing the given user handle.
 */
void show_view(view_type_t view, object_handle_t handle)
{
	struct clist *clist = view_clist[view];

	populate_view(view);
	if (is_clist_empty(clist) == FALSE) {
		GList *element;

		element = find_item_from_handle(clist, handle);
		if (element != NULL) {
			curr_view = view;
			if (!is_item_visible(clist, element))
				set_clist_top_item(clist, element);

			set_clist_focus_item(clist, element);
			draw_clist(clist);
			show_panel_window(view_panel[view], TRUE);
		}
	}
}

/**
 *	jump_to_object - sets focus on the given object in a view; switching views if necessary
 *	@handle: the object to jump to
 *
 *	This routine makes the object corresponding to the handle the focus item in the
 *	view matching the object type. It makes the corresponding view the current view
 *	if it isn't already.
 */
int jump_to_object(object_handle_t handle)
{
	int rc;
	view_type_t view;
	object_type_t type;

	rc = evms_get_handle_object_type(handle, &type);
	if (rc == 0) {
		switch (type) {
		case EVMS_OBJECT:
			view = FEATURE_VIEW;
			break;
		case REGION:
			view = REGION_VIEW;
			break;
		case SEGMENT:
			view = SEGMENT_VIEW;
			break;
		case DISK:
			view = DISK_VIEW;
			break;
		case VOLUME:
			view = VOLUME_VIEW;
			break;
		case CONTAINER:
			view = CONTAINER_VIEW;
			break;
		case PLUGIN:
			view = PLUGIN_VIEW;
		default:
			view = curr_view;
			break;
		}
		show_view(view, handle);
	}
	return rc;
}
/**
 *	jump_to_view_by_handle - function activated to change a view to a certain object
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is a callback routine invoked when one of the navigation
 *	menu items in the context popup menu is selected. It switches to the view
 *	containing the object type and makes that object the focus object.
 */
int jump_to_view_by_handle(struct menu_item *item)
{
	return jump_to_object(GPOINTER_TO_UINT(item->user_data));
}

