/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <string.h>
#include <frontend.h>
#include <ncurses.h>
#include <panel.h>
#include <glib.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "dialog.h"
#include "clist.h"
#include "selwin.h"
#include "options.h"
#include "task.h"
#include "readable.h"
#include "value.h"
#include "logging.h"

/**
 *	get_active_option - return option descriptor for an activate option
 *	@task: the task handle
 *	@dialog: the dialog containing the option
 *	@index: the option index
 *
 *	This routine gets the active option descriptor for a given index and returns
 *	it. If the option is for an inactive descriptor or an error occurs retrieving
 *	it, we return NULL instead.
 **/
option_descriptor_t *get_active_option(task_handle_t task, struct dialog_window *dialog, int index)
{
	int rc;
	option_descriptor_t *option;

	rc = evms_get_option_descriptor(task, index, &option);
	if (rc == 0) {
		if (!(EVMS_OPTION_IS_ACTIVE(option->flags))) {
			evms_free(option);
			option = NULL;
		}
	} else {
		option = NULL;
		log_error("%s: evms_get_option_descriptor() returned error code %d.\n", __FUNCTION__, rc);
	}
	return option;
}

/**
 *	create_options_list - get option descriptors for a task
 *	@task: the task handle
 *	@dialog: the dialog window to contain the options
 *
 *	This routine finds out the maximum number of options for a task and
 *	then proceeds to retrieve the active options for the task, placing
 *	the option descriptors in a singly linked list.
 */
GSList *create_options_list(task_handle_t task, struct dialog_window *dialog)
{
	int rc, count;
	GSList *options = NULL;

	rc = evms_get_option_count(task, &count);
	if (rc == 0) {
		int i;

		for (i = 0; i < count; i++) {
			option_descriptor_t *option;

			option = get_active_option(task, dialog, i);
			if (option != NULL)
				options = g_slist_append(options, option);
		}
	} else {
		log_error("%s: evms_get_option_count() encountered error code %d.\n", __FUNCTION__, rc);
	}
	return options;
}

/**
 *	required_options_have_values - checks to see if all required options have values
 *	@options: the list of active option descriptors
 *
 *	This routine scans the list of option descriptors to see if all required
 *	options have values associated with them. It returns FALSE if any of them
 *	still have no value.
 */
gboolean required_options_have_values(GSList *options)
{
	gboolean results = TRUE;

	while (options != NULL && results == TRUE) {
		if (options->data != NULL) {
			option_descriptor_t *option;

			option = options->data;
			if (EVMS_OPTION_IS_REQUIRED(option->flags) &&
				!EVMS_OPTION_HAS_VALUE(option->flags))
					results = FALSE;
		}
		options = g_slist_next(options);
	}
	return results;
}

/**
 *	check_required_options - checks to see if all required options have values
 *	@dialog: the options dialog
 *	@options: the list of option descriptors
 *
 *	This routine checks to see if all required options have values. If they
 *	all do then the task button is made sensitive to 
 */
void check_required_options(struct dialog_window *dialog, GSList *options)
{
	if (required_options_have_values(options))
		make_next_button_sensitive(dialog);
	else
		make_next_button_insensitive(dialog);
}

/**
 *	get_title_column_width - get optimum width for the option title column
 *	@options: the list of option descriptors
 *
 *	This routine finds the longest option title to determine the width of
 *	the title column.
 */
int get_title_column_width(GSList *options)
{
	int width = 0;

	while (options != NULL) {
		int len;
		option_descriptor_t *option;

		option = options->data;
		len = strlen(option->title);
		if (len > width)
			width = len;
		options = g_slist_next(options);
	}
	width = MAX(width, strlen(_("Option")));
	width += 2;
	return width;
}

/**
 *	get_info_title_column_width - get optimum width for the extended info title column
 *	@info: the array of extended info structures
 *
 *	This routine finds the longest extended info title to determine the width of
 *	the title column.
 */
int get_info_title_column_width(extended_info_array_t *info)
{
	int i, width = 0;

	for (i = 0; i < info->count; i++) {
		if (info->info[i].title != NULL) {
			int len;

			len = strlen(info->info[i].title);
			if (len > width)
				width = len;
		}
	}
	width = MAX(width, strlen(_("Name")));
	width += 2;
	return width;
}

/**
 *	create_value_string - common routine used to generate a string for a value or collection
 *	@value: the value_t looked at if collection_type is EVMS_Collection_None
 *	@unit: the unit type for the value
 *	@type: the value type, i.e a string, or int64, etc.
 *	@format: special formatting needs
 *	@collection: looked at if collection_type is either EVMS_Collection_List or EVMS_Collection_Range
 *	@collection_type: the collection type
 *	@can_convert: whether plugins allows us to convert to a more readable format
 *
 *	This routine is the common routine invoked to produce the string for either a value if 
 *	it is not a collection type or the first value in a list or the min and max values for
 *	a range if they are a collection type.
 */
char *create_value_string(value_t value, value_unit_t unit, value_type_t type, value_format_t format,
				value_collection_t collection, collection_type_t collection_type,
				gboolean can_convert)
{
	char *str = NULL;

	if (can_convert && unit == EVMS_Unit_Sectors) {
		switch (collection_type) {
		case EVMS_Collection_None:
			str = make_sectors_readable_string(convert_value_to_ui64(value, type));
			break;
		case EVMS_Collection_Range: {
			char *min, *max;
			min = make_sectors_readable_string(
						convert_value_to_ui64(collection.range->min, type));
			max = make_sectors_readable_string(
						convert_value_to_ui64(collection.range->max, type));
			str = g_strdup_printf(_("%s to %s"), min, max);
			g_free(min);
			g_free(max);
			}
			break;
		case EVMS_Collection_List:
			if (collection.list->count > 0)
				str = make_sectors_readable_string(convert_value_to_ui64(collection.list->value[0], 
				type));
			else
				str = make_sectors_readable_string(0);
			break;
		}
	} else {
		switch (collection_type) {
		case EVMS_Collection_None:
			str = convert_value_to_string(value, type, format);
			break;
		case EVMS_Collection_Range: {
			char *min, *max;

			min = convert_value_to_string(collection.range->min,
						type, format);
			max = convert_value_to_string(collection.range->max,
						type, format);
			str = g_strdup_printf(_("%s to %s"), min, max);
			g_free(min);
			g_free(max);
			}	
			break;
		case EVMS_Collection_List:
			if (collection.list->count > 0)
				str = convert_value_to_string(collection.list->value[0],
							type, format);
			else {
				memset(&value, 0, sizeof(value_t));
				str = convert_value_to_string(value, type, format);
			}
			break;
		}
	}
	if (str == NULL)
		str = g_strdup("");
	return str;
}

/**
 *	create_option_value_string - convert option value or first value in a multi-value list to a string
 *	@option: the option descriptor
 *
 *	This routine converts an option value to a readable string or the first value in a multi-value list.
 */
char *create_option_value_string(option_descriptor_t *option)
{
	char *str;

	if (EVMS_OPTION_HAS_VALUE(option->flags)) {
		value_collection_t collection;
		collection_type_t collection_type;

		/*
		 * Weirdness to handle multi value option value.
		 * Basically we send over the value list
		 * through the collection list parm and set it
		 * up so it is looked (like an extended info 
		 * collection) and the option->value is ignored.
		 */
		if (EVMS_OPTION_VALUE_IS_LIST(option->flags)) {
			collection_type = EVMS_Collection_List;
			collection.list = option->value.list;
		} else {
			collection_type = EVMS_Collection_None;
		}

		str = create_value_string(option->value,
					option->unit,
					option->type,
					option->format,
					collection,
					collection_type,
					!(option->flags & EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION));
	} else {
		if (EVMS_OPTION_IS_REQUIRED(option->flags))
			str = g_strdup(_("-- Required value not set --"));
		else
			str = g_strdup(_("-- Optional value not set --"));
	}
	return str;
}

/**
 *	format_option_item - return column strings for a row in the option window clist
 *	@option: the address of the option descriptor structure
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for an option placed
 *	in the options window clist containing the title of the option and its
 *	current value.
 */
void format_option_item(option_descriptor_t *option, GPtrArray *text)
{
	g_ptr_array_add(text, g_strdup(option->title));
	g_ptr_array_add(text, create_option_value_string(option));
}

/**
 *	format_info_item - return column strings for a row in the extended info window clist
 *	@option: the address of the extended info structure
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for an option placed
 *	in the options window clist containing the title of the option and its
 *	current value.
 */
void format_info_item(extended_info_t *info, GPtrArray *text)
{
	char *str;

	if (info->flags & EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE ||
		info->collection_type == EVMS_Collection_List)
		g_ptr_array_add(text, g_strdup("+"));
	else
		g_ptr_array_add(text, g_strdup(" "));

	if (info->title != NULL)
		g_ptr_array_add(text, g_strdup(info->title));
	else
		g_ptr_array_add(text, g_strdup(" "));

	str = create_value_string(info->value,
				info->unit,
				info->type,
				info->format,
				info->collection,
				info->collection_type,
				!(info->flags & EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION));
	g_ptr_array_add(text, str);
}

/**
 *	delete_option - callback invoked when clist item is being deleted
 *	@item: the clist item being deleted
 *
 *	This routine frees the option descriptor associated with an item in the
 *	options clist when the item is being deleted.
 */
void delete_option(struct clist_item *item)
{
	evms_free(item->user_data);
}

/**
 *	on_option_item_focus - callback invoked when an option in the options clist receives focus
 *	@clist: the options clist
 *	@item: the clist_item that received focus
 *
 *	This routine is invoked when an item in the options clist receives focus. We display the
 *	options tip in the status bar for the user to see.
 */
void on_option_item_focus(struct clist *clist, struct clist_item *item)
{
	option_descriptor_t *option = item->user_data;

	if (option->tip != NULL)
		print_statusbar_text(option->tip);
	else
		print_statusbar_text("");
}

/**
 *	populate_options_clist - populate the given options clist with option descriptor info
 *	@task: the task handle
 *	@dialog: the dialog window to contain the options
 *	@clist: the options clist
 *
 *	This routine creates a list of active option descriptors and associates this with
 *	the options dialog. It also sets the task button sensitivity depending on whether
 *	all required options have values or not.
 */
void populate_options_clist(task_handle_t task, struct dialog_window *dialog, struct clist *clist)
{
	int width;
	GSList *options;
	GHashTable *hash_table = dialog->user_data;

	options = create_options_list(task, dialog);
	g_hash_table_insert(hash_table, "options", options);
	check_required_options(dialog, options);

	width = get_title_column_width(options);

	set_clist_column_info(clist, 0, width, 0, CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 1, getmaxx(clist->win) - width,
				get_clist_column_end(clist, 0),
				CLIST_TEXT_JUSTIFY_LEFT);
	print_clist_column_title(clist, 0, "Option");
	print_clist_column_title(clist, 1, "Value");

	set_clist_focus_item_cb(clist, (clist_focus_item_cb)on_option_item_focus);

	while (options != NULL) {
		GPtrArray *text;

		text = g_ptr_array_new();
		format_option_item(((option_descriptor_t *)(options->data)), text);
		append_item(clist, text, options->data, (clist_item_delete_cb)delete_option);
		g_ptr_array_free(text, TRUE);

		options = g_slist_next(options);
	}
}

/**
 *	on_clist_idle_reload_options - re-populate the options clist and clist event idle time
 *	@clist: the options clist
 *
 *	This routine deletes all the items in the clist and then re-populates
 *	the clist with the current active options and their current values.
 *	This routine is intended to be invoked as a clist idle function.
 */
int on_clist_idle_reload_options(struct clist *clist)
{
	GSList *options;
	task_handle_t task;
	GHashTable *hash_table;
	struct dialog_window *dialog;

	dialog = clist->user_data;
	hash_table = dialog->user_data;

	clear_clist(clist);

	options = g_hash_table_lookup(hash_table, "options");
	g_slist_free(options);
	
	task = get_task_from_dialog_data(dialog);
	populate_options_clist(task, dialog, clist);

	return 0;
}

/**
 *	populate_extended_info_clist - populate the given extended info clist with extended info
 *	@info: the extended info array
 *	@dialog: the dialog window to display the extended info
 *	@clist: the extended info clist
 *
 *	This routine gets an array of extended info descriptors and populates the clist in the
 *	dialog window with them.
 */
void populate_extended_info_clist(extended_info_array_t *info, struct dialog_window *dialog,
					struct clist *clist)
{
	int i, width;

	width = get_info_title_column_width(info);

	set_clist_column_info(clist, 0, 3, 0, CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(clist, 1, width,
			get_clist_column_end(clist, 0),
			CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(clist, 2, getmaxx(clist->win) - width,
			get_clist_column_end(clist, 1),
			CLIST_TEXT_JUSTIFY_LEFT);
	print_clist_column_title(clist, 0, " ");
	print_clist_column_title(clist, 1, "Name");
	print_clist_column_title(clist, 2, "Value");

	for (i = 0; i < info->count; i++) {
		GPtrArray *text;

		text = g_ptr_array_new();
		format_info_item(&(info->info[i]), text);
		append_item(clist, text, &(info->info[i]), NULL);
		g_ptr_array_free(text, TRUE);
	}
}

/**
 *	display_info_list_field - display a dialog window for an extended info list field
 *	@info: the extended info containing
 *
 *	This routine displays a dialog with the list of values for an extended info field
 *	that is a collection list type.
 */
void display_info_list_field(extended_info_t *info)
{
	int i;
	struct selwin *list_dialog;
	struct dialog_window *dialog;

	list_dialog = create_list_dialog(info->title,
				NULL,
				"",
				1, NULL);
	dialog = (struct dialog_window *)list_dialog;

	set_clist_column_count(list_dialog->clist, 1);
	set_clist_column_info(list_dialog->clist, 0, calc_clist_column_width(list_dialog->clist, 1.00),
				0,
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(list_dialog->clist, 0, _("Values"));

	for (i = 0; i < info->collection.list->count; i++) {
		GPtrArray *text;

		text = g_ptr_array_new();
		g_ptr_array_add(text, create_value_string(info->collection.list->value[i],
				info->unit,
				info->type,
				info->format,
				info->collection,
				EVMS_Collection_None,
				!(info->flags & EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION)));
		append_item(list_dialog->clist, text, NULL, NULL);
		g_ptr_array_free(text, TRUE);
	}
	process_modal_dialog(dialog);
}

/**
 *	expand_info_array - allocate an extended info array with space for new entries
 *	@info_in: the original extended info array to copy
 *	@extra_entries: the count of new entries needed in the array
 *
 *	This routine allocates a new extended info array with space for additional
 *	entries and does a shallow copy of the original extended info array into
 *	the newly allocated extended info array.
 */
inline extended_info_array_t *expand_info_array(extended_info_array_t *info_in, int extra_entries)
{
	extended_info_array_t *info_out;
	
	info_out = g_malloc0(sizeof(extended_info_array_t) + (sizeof(extended_info_t) *
				(info_in->count + extra_entries)));
				
	info_out->count = info_in->count + extra_entries;
	
	memcpy(&(info_out->info[extra_entries]), &(info_in->info[0]),
		sizeof(extended_info_t) * info_in->count);

	return info_out;
}
/**
 *	add_extra_volume_info - adds supplemental information about a volume to extended info
 *	@vinfo: the volume information
 *	@einfo: the original extended information
 *
 *	This routine creates a completely fake extended information array made up of
 *	information from the volume information.
 */
extended_info_array_t *add_extra_volume_info(logical_volume_info_t *vinfo, extended_info_array_t *einfo)
{
	extended_info_array_t *info_out;

	/*
	 * Allocate space for the volume related information and copy FSIM related information
	 * at the end.
	 */
	info_out = expand_info_array(einfo, VOL_INFO_MAX);

	info_out->info[VOL_INFO_MAJOR].name = "major_no";
	info_out->info[VOL_INFO_MAJOR].title = _("Major Number");
	info_out->info[VOL_INFO_MAJOR].desc = _("This field describes the volume's device major number.");
	info_out->info[VOL_INFO_MAJOR].type = EVMS_Type_Unsigned_Int32;
	info_out->info[VOL_INFO_MAJOR].unit = EVMS_Unit_None;
	info_out->info[VOL_INFO_MAJOR].format = EVMS_Format_Normal;
	info_out->info[VOL_INFO_MAJOR].collection_type = EVMS_Collection_None;
	info_out->info[VOL_INFO_MAJOR].value.ui32 = vinfo->dev_major;
	
	info_out->info[VOL_INFO_MINOR].name = "minor_no";
	info_out->info[VOL_INFO_MINOR].title = _("Minor Number");
	info_out->info[VOL_INFO_MINOR].desc = _("This field describes the volume's device minor number.");
	info_out->info[VOL_INFO_MINOR].type = EVMS_Type_Unsigned_Int32;
	info_out->info[VOL_INFO_MINOR].unit = EVMS_Unit_None;
	info_out->info[VOL_INFO_MINOR].format = EVMS_Format_Normal;
	info_out->info[VOL_INFO_MINOR].collection_type = EVMS_Collection_None;
	info_out->info[VOL_INFO_MINOR].value.ui32 = vinfo->dev_minor;

	info_out->info[VOL_INFO_MOUNTPOINT].name = "mount_point";
	info_out->info[VOL_INFO_MOUNTPOINT].title = _("Mount Point");
	info_out->info[VOL_INFO_MOUNTPOINT].desc = _("This field describes the mount point for the volume.");
	info_out->info[VOL_INFO_MOUNTPOINT].type = EVMS_Type_String;
	info_out->info[VOL_INFO_MOUNTPOINT].unit = EVMS_Unit_None;
	info_out->info[VOL_INFO_MOUNTPOINT].format = EVMS_Format_Normal;
	info_out->info[VOL_INFO_MOUNTPOINT].collection_type = EVMS_Collection_None;
	info_out->info[VOL_INFO_MOUNTPOINT].value.s = vinfo->mount_point ? vinfo->mount_point : "Not mounted";

	info_out->info[VOL_INFO_FS_SIZE].name = "fs_size";
	info_out->info[VOL_INFO_FS_SIZE].title = _("File System Size");
	info_out->info[VOL_INFO_FS_SIZE].desc = _("This field describes the current size of the file system.");
	info_out->info[VOL_INFO_FS_SIZE].type = EVMS_Type_Unsigned_Int64;
	info_out->info[VOL_INFO_FS_SIZE].unit = EVMS_Unit_Sectors;
	info_out->info[VOL_INFO_FS_SIZE].format = EVMS_Format_Normal;
	info_out->info[VOL_INFO_FS_SIZE].collection_type = EVMS_Collection_None;
	info_out->info[VOL_INFO_FS_SIZE].value.ui64 = vinfo->fs_size;

	info_out->info[VOL_INFO_FS_MIN_SIZE].name = "min_fs_size";
	info_out->info[VOL_INFO_FS_MIN_SIZE].title = _("Minimum File System Size");
	info_out->info[VOL_INFO_FS_MIN_SIZE].desc = _("This field describes the minimum size of the file system.");
	info_out->info[VOL_INFO_FS_MIN_SIZE].type = EVMS_Type_Unsigned_Int64;
	info_out->info[VOL_INFO_FS_MIN_SIZE].unit = EVMS_Unit_Sectors;
	info_out->info[VOL_INFO_FS_MIN_SIZE].format = EVMS_Format_Normal;
	info_out->info[VOL_INFO_FS_MIN_SIZE].collection_type = EVMS_Collection_None;
	info_out->info[VOL_INFO_FS_MIN_SIZE].value.ui64 = vinfo->min_fs_size;
	    
	info_out->info[VOL_INFO_FS_MAX_SIZE].name = "max_fs_size";
	info_out->info[VOL_INFO_FS_MAX_SIZE].title = _("Maximum File System Size");
	info_out->info[VOL_INFO_FS_MAX_SIZE].desc = _("This field describes the maximum size of the file system.");
	info_out->info[VOL_INFO_FS_MAX_SIZE].type = EVMS_Type_Unsigned_Int64;
	info_out->info[VOL_INFO_FS_MAX_SIZE].unit = EVMS_Unit_Sectors;
	info_out->info[VOL_INFO_FS_MAX_SIZE].format = EVMS_Format_Normal;
	info_out->info[VOL_INFO_FS_MAX_SIZE].collection_type = EVMS_Collection_None;
	info_out->info[VOL_INFO_FS_MAX_SIZE].value.ui64 = vinfo->max_fs_size;

	info_out->info[VOL_INFO_VOL_SIZE].name = "vol_size";
	info_out->info[VOL_INFO_VOL_SIZE].title = _("Volume Size");
	info_out->info[VOL_INFO_VOL_SIZE].desc = _("This field describes the current size of the volume object.");
	info_out->info[VOL_INFO_VOL_SIZE].type = EVMS_Type_Unsigned_Int64;
	info_out->info[VOL_INFO_VOL_SIZE].unit = EVMS_Unit_Sectors;
	info_out->info[VOL_INFO_VOL_SIZE].format = EVMS_Format_Normal;
	info_out->info[VOL_INFO_VOL_SIZE].collection_type = EVMS_Collection_None;
	info_out->info[VOL_INFO_VOL_SIZE].value.ui64 = vinfo->vol_size;
	
	info_out->info[VOL_INFO_VOL_MAX_SIZE].name = "max_vol_size";
	info_out->info[VOL_INFO_VOL_MAX_SIZE].title = _("Maximum Volume Size");
	info_out->info[VOL_INFO_VOL_MAX_SIZE].desc = _("This field describes the maximum size of the volume object.");
	info_out->info[VOL_INFO_VOL_MAX_SIZE].type = EVMS_Type_Unsigned_Int64;
	info_out->info[VOL_INFO_VOL_MAX_SIZE].unit = EVMS_Unit_Sectors;
	info_out->info[VOL_INFO_VOL_MAX_SIZE].format = EVMS_Format_Normal;
	info_out->info[VOL_INFO_VOL_MAX_SIZE].collection_type = EVMS_Collection_None;
	info_out->info[VOL_INFO_VOL_MAX_SIZE].value.ui64 = vinfo->max_vol_size;

	return info_out;
}

/**
 *	add_extra_object_info - adds supplemental information about a storage object to extended info
 *	@oinfo: the storage object info
 *	@einfo: the original extended info
 *
 *	This routine creates a supplemental extended info array containing the original
 *	extended information for a storage object along with some common information
 *	available for a storage object such as the device major and device minor.
 */
extended_info_array_t *add_extra_object_info(storage_object_info_t *oinfo, extended_info_array_t *einfo)
{
	extended_info_array_t *info_out;

	/*
	 * Allocate space for the device major and minor fields and copy the
	 * existing extended_info after the additional fields.
	 */
	info_out = expand_info_array(einfo, OBJ_INFO_MAX);
	
	info_out->info[OBJ_INFO_MAJOR].name = "major_no";
	info_out->info[OBJ_INFO_MAJOR].title = _("Major Number");
	info_out->info[OBJ_INFO_MAJOR].desc = _("This field describes the object's device major number.");
	info_out->info[OBJ_INFO_MAJOR].type = EVMS_Type_Unsigned_Int32;
	info_out->info[OBJ_INFO_MAJOR].unit = EVMS_Unit_None;
	info_out->info[OBJ_INFO_MAJOR].format = EVMS_Format_Normal;
	info_out->info[OBJ_INFO_MAJOR].collection_type = EVMS_Collection_None;
	info_out->info[OBJ_INFO_MAJOR].value.ui32 = oinfo->dev_major;
	
	info_out->info[OBJ_INFO_MINOR].name = "minor_no";
	info_out->info[OBJ_INFO_MINOR].title = _("Minor Number");
	info_out->info[OBJ_INFO_MINOR].desc = _("This field describes the object's device minor number.");
	info_out->info[OBJ_INFO_MINOR].type = EVMS_Type_Unsigned_Int32;
	info_out->info[OBJ_INFO_MINOR].unit = EVMS_Unit_None;
	info_out->info[OBJ_INFO_MINOR].format = EVMS_Format_Normal;
	info_out->info[OBJ_INFO_MINOR].collection_type = EVMS_Collection_None;
	info_out->info[OBJ_INFO_MINOR].value.ui32 = oinfo->dev_minor;

	return info_out;
}

/**
 *	duplicate_extended_info - returns a shallow copy of an extended info array structure
 *	@info: the extended info array
 *
 *	This routine returns a dynamically allocated shallow copy of an extended info
 *	array.
 */
inline extended_info_array_t *duplicate_extended_info(extended_info_array_t *info)
{
	return g_memdup(info, sizeof(extended_info_array_t) + (sizeof(extended_info_t) *
			(info->count - 1)));
}

/**
 *	add_extra_info - add some common supplementary information to details
 *	@object: contains the object type and other object information
 *	@info_in: the original extended info for the object
 *
 *	This routine creates a new extended_info_array_t structure that in
 *	addition to the original extended info from the plug-in, contains
 *	information about the object such and device major and minor.
 *	For volumes we add even more information such as filesystem
 *	size statistics, etc.
 */
extended_info_array_t *add_extra_info(handle_object_info_t *object, extended_info_array_t *info_in)
{
	extended_info_array_t *info_out;
	
	if (object != NULL && object->type != CONTAINER && object->type != PLUGIN) {	
		if (object->type == VOLUME) {
			info_out = add_extra_volume_info(&(object->info.volume), info_in);
		} else {
			info_out = add_extra_object_info(&(object->info.object), info_in);
		}
	} else {
		info_out = duplicate_extended_info(info_in);
	}
	return info_out;
}

/**
 *	view_more_info - display a dialog to view more details about an extended info field
 *	@clist: the selection list
 *	@item: the item/row selected
 *	
 *	This routine is invoked when a row in the extended info clist is selected.
 *	If the field indicates there are more details to view, we display another
 *	dialog containing the additional information.
 */
int view_more_info(struct clist *clist, struct clist_item *item)
{
	extended_info_t *info = item->user_data;
	struct dialog_window *dialog = clist->user_data;

	if (info->collection_type == EVMS_Collection_List)
		display_info_list_field(info);
	else if (info->flags & EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE)
		show_extended_info(GPOINTER_TO_UINT(dialog->user_data), info->name, info->title);
	return 1;
}

/**
 *	on_info_item_focus - callback invoked when an extended info item in the clist receives focus
 *	@clist: the options clist
 *	@item: the clist_item that received focus
 *
 *	This routine is invoked when an item in the extended info clist receives focus. 
 *	We display the extended info field description in the status bar for the user to see.
 */
void on_info_item_focus(struct clist *clist, struct clist_item *item)
{
	extended_info_t *info = item->user_data;

	if (info->desc != NULL)
		print_statusbar_text(info->desc);
	else
		print_statusbar_text("");	
}

/**
 *	on_delete_extended_info_dialog - callback invoked when extended info dialog is deleted
 *	@dialog: the extended info dialog
 *
 *	This routine is invoked when an extended info dialog is deleted. We simply
 *	make sure that the statusbar text is cleared.
 */
void on_delete_extended_info_dialog(struct dialog_window *dialog)
{
	/*
	 * BUGBUG: If there are stacked extended info dialogs, this will
	 * likely clear the text for a focus item in the window underneath.
	 */
	print_statusbar_text("");
}

/**
 *	show_extended_info - create a dialog to allow displaying extended info
 *	@handle: the engine handle
 *	@field_name: the optional field name to gather detailed info on
 *	@field_title: if drilling down to a field, the field's title
 *
 *	This routine creates a dialog window for display detailed information
 *	for a handle or one a particular field known by that handle.
 **/
void show_extended_info(engine_handle_t handle, char *field_name, char *field_title)
{
	int rc;
	extended_info_array_t *info = NULL;

	rc = evms_get_extended_info(handle, field_name, &info);
	if (rc == 0) {
		char *field = NULL;
		struct selwin *selwin;
		struct dialog_window *dialog;
		handle_object_info_t *object = NULL;
		char *title, *thing_name, *plugin_name;
		extended_info_array_t *supplemented_info;

		evms_get_info(handle, &object);	
		supplemented_info = add_extra_info(object, info);
		
		get_object_and_plugin_names(handle, &thing_name, &plugin_name);

		if (field_name != NULL && field_title != NULL)
			field = g_strdup_printf(" - %s", field_title);

		title = g_strconcat(_("Detailed Information - "), thing_name, field, NULL);
		
		selwin = create_selection_window(title, NULL,
				_("Use spacebar on fields marked with \"+\" to view more information"),
				_("_OK"),
				(menuitem_activate_cb)close_window_button_activated,
				NULL, NULL,
				GUINT_TO_POINTER(handle));

		dialog = (struct dialog_window *)selwin;

		set_dialog_delete_cb(dialog, (dialog_delete_cb)on_delete_extended_info_dialog);
		set_clist_select_item_cb(selwin->clist, (clist_select_item_cb)view_more_info);
		set_clist_unselect_item_cb(selwin->clist, (clist_unselect_item_cb)NULL);
		set_clist_focus_item_cb(selwin->clist, (clist_focus_item_cb)on_info_item_focus);
		set_menu_item_visibility(dialog->prev_button, FALSE);
		set_menu_item_visibility(dialog->cancel_button, FALSE);
		set_menu_item_sensitivity(dialog->next_button, TRUE);
		set_horizontal_menu_focus(dialog->menu, dialog->next_button);

		populate_extended_info_clist(supplemented_info, dialog, selwin->clist);
		process_modal_dialog(dialog);

		g_free(title);
		g_free(field);
		g_free(thing_name);
		g_free(plugin_name);
		g_free(supplemented_info);
		if (object != NULL)
			evms_free(object);
		evms_free(info);
	} else {
		show_message_dialog(_("View Detailed Information"),
				_("No detailed information is available."));
	}
}

/**
 *	display_details_menuitem_activated - invoked when "Display Details" menuitem activated
 *	@item: the menu item that was activated
 *
 *	This routine is invoked when the "Display Details" menu item is activated in the
 *	popup context menu.
 */
int display_details_menuitem_activated(struct menu_item *item)
{
	show_extended_info(GPOINTER_TO_UINT(item->user_data), NULL, NULL);
	return 0;
}

/**
 *	replace_option - replace the changed option descriptor with the latest update
 *	@task: the task handle
 *	@item: the clist item corresponding to the option
 *	@option: the option descriptor
 *	@options: the linked list containing the option descriptor including the element pointing to ours
 *
 *	This routine replaces an option descriptor after having had it updated successfully. We update
 *	the pointer to the option descriptor in the options linked list and in the options clisy item
 *	as well as updating the value text for the clist item.
 *
 *	Care should be take not to reference the old option descriptor pointer after this
 *	call as the the structure will have been freed.
 */
void replace_option(task_handle_t task, struct clist_item *item, option_descriptor_t *option, GSList *options)
{
	int rc; 
	option_descriptor_t *option_redux;

	rc = evms_get_option_descriptor_by_name(task, option->name, &option_redux);
	if (rc == 0) {
		GSList *element;

		element = g_slist_find(options, option);
	
		element->data = option_redux;
		item->user_data = option_redux;

		g_free(item->row_text[1]);
		evms_free(option);

		item->row_text[1] = create_option_value_string(option_redux); 
	} else {
		log_error("%s: evms_get_option_descriptor_by_name() returned error code %d.\n", __FUNCTION__, rc);
	}
}

/**
 *	set_option_value - convert the string buffer to an option value and provide it to the engine
 *	@task: the task handle
 *	@option: the option descriptor whose value we are changing
 *	@options: the linked of all option descriptors including the element pointing to ours
 *	@item: the clist item corresponding to the option
 *	@buffer: contains the characters entered by the user
 *
 *	This routine takes the user's string input an converts it to a value_t that we
 *	can then pass to the engine via the evms_set_option_value_by_name() to provide
 *	the new value to the plugin.
 *
 *	We replace the clist item value displayed on success and if the effect indicates
 *	we need to reload options we just let the reload handle the refresh.
 */
gboolean set_option_value(task_handle_t task, option_descriptor_t *option, GSList *options, 
				struct clist_item *item, char *buffer)
{
	int rc;
	value_t value;
	task_effect_t effect = 0;
	gboolean reload_options = FALSE;

	if (option->unit == EVMS_Unit_Sectors)
		convert_size_string_to_sector_value(buffer, option->unit, option->type, &value);
	else
		convert_string_to_value(buffer, option->type, option->max_len, &value);
		
	if (option->constraint_type == EVMS_Collection_Range) {
		value = clamp_value(value, option->constraint.range, option->type);
		value = snap_value_to_constraint_increment(value, option->constraint.range, option->type);
	}

	rc = evms_set_option_value_by_name(task, option->name, &value, &effect);
	if (rc == 0) {
		if (effect & EVMS_Effect_Reload_Options) {
			reload_options = TRUE;
		} else {
			replace_option(task, item, option, options);
		}
	} else {
		print_statusbar_text(_("The plug-in did not accept the new value!"));
		log_error("%s: set_option_value_by_name() returned error code %d.\n", __FUNCTION__, rc);
	}
	return reload_options;
}

/**
 *	format_constraint_list_item - return column strings for a row in the constraint list
 *	@option: the address of the option descriptor structure
 *	@value: a value in the constraint list
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for an option value
 *	from a constraint value list.
 */
void format_constraint_list_item(option_descriptor_t *option, value_t value, GPtrArray *text)
{
	g_ptr_array_add(text, g_strdup(" "));
	g_ptr_array_add(text, create_value_string(value,
				option->unit,
				option->type,
				option->format,
				option->constraint,
				EVMS_Collection_None,
				!(option->flags & EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION)));
}

/**
 *	create_option_value_from_selection - create a value_t from a clist selection
 *	@selection: the list element containing the clist_item
 *	@option: the option descriptor
 *	@value: the value_t to return
 *
 *	This routine retrieves the constraint list index from the clist_item user data
 *	and makes a duplicate of the value in the constraint list corresponding to the
 *	index.
 */
void create_option_value_from_selection(GList *selection, option_descriptor_t *option, value_t *value)
{
	int index;
	struct clist_item *item;

	item = selection->data;
	index = GPOINTER_TO_UINT(item->user_data);

	duplicate_value(option->constraint.list->value[index], option->type,
			FALSE, 0, option->max_len, value);

}

/**
 *	create_constraint_selection_value - create the proper value_t from the clist selection(s)
 *	@selections: list of selections made
 *	@list_size: length of selection list
 *	@is_multi_value: whether value
 *	@option: option descriptor
 *	@value: the value created from the selection(s)
 *
 *	This routine creates the value_t corresponding to the selection(s) made. For an option
 *	that expects a single value we simply duplicate the constraint value selected. For
 *	a multi-value type we create a value list with the selections.
 */
void create_constraint_selection_value(GList *selections, int list_size, gboolean is_multi_value,
					option_descriptor_t *option, value_t *value)
{
	if (is_multi_value) {
		value_list_t *list;

		list = allocate_value_list(list_size, option->type, option->max_len);
		list->count = 0;
		value->list = list;

		while (selections != NULL) {
			create_option_value_from_selection(selections, option, &(list->value[list->count]));
			list->count++;
			selections = g_list_next(selections);
		}
	} else {
		create_option_value_from_selection(selections, option, value);
	}
}

/**
 *	set_list_option_value - set the option value for a value(s) selected from a constraint list
 *	@item: the OK button that was activated
 *
 *	This routine determines which value or values (if value is a list of potential values)
 *	were selected in the constraint list and then invokes evms_set_option_value_by_name()
 *	to provide the engine/plugin with the new value(s).
 */
int set_list_option_value(struct menu_item *item)
{
	int list_size;
	GSList *options;
	task_handle_t task;
	struct selwin *selwin;
	GHashTable *hash_table;
	gboolean *reload_options;
	option_descriptor_t *option;
	struct dialog_window *dialog;
	struct clist_item *clist_item;

	selwin = item->user_data;
	dialog = item->user_data;
	hash_table = dialog->user_data;
	
	task = get_task_from_dialog_data(dialog);
	option = g_hash_table_lookup(hash_table, "option");
	options = g_hash_table_lookup(hash_table, "options");
	clist_item = g_hash_table_lookup(hash_table, "item");
	reload_options = g_hash_table_lookup(hash_table, "reload");

	*reload_options = FALSE;
	list_size = g_list_length(selwin->clist->selections);

	if (list_size > 0) {
		int rc;
		value_t value;
		value_type_t type;
		gboolean is_multi_value;
		task_effect_t effect = 0;

		type = option->type;
		is_multi_value = EVMS_OPTION_VALUE_IS_LIST(option->flags);
		create_constraint_selection_value(selwin->clist->selections, list_size,
					is_multi_value, option, &value);
		rc = evms_set_option_value_by_name(task, option->name, &value, &effect);
		if (rc == 0) {
			if (effect & EVMS_Effect_Reload_Options) {
				*reload_options = TRUE;				
			} else {
				replace_option(task, clist_item, option, options);
			}
		}
		clear_value(&value, type, is_multi_value, list_size);
	}
	dialog->status = DLG_STATUS_CLOSING;
	return 0;
}

/**
 *	populate_list_option_value_clist - populate clist containing constraint list values
 *	@clist: the constraint value clist
 *	@option: the option descriptor containing a constraint list
 *
 *	This routine populates a clist with the values found in an option's constraint
 *	list. It pre-selects values in the list that are current/initial values for the
 *	option as well.
 */
void populate_list_option_value_clist(struct clist *clist, option_descriptor_t *option)
{
	int i;
	
	for (i = 0; i < option->constraint.list->count; i++) {
		GPtrArray *text;
		struct clist_item *item;

		text = g_ptr_array_new();
		format_constraint_list_item(option, option->constraint.list->value[i], text);
		item = append_item(clist, text, GUINT_TO_POINTER(i), NULL);
		
		/*
		 * If the option has a value or values, select this new item if it is
		 * the or one of the current value(s).
		 */
		if (EVMS_OPTION_HAS_VALUE(option->flags)) {
			gboolean select;
			
			if (EVMS_OPTION_VALUE_IS_LIST(option->flags)) {
				select = value_in_value_list(option->constraint.list->value[i],
							option->value.list,
							option->type);
			} else {
				select = values_are_equal(option->constraint.list->value[i],
							option->value,
							option->type);
			}
			if (select)
				select_item(clist, item);
		}
		g_ptr_array_free(text, TRUE);
	}
}

/**
 *	edit_list_option_value - present a dialog window for editing a multi-value list
 *	@task: the task handle
 *	@option: the option descriptor being changed
 *	@options: the linked list of all option descriptors including the element pointing to ours
 *	@item: the clist item corresponding to the option being changed
 *
 *	This routine creates a dialog window with a selection list for a constraint list.
 *	If this is a multi-value list we allow multiple selections otherwise the selection
 *	list is placed in single selection mode.
 */
gboolean edit_list_option_value(task_handle_t task, option_descriptor_t *option, GSList *options,
				struct clist_item *item)
{
	struct selwin *selwin;
	GHashTable *hash_table;
	struct dialog_window *dialog;
	gboolean reload_options = FALSE;

	hash_table = g_hash_table_new(g_str_hash, g_str_equal);
	g_hash_table_insert(hash_table, "item", item);
	g_hash_table_insert(hash_table, "option", option);
	g_hash_table_insert(hash_table, "options", options);
	g_hash_table_insert(hash_table, "reload", &reload_options);
	g_hash_table_insert(hash_table, "task", GUINT_TO_POINTER(task));

	selwin = create_selection_window(_("Set Value from Constraint List"), NULL, NULL,
					_("OK"),
					(menuitem_activate_cb)set_list_option_value,
					NULL, NULL, hash_table);

	dialog = (struct dialog_window *)selwin;

	set_clist_column_count(selwin->clist, 2);
	set_clist_column_info(selwin->clist, 0, calc_clist_column_width(selwin->clist, 0.05),
				0,
				CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(selwin->clist, 1, calc_clist_column_width(selwin->clist, 0.95),
				get_clist_column_end(selwin->clist, 0),
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(selwin->clist, 0, "");
	print_clist_column_title(selwin->clist, 1, "Value");

	set_menu_item_visibility(dialog->prev_button, FALSE);
	set_menu_item_activate_cb(dialog->cancel_button,
				(menuitem_activate_cb)close_window_button_activated);

	set_selection_mode(selwin->clist, (EVMS_OPTION_VALUE_IS_LIST(option->flags) ?
				CLIST_MODE_MULTI : CLIST_MODE_SINGLE));
	populate_list_option_value_clist(selwin->clist, option);
	process_modal_dialog(dialog);
	g_hash_table_destroy(hash_table);

	return reload_options;
}

/**
 *	change_boolean_value - change the value of a boolean type option
 *	@task: the task handle
 *	@option: the option descriptor being changed
 *	@options: the linked list of all option descriptors including the element pointing to ours
 *	@item: the clist item corresponding to the option being changed
 *
 *	This routine "flips" the value of a boolean type option.
 */
gboolean change_boolean_value(task_handle_t task, option_descriptor_t *option, GSList *options,
				struct clist_item *item)
{
	return set_option_value(task, option, options, item, option->value.i ? "False" : "True");
}

/**
 *	edit_option_value - get the new option value from user and call engine to set it
 *	@task: the task handle
 *	@option: the option descriptor being changed
 *	@options: the linked list of all option descriptors including the element pointing to ours
 *	@item: the clist item corresponding to the option being changed
 *
 *	This routine handles input for a single value that could be unconstrained
 *	or constrained by a range (min/max/step).
 */
gboolean edit_option_value(task_handle_t task, option_descriptor_t *option, GSList *options,
				struct clist_item *item)
{
	char *prompt, *buffer;
	int starty, len, bufsize;
	WINDOW *win, *border_win;
	PANEL *panel, *border_panel;
	gboolean reload_options = FALSE;

	panel = create_centered_popup_window(getmaxy(stdscr) - 8, getmaxx(stdscr) - 10, WHITE_BKGD);
	border_panel = (PANEL *)panel_userptr(panel);
	win = panel_window(panel);
	border_win = panel_window(border_panel);

	starty = getmaxy(win) / 2;
	prompt = g_strdup_printf("%s:: ", option->title);
	len = strlen(prompt);

	print_centered(border_win, 0, _("Change Option Value"));
	if (!(option->flags & EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION) && option->unit == EVMS_Unit_Sectors)
		print_centered(win, 2, _("Enter number expressed in sectors or a number with KB, MB, GB appended."));
	else if (option->unit != EVMS_Unit_None) {
		char *prompt_unit, *units;

		units = make_unit_readable_string(option->unit, FALSE);
		prompt_unit = g_strdup_printf(_("Enter value expressed in units of %s."), units);
		print_centered(win, 4, prompt_unit);
		g_free(units);
		g_free(prompt_unit);
	}
	if (option->constraint_type == EVMS_Collection_Range) {
		char *range, *step, *prompt_range;

		step = create_value_string(option->constraint.range->increment,
				option->unit,
				option->type,
				option->format,
				option->constraint,
				EVMS_Collection_None,
				!(option->flags & EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION));

		range = create_value_string(option->value,
				option->unit,
				option->type,
				option->format,
				option->constraint,
				option->constraint_type,
				!(option->flags & EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION));

		prompt_range = g_strdup_printf(_("Enter a value from %s in increments of %s."), range, step);
		print_centered(win, 3, prompt_range);
		g_free(step);
		g_free(range);
		g_free(prompt_range);
	}
	mvwaddnstr(win, starty, 2, prompt, len);

	show_popup_window(panel);

	if (option->type == EVMS_Type_String)
		bufsize = option->max_len;
	else
		bufsize = 100;

	buffer = g_malloc0(bufsize);
	if (read_string(win, starty, len + 2, buffer, bufsize) == OK)
		reload_options = set_option_value(task, option, options, item, buffer);

	g_free(buffer);
	delete_popup_window(panel);

	return reload_options;
}

/**
 *	edit_option - edit an option in the options dialog clist
 *	@clist: the options clist
 *	@item: the clist item corresponding to an option
 *
 *	This routine is invoked when the user selects an option item from
 *	the options clist. We create a dialog tailored for editing a specific type of
 *	option type. For example, it would display a clist for a list type value
 *	or handle conversion for a size or rounding, etc. before setting the
 *	option value.
 *
 *	If it turns out that the options need to be reloaded, we also register
 *	a clist idle function to handle that after the clist events have been
 *	processed.
 */
int edit_option(struct clist *clist, struct clist_item *item)
{
	GSList *options;
	task_handle_t task;
	GHashTable *hash_table;
	gboolean reload_options;
	option_descriptor_t *option = item->user_data;
	struct dialog_window *dialog = clist->user_data;

	hash_table = dialog->user_data;
	options = g_hash_table_lookup(hash_table, "options");
	task = get_task_from_dialog_data(dialog);

	if (EVMS_OPTION_VALUE_IS_LIST(option->flags) ||
		option->constraint_type == EVMS_Collection_List)
		reload_options = edit_list_option_value(task, option, options, item);
	else if (option->type == EVMS_Type_Boolean)
		reload_options = change_boolean_value(task, option, options, item);
	else
		reload_options = edit_option_value(task, option, options, item);

	if (reload_options)
		sched_clist_idle_func(clist, (clist_idle_func)on_clist_idle_reload_options);
	else
		check_required_options(dialog, options);

	return 1;
}
