/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: task.c
 */

#include <unistd.h>
#include <sys/types.h>
#include <pthread.h>
#include <frontend.h>
#include <string.h>
#include <gtk/gtk.h>

#include "support.h"
#include "widget.h"
#include "task.h"
#include "thing.h"
#include "readable.h"
#include "pixmap.h"
#include "task_window.h"
#include "selection_cb.h"
#include "views.h"
#include "logging.h"

/*
 * Structure containing information needed by invoke task thread.
 */
typedef struct evmsgui_task_info_s {
	gint rc;
	task_handle_t task;
	handle_array_t *resulting_objects;
} evmsgui_task_info_t;

/*
 * This idle function simply allows quitting an inner
 * event loop when it regains control.
 */
gboolean exit_from_event_loop(gpointer not_used)
{
	gdk_threads_enter();
	gtk_main_quit();
	gdk_threads_leave();

	return FALSE;
}

/*
 *
 *   void *task_thread (void *)
 *
 *   Description:
 *      This routine calls the evms_invoke_task API. Once
 *      the API call is complete it updates the return
 *      code in the task info structure it was given,
 *      calls gtk_main_quit() to terminate the waiter's
 *      nested event loop and then the thread terminates.
 *
 *   Entry:
 *      args - address of argument given at thread create
 *
 *   Exit:
 *      See description
 *
 */
void *task_thread(void *arg)
{
	evmsgui_task_info_t *info;

	info = (evmsgui_task_info_t *) arg;

	info->rc = evms_invoke_task(info->task, &(info->resulting_objects));

	/*
	 * Our caller should have called gtk_main() to continue processing
	 * GTK events while the task was processed. Now is a good time to let
	 * them out of their nested loop.
	 */

	gdk_threads_enter();
	gtk_idle_add((GtkFunction) exit_from_event_loop, NULL);
	gdk_threads_leave();

	return NULL;
}

/*
 *
 *   void on_task_window_destroy (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine frees engine resources associated with a task
 *      context by calling evms_destroy_task(). This routine is
 *      invoked when the window that created the task is destroyed.
 *
 *   Entry:
 *      widget    - the window that created the task context
 *      user_data - contains the task handle
 *
 *   Exit:
 *      The task context is destroyed.
 *
 */
void on_task_window_destroy(GtkWidget * widget, gpointer user_data)
{
	evms_destroy_task(GPOINTER_TO_UINT(user_data));
}

/*
 *
 *   void on_window_destroy_free_selection_array (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine frees the a handle array of current selections
 *
 *   Entry:
 *      widget     - the window that created the task context
 *      selections - pointer to handle array of current selections
 *
 *   Exit:
 *      The handle array of current selections is freed.
 *
 */
void on_window_destroy_free_selection_array(GtkWidget * widget, gpointer selections)
{
	log_debug("%s: Freeing handle array of current selections: %p.\n", __FUNCTION__,
		  selections);
	g_free(selections);
}

/*
 *
 *   void on_options_notebook_destroy (GtkWidget *, GSList *)
 *
 *   Description:
 *      This routine frees the linked list structures used to
 *      track option widget information. The option widget info
 *      is not removed only the links since by this time the
 *      widget destroy handler has freed the widget info structure
 *      already.
 *
 *   Entry:
 *      widget     - the notebook that contained the option widgets
 *      selections - head of the option widget info linked list
 *
 *   Exit:
 *      The option widget info linked list is freed.
 *
 */
void on_options_notebook_destroy(GtkWidget * widget, GSList * options)
{
	log_debug("%s: Freeing the linked list containing option widget info: %p.\n", __FUNCTION__,
		  options);
	g_slist_free(options);
}

/*
 *
 *   void create_option_widget_list (task_handle_t, gint, GSList **, GSList **, gboolean)
 *
 *   Description:
 *      This routine generates the widget list for option descriptors for the
 *      the given task. It also returns the option widget info structures
 *      created to track information on the option widget/descriptor.
 *
 *   Entry:
 *      handle        - the task handle
 *      count         - the count of option descriptors available
 *      options       - the address of the list to place the option widget info structures
 *      widgets       - the address of the list to insert the generated widgets
 *      show_inactive - TRUE if widgets for inactive options are wanted
 *
 *   Exit:
 *      The option widgets and option descriptor lists are returned.
 *
 */
void create_option_widget_list(task_handle_t handle, gint count, GSList ** options,
			       GSList ** widgets, gboolean show_inactive)
{
	gint i;
	gint rc;
	option_descriptor_t *option;
	option_widget_info_t *info;

	*widgets = NULL;
	*options = NULL;

	for (i = 0; i < count; i++) {
		rc = evms_get_option_descriptor(handle, i, &option);

		if (rc == SUCCESS) {
			if (show_inactive == TRUE || EVMS_OPTION_IS_ACTIVE(option->flags)) {
				info = create_widget_from_option_descriptor(option);

				if (info != NULL && info->widget != NULL)
					*widgets = g_slist_append(*widgets, info->widget);

				*options = g_slist_append(*options, info);
			} else {
				/*
				 * We're not going to show this option so free it.
				 */
				evms_free(option);
			}
		} else
			log_error("%s: evms_get_option_descriptor() returned error code %d.\n",
				  __FUNCTION__, rc);
	}
}

/*
 *
 *   gboolean have_required_options_been_set (GSList *)
 *
 *   Description:
 *      This routine walks a linked list of option widget info
 *      structures and checks to see if all active and required
 *      options have had their value set/initialized.
 *
 *   Entry:
 *      options - linked list containing option widget info
 *
 *   Exit:
 *      Returns TRUE if all active/required options have
 *      their values set. If any one of the required options
 *      has not had their value set then we return FALSE.
 *
 */
gboolean have_required_options_been_set(GSList * options)
{
	gboolean results = TRUE;
	option_widget_info_t *info;

	while (options != NULL && results == TRUE) {
		if (options->data != NULL) {
			info = options->data;

			if (EVMS_OPTION_IS_ACTIVE(info->option->flags) &&
			    EVMS_OPTION_IS_REQUIRED(info->option->flags) &&
			    info->has_value == FALSE)
				results = FALSE;
		}

		options = options->next;
	}

	return results;
}

/*
 *
 *   void destroy_option_widgets (GSList *)
 *
 *   Description:
 *      This routine walks a linked list of option widget info
 *      structures and destroys them with special attention to
 *      GtkSpinButton widgets in order to remove any timeout
 *      functions it may have registered. We also disconnect
 *      the "focus-out-event" signal handler for text entry
 *      fields to avoid it executing any update signal handlers
 *      as we destroy the widget.
 *
 *   Entry:
 *      options - linked list containing option widget info
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void destroy_option_widgets(GSList * options)
{
	option_widget_info_t *info;

	while (options != NULL) {
		if (options->data != NULL) {
			info = options->data;

			if (GTK_IS_SPIN_BUTTON(info->widget)) {
				GtkSpinButton *spin = GTK_SPIN_BUTTON(info->widget);

				if (spin->timer) {
					gtk_timeout_remove(spin->timer);
					spin->timer = 0;
					spin->timer_calls = 0;
					spin->need_timer = FALSE;
				}
			} else if (GTK_IS_ENTRY(info->widget)
				       && EVMS_OPTION_IS_ACTIVE(info->option->flags)) {
				    gtk_signal_disconnect_by_func(GTK_OBJECT(info->widget),
								  GTK_SIGNAL_FUNC
								  (on_option_entry_focus_out), info);
			} else if ((GTK_IS_RANGE(info->widget))) {
				GtkRange * range = GTK_RANGE(info->widget);
				if (range->timer) {
					gtk_timeout_remove(range->timer);
					range->timer = 0;
					range->need_timer = FALSE;
                                }
			}

			gtk_widget_destroy(info->widget);
		}

		options = options->next;
	}
}

/*
 *
 *   gchar* get_task_action_string (task_handle_t)
 *
 *   Description:
 *      This routine returns a string corresponding to the task
 *      operation at hand, i.e. "Create", "Expand", etc.
 *
 *   Entry:
 *      task - the task context handle
 *
 *   Exit:
 *      Returns a dynamically allocated string containing the task
 *      operation action.
 *
 */
gchar *get_task_action_string(task_handle_t task)
{
	gint rc;
	gchar *action_text = NULL;
	task_action_t action;

	rc = evms_get_task_action(task, &action);

	if (rc == SUCCESS) {
		switch (action) {
		case EVMS_Task_Create:
		case EVMS_Task_Create_Container:
			action_text = g_strdup(_("Create"));
			break;

		case EVMS_Task_Assign_Plugin:
			action_text = g_strdup(_("Add"));
			break;

		case EVMS_Task_Expand:
		case EVMS_Task_Expand_Container:
			action_text = g_strdup(_("Expand"));
			break;

		case EVMS_Task_Shrink:
		case EVMS_Task_Shrink_Container:
			action_text = g_strdup(_("Shrink"));
			break;

		case EVMS_Task_Slide:
			action_text = g_strdup(_("Slide"));
			break;

		case EVMS_Task_Move:
			action_text = g_strdup(_("Move"));
			break;

		case EVMS_Task_fsck:
			action_text = g_strdup(_("Check"));
			break;

		case EVMS_Task_mkfs:
			action_text = g_strdup(_("Make"));
			break;

		case EVMS_Task_defrag:
			action_text = g_strdup(_("Defragment"));
			break;

		default:
			log_debug("%s: Uknown task action %d received.\n", __FUNCTION__, action);
			break;
		}
	}

	return action_text;
}

/*
 *
 *   handle_array_t* make_handle_array_from_selection_list (GSList *selections)
 *
 *   Description:
 *      This routine takes a selection list from GtkCList widget and extracts
 *      the data (object handles associated with the list row) and generates
 *      a handle_array_t filled with valid (non-zero) handles.
 *
 *   Entry:
 *      selections - linked list with handles corresponding to rows selected
 *
 *   Exit:
 *      The address of the handle array containing the object
 *      handles is returned or NULL if the array was not created.
 *
 */
handle_array_t *make_handle_array_from_selection_list(GSList * selections)
{
	gint count;
	gint i;
	handle_array_t *harray;

	count = g_slist_length(selections);

	harray = g_malloc0(sizeof(handle_array_t) + (sizeof(object_handle_t) * count));

	for (i = 0; i < count; i++) {
		object_handle_t handle;

		handle = GPOINTER_TO_UINT(g_slist_nth_data(selections, i));

		if (handle != 0) {
			harray->handle[harray->count] = handle;
			harray->count++;
		}
	}

	return harray;
}

/*
 *
 *   gboolean are_handle_arrays_equal (handle_array_t *, handle_array_t *)
 *
 *   Description:
 *      This routine takes two handle arrays and checks to see if the counts
 *      match. If they do, we memcmp both to see if they contain the exact
 *      same handles. We should probably change this code to see if the same
 *      handles exist but in a different order but so what? This is really
 *      an optimized check anyways.
 *
 *   Entry:
 *      array1 - the first handle array
 *      array2 - the second handle array
 *
 *   Exit:
 *      Returns TRUE if the handle arrays are of the same length and contents.
 *
 */
gboolean are_handle_arrays_equal(handle_array_t * array1, handle_array_t * array2)
{
	gboolean arrays_equal = FALSE;

	if (array1 != NULL && array2 != NULL && array1->count == array2->count) {
		gint array_size;

		array_size =
		    sizeof(handle_array_t) + ((array1->count - 1) * sizeof(object_handle_t));
		arrays_equal = memcmp(array1, array2, array_size) == 0;
	}

	return arrays_equal;
}

/*
 *
 *   void make_next_button_sensitive_if_necessary (GtkWidget *, GSList *)
 *
 *   Description:
 *      This routine runs through the descriptors. If all the active
 *      and required options have set values then make the "Next"
 *      button sensitive otherwise it will go sensitive when at a
 *      minimum all required options have been set.
 *
 *   Entry:
 *      widget  - a widget to allow us to lookup the next button widget
 *      options - the list of current option descriptors
 *
 *   Exit:
 *      The "Next" button is made sensitive if all required/active
 *      option values have been set. Otherwise, it is made insensitive.
 *
 */
void make_next_button_sensitive_if_necessary(GtkWidget * widget, GSList * options)
{
	GtkWidget *next_button;

	next_button = lookup_widget(widget, "selection_window_next_button");

	if (have_required_options_been_set(options) == TRUE) {
		gtk_widget_set_sensitive(next_button, TRUE);
		gtk_widget_grab_default(next_button);
	} else
		gtk_widget_set_sensitive(next_button, FALSE);
}

/*
 *
 *   inline void indicate_plugin_supplied_no_options (GtkWidget *)
 *
 *   Description:
 *      This routine places a label in the options window that
 *      indicates that the plugin provided no changeable options.
 *
 *   Entry:
 *      container - widget to contain label
 *
 *   Exit:
 *      See description
 *
 */
inline void indicate_plugin_supplied_no_options(GtkWidget * container)
{
	GtkWidget *label;

	label = gtk_label_new(_("The plugin did not provide any configuration options."));

	gtk_container_add(GTK_CONTAINER(container), label);
	gtk_widget_show(label);
}

/*
 *
 *   void give_focus_to_first_empty_option_entry (GtkNotebook *)
 *
 *   Description:
 *      This routine goes through the option widgets in the current
 *      page of the notebook and gives focus to the first option entry
 *      widget that has no value set.
 *
 *   Entry:
 *      notebook - the option widget notebook
 *
 *   Exit:
 *      See description
 *
 */
void give_focus_to_first_empty_option_entry(GtkNotebook * notebook)
{
	GSList *children;
	GSList *child;
	option_widget_info_t *info;

	children = child = get_widget_notebook_page_children(notebook,
							     gtk_notebook_get_current_page
							     (notebook));

	while (child != NULL) {
		if (child->data != NULL && GTK_IS_ENTRY(child->data)) {
			info = gtk_object_get_user_data(GTK_OBJECT(child->data));

			if (info && EVMS_OPTION_IS_ACTIVE(info->option->flags) &&
			    info->has_value == FALSE) {
				gtk_widget_grab_focus(GTK_WIDGET(child->data));
				break;
			}
		}
		child = child->next;
	}

	g_slist_free(children);
}

/*
 *
 *   gint build_options_notebook (GtkWidget *, task_handle_t, gint, gboolean)
 *
 *   Description:
 *      This routine builds the options widget notebook
 *      which is placed in the options frame container
 *      widget of the options/task window.
 *
 *   Entry:
 *      window        - the task/options window id
 *      handle        - the task handle
 *      currpage      - page to be made the current page
 *      show_inactive - TRUE if we should show inactive options
 *
 *   Exit:
 *      Return SUCCESS if the option count was retrieved and
 *      sets up the option notebook. Option list and option
 *      notebook id are also bound to the toplevel window.
 *
 */
gint build_options_notebook(GtkWidget * window, task_handle_t handle, gint currpage,
			    gboolean show_inactive)
{
	gint rc;
	gint option_count;

	rc = evms_get_option_count(handle, &option_count);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_option_count() returned error code %d.\n", __FUNCTION__,
			  rc);

		display_results_window(rc, NULL, _("Problem retrieving plugin option count."),
				       NULL, FALSE, window);
	} else {
		GtkWidget *options_frame;

		log_debug("%s: option count == %d\n", __FUNCTION__, option_count);

		options_frame = gtk_object_get_data(GTK_OBJECT(window), "options_frame");

		if (option_count > 0) {
			GSList *widgets = NULL;
			GSList *options = NULL;
			GtkWidget *notebook;

			create_option_widget_list(handle, option_count, &options, &widgets,
						  show_inactive);

			if (widgets) {
				notebook = create_widget_notebook(widgets, NULL, 6, currpage);

				gtk_container_add(GTK_CONTAINER(options_frame), notebook);
				gtk_container_set_border_width(GTK_CONTAINER(notebook), 10);

				gtk_object_set_data(GTK_OBJECT(window), "options_notebook",
						    notebook);

				gtk_widget_show(notebook);

				g_slist_free(widgets);

				bind_option_list_to_toplevel_widget(window, options);

				make_next_button_sensitive_if_necessary(window, options);

				gtk_signal_connect(GTK_OBJECT(notebook), "destroy",
						   GTK_SIGNAL_FUNC(on_options_notebook_destroy),
						   options);

				give_focus_to_first_empty_option_entry(GTK_NOTEBOOK(notebook));

			} else {
				indicate_plugin_supplied_no_options(options_frame);
			}
		} else {
			indicate_plugin_supplied_no_options(options_frame);
		}
	}

	return rc;
}

/*
 *
 *   gboolean rebuild_options_notebook (GtkWidget *)
 *
 *   Description:
 *      This routine handles destroying the options notebook and building
 *      a new one. It is typically used invoked as an idle function so
 *      that we don't destroy the widgets or notebook while in the middle
 *      of a options widget signal handler invocation which causes bad
 *      side-effects.
 *
 *   Entry:
 *      notebook - the id of the options notebook
 *
 *   Exit:
 *      Always return FALSE to remove this function from being called
 *      again as an idle function.
 *
 */
gboolean rebuild_options_notebook(GtkWidget * toplevel)
{
	GtkWidget *notebook;

	gdk_threads_enter();

	notebook = lookup_widget(toplevel, "options_notebook");

	if (notebook) {
		gint currpage;
		GSList *options;
		GtkWidget *checkbox;
		task_handle_t task;

		log_debug("%s: Rebuilding the options notebook now. notebook == %p.\n",
			  __FUNCTION__, notebook);

		checkbox =
		    gtk_object_get_data(GTK_OBJECT(toplevel), "inactive_options_checkbutton");
		task = retrieve_task_handle_from_toplevel_widget(toplevel);
		options = retrieve_option_list_from_toplevel_widget(toplevel);
		currpage = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));

		/*
		 * Destroy the options notebook and free the original option
		 * descriptor list. Note that the old option descriptors
		 * will be freed through evms_free() when each widget with
		 * an associated option descriptor is given the "destroy"
		 * signal.
		 *
		 * HACK! HACK! Walk the widget list ourselves to do the
		 * gtk_widget_destroy. If the widget is a spinbutton,
		 * check to see if it has a timeout function and if
		 * so remove it so we avoid problems later on.
		 *
		 * See bugzilla.gnome.org bug report #62539 reported by
		 * me (lnx1138@us.ibm.com) that details the problem.
		 */

		destroy_option_widgets(options);
		gtk_widget_destroy(notebook);

		build_options_notebook(toplevel, task, currpage,
				       gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkbox)));

		/*
		 * Clear the handler id as we are done.
		 */
		gtk_object_set_data(GTK_OBJECT(toplevel), "rebuild_options_handler_id",
				    GUINT_TO_POINTER(0));
	}

	gdk_threads_leave();

	return FALSE;
}

/*
 *
 *   void on_display_inactive_options_check_button_toggled (GtkToggleButton *, GtkWidget *window)
 *
 *   Description:
 *      This routine gets called when an the display inactive options
 *      checkbutton is toggled. We schedule a rebuild of the options
 *      notebook. The rebuild routine checks the current state of the
 *      this checkbox to decide whether to display inactive options in
 *      the options notebook.
 *
 *   Entry:
 *      button - the id of the toggle button that was pressed
 *      window - the id of the toplevel window
 *
 *   Exit:
 *      The rebuild_options_notebook is scheduled for execution on the
 *      next idle loop time.
 *
 */
void on_display_inactive_options_check_button_toggled(GtkToggleButton * button, GtkWidget * window)
{
	gtk_idle_add((GtkFunction) rebuild_options_notebook, window);
}

/*
 *
 *   void options_window_callout (option_widget_info_t *, value_t, task_effect_t)
 *
 *   Description:
 *      This routine is the callout function called by the option widgets
 *      to allow us to handle management of the task/options window. This
 *      includes rebuilding the options notebook if EVMS_Effect_Reload_Options
 *      is received. We also, run through all the required options to see
 *      if they have been set in order to make the "Next" button sensitive.
 *
 *   Entry:
 *      info            - contains options descriptor, current value, and state
 *      value           - new value
 *      set_side_effect - flags of sideaffects from evms_set_option_value()
 *
 *   Exit:
 *      Nothing returned but either options notebook is rebuilt or "Next"
 *      button sensitivity is changed.
 *
 */
void options_window_callout(option_widget_info_t * info, value_t value,
			    task_effect_t set_side_effect)
{
	if (set_side_effect & EVMS_Effect_Reload_Options) {
		guint handler_id;
		GtkWidget *toplevel;

		toplevel = gtk_widget_get_toplevel(info->widget);

		handler_id =
		    GPOINTER_TO_UINT(gtk_object_get_data
				     (GTK_OBJECT(toplevel), "rebuild_options_handler_id"));

		if (handler_id == 0) {
			handler_id = gtk_idle_add((GtkFunction) rebuild_options_notebook, toplevel);

			gtk_object_set_data(GTK_OBJECT(toplevel), "rebuild_options_handler_id",
					    GUINT_TO_POINTER(handler_id));

			log_debug
			    ("%s: Registered idle function to rebuild options notebook %p. handler id = %u.\n",
			     __FUNCTION__, toplevel, handler_id);
		} else {
			log_debug("%s: Rebuild of options notebook already pending.\n",
				  __FUNCTION__);
		}
	} else {
		GSList *options;

		options = retrieve_option_list_from_toplevel_widget(info->widget);
		make_next_button_sensitive_if_necessary(info->widget, options);
	}
}

/*
 *
 *   void populate_clist_with_resulting_objects (GtkCList *, handle_array_t *)
 *
 *   Description:
 *      This routine populates the given resulting objects clist with
 *      selected information about the object.
 *
 *   Entry:
 *      objects - handle array contining objects produced
 *
 *   Exit:
 *     The clist is populated with information on the resulting objects.
 *
 */
void populate_clist_with_resulting_objects(GtkCList * clist, handle_array_t * objects)
{
	gint i;
	gint rc;
	gint row;
	gchar *text[MAX_RO_COLUMNS];
	handle_object_info_t *object;

	for (i = 0; i < objects->count; i++) {
		rc = evms_get_info(objects->handle[i], &object);

		if (rc != SUCCESS) {
			log_error("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
		} else {
			text[RO_ICON_COLUMN] = "";

			switch (object->type) {
			case EVMS_OBJECT:
			case REGION:
			case SEGMENT:
			case DISK:
				text[RO_SIZE_COLUMN] =
				    make_sectors_readable_string(object->info.object.size);
				text[RO_NAME_COLUMN] = object->info.object.name;
				text[RO_TYPE_COLUMN] =
				    make_data_type_readable_string(object->info.object.data_type);
				break;

			case CONTAINER:
				text[RO_SIZE_COLUMN] =
				    make_sectors_readable_string(object->info.container.size);
				text[RO_NAME_COLUMN] = object->info.container.name;
				text[RO_TYPE_COLUMN] = _("Container");
				break;

			default:
				text[RO_SIZE_COLUMN] = g_strdup(_("N/A"));
				text[RO_NAME_COLUMN] = _("Unknown");
				text[RO_TYPE_COLUMN] = _("Unknown");
				log_debug("%s: Unhandled type %d.\n", __FUNCTION__, object->type);
				break;
			}

			row = clist_append_row(clist, text);

			if (row != -1)
				set_clist_row_pixmap(clist, row, object->type);

			g_free(text[RO_SIZE_COLUMN]);
			evms_free(object);
		}
	}
}

/*
 *
 *   void populate_clist_with_declined_objects (GtkCList *, declined_handle_array_t *)
 *
 *   Description:
 *      This routine populates the given declined objects clist with
 *      the name of the declined object and the declination reason.
 *
 *   Entry:
 *      declined_list - handle array containing declined objects
 *                      with the corresponding reason codes.
 *
 *   Exit:
 *     The clist is populated with information on the declined objects.
 *
 */
void populate_clist_with_declined_objects(GtkCList * clist, declined_handle_array_t * declined_list)
{
	gint i;
	gint rc;
	gint row;
	gchar *text[MAX_DO_COLUMNS];
	handle_object_info_t *object;

	for (i = 0; i < declined_list->count; i++) {
		rc = evms_get_info(declined_list->declined[i].handle, &object);

		if (rc != SUCCESS) {
			log_error("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
		} else {
			text[DO_ICON_COLUMN] = "";
			text[DO_REASON_COLUMN] =
			    (gchar *) evms_strerror(ABS(declined_list->declined[i].reason));

			switch (object->type) {
			case VOLUME:
				text[DO_NAME_COLUMN] = object->info.volume.name;
				break;

			case EVMS_OBJECT:
			case REGION:
			case SEGMENT:
			case DISK:
				text[DO_NAME_COLUMN] = object->info.object.name;
				break;

			case CONTAINER:
				text[DO_NAME_COLUMN] = object->info.container.name;
				break;

			default:
				text[DO_NAME_COLUMN] = _("Unknown");
				log_debug("%s: Unhandled type %d.\n", __FUNCTION__, object->type);
				break;
			}

			row = clist_append_row(clist, text);

			if (row != -1)
				set_clist_row_pixmap(clist, row, object->type);

			evms_free(object);
		}
	}
}

/*
 *
 *   void display_declination_window (declined_handle_array_t *)
 *
 *   Description:
 *      This routine displays a dialog window containing
 *      a read-only list of selection objects declined
 *      by the plugin.
 *
 *   Entry:
 *      declined_list - handle array containing declined objects
 *                      with the corresponding reason codes.
 *
 *   Exit:
 *      The window is displayed with the list of declined
 *      objects and the declination reasons.
 *
 */
void display_declination_window(declined_handle_array_t * declined_list)
{
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *text_box_frame;
	GtkWidget *scrolledwindow;
	GtkWidget *clist;
	GtkWidget *icon_col_label;
	GtkWidget *object_col_label;
	GtkWidget *reason_col_label;
	GtkWidget *hseparator;
	GtkWidget *hbuttonbox;
	GtkWidget *button;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkAccelGroup *accel_group;
	GtkTooltips *tooltips;

	tooltips = gtk_tooltips_new();
	accel_group = gtk_accel_group_new();
	window = gtk_window_new(GTK_WINDOW_DIALOG);
	vbox = gtk_vbox_new(FALSE, 0);
	text_box_frame = gtk_frame_new(NULL);
	scrolledwindow = gtk_scrolled_window_new(NULL, NULL);
	clist = gtk_clist_new(MAX_DO_COLUMNS);
	icon_col_label = gtk_label_new("");
	object_col_label = gtk_label_new(_("Storage Object"));
	reason_col_label = gtk_label_new(_("Declination Reason"));
	hseparator = gtk_hseparator_new();
	hbuttonbox = gtk_hbutton_box_new();
	button = gtk_button_new();

	gtk_accel_group_attach(accel_group, GTK_OBJECT(window));

	get_ok_pixmap(&pixmap, &mask);
	add_pixmap_label_to_button(button, pixmap, mask, _("_OK"), accel_group);

	gtk_widget_show(vbox);
	gtk_widget_show(text_box_frame);
	gtk_widget_show(scrolledwindow);
	gtk_widget_show(clist);
	gtk_widget_show(icon_col_label);
	gtk_widget_show(object_col_label);
	gtk_widget_show(reason_col_label);
	gtk_widget_show(hseparator);
	gtk_widget_show(hbuttonbox);
	gtk_widget_show(button);

	gtk_window_set_title(GTK_WINDOW(window), _("Selections Declined"));
	gtk_window_set_default_size(GTK_WINDOW(window), 475, -1);

	gtk_container_add(GTK_CONTAINER(window), vbox);

	gtk_box_pack_start(GTK_BOX(vbox), text_box_frame, TRUE, TRUE, 0);
	gtk_widget_set_usize(text_box_frame, 213, 300);
	gtk_container_set_border_width(GTK_CONTAINER(text_box_frame), 6);

	gtk_container_add(GTK_CONTAINER(text_box_frame), scrolledwindow);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow),
				       GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);

	gtk_container_add(GTK_CONTAINER(scrolledwindow), clist);
	gtk_clist_set_column_width(GTK_CLIST(clist), DO_ICON_COLUMN, 24);
	gtk_clist_set_column_width(GTK_CLIST(clist), DO_NAME_COLUMN, 151);
	gtk_clist_set_column_width(GTK_CLIST(clist), DO_REASON_COLUMN, 80);
	gtk_clist_column_titles_show(GTK_CLIST(clist));

	gtk_clist_set_column_widget(GTK_CLIST(clist), DO_ICON_COLUMN, icon_col_label);
	gtk_clist_set_column_widget(GTK_CLIST(clist), DO_NAME_COLUMN, object_col_label);
	gtk_clist_set_column_widget(GTK_CLIST(clist), DO_REASON_COLUMN, reason_col_label);

	gtk_box_pack_start(GTK_BOX(vbox), hseparator, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbuttonbox, FALSE, FALSE, 0);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbuttonbox), 18);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbuttonbox), 108, 46);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);

	populate_clist_with_declined_objects(GTK_CLIST(clist), declined_list);

	gtk_tooltips_set_tip(tooltips, button, _("Close this window"), NULL);

	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  gtk_widget_destroy, GTK_OBJECT(window));

	gtk_signal_connect(GTK_OBJECT(clist), "select_row",
			   GTK_SIGNAL_FUNC(on_readonly_clist_select_row), NULL);

	gtk_widget_show(window);
}

/*
 *
 *   GtkWidget* create_resulting_objects_window (handle_array_t *)
 *
 *   Description:
 *      This routine displays a results window with a list of
 *      objects produced by the operation.
 *
 *   Entry:
 *      objects - handle array containing objects produced
 *
 *   Exit:
 *      The window is created and the list of objects is populated.
 *
 */
GtkWidget *create_resulting_objects_window(handle_array_t * objects)
{
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *frame;
	GtkWidget *scrolled_window;
	GtkWidget *clist;
	GtkWidget *label;
	GtkWidget *hseparator;
	GtkWidget *hbuttonbox;
	GtkWidget *button;
	GtkWidget *pixmap;
	GdkBitmap *mask;
	GdkPixmap *gdk_pixmap;
	GtkAccelGroup *accel_group;
	GtkTooltips *tooltips;

	get_dialog_pixmap(INFO_PIXMAP, &gdk_pixmap, &mask);

	accel_group = gtk_accel_group_new();
	tooltips = gtk_tooltips_new();
	window = gtk_window_new(GTK_WINDOW_DIALOG);
	vbox = gtk_vbox_new(FALSE, 0);
	hbox = gtk_hbox_new(FALSE, 10);
	frame = gtk_frame_new(NULL);
	scrolled_window = gtk_scrolled_window_new(NULL, NULL);
	clist = gtk_clist_new(MAX_RO_COLUMNS);
	label =
	    gtk_label_new(_
			  ("The operation completed successfully producing the objects listed below:"));
	hseparator = gtk_hseparator_new();
	hbuttonbox = gtk_hbutton_box_new();
	button = gtk_button_new();
	pixmap = gtk_pixmap_new(gdk_pixmap, mask);

	gtk_accel_group_attach(accel_group, GTK_OBJECT(window));

	get_ok_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(button, gdk_pixmap, mask, _("_OK"), accel_group);

	gtk_window_set_title(GTK_WINDOW(window), _("Operation Completed Successfully"));
	gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);
	gtk_window_set_default_size(GTK_WINDOW(window), 475, -1);

	gtk_widget_show(vbox);
	gtk_widget_show(frame);
	gtk_widget_show(scrolled_window);
	gtk_widget_show(clist);
	gtk_widget_show(label);
	gtk_widget_show(hseparator);
	gtk_widget_show(hbuttonbox);
	gtk_widget_show(button);
	gtk_widget_show(pixmap);
	gtk_widget_show(hbox);

	gtk_container_add(GTK_CONTAINER(window), vbox);

	gtk_box_pack_start(GTK_BOX(hbox), pixmap, FALSE, FALSE, 8);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 10);

	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
	gtk_widget_set_usize(frame, 213, 300);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 3);

	gtk_container_add(GTK_CONTAINER(frame), scrolled_window);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
				       GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	gtk_container_add(GTK_CONTAINER(scrolled_window), clist);
	gtk_container_set_border_width(GTK_CONTAINER(clist), 12);
	gtk_clist_set_column_width(GTK_CLIST(clist), RO_ICON_COLUMN, 24);
	gtk_clist_set_column_width(GTK_CLIST(clist), RO_SIZE_COLUMN, 61);
	gtk_clist_set_column_width(GTK_CLIST(clist), RO_TYPE_COLUMN, 80);
	gtk_clist_set_column_width(GTK_CLIST(clist), RO_NAME_COLUMN, 80);

	label = gtk_label_new("");
	gtk_widget_show(label);
	gtk_clist_set_column_widget(GTK_CLIST(clist), RO_ICON_COLUMN, label);

	label = gtk_label_new("Size");
	gtk_widget_show(label);
	gtk_clist_set_column_widget(GTK_CLIST(clist), RO_SIZE_COLUMN, label);

	label = gtk_label_new("Type");
	gtk_widget_show(label);
	gtk_clist_set_column_widget(GTK_CLIST(clist), RO_TYPE_COLUMN, label);

	label = gtk_label_new("Name");
	gtk_widget_show(label);
	gtk_clist_set_column_widget(GTK_CLIST(clist), RO_NAME_COLUMN, label);

	gtk_clist_column_titles_show(GTK_CLIST(clist));

	gtk_box_pack_start(GTK_BOX(vbox), hseparator, FALSE, FALSE, 20);

	gtk_box_pack_start(GTK_BOX(vbox), hbuttonbox, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbuttonbox), 13);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbuttonbox), 9);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbuttonbox), 118, 47);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), button);
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);

	gtk_tooltips_set_tip(tooltips, button, _("Close this window"), NULL);

	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
				  gtk_widget_destroy, GTK_OBJECT(window));

	gtk_signal_connect(GTK_OBJECT(clist), "select_row",
			   GTK_SIGNAL_FUNC(on_readonly_clist_select_row), NULL);

	populate_clist_with_resulting_objects(GTK_CLIST(clist), objects);

	return window;
}

/*
 *
 *   void unschedule_pending_options_notebook_rebuild (GtkWidget *)
 *
 *   Description:
 *      This routine removes the idle function that is scheduled
 *      to rebuild the options notebook.
 *
 *   Entry:
 *      widget - a child widget of the toplevel widget
 *
 *   Exit:
 *      Rebuild of options notebook is unscheduled if it existed.
 *
 */
void unschedule_pending_options_notebook_rebuild(GtkWidget * widget)
{
	guint handler_id;
	GtkWidget *toplevel;

	toplevel = gtk_widget_get_toplevel(widget);

	handler_id =
	    GPOINTER_TO_UINT(gtk_object_get_data
			     (GTK_OBJECT(toplevel), "rebuild_options_handler_id"));

	if (handler_id) {
		gtk_idle_remove(handler_id);

		gtk_object_set_data(GTK_OBJECT(toplevel), "rebuild_options_handler_id",
				    GUINT_TO_POINTER(0));

		log_debug("%s: Unregistered pending options notebook rebuild. handle id = %u.\n",
			  __FUNCTION__, handler_id);
	}
}

/*
 *
 *   inline gint invoke_evms_task (task_handle_t, handle_array_t **)
 *
 *   Description:
 *      This routine attempts to create a thread to process
 *      a call to evms_invoke_task(). Having evms_invoke_task()
 *      run on it's own thread has the following advantages:
 *
 *        1. Allows us to start an inner event loop and handle
 *           GUI related events such as window repaints, etc.
 *
 *        2. Allows alert messages to get immediately displayed
 *           (not delayed until returned from call)
 *
 *        3. Allows progress reports from plugin to get immediately
 *           updated.
 *
 *      If the thread can not be started, we instead simply call
 *      evms_invoke_task() directly.
 *
 *   Entry:
 *      task              - the task handle
 *      resulting_objects - the address to store the handle array pointer
 *                          for any resulting objects from task
 *
 *   Exit:
 *      The return code from evms_invoke_task() is returned.
 *
 */
inline gint invoke_evms_task(task_handle_t task, handle_array_t ** resulting_objects)
{
	gint rc;
	pthread_t tid;
	evmsgui_task_info_t *info;

	info = g_new0(evmsgui_task_info_t, 1);
	info->task = task;

	rc = pthread_create(&tid, NULL, task_thread, info);

	if (rc == SUCCESS) {
		gtk_main();

		rc = info->rc;
		*resulting_objects = info->resulting_objects;
	} else {
		rc = evms_invoke_task(task, resulting_objects);
	}

	g_free(info);

	return rc;
}

/*
 *
 *   void on_invoke_task_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine calls the function that initiates the
 *      task operation (whatever that may be). It then
 *      displays a results screen.
 *
 *   Entry:
 *      button    - the button that was clicked
 *      user_data - contains the task handle
 *
 *   Exit:
 *      Nothing is returned but the results screen will destroy
 *      all the windows in the window list.
 *
 */
void on_invoke_task_button_clicked(GtkButton * button, gpointer user_data)
{
	gint rc;
	gchar *message;
	GList *window_list = NULL;
	task_handle_t task;
	handle_array_t *resulting_objects = NULL;

	task = GPOINTER_TO_UINT(user_data);

	gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

#ifdef EVMS_DEBUG
	rc = evms_invoke_task(task, &resulting_objects);
#else
	rc = invoke_evms_task(task, &resulting_objects);
#endif

	gtk_widget_set_sensitive(GTK_WIDGET(button), TRUE);

	if (rc != SUCCESS) {
		log_error("%s: evms_invoke_task() returned error code %d.\n", __FUNCTION__, rc);
		message = _("The operation completed with an error.");
	} else {
		task_action_t action;

		if (evms_get_task_action(task, &action) == SUCCESS) {
			/*
			 * Make the message clearer for certain operations
			 * such as file system operations, e.g. FSCK and
			 * MKFS that the operation is scheduled to be done
			 * when the user commits changes.
			 */

			switch (action) {
			case EVMS_Task_mkfs:
			case EVMS_Task_fsck:
			case EVMS_Task_defrag:
				message =
				    _
				    ("The operation has been scheduled for completion when changes are saved.");
				break;

			default:
				message = _("The operation completed successfully.");
				break;
			}
		} else
			message = _("The operation completed successfully.");

		window_list = get_window_list(GTK_WIDGET(button));
	}

	/*
	 * If the operation was a success then the window list
	 * will be destroyed. As part of this, the window that
	 * created the task will have destroyed the task upon
	 * invocation of its destroy function, therefore we
	 * make no explicit evms_destroy_task() call here.
	 *
	 * Also note that if the operation errored then the
	 * task context is still active and the previous
	 * dialog is still visible. This is to allow the
	 * user the chance to retry the operation or cancel
	 * it at their discretion.
	 */

	if (rc == SUCCESS) {
		/*
		 * Since we already successfully invoked the task and are about
		 * to destroy the window list and the task, make sure we abort
		 * any pending options notebook rebuild idle function executions.
		 */

		unschedule_pending_options_notebook_rebuild(GTK_WIDGET(button));

		/*
		 * If this task produced some objects then display some
		 * info on them. Otherwise, display the boring results
		 * window.
		 */

		if (resulting_objects) {
			if (resulting_objects->count > 0) {
				GtkWidget *window;

				window = create_resulting_objects_window(resulting_objects);

				complete_display_results_window_work(window, message, window_list,
								     TRUE);

				gtk_widget_show(window);
			} else {
				display_results_window(rc, NULL, message, window_list, TRUE, NULL);
			}

			evms_free(resulting_objects);
		} else {
			display_results_window(rc, NULL, message, window_list, TRUE, NULL);
		}

		evms_destroy_task(task);
	} else {
		display_results_window(rc, NULL, message, window_list, FALSE,
				       gtk_widget_get_toplevel(GTK_WIDGET(button)));
	}
}

/*
 *
 *   void on_acceptable_objects_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine does all the relevant checks to see if the task window
 *      containing options should be created or redisplayed. For certain tasks,
 *      we need to create a target selection list of target acceptable objects
 *      (like for move). Normally, we move on to the task window where options
 *      for the source selection will be available. We also check now and save
 *      ourselves the grief if there are no options or none of them are active
 *      so that the we simply call on_invoke_task_button_clicked() to get the
 *      operation done with.
 *
 *   Entry:
 *      button    - the button that was clicked
 *      user_data - contains the task handle
 *
 *   Exit:
 *
 */
void on_acceptable_objects_button_clicked(GtkButton * button, gpointer user_data)
{
	GSList *selections;
	GtkWidget *clist;
	task_handle_t task;

	task = GPOINTER_TO_UINT(user_data);

	clist = lookup_widget(GTK_WIDGET(button), "selection_window_clist");
	selections = get_clist_selection_data(GTK_CLIST(clist));

	if (selections != NULL) {
		GList *window_list;
		GtkWidget *curr_next_window;
		GtkWidget *selection_window;
		handle_array_t *curr_selections;
		handle_array_t *prev_selections;

		selection_window = gtk_widget_get_toplevel(GTK_WIDGET(button));

		window_list = get_window_list(GTK_WIDGET(selection_window));
		curr_next_window =
		    gtk_object_get_data(GTK_OBJECT(selection_window), "next_window_id");
		prev_selections =
		    gtk_object_get_data(GTK_OBJECT(selection_window), "prev_selections");

		curr_selections = make_handle_array_from_selection_list(selections);

		if (are_handle_arrays_equal(curr_selections, prev_selections) == FALSE) {
			gint count;

			/*
			 * Decide a) if we need another window or can just call invoke task
			 * b) the next window should be the target list or c) the next window
			 * should be the task/options window.
			 */

			if (evms_get_option_count(task, &count) == SUCCESS && count == 0) {
				on_invoke_task_button_clicked(button, user_data);
			} else {
				gchar *next_button_text;
				GtkWidget *next_window;

				next_button_text =
				    gtk_object_get_data(GTK_OBJECT(selection_window),
							"function_verb");

				if (next_button_text == NULL)
					next_button_text = get_task_action_string(task);

				next_window = create_standard_task_window("Configuration Options",
									  next_button_text,
									  NULL,
									  on_button_clicked_display_prev_window,
									  on_button_clicked_destroy_window_list,
									  task);

				if (next_window != NULL) {
					/*
					 * Ensure that we remove all windows starting from
					 * the current next window. Once that is done, add
					 * the new next window to the window list.
					 */

					if (curr_next_window != NULL)
						destroy_window_list(g_list_find
								    (window_list,
								     curr_next_window));

					window_list = g_list_append(window_list, next_window);

					set_window_list(next_window, window_list);
					set_window_list(selection_window, window_list);

					gtk_widget_show(next_window);
					gtk_widget_hide(selection_window);

					bind_options_callout_to_toplevel_widget(next_window,
										options_window_callout);

					/*
					 * Keep track of the current selections. If this window should
					 * get destroyed, make sure we free up the selection list.
					 */

					gtk_object_set_data(GTK_OBJECT(selection_window),
							    "prev_selections", curr_selections);
					gtk_object_set_data(GTK_OBJECT(selection_window),
							    "next_window_id", next_window);

					gtk_signal_connect(GTK_OBJECT(next_window), "destroy",
							   GTK_SIGNAL_FUNC
							   (on_window_destroy_free_selection_array),
							   curr_selections);
				}

				g_free(next_button_text);
			}
		} else {
			g_free(curr_selections);
			gtk_widget_show(curr_next_window);
			gtk_widget_hide(selection_window);
		}

		g_slist_free(selections);
	} else
		log_error("%s: No acceptable objects selected!\n", __FUNCTION__);
}

/*
 *
 *   void reload_acceptable_objects (GtkCList *, task_handle_t, gint *)
 *
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of things that are considered acceptable by the plugin
 *      and then selects the rows in the list corresponding to
 *      the objects in the selection list provided by the plugin.
 *
 *   Entry:
 *      clist - address of the acceptable objects GtkCList
 *      task  - the task context handle
 *      count - count of rows selected to be updated by us
 *
 *   Exit:
 *      Task acceptable objects are populated and selected
 *      according to the current task context information.
 *
 */
void reload_acceptable_objects(GtkCList * clist, task_handle_t task, gint * count)
{
	gint rc;
	handle_array_t *acceptable_list;
	handle_array_t *selected_list;

	log_entry;

	rc = evms_get_acceptable_objects(task, &acceptable_list);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_acceptable_objects() returned error code %d.\n",
			  __FUNCTION__, rc);
	} else {
		guint i;
		guint min;
		guint max;

		gtk_signal_handler_block_by_func(GTK_OBJECT(clist),
						 GTK_SIGNAL_FUNC
						 (on_acceptable_objects_clist_select_row),
						 GUINT_TO_POINTER(task));

		gtk_signal_handler_block_by_func(GTK_OBJECT(clist),
						 GTK_SIGNAL_FUNC
						 (on_acceptable_objects_clist_unselect_row),
						 GUINT_TO_POINTER(task));

		gtk_clist_freeze(clist);
		gtk_clist_clear(clist);

		rc = evms_get_selected_object_limits(task, &min, &max);

		if (rc == SUCCESS)
			gtk_clist_set_selection_mode(clist,
						     ((min != 1
						       || max !=
						       1) ? GTK_SELECTION_MULTIPLE :
						      GTK_SELECTION_SINGLE));

		for (i = 0; i < acceptable_list->count; i++)
			add_thing_to_selection_list(clist, acceptable_list->handle[i], FALSE);

		rc = evms_get_selected_objects(task, &selected_list);

		if (rc != SUCCESS) {
			log_error("%s: evms_get_selected_objects () returned error code %d.\n",
				  __FUNCTION__, rc);
		} else {
			gint row;
			GSList *selections;

			/*
			 * Find the row corresponding to each object handle in the
			 * selected list and mark those rows as selected in the
			 * order provided by the list.
			 */

			for (i = 0; i < selected_list->count; i++) {
				row = gtk_clist_find_row_from_data(clist,
								   GUINT_TO_POINTER(selected_list->
										    handle[i]));

				if (row != -1)
					gtk_clist_select_row(clist, row, 0);
			}

			/*
			 * Determine if anything selected make Next button
			 * insensitive if now below our selection minimum.
			 */

			selections = get_clist_selection_data(clist);

			if (selections) {
				*count = g_slist_length(selections);

				if (*count < min || *count == 0)
					inactivate_next_button(GTK_WIDGET(clist));

				g_slist_free(selections);
			} else {
				*count = 0;
				inactivate_next_button(GTK_WIDGET(clist));
			}

			evms_free(selected_list);
		}

		gtk_clist_thaw(clist);

		gtk_signal_handler_unblock_by_func(GTK_OBJECT(clist),
						   GTK_SIGNAL_FUNC
						   (on_acceptable_objects_clist_select_row),
						   GUINT_TO_POINTER(task));

		gtk_signal_handler_unblock_by_func(GTK_OBJECT(clist),
						   GTK_SIGNAL_FUNC
						   (on_acceptable_objects_clist_unselect_row),
						   GUINT_TO_POINTER(task));

		evms_free(acceptable_list);
	}

	log_exit;
}

/*
 *
 *   gint update_selected_objects (GtkCList *, task_handle_t, GSList *, gint *)
 *
 *   Description:
 *      This routine calls the engine to update the selected objects
 *      list. It handles any side-effects from the operation such
 *      dealing with declined lists or having to reload the acceptable
 *      objects.
 *
 *   Entry:
 *      clist      - the selections window GtkCList *
 *      task       - handle for the task at hand
 *      selections - linked list of object handles corresponding to
 *                   row selections made (in the order selected)
 *      count      - address of count of rows selected to update if necessary
 *
 *   Exit:
 *      Selected objects relayed to the plugin.
 *
 */
gint update_selected_objects(GtkCList * clist, task_handle_t task, GSList * selections,
			     gint * count)
{
	gint rc;
	task_effect_t effect = 0;
	handle_array_t *handle_array;
	declined_handle_array_t *declined_list = NULL;

	handle_array = make_handle_array_from_selection_list(selections);

	rc = evms_set_selected_objects(task, handle_array, &declined_list, &effect);

	if (rc == SUCCESS) {
		/*
		 * Check to see if we ended up with a declined list or/and
		 * we need to reload our acceptable list and sync with the
		 * plugin on the selections.
		 */

		if ((declined_list && declined_list->count > 0) ||
		    (effect & EVMS_Effect_Reload_Objects)) {
			if (declined_list) {
				display_declination_window(declined_list);
				log_warning("%s: Some of the object selections were declined.\n",
					    __FUNCTION__);

				evms_free(declined_list);
			}

			reload_acceptable_objects(clist, task, count);
		}
	} else {
		gchar *message = _("Unable to tell plugin what our selections were.");

		display_results_window(rc, NULL, message, NULL, FALSE,
				       gtk_widget_get_toplevel(GTK_WIDGET(clist)));

		log_error("%s: evms_set_selected_objects() returned error code %d.\n", __FUNCTION__,
			  rc);
	}

	g_free(handle_array);

	return rc;
}

/*
 *
 *   void on_acceptable_objects_clist_select_row (GtkCList *, gint, gint, GdkEventButton *, gpointer)
 *
 *   Description:
 *      This routine is called whenever a row is selected.
 *      We make sure that we are within the min and max
 *      selectable objects. If we reach the minimum number
 *      of selected objects, we make the "Next" button
 *      sensitive/active. If we attempt to exceed the max
 *      selectable objects, we display a friendly popup
 *      and nullify the select_row signal.
 *
 *      As long as the uselect_row does not cause us to exceed
 *      the maximum selections indicated by the plugin then we
 *      call evms_set_selected_object () to update our selections.
 *
 *   Entry:
 *      clist     - address of GtkCList object that received select-row signal
 *      row       - index of row selected
 *      column    - index of column selected on current row
 *      event     - address of structure to obtain additional context information
 *      user_data - contains task handle
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_acceptable_objects_clist_select_row(GtkCList * clist, gint row, gint column,
					    GdkEventButton * event, gpointer user_data)
{
	gint rc;
	guint min;
	guint max;
	task_handle_t task = GPOINTER_TO_UINT(user_data);

	rc = evms_get_selected_object_limits(task, &min, &max);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_selected_object_limits() returned error code.\n",
			  __FUNCTION__, rc);
	} else {
		GSList *selections;

		selections = get_clist_selection_data(clist);

		if (selections) {
			gint length = g_slist_length(selections);

			log_debug("%s: selection len = %d, min = %d, max = %d\n",
				  __FUNCTION__, length, min, max);

			if (length <= max || length == -1) {
				if (update_selected_objects(clist, task, selections, &length) ==
				    SUCCESS) {
					GtkWidget *next_button;

					next_button = lookup_widget(GTK_WIDGET(clist),
								    "selection_window_next_button");

					if (length >= min
					    && !(GTK_WIDGET_IS_SENSITIVE(next_button))) {
						gtk_widget_set_sensitive(next_button, TRUE);
						gtk_widget_grab_default(next_button);
					}
				}
			} else {
				/*
				 * Ignore the selection as we have exceeded our max
				 */

				gtk_signal_handler_block_by_func(GTK_OBJECT(clist),
								 GTK_SIGNAL_FUNC
								 (on_acceptable_objects_clist_unselect_row),
								 GUINT_TO_POINTER(task));

				gtk_clist_unselect_row(clist, row, column);

				gtk_signal_handler_unblock_by_func(GTK_OBJECT(clist),
								   GTK_SIGNAL_FUNC
								   (on_acceptable_objects_clist_unselect_row),
								   GUINT_TO_POINTER(task));
			}

			g_slist_free(selections);
		} else {
			log_debug("%s: No selections exist.\n", __FUNCTION__);

			inactivate_next_button(GTK_WIDGET(clist));
		}
	}
}

/*
 *
 *   void on_acceptable_objects_clist_unselect_row (GtkCList *, gint, gint, GdkEventButton *, gpointer)
 *
 *   Description:
 *      This routine is called whenever a row is unselected.
 *      If the number of selections falls below the minimum
 *      then we make the "Next" button insensitive.
 *
 *      As long as the unselect_row does not cause us to fall
 *      below the minimum indicated by the plugin then we
 *      call evms_set_selected_object () to update our selections.
 *
 *   Entry:
 *      clist     - address of GtkCList object that received unselect-row signal
 *      row       - index of row deselected
 *      column    - index of column deselected on current row
 *      event     - address of structure to obtain additional context information
 *      user_data - contains tasks handle
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_acceptable_objects_clist_unselect_row(GtkCList * clist, gint row, gint column,
					      GdkEventButton * event, gpointer user_data)
{
	gint rc;
	guint min;
	guint max;
	task_handle_t task = GPOINTER_TO_UINT(user_data);

	rc = evms_get_selected_object_limits(task, &min, &max);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_selected_object_limits() returned error code.\n",
			  __FUNCTION__, rc);
	} else {
		gint length;
		GSList *selections;

		selections = get_clist_selection_data(clist);

		length = g_slist_length(selections);

		log_debug("%s: selection len = %d, min = %d, max = %d\n",
			  __FUNCTION__, length, min, max);

		rc = update_selected_objects(clist, task, selections, &length);

		if (rc == SUCCESS && (length < min || length == 0))
			inactivate_next_button(GTK_WIDGET(clist));
	}
}

/*
 *
 *   void on_acceptable_objects_clist_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of acceptable "things" for a task.
 *
 *   Entry:
 *      widget    - address of the task acceptable parameters GtkCList widget
 *      user_data - address of user data bound with signal
 *
 *   Exit:
 *      Task acceptable parameters list populated with acceptable "things".
 *
 */
void on_acceptable_objects_clist_realize(GtkWidget * widget, gpointer user_data)
{
	gint rc;
	guint min;
	guint max;
	GtkCList *clist = GTK_CLIST(widget);
	task_handle_t task = GPOINTER_TO_UINT(user_data);

	set_selection_window_clist_column_titles(clist, _("Size"), _("Storage Object"), NULL);

	rc = evms_get_selected_object_limits(task, &min, &max);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_selected_object_limits() returned error code %d.\n",
			  __FUNCTION__, rc);
	} else {
		gint count = 0;

		/*
		 * Load objects. Select those that are part of the selected objects list.
		 * This could be because the creator of the selection window may have
		 * already pre-selected a set of acceptable objects.
		 */
		reload_acceptable_objects(clist, task, &count);

		if (count == 0 && clist->rows == 1) {
			gtk_clist_select_row(clist, 0, 0);
			on_acceptable_objects_clist_select_row(clist, 0, 0, NULL, user_data);
		} else if (count >= min) {
			GtkWidget *next_button;

			next_button = lookup_widget(GTK_WIDGET(clist),
						    "selection_window_next_button");

			if (!(GTK_WIDGET_IS_SENSITIVE(next_button))) {
				gtk_widget_set_sensitive(next_button, TRUE);
				gtk_widget_grab_default(next_button);
			}
		}
	}
}

/*
 *
 *   void set_task_window_instructions_label (GtkWidget *, gchar *)
 *
 *   Description:
 *      This routine sets the text in the task dialog window that instructs
 *      the user what to do.
 *
 *   Entry:
 *      window           - the task dialog window id
 *      next_button_text - the text to be displayed on the "Next" button
 *
 *   Exit:
 *      The instructions label text is set.
 *
 */
void set_task_window_instructions_label(GtkWidget * window, gchar * next_button_text)
{
	gchar *instruction_label_text;
	GtkLabel *button_label;
	GtkLabel *instruction_label;

	button_label = gtk_object_get_data(GTK_OBJECT(window),
					   "selection_window_next_button_label");

	instruction_label = gtk_object_get_data(GTK_OBJECT(window),
						"selection_window_instruction_label");

	instruction_label_text =
	    g_strdup_printf(_
			    ("Make your configuration choices then select %s to complete the operation"),
			    next_button_text);

	gtk_label_set_text(button_label, next_button_text);
	gtk_label_set_text(instruction_label, instruction_label_text);

	g_free(instruction_label_text);

}

/*
 *
 *   GtkWidget* create_standard_task_window (gchar *, gchar *
 *                                           GtkSignalFunc, GtkSignalFunc,
 *                                           GtkSignalFunc,
 *                                           task_handle_t)
 *
 *   Description:
 *      This routine creates the generic task dialog and connects the
 *      appropriate signal handlers that will initiate the operation actions.
 *
 *   Entry:
 *      window_title             - title for the selection window
 *      next_button_text         - label text for next button (default is "Next")
 *      next_button_sighandler   - function to call when next button clicked
 *      prev_button_sighandler   - function to call when prev button clicked
 *      cancel_button_sighandler - function to call when cancel button clicked
 *      handle                   - handle for specific task
 *
 *   Exit:
 *      Generic task dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
GtkWidget *create_standard_task_window(gchar * window_title, char *next_button_text,
				       GtkSignalFunc next_button_sighandler,
				       GtkSignalFunc prev_button_sighandler,
				       GtkSignalFunc cancel_button_sighandler, task_handle_t handle)
{
	GtkWidget *window;

	window = create_task_window();

	if (window != NULL) {
		if (build_options_notebook(window, handle, 0, FALSE) == SUCCESS) {
			GtkWidget *asterik_label;
			GtkWidget *cancel_button;
			GtkWidget *prev_button;
			GtkWidget *next_button;
			GtkWidget *checkbox;

			bind_task_handle_to_toplevel_widget(window, handle);

			prev_button =
			    gtk_object_get_data(GTK_OBJECT(window), "selection_window_prev_button");
			next_button =
			    gtk_object_get_data(GTK_OBJECT(window), "selection_window_next_button");
			cancel_button =
			    gtk_object_get_data(GTK_OBJECT(window),
						"selection_window_cancel_button");
			checkbox =
			    gtk_object_get_data(GTK_OBJECT(window), "inactive_options_checkbutton");
			asterik_label = gtk_object_get_data(GTK_OBJECT(window), "asterik_label");

			set_widget_foreground_color(asterik_label, "red");

			if (window_title != NULL)
				gtk_window_set_title(GTK_WINDOW(window), window_title);

			if (next_button_text != NULL)
				set_task_window_instructions_label(window, next_button_text);

			/*
			 * Initialize the "Cancel" button to dismiss the window
			 * when no signal handler is provided. Setup the "Next"
			 * button with the signal handler given.
			 */

			if (next_button_sighandler != NULL)
				gtk_signal_connect(GTK_OBJECT(next_button), "clicked",
						   next_button_sighandler,
						   GUINT_TO_POINTER(handle));
			else
				gtk_signal_connect(GTK_OBJECT(next_button), "clicked",
						   on_invoke_task_button_clicked,
						   GUINT_TO_POINTER(handle));

			if (prev_button_sighandler != NULL)
				gtk_signal_connect(GTK_OBJECT(prev_button), "clicked",
						   prev_button_sighandler,
						   GUINT_TO_POINTER(handle));
			else
				gtk_widget_hide(prev_button);

			if (cancel_button_sighandler != NULL)
				gtk_signal_connect(GTK_OBJECT(cancel_button), "clicked",
						   cancel_button_sighandler,
						   GUINT_TO_POINTER(handle));
			else
				gtk_signal_connect(GTK_OBJECT(cancel_button), "clicked",
						   on_button_clicked_destroy_window, NULL);

			gtk_signal_connect(GTK_OBJECT(checkbox), "toggled",
					   GTK_SIGNAL_FUNC
					   (on_display_inactive_options_check_button_toggled),
					   window);
			gtk_window_set_transient_for(GTK_WINDOW(window),
						     GTK_WINDOW(get_main_window_id()));
		} else {
			gtk_widget_destroy(window);
			window = NULL;
		}
	}

	return window;
}

/*
 * The following routines are basically getter/setter functions
 * to allow binding and retrieving a task information from a toplevel
 * widget.
 */

inline void bind_task_handle_to_toplevel_widget(GtkWidget * widget, task_handle_t task)
{
	GtkWidget *toplevel = gtk_widget_get_toplevel(widget);

	gtk_object_set_data(GTK_OBJECT(toplevel), "task_handle", GUINT_TO_POINTER(task));
}

inline task_handle_t retrieve_task_handle_from_toplevel_widget(GtkWidget * widget)
{
	GtkWidget *toplevel = gtk_widget_get_toplevel(widget);

	return GPOINTER_TO_UINT(gtk_object_get_data(GTK_OBJECT(toplevel), "task_handle"));
}

inline void bind_option_list_to_toplevel_widget(GtkWidget * widget, GSList * options)
{
	GtkWidget *toplevel = gtk_widget_get_toplevel(widget);

	gtk_object_set_data(GTK_OBJECT(toplevel), "option_widget_info_list", options);
}

inline GSList *retrieve_option_list_from_toplevel_widget(GtkWidget * widget)
{
	GtkWidget *toplevel = gtk_widget_get_toplevel(widget);

	return (GSList *) gtk_object_get_data(GTK_OBJECT(toplevel), "option_widget_info_list");
}


/*
 *
 *   inline gboolean object_is_only_available (task_handle_t, engine_handle_t)
 *
 *   Description:
 *      This routine checks to see if the acceptable objects lists contains one
 *      object and whether the handle matches that of the given handle.
 *
 *   Entry:
 *      task   - the task handle
 *      handle - the handle to check
 *
 *   Exit:
 *      Returns TRUE if there is only one acceptable object for the task and
 *      it matches the handle given.
 *
 */
inline gboolean object_is_only_available(task_handle_t task, engine_handle_t handle)
{
	gboolean result = FALSE;
	handle_array_t *acceptable_list;

	if (evms_get_acceptable_objects(task, &acceptable_list) == SUCCESS &&
	    acceptable_list->count == 1 && acceptable_list->handle[0] == handle) {
		result = TRUE;
	}

	return result;
}

/*
 *
 *   gint create_task (engine_handle_t, task_action_t, task_requirements_t *, task_handle_t *)
 *
 *   Description:
 *      This routine creates a task and determines, given a handle and task action,
 *      whether the task will need to present an acceptable objects selection list
 *      and/or an options configuration panels or none at all.
 *
 *   Entry:
 *      handle       - the handle of the thing typically used to initiate this task
 *      action       - the task function number
 *      requirements - the address to store the returned task requirements
 *      task         - the address to store the returned task handle
 *
 *   Exit:
 *      The task requirements and handle are returned with a successful return code
 *      or an error code otherwise.
 *
 */
gint create_task(engine_handle_t handle, task_action_t action,
		 task_requirements_t * requirements, task_handle_t * task)
{
	gint rc;
	gint objects_min;
	gint objects_max;
	gint option_count;

	*requirements = TASK_REQUIREMENTS_UNKNOWN;

	rc = evms_create_task(handle, action, task);

	if (rc == SUCCESS) {
		rc = evms_get_selected_object_limits(*task, &objects_min, &objects_max);

		if (rc == SUCCESS) {
			rc = evms_get_option_count(*task, &option_count);

			if (rc == SUCCESS) {
				if (objects_max != 0 && option_count == 0) {
					if (objects_min <= 1
					    && object_is_only_available(*task, handle) == TRUE) {
						/*
						 * It looks like we have only one acceptable object and it's the
						 * one for the handle given. Make it the selected object and simply
						 * provide a confirmation dialog for the operation.
						 */
						if (set_selected_object(*task, handle) == SUCCESS)
							*requirements =
							    TASK_REQUIRES_NO_ADDITIONAL_ARGS;
						else
							*requirements = TASK_REQUIRES_OBJECTS_ONLY;
					} else
						*requirements = TASK_REQUIRES_OBJECTS_ONLY;
				} else if (objects_max == 0 && option_count > 0)
					*requirements = TASK_REQUIRES_OPTIONS_ONLY;
				else if (objects_max != 0 && option_count > 0) {
					if (objects_min <= 1
					    && object_is_only_available(*task, handle) == TRUE) {
						/*
						 * It looks like we have only one acceptable object and the plugin
						 * can accept either any minimum or a minimum of one. Go ahead and
						 * set our only selection and skip ahead to option configuration.
						 */
						if (set_selected_object(*task, handle) == SUCCESS)
							*requirements = TASK_REQUIRES_OPTIONS_ONLY;
						else
							*requirements =
							    TASK_REQUIRES_OBJECTS_AND_OPTIONS;
					} else
						*requirements = TASK_REQUIRES_OBJECTS_AND_OPTIONS;
				} else
					*requirements = TASK_REQUIRES_NO_ADDITIONAL_ARGS;
			} else {
				evms_destroy_task(*task);
				log_error("%s: evms_get_option_count() returned error code %d.\n",
					  __FUNCTION__, rc);
			}
		} else {
			evms_destroy_task(*task);
			log_error("%s: evms_get_selected_object_limits() returned error code %d.\n",
				  __FUNCTION__, rc);
		}
	} else {
		log_error("%s: evms_create_task() returned error code %d.\n", __FUNCTION__, rc);
	}

	log_debug("%s: The task requirement code is %d\n", __FUNCTION__, *requirements);

	return rc;
}

/*
 *
 *   GtkWidget *create_task_confirmation_window (task_handle_t, gchar *, gchar *, gchar *)
 *
 *   Description:
 *      This routine is creates a confirmation dialog for a task that
 *      requires no selectable objects or options. In other words,
 *      invoking the task is all that is necessary. Should not be too common.
 *
 *   Entry:
 *      task  -
 *      title - the title string for the operation
 *      verb  - the string to use as the text for the button to complete the
 *              operation, e.g. "Add"
 *      help  - address of string with detailed help on the operation
 *
 *   Exit:
 *      Returns the id of the confirmation window created.
 *
 */
GtkWidget *create_task_confirmation_window(task_handle_t task, gchar * title, gchar * verb,
					   gchar * help)
{
	gchar *question;
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *question_label;
	GtkWidget *hbuttonbox;
	GtkWidget *invoke_button;
	GtkWidget *cancel_button;
	GtkWidget *help_button;
	GtkWidget *pixmap;
	GdkBitmap *mask;
	GdkPixmap *gdk_pixmap;
	GtkAccelGroup *accel_group;
	GtkTooltips *tooltips;

	tooltips = gtk_tooltips_new();
	accel_group = gtk_accel_group_new();

	question = g_strdup_printf(_("%s or cancel the operation?"), title);

	get_dialog_pixmap(QUESTION_PIXMAP, &gdk_pixmap, &mask);

	window = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_window_set_title(GTK_WINDOW(window), title);
	gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);
	gtk_accel_group_attach(accel_group, GTK_OBJECT(window));

	vbox = gtk_vbox_new(FALSE, 0);
	question_label = gtk_label_new(question);
	hbuttonbox = gtk_hbutton_box_new();
	invoke_button = gtk_button_new_with_label(verb);
	cancel_button = gtk_button_new();
	help_button = gtk_button_new();
	pixmap = gtk_pixmap_new(gdk_pixmap, mask);

	gtk_widget_show(vbox);
	gtk_widget_show(question_label);
	gtk_widget_show(hbuttonbox);
	gtk_widget_show(invoke_button);
	gtk_widget_show(cancel_button);
	gtk_widget_show(help_button);
	gtk_widget_show(pixmap);

	get_cancel_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(cancel_button, gdk_pixmap, mask, _("_Cancel"), accel_group);

	get_help_16_pixmap(&gdk_pixmap, &mask);
	add_pixmap_label_to_button(help_button, gdk_pixmap, mask, _("_Help"), accel_group);

	gtk_container_add(GTK_CONTAINER(window), vbox);

	gtk_box_pack_start(GTK_BOX(vbox), pixmap, FALSE, FALSE, 0);
	gtk_misc_set_alignment(GTK_MISC(pixmap), 0.10, 1);
	gtk_misc_set_padding(GTK_MISC(pixmap), 0, 5);

	gtk_box_pack_start(GTK_BOX(vbox), question_label, FALSE, FALSE, 0);
	gtk_label_set_line_wrap(GTK_LABEL(question_label), TRUE);
	gtk_misc_set_padding(GTK_MISC(question_label), 63, 20);

	gtk_box_pack_start(GTK_BOX(vbox), hbuttonbox, TRUE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbuttonbox), 10);
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbuttonbox), 14);
	gtk_button_box_set_child_size(GTK_BUTTON_BOX(hbuttonbox), -1, 50);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), help_button);
	GTK_WIDGET_SET_FLAGS(help_button, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip(tooltips, help_button, _("Get help on this operation"), NULL);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), cancel_button);
	GTK_WIDGET_SET_FLAGS(cancel_button, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip(tooltips, cancel_button,
			     _("Cancel the operation and dismiss this dialog"), NULL);

	gtk_container_add(GTK_CONTAINER(hbuttonbox), invoke_button);
	GTK_WIDGET_SET_FLAGS(invoke_button, GTK_CAN_DEFAULT);
	gtk_tooltips_set_tip(tooltips, invoke_button, _("Complete the operation"), NULL);

	gtk_signal_connect(GTK_OBJECT(invoke_button), "clicked", on_invoke_task_button_clicked,
			   GUINT_TO_POINTER(task));
	gtk_signal_connect(GTK_OBJECT(cancel_button), "clicked",
			   on_button_clicked_destroy_window_list, NULL);
	gtk_signal_connect(GTK_OBJECT(help_button), "clicked",
			   on_selection_window_help_button_clicked, help);

	g_free(question);

	gtk_widget_grab_default(invoke_button);

	return window;
}

/*
 *
 *   inline void display_task_window (GtkWidget *, GtkWidget *, task_handle_t)
 *
 *   Description:
 *      This routine determines attaches some properties to the new task window,
 *      sets up signal handlers and displays the new window.
 *
 *   Entry:
 *      currwin - the toplevel window id of the window that is creating the task (optional)
 *      newwin  - the window id of the first task window for us to display
 *      task    - the handle to the task created
 *
 *   Exit:
 *      Returns nothing.
 *
 */
inline void display_task_window(GtkWidget * currwin, GtkWidget * newwin, task_handle_t task)
{
	GList *window_list = NULL;

	if (currwin != NULL) {
		GtkWidget *next_window;

		window_list = get_window_list(currwin);
		next_window = gtk_object_get_data(GTK_OBJECT(currwin), "next_window_id");

		if (window_list == NULL)
			window_list = g_list_append(window_list, currwin);

		if (next_window != NULL)
			destroy_window_list(g_list_find(window_list, next_window));
	}

	window_list = g_list_append(window_list, newwin);

	set_window_list(newwin, window_list);

	if (currwin != NULL) {
		set_window_list(currwin, window_list);
		gtk_object_set_data(GTK_OBJECT(currwin), "next_window_id", newwin);
	}

	bind_options_callout_to_toplevel_widget(newwin, options_window_callout);

	/*
	 * Connect the destroy signal to the window. If the window is
	 * destroyed, so is the task context.
	 */

	gtk_signal_connect(GTK_OBJECT(newwin), "destroy",
			   GTK_SIGNAL_FUNC(on_task_window_destroy), GUINT_TO_POINTER(task));

	gtk_widget_show(newwin);
}

/*
 *
 *   gint initiate_task (GtkWidget *, engine_handle_t, task_action_t, gchar *, gchar *, gchar *)
 *
 *   Description:
 *      This routine determines the requirements for a task and creates the
 *      appropriate dialog window(s) to initiate the task.
 *
 *   Entry:
 *      currwin - the toplevel window id of the window that is creating the task (optional)
 *      handle  - the handle of the object to act upon or as reference
 *      action  - the id of the function to perform
 *      title   - the title string for the operation
 *      verb    - the string to use as the text for the button to complete the
 *                operation, e.g. "Add"
 *      help    - address of string with detailed help on the operation
 *
 *   Exit:
 *      Returns SUCCESS or some other return code on error
 *
 */
gint initiate_task(GtkWidget * currwin, engine_handle_t handle, task_action_t action,
		   gchar * title, gchar * verb, gchar * help)
{
	gint rc;
	task_handle_t task;
	task_requirements_t requirements;

	rc = create_task(handle, action, &requirements, &task);

	if (rc == SUCCESS) {
		GtkWidget *window = NULL;

		switch (requirements) {
		case TASK_REQUIRES_OPTIONS_ONLY:
			window = create_standard_task_window(title, verb, NULL,
							     currwin ==
							     NULL ? NULL :
							     on_button_clicked_display_prev_window,
							     NULL, task);
			break;

		case TASK_REQUIRES_OBJECTS_ONLY:
		case TASK_REQUIRES_OBJECTS_AND_OPTIONS:
			window = create_standard_selection_window(title,
								  requirements ==
								  TASK_REQUIRES_OBJECTS_ONLY ? verb
								  : NULL,
								  help ==
								  NULL ? NULL : g_strdup(help),
								  on_acceptable_objects_clist_realize,
								  on_acceptable_objects_button_clicked,
								  currwin ==
								  NULL ? NULL :
								  on_button_clicked_display_prev_window,
								  on_button_clicked_destroy_window_list,
								  on_acceptable_objects_clist_select_row,
								  on_acceptable_objects_clist_unselect_row,
								  GUINT_TO_POINTER(task));

			/*
			 * Attach a copy of the verb string to the window for it to use
			 * when creating the last window, the options dialog, if an
			 * options dialog looks to be required.
			 */

			if (requirements == TASK_REQUIRES_OBJECTS_AND_OPTIONS)
				gtk_object_set_data(GTK_OBJECT(window), "function_verb",
						    g_strdup(verb));

			break;

		case TASK_REQUIRES_NO_ADDITIONAL_ARGS:
			window = create_task_confirmation_window(task, title, verb,
								 help ==
								 NULL ? NULL : g_strdup(help));
			break;

		default:
			log_error("%s: Unable to determine task requirements.\n", __FUNCTION__);
			break;
		}

		if (window)
			display_task_window(currwin, window, task);
	}

	return rc;
}

/*
 *
 *   gboolean can_plugin_do_action (plugin_handle_t, task_action_t)
 *
 *   Description:
 *      This routine attempts to see if the plugin could attempt
 *      the given task action. We basically make a determination
 *      by creating a task, checking the acceptable object count
 *      and the minimum selection count.
 *
 *   Entry:
 *      plugin - the plugin to check for the task action
 *      action - the task action code
 *
 *   Exit:
 *      Returns TRUE if the plugin could possibly do the task.
 *
 */
gboolean can_plugin_do_action(plugin_handle_t plugin, task_action_t action)
{
	gboolean result = FALSE;
	task_handle_t task;

	if ((evms_create_task(plugin, action, &task)) == SUCCESS) {
		handle_array_t *acceptable_list;

		if ((evms_get_acceptable_objects(task, &acceptable_list) == SUCCESS) &&
		    acceptable_list->count > 0) {
			guint min;
			guint max;

			if ((evms_get_selected_object_limits(task, &min, &max) == SUCCESS) &&
			    acceptable_list->count >= min) {
				result = TRUE;
			}
		}
		evms_free(acceptable_list);
		evms_destroy_task(task);
	}

	return result;
}

/*
 *
 *   gint set_selected_object (task_handle_t, object_handle_t)
 *
 *   Description:
 *      This routine attempts to set the selected object
 *      for a task.
 *
 *   Entry:
 *      task   - the task handle
 *      handle - the object handle
 *
 *   Exit:
 *      Returns 0 on success or error code on failure.
 *
 */
gint set_selected_object(task_handle_t task, object_handle_t handle)
{
	gint rc;
	handle_array_t ha;
	declined_handle_array_t *declined_list;

	ha.count = 1;
	ha.handle[0] = handle;

	rc = evms_set_selected_objects(task, &ha, &declined_list, NULL);

	if (rc == SUCCESS && declined_list) {
		if (declined_list->count > 0)
			rc = EINVAL;

		evms_free(declined_list);
	}

	return rc;
}
