/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: create.c
 */

#include <frontend.h>
#include <gtk/gtk.h>
#include <string.h>

#include "support.h"
#include "create.h"
#include "thing.h"
#include "widget.h"
#include "task.h"
#include "logging.h"
#include "help.h"

/*
 *
 *   void on_create_volume_button_clicked (GtkButton *, gchar *, gchar *, gboolean)
 *
 *   Description:
 *      This routine initiates the create volume or create compatibility volume
 *      API call using the handle of the object associated with the last row
 *      selected. It then creates and displays the results popup which destroys
 *      both windows when dismissed.
 * 
 *   Entry:
 *      button        - address of the GtkButton widget
 *      error_msg     - string to use for additional error message on failure
 *      success_msg   - string to use as additional message on success
 *      compat_create - boolean that indicates if we are creating a 
 *                      compatability volume
 *
 *   Exit:
 *      Create volume is invoked and results window is displayed unless error is
 *      recoverable (such as EVMS name is not unique) to allow user to try again.
 *
 */
void on_create_volume_button_clicked(GtkButton * button, gchar * error_msg, gchar * success_msg,
				     gboolean compat_create)
{
	GtkCList *clist;
	object_handle_t handle;

	clist = GTK_CLIST(lookup_widget(GTK_WIDGET(button), "selection_window_clist"));

	handle = GPOINTER_TO_UINT(get_single_select_current_row_data(clist));

	if (handle != 0) {
		gint rc;

		/*
		 * If this is not a compatibility volume create then
		 * we need to extract the name from the selection
		 * window before proceeding.
		 */

		if (compat_create)
			rc = evms_create_compatibility_volume(handle);
		else {
			gchar *name;
			GtkWidget *entry;

			/*
			 * Extract the text to use as the volume name from
			 * the modified selection window text entry field.
			 */

			entry = lookup_widget(GTK_WIDGET(button), "selection_window_entry");

			name = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1);

			rc = evms_create_volume(handle, g_strstrip(name));

			g_free(name);
		}

		display_selection_window_results(GTK_WIDGET(button), rc, error_msg, success_msg);
	}
}

/*
 *
 *   void on_create_volume_clist_realize (GtkWidget *, gboolean)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of objects that can be used on a create volume operation.
 * 
 *   Entry:
 *      widget        - address of the selections GtkCList widget
 *      compat_create - boolean that indicates if we are creating a 
 *                      compatability volume
 *
 *   Exit:
 *      Selection list populated with acceptable objects
 *
 */
void on_create_volume_clist_realize(GtkWidget * widget, gboolean compat_create)
{
	gint rc;
	GtkCList *clist = GTK_CLIST(widget);
	handle_array_t *objects;

	rc = evms_get_object_list(0, DATA_TYPE, 0, 0, TOPMOST, &objects);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_object_list() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		guint i;
		gboolean is_selected = (objects->count == 1);
		gboolean object_acceptable;

		set_selection_window_clist_column_titles(clist, _("Size"), _("Storage Object"),
							 NULL);

		for (i = 0; i < objects->count; i++) {
			if (compat_create)
				object_acceptable =
				    (evms_can_create_compatibility_volume(objects->handle[i]) == 0);
			else
				object_acceptable =
				    (evms_can_create_volume(objects->handle[i]) == 0);

			if (object_acceptable)
				add_thing_to_selection_list(clist, objects->handle[i], is_selected);
		}

		evms_free(objects);
	}
}

/*
 *
 *   gboolean is_create_task_acceptable_object (plugin_handle_t, object_handle_t)
 *   
 *   Description:
 *      This routine checks to see if the given object is an
 *      acceptable object for the create task of the given plugin.
 * 
 *   Entry:
 *      plugin - the plugin to check against its create task
 *      handle - the handle of the object to see if acceptable
 *
 *   Exit:
 *      Returns TRUE if the plugin finds the object acceptable for a create task.
 *
 */
gboolean is_create_task_acceptable_object(plugin_handle_t plugin, object_handle_t object)
{
	gboolean result = FALSE;
	task_handle_t task;

	if ((evms_create_task(plugin, EVMS_Task_Create, &task)) == SUCCESS) {
		result = set_selected_object(task, object) == 0;
		evms_destroy_task(task);
	}

	return result;
}

/*
 *
 *   gboolean can_create_from_freespace_object (object_handle_t)
 *   
 *   Description:
 *      This routine determines if a create task on the plugin for
 *      the given freespace storage object could work.
 * 
 *   Entry:
 *      handle - the handle of a freespace object to see if suitable for create
 *
 *   Exit:
 *      Returns TRUE if a create task finds the free space acceptable
 *      else returns FALSE
 *
 */
gboolean can_create_from_freespace_object(object_handle_t handle)
{
	gboolean result = FALSE;
	handle_object_info_t *object;

	if ((evms_get_info(handle, &object)) == SUCCESS) {
		if (object->info.object.data_type == FREE_SPACE_TYPE)
			result =
			    is_create_task_acceptable_object(object->info.object.plugin, handle);

		evms_free(object);
	}

	return result;
}

/*
 *
 *   plugin_handle_t get_plugin_handle_for_object (object_handle_t)
 *   
 *   Description:
 *      This routine returns the plugin handle of the plugin that
 *      manages the supplied storage object.
 * 
 *   Entry:
 *      handle - the handle of a storage object
 *
 *   Exit:
 *      Returns the plugin handle for storage object otherwise 0
 *
 */
plugin_handle_t get_plugin_handle_for_object(object_handle_t handle)
{
	plugin_handle_t plugin = 0;
	handle_object_info_t *info;

	if ((evms_get_info(handle, &info)) == SUCCESS) {
		plugin = info->info.object.plugin;
		evms_free(info);
	}

	return plugin;
}

/*
 *
 *   gchar *get_create_menu_item_text (object_type_t)
 *   
 *   Description:
 *      This routine returns a string containing the menu
 *      item for a create from freespace.
 * 
 *   Entry:
 *      type - the object type (SEGMENT or REGION)
 *
 *   Exit:
 *      Returns NULL if a problem or a dynamically allocated string
 *      containing the proper menu item text.
 *
 */
gchar *get_create_menu_item_text(object_type_t type)
{
	gchar *text = NULL;

	switch (type) {
	case REGION:
		text = g_strdup("Create Region...");
		break;
	case SEGMENT:
		text = g_strdup("Create Segment...");
		break;
	default:
		log_warning("%s: %d is not a valid create from freespace type.\n", __FUNCTION__,
			    type);
		break;
	}

	return text;
}

/*
 *
 *   gchar *get_create_from_freespace_window_title (object_handle_t)
 *   
 *   Description:
 *      This routine returns a string containing the title for 
 *      the window for the create from freespace action.
 * 
 *   Entry:
 *      handle - the handle of the free space storage object
 *
 *   Exit:
 *      Returns NULL if a problem or a dynamically allocated string
 *      containing the proper window title text.
 *
 */
gchar *get_create_from_freespace_window_title(object_handle_t handle)
{
	gchar *text = NULL;
	object_type_t type;

	if (evms_get_handle_object_type(handle, &type) == SUCCESS) {
		switch (type) {
		case REGION:
			text = g_strdup("Create Region Configuration Options");
			break;
		case SEGMENT:
			text = g_strdup("Create Segment Configuration Options");
			break;
		default:
			log_warning("%s: %d is not a valid create from freespace type.\n",
				    __FUNCTION__, type);
			break;
		}
	}

	return text;
}

/*
 *
 *   void on_create_from_freespace_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine initiates the create task to create a new object from
 *      the supplied freespace storage object.
 * 
 *   Entry:
 *      handle - the handle of a freespace object to create a new object from
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_create_from_freespace_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	plugin_handle_t plugin;
	object_handle_t handle = GPOINTER_TO_UINT(user_data);

	if ((plugin = get_plugin_handle_for_object(handle))) {
		task_handle_t task;
		task_requirements_t requirements;

		if ((create_task(plugin, EVMS_Task_Create, &requirements, &task)) == 0) {
			gint rc;

			rc = set_selected_object(task, handle);

			if (rc == SUCCESS) {
				gchar *title;
				GtkWidget *window;

				title = get_create_from_freespace_window_title(handle);

				if (requirements == TASK_REQUIRES_OBJECTS_ONLY)
					window =
					    create_task_confirmation_window(task, title,
									    _("Create"), NULL);
				else
					window =
					    create_standard_task_window(title, _("Create"), NULL,
									NULL, NULL, task);

				g_free(title);
				display_task_window(NULL, window, task);
			} else {
				log_error("%s: Error %d setting freespace object for create.\n",
					  __FUNCTION__, rc);
				evms_destroy_task(task);
			}
		}
	}
}

/*
 *
 *   handle_array_t *get_all_feature_plugins_list (void)
 *   
 *   Description:
 *      This routine returns a single, merged handle array of all
 *      the different feature type plugins.
 * 
 *   Entry:
 *      None
 *
 *   Exit:
 *      Returns the address of the handle array containing all
 *      feauture plugins or NULL on error.
 *
 */
handle_array_t *get_all_feature_plugins_list(void)
{
	gint rc = 0;
	handle_array_t *plugins = NULL;
	handle_array_t *features;
	handle_array_t *assoc_features;

	rc = evms_get_plugin_list(EVMS_FEATURE, 0, &features);

	if (rc == SUCCESS) {
		rc = evms_get_plugin_list(EVMS_ASSOCIATIVE_FEATURE, 0, &assoc_features);

		if (rc == SUCCESS) {
			plugins = merge_handle_array(features, assoc_features);
			evms_free(features);
			evms_free(assoc_features);
		} else {
			evms_free(features);
			log_error("%s: evms_get_plugin_list() returned error code %d.\n",
				  __FUNCTION__, rc);
		}
	} else {
		log_error("%s: evms_get_plugin_list() returned error code %d.\n", __FUNCTION__, rc);
	}

	return plugins;
}

/*
 *
 *   gboolean can_create_feature_object (object_handle_t)
 *   
 *   Description:
 *      This routine determines if the given object can be
 *      used to create any feature object.
 * 
 *   Entry:
 *      handle - the handle of the storage object to check
 *
 *   Exit:
 *      Returns TRUE if at least one feature plugin finds
 *      the object acceptable for a create task.
 *
 */
gboolean can_create_feature_object(object_handle_t handle)
{
	gboolean result = FALSE;
	handle_object_info_t *object;

	if ((evms_get_info(handle, &object)) == SUCCESS) {
		/*
		 * Check only topmost objects
		 */
		if (object->info.object.parent_objects == NULL ||
		    object->info.object.parent_objects->count == 0) {
			gint i;
			handle_array_t *plugins;

			plugins = get_all_feature_plugins_list();

			for (i = 0; i < plugins->count && result == FALSE; i++)
				result =
				    is_create_task_acceptable_object(plugins->handle[i], handle);

			g_free(plugins);
		}
		evms_free(object);
	}

	return result;
}

/*
 *
 *   void on_feature_create_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of feature plugins that say they can create something with
 *      the given object handle.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - contains object handle
 *
 *   Exit:
 *      Selection list populated with suitable feature plugins
 *
 */
void on_feature_create_clist_realize(GtkWidget * widget, gpointer user_data)
{
	GtkCList *clist = GTK_CLIST(widget);
	object_handle_t handle = GPOINTER_TO_UINT(user_data);
	handle_array_t *plugins;

	log_entry;

	set_selection_window_clist_column_titles(clist, NULL, _("EVMS Feature Plugin"), NULL);

	plugins = get_all_feature_plugins_list();

	if (plugins) {
		guint i;
		gboolean is_selected = (plugins->count == 1);

		for (i = 0; i < plugins->count; i++) {
			if (is_create_task_acceptable_object(plugins->handle[i], handle))
				add_thing_to_selection_list(clist, plugins->handle[i], is_selected);
		}

		if (clist->rows == 1)
			gtk_clist_select_row(clist, 0, 0);

		g_free(plugins);
	}

	gtk_clist_set_column_visibility(clist, SL_SIZE_COLUMN, FALSE);

	log_exit;
}

/*
 *
 *   void on_feature_plugin_selection_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine responds to feature plugin selection with creating the
 *      task given and creating the acceptable object selection dialog
 *      window to proceed with acceptable parameter and option selections.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - contains the handle of the context object create initiated on
 *
 *   Exit:
 *      The task context is created with the given plugin and task code.
 *      This window is hidden and another standard selection window is displayed.
 *
 */
void on_feature_plugin_selection_button_clicked(GtkButton * button, gpointer user_data)
{
	GList *window_list;
	GtkWidget *clist;
	GtkWidget *next_window;
	GtkWidget *selection_window;
	task_action_t action;
	plugin_handle_t plugin;
	plugin_handle_t prev_plugin;

	log_entry;

	/*
	 * Hide the selection window and either redisplay the existing task window
	 * if it exists and the plugin selection did not change. If the task window
	 * does not exist, create it. However, if the plugin selections changed and
	 * the task window existed then destroy the current task window and create
	 * a new one.
	 */

	action = GPOINTER_TO_UINT(user_data);
	selection_window = gtk_widget_get_toplevel(GTK_WIDGET(button));
	clist = lookup_widget(GTK_WIDGET(button), "selection_window_clist");
	plugin = GPOINTER_TO_UINT(get_single_select_current_row_data(GTK_CLIST(clist)));
	window_list = get_window_list(selection_window);
	next_window = gtk_object_get_data(GTK_OBJECT(selection_window), "next_window_id");
	prev_plugin =
	    GPOINTER_TO_UINT(gtk_object_get_data(GTK_OBJECT(selection_window), "previous_plugin"));

	if (window_list == NULL)
		window_list = g_list_append(window_list, selection_window);

	if (plugin != prev_plugin) {
		gint rc;
		task_handle_t handle;

		rc = evms_create_task(plugin, EVMS_Task_Create, &handle);

		if (rc != SUCCESS) {
			log_error("%s: evms_create_task() returned error code %d.\n", __FUNCTION__,
				  rc);

			display_results_window(rc, NULL, _("Unable to create the task context."),
					       NULL, FALSE, selection_window);
		} else {
			guint min = 0;
			guint max = 0;

			set_selected_object(handle, GPOINTER_TO_UINT(user_data));

			/*
			 * If we already created the next window for selecting
			 * the acceptable objects (and maybe even other follow-on 
			 * windows like for options), make sure we destroy them
			 * all and clean up the list since our plugin selections
			 * have apperently changed and so all the following panels
			 * need to be recreated.
			 */

			if (next_window != NULL)
				destroy_window_list(g_list_find(window_list, next_window));

			/*
			 * If the plug-in can only accept one selected object and we've already
			 * done that then skip the acceptable objects selection window and just
			 * go straight to options configuration.
			 */
			evms_get_selected_object_limits(handle, &min, &max);

			if (min == 1 && max == 1) {
				next_window =
				    create_standard_task_window(_("Create Feature Configuration"),
								_("Create"), NULL,
								on_button_clicked_display_prev_window,
								on_button_clicked_destroy_window_list,
								handle);

				bind_options_callout_to_toplevel_widget(next_window,
									options_window_callout);
			} else {
				gint count;
				gchar *next_button_text = NULL;

				/*
				 * If we have no options then set the Next button label to
				 * indicate the actual task action rather than "Next".
				 */

				if (evms_get_option_count(handle, &count) == SUCCESS && count == 0)
					next_button_text = _("Create");

				next_window =
				    create_standard_selection_window(_
								     ("Select Storage Objects for Create"),
								     next_button_text,
								     get_acceptable_objects_help_text
								     (),
								     on_acceptable_objects_clist_realize,
								     on_acceptable_objects_button_clicked,
								     on_button_clicked_display_prev_window,
								     on_button_clicked_destroy_window_list,
								     on_acceptable_objects_clist_select_row,
								     on_acceptable_objects_clist_unselect_row,
								     GUINT_TO_POINTER(handle));
			}

			window_list = g_list_append(window_list, next_window);

			set_window_list(next_window, window_list);
			set_window_list(selection_window, window_list);

			gtk_object_set_data(GTK_OBJECT(selection_window), "next_window_id",
					    next_window);
			gtk_object_set_data(GTK_OBJECT(selection_window), "previous_plugin",
					    GUINT_TO_POINTER(plugin));

			/*
			 * Connect the destroy signal to the next window. If the window is
			 * destroyed, so is the task context.
			 */

			gtk_signal_connect(GTK_OBJECT(next_window), "destroy",
					   GTK_SIGNAL_FUNC(on_task_window_destroy),
					   GUINT_TO_POINTER(handle));

			gtk_widget_show(next_window);
			gtk_widget_hide(selection_window);
		}
	} else {
		gtk_widget_show(next_window);
		gtk_widget_hide(selection_window);
	}

	log_exit;
}

/*
 *
 *   void on_create_feature_object_menu_item_activate (GtkMenuItem *, gpointer)
 *   
 *   Description:
 *      This routine initiates process to create a new feature object
 *      starting with the given selected object.
 * 
 *   Entry:
 *      menuitem  - the menu item that initiated the action
 *      user_data - contains the handle of a object to use in creating a feature object
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_create_feature_object_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Create EVMS Feature Object"),
						  NULL,
						  get_create_feature_help_text(),
						  on_feature_create_clist_realize,
						  on_feature_plugin_selection_button_clicked,
						  NULL, NULL, NULL, NULL, user_data);

	gtk_widget_show(window);
}
