/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: container_cb.c
 */

#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "thing.h"
#include "readable.h"
#include "logging.h"
#include "plugin_cb.h"
#include "container_cb.h"

/*
 *
 *   gboolean can_expand_container (object_handle_t)
 *   
 *   Description:
 *      This routine creates an expand container task
 *      to probe if the container can possibly expand.
 * 
 *   Entry:
 *      container - the target container to determine if expandable
 *
 *   Exit:
 *      Returns TRUE if container can expand
 *
 */
gboolean can_expand_container(object_handle_t container)
{
	gint rc;
	gboolean result = FALSE;
	task_handle_t task;

	rc = evms_create_task(container, EVMS_Task_Expand_Container, &task);

	if (rc != SUCCESS) {
		log_error("%s: evms_create_task() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		handle_array_t *acceptable_list;

		rc = evms_get_acceptable_objects(task, &acceptable_list);

		if (rc != SUCCESS) {
			log_error("%s: evms_get_acceptable_objects() returned error code %d.\n",
				  __FUNCTION__, rc);
		} else {
			/*
			 * If the list is non-empty then we can expand the container
			 */
			if (acceptable_list->count > 0)
				result = TRUE;

			evms_free(acceptable_list);
		}
		evms_destroy_task(task);
	}

	return result;
}

/*
 *
 *   void on_expand_container_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of containers that can be expanded.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Selection list populated with suitable containers
 *
 */
void on_expand_container_clist_realize(GtkWidget * widget, gpointer user_data)
{
	gint rc = 0;
	GtkCList *clist = GTK_CLIST(widget);
	handle_array_t *containers;

	rc = evms_get_container_list(0, 0, 0, &containers);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_container_list() returned error code %d.\n", __FUNCTION__,
			  rc);
	} else {
		guint i;
		gboolean is_selected = (containers->count == 1);

		set_selection_window_clist_column_titles(clist, _("Size"),
							 make_object_type_readable_string
							 (CONTAINER), NULL);

		for (i = 0; i < containers->count; i++) {
			if (can_expand_container(containers->handle[i]))
				add_thing_to_selection_list(clist, containers->handle[i],
							    is_selected);
		}

		if (clist->rows == 1)
			gtk_clist_select_row(clist, 0, 0);

		evms_free(containers);
	}
}

/*
 *
 *   gboolean can_shrink_container (object_handle_t)
 *   
 *   Description:
 *      This routine creates an shrink container task
 *      to probe if the container can possibly shrink.
 * 
 *   Entry:
 *      container - the target container to determine if shrinkable
 *
 *   Exit:
 *      Returns TRUE if container can shrink
 *
 */
gboolean can_shrink_container(object_handle_t container)
{
	gint rc;
	gboolean result = FALSE;
	task_handle_t task;

	rc = evms_create_task(container, EVMS_Task_Shrink_Container, &task);

	if (rc != SUCCESS) {
		log_error("%s: evms_create_task() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		handle_array_t *acceptable_list;

		rc = evms_get_acceptable_objects(task, &acceptable_list);

		if (rc != SUCCESS) {
			log_error("%s: evms_get_acceptable_objects() returned error code %d.\n",
				  __FUNCTION__, rc);
		} else {
			/*
			 * If the list is non-empty then we can shrink the container
			 */
			if (acceptable_list->count > 0)
				result = TRUE;

			evms_free(acceptable_list);
		}
		evms_destroy_task(task);
	}

	return result;
}

/*
 *
 *   void on_shrink_container_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of containers that can be shrunk.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Selection list populated with suitable containers
 *
 */
void on_shrink_container_clist_realize(GtkWidget * widget, gpointer user_data)
{
	gint rc = 0;
	GtkCList *clist = GTK_CLIST(widget);
	handle_array_t *containers;

	rc = evms_get_container_list(0, 0, 0, &containers);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_container_list() returned error code %d.\n", __FUNCTION__,
			  rc);
	} else {
		guint i;
		gboolean is_selected = (containers->count == 1);

		set_selection_window_clist_column_titles(clist, _("Size"),
							 make_object_type_readable_string
							 (CONTAINER), NULL);

		for (i = 0; i < containers->count; i++) {
			if (can_shrink_container(containers->handle[i]))
				add_thing_to_selection_list(clist, containers->handle[i],
							    is_selected);
		}

		if (clist->rows == 1)
			gtk_clist_select_row(clist, 0, 0);

		evms_free(containers);
	}
}

/*
 *
 *   void on_container_plugin_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of plugins that support containers.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - contains task action code for action to use list
 *
 *   Exit:
 *      Selection list populated with available plugins that support containers
 *
 */
void on_container_plugin_clist_realize(GtkWidget * widget, gpointer user_data)
{
	set_selection_window_clist_column_titles(GTK_CLIST(widget), NULL, _("Container Plugin"),
						 NULL);

	on_plugin_selection_clist_realize(GTK_CLIST(widget), 0, SUPPORTS_CONTAINERS,
					  GPOINTER_TO_UINT(user_data));
}
