# (C) Copyright IBM Corp. 2004
#
# This program is free software;  you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY;  without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
# the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program;  if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#
# Common subroutines for getting information about EVMS objects.

package Evms::Object;

use strict;
use warnings;

use Evms::Common;
use Evms::Log;

BEGIN {
	use Exporter();
	our (@ISA, @EXPORT);
        @ISA = qw(Exporter);
        @EXPORT = qw(&create_object
		     &query_object
		     &extended_query_object
		     &get_object_details
		     &get_object_size
		     &check_minimum_object_size
		    );
}

# create_object
#
# Arguements
#    plugin:  Short name of plugin to do the creating
#    objects: Reference to an array of input objects
#    options: Optional reference to a hash of options (key,value pairs)
# Returns:
#    Array with names of new objects created.
sub create_object
{
	my $plugin      = $_[0];
	my $objects_ref = $_[1];
	my $options_ref = $_[2];
	my @objects;
	my $command = "create:object,$plugin={";
	my $i;
	my $rc;
	my @output;
	my @lines;
	my @new_objects;

	if (ref($objects_ref)) {
		@objects = @{$objects_ref};
	} else {
		log_error("Argument 2 is not a reference to an array.\n");
		return;
	}

	if (ref($options_ref)) {
		my %options = %{$options_ref};
		if (%options) {
			my @names = keys(%options);
			my $i;
			my $name;

			for ($i = 0; $i < @names; $i++) {
				$name = $names[$i];
				$command .= "$name=$options{$name},";
			}
		}
	}

	$command =~ s/^(.*),$/$1/;
	$command .= "}";

	for ($i = 0; $i < @objects; $i++) {
		$command .= ",$objects[$i]";
	}

	$rc = run_evms_command($command, \@output);
	if ($rc) {
		log_error("Creation of object failed with error $rc.\n");
	} else {
		$i = 0;
                @lines = grep(/created object:/, @output);
                foreach my $line (@lines) {
			if ($line =~ /created object:\s+(\S+)/) {
				$new_objects[$i] = $1;
				$i++;
			}
		}
	}

	return @new_objects;
}

# query_object
# Get the basic information for the specified object
#
# Arguments:
#    object: Name of object to query.
# Returns:
#    Array with query output.
sub query_object($)
{
	my $object = $_[0];
	my (@output, @query);
	my ($i, $rc);
	#my $command = "query:objects,object=$object";
	my $command = "query:objects";

	$rc = run_evms_command($command, \@output);
	if ($rc) {
		return;
	}

	for ($i = 0; $i < @output; $i++) {
		if ($output[$i] =~ /Name:\s+$object/) {
			while ($output[$i] !~ /^$/) {
				push(@query, $output[$i]);
				$i++;
			}
			last;
		}
	}

	return @query;
}

# extended_query_object
# Get the extended information for the specified object
#
# Arguments:
#    object: Name of object to query.
# Returns:
#    Array with extended info output.
sub extended_query_object($)
{
	my $object = $_[0];
	my @output;
	my $command = "query:ei,$object";
	my $rc;

	$rc = run_evms_command($command, \@output);
	if ($rc) {
		return;
	}

	return @output;
}

# get_object_details
# Return a hash containing standard info about the specified object.
#
# Arguments:
#    object: Name of object to get details for.
sub get_object_details
{
	my $object = $_[0];
	my @query_output;
	my %details;
	my @keys = ("Name", "Major", "Minor", "Active", "Size", "Start", "Type");
	my ($key, $line);

	@query_output = query_object($object);

	foreach $key (@keys) {
		($line) = grep(/$key/, @query_output);
		if (defined($line) &&
		    $line =~ /^.*$key.*:\s+(.*)$/) {
			$details{$key} = $1;
		}
	}

	$details{"Sectors"} = get_blkdev_sectors($object);

	return %details;
}

# get_object_size
#
# Arguments:
#    object: Name of object to query.
sub get_object_size($)
{
        my $object = $_[0];
	my %details;

	%details = get_object_details($object);

	return $details{"Size"};
}

# check_minimum_object_size
# Check that the specified object is at least as big as the
# specified minimum size.
#
# Arguments:
#    object: Name of object to check.
#    size: Minimum required size. Must contain appropriate suffix.
sub check_minimum_object_size($ $)
{
	my $object = $_[0];
	my $size = $_[1];
	my $real_size;
	my $rc;

	$real_size = get_object_size($object);

	$rc = compare_sizes($real_size, $size, 0);
	if ($rc < 0) {
		log_error("Object $object is smaller than " .
			  "required size ($size).\n");
	} else {
		$rc = 0;
	}

	return $rc;
}

1;

