/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * plugins/xfs/fs_xfs.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <plugin.h>
#include "fsimxfs.h"

plugin_record_t * my_plugin_record = &xfs_plugin_record;
engine_functions_t * EngFncs = NULL;
char xfsutils_version[10];
u_int32_t xfsutils_support = 0;

/**
 * xfs_setup
 *
 * Verify that the XFS utilities are present on the system.
 **/
static int xfs_setup(engine_functions_t * engine_function_table)
{
	int rc;

	EngFncs = engine_function_table;
	LOG_ENTRY();

	/* xfsutils must be at version 2.0.0 or greater to work properly with
	 * this FSIM. This also initializes xfsutils_version and xfsutils_support.
	 */
	rc = xfs_test_version();
	if (rc) {
		LOG_WARNING("xfsutils must be version 2.0.0 or later to function properly with this FSIM.\n");
		LOG_WARNING("xfsutils must also reside in the search path specified by PATH.\n");
		LOG_WARNING("Please get/install the current version of xfsutils from http://oss.sgi.com/projects/xfs\n");
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_cleanup
 *
 * Free all of the private data item we have left on volumes.
 **/
static void xfs_cleanup()
{
	list_anchor_t global_volumes;
	list_element_t vol_list_iter;
	logical_volume_t * volume;
	int rc;

	LOG_ENTRY();
	rc = EngFncs->get_volume_list(my_plugin_record, NULL, 0, &global_volumes);
	if (!rc) {
		LIST_FOR_EACH(global_volumes, vol_list_iter, volume) {
			EngFncs->engine_free(volume->private_data);
		}
		EngFncs->destroy_list(global_volumes);
	}
	LOG_EXIT_VOID();
}

/**
 * xfs_get_fs_limits
 *
 * Get the size limits for the filesystem on this volume.
 **/
static int xfs_get_fs_limits(logical_volume_t * volume,
			     sector_count_t * min_fs_size,
			     sector_count_t * max_fs_size,
			     sector_count_t * max_object_size)
{
	xfs_volume_t * xfs_vol = volume->private_data;
	int rc = 0;

	LOG_ENTRY();

	if (!xfs_vol) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (xfs_vol->sb) {
		/* Get and validate current XFS superblock */
		rc = xfs_get_superblock(volume, xfs_vol->sb);
		if (!rc) {
			*max_fs_size = (u_int64_t)1 << 63;
			*max_object_size = (u_int64_t)1 << 63;
			*min_fs_size = xfs_vol->sb->sb_dblocks *
				       (xfs_vol->sb->sb_blocksize >>
					EVMS_VSECTOR_SIZE_SHIFT);
			LOG_EXTRA("volume:%s, min:%"PRIu64", max:%"PRIu64"\n",
				  volume->dev_node, *min_fs_size, *max_fs_size);
			LOG_EXTRA("fs size:%"PRIu64", vol size:%"PRIu64"\n",
				  volume->fs_size, volume->vol_size);

			if (*min_fs_size > volume->vol_size) {
				LOG_ERROR("XFS returned min size > volume size."
					  " Setting min size to volume size\n");
				*min_fs_size = volume->vol_size;
			}
		}
	} else if (xfs_vol->fs_vol) {
		xfs_volume_t *external_vol = xfs_vol->fs_vol->private_data;
		if (external_vol) {
			/* If an external log is associated with the fs. */
		        *min_fs_size = external_vol->sb->sb_logblocks *
				       (external_vol->sb->sb_blocksize >>
					EVMS_VSECTOR_SIZE_SHIFT);
			*max_fs_size = *min_fs_size;
			*max_object_size = (u_int64_t)1 << 63;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_probe
 *
 * Check for an XFS filesystem on this volume.
 */
static int xfs_probe(logical_volume_t * volume)
{
	xfs_volume_t * xfs_vol, * tmp_vol;
	logical_volume_t * tmp;
	list_anchor_t xfs_vols;
	list_element_t vol_list_iter;
	int rc;

	LOG_ENTRY();

	/* allocate space for copy of superblock in private data */
	xfs_vol = EngFncs->engine_alloc(sizeof(*xfs_vol));
	if (!xfs_vol) {
		rc = ENOMEM;
		goto out;
	}

	xfs_vol->sb = EngFncs->engine_alloc(SIZE_OF_SUPER);
	if (!xfs_vol->sb) {
		EngFncs->engine_free(xfs_vol);
		rc = ENOMEM;
		goto out;
	}

	/* Get and validate XFS superblock */
	rc = xfs_get_superblock(volume, xfs_vol->sb);
	if (rc) {
		/* Didn't find valid XFS superblock,
		 * look for XFS log superblock.
		 */
		EngFncs->engine_free(xfs_vol->sb);
		xfs_vol->sb = NULL;
		xfs_vol->log_sb = EngFncs->engine_alloc(sizeof(xlog_rec_header_t));
		if (!xfs_vol->log_sb) {
			EngFncs->engine_free(xfs_vol);
			rc = ENOMEM;
			goto out;
		}
		rc = xfs_get_log_superblock(volume, xfs_vol->log_sb);
		if (rc) {
			EngFncs->engine_free(xfs_vol->log_sb);
			EngFncs->engine_free(xfs_vol);
			goto out;
		}
	}

	/* Store copy of valid XFS superblock in private data */
	volume->private_data = xfs_vol;

	/* Get a list of XFS claimed volumes */
	rc = EngFncs->get_volume_list(my_plugin_record, NULL, 0, &xfs_vols);

	if (rc) {
		EngFncs->engine_free(xfs_vol->log_sb);
		EngFncs->engine_free(xfs_vol);
		goto out;
	}

	/* Look through all of the volumes to find one that is an external
	 * log for this XFS volume or the FS vol for this external log. We do
	 * this by checking for a match of the actual uuid of the external log,
	 * and the uuid of the external log that is stored in the XFS volume's
	 * superblock.
	 */
	LIST_FOR_EACH(xfs_vols, vol_list_iter, tmp) {
		tmp_vol = tmp->private_data;

		if (xfs_vol->sb && tmp_vol->log_sb) {
			/* Look for a pre-discovered log volume that matches
			 * this new XFS volume. Compare the log uuid stored in
			 * the XFS superblock with the actual uuid of the
			 * external log volume to see if we have a match.
			 */
			if (!uuid_compare(xfs_vol->sb->sb_uuid,
					  tmp_vol->log_sb->h_fs_uuid)) {
				/* Add the XFS vol/external log
				 * vol pair to the list */
				xfs_vol->log_vol = tmp;
				tmp_vol->fs_vol = volume;
				/* Mark the log volume as not mountable. */
				tmp->flags |= VOLFLAG_NOT_MOUNTABLE;
				/* since we discovered the external log before
				 * the fs volume, the size information for the
				 * log volume is incorrect due the the log size
				 * only being stored if the FS supoerblock so go
				 * update the size infor for the external log
				 * volume.
				 */
				xfs_get_fs_limits(tmp, &tmp->min_fs_size,
						  &tmp->max_fs_size,
						  &tmp->max_vol_size);
			}
		} else if (xfs_vol->log_sb && tmp_vol->sb) {
			/* log volume, look for fs volume */
			if (!uuid_compare(xfs_vol->log_sb->h_fs_uuid,
					  tmp_vol->sb->sb_uuid)) {
				xfs_vol->fs_vol = tmp;
				tmp_vol->log_vol = volume;
				/* Mark the log volume as not mountable. */
				volume->flags |= VOLFLAG_NOT_MOUNTABLE;
			}
		}
	}
	EngFncs->destroy_list(xfs_vols);
	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_can_mkfs
 *
 * Can we create an XFS filesystem on this volume?
 **/
static int xfs_can_mkfs(logical_volume_t * volume)
{
	int rc = 0;

	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* Can't format if mounted. */
		rc = EBUSY;
	} else if ((volume->vol_size * EVMS_VSECTOR_SIZE) < MINXFS) {
		/* Volume size must be >= MINXFS */
		rc = EPERM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_can_unmkfs
 *
 * Can we remove the XFS filesystem from this volume?
 **/
static int xfs_can_unmkfs(logical_volume_t * volume)
{
	xfs_volume_t * xfs_vol = volume->private_data;
	int rc = 0;

	LOG_ENTRY();

	if (!xfs_vol) {
		rc = EINVAL;
	} else if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't unmkfs. */
		rc = EBUSY;
	} else if (xfs_vol->log_sb && xfs_vol->fs_vol) {
		/* If XFS external log volume, and has an associated
		 * filesystem, can't unmkfs */
		rc = EPERM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_can_fsck
 *
 * Can we check the XFS filesystem on this volume?
 **/
static int xfs_can_fsck(logical_volume_t * volume)
{
	xfs_volume_t * xfs_vol = volume->private_data;
	int rc = 0;

	LOG_ENTRY();

	if (!xfs_vol) {
		rc = EINVAL;
	} else if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't unmkfs. */
		rc = EBUSY;
	} else if (xfs_vol->log_sb) {
		/* If XFS external log volume, can't fsck */
		rc = EPERM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_get_fs_size
 *
 * Get the current size of the filesystem on this volume.
 **/
static int xfs_get_fs_size(logical_volume_t * volume,
			   sector_count_t * size)
{
	xfs_volume_t * xfs_vol = volume->private_data;
	int rc = 0;

	LOG_ENTRY();

	if (!xfs_vol) {
		rc = EINVAL;
	} else if (xfs_vol->sb) {
		/* get and validate current XFS superblock */
		rc = xfs_get_superblock(volume, xfs_vol->sb);
		if (!rc) {
			*size = xfs_vol->sb->sb_dblocks *
				(xfs_vol->sb->sb_blocksize >>
				 EVMS_VSECTOR_SIZE_SHIFT);
		}
	} else {
		if (xfs_vol->fs_vol) {
			/* if log associated with a fs */
			xfs_volume_t * ext_vol = xfs_vol->fs_vol->private_data;
			*size = ext_vol->sb->sb_logblocks *
				(ext_vol->sb->sb_blocksize >>
			     	 EVMS_VSECTOR_SIZE_SHIFT);
		} else {
			rc = EINVAL;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_expand
 *
 * Expand the filesystem to new_size. If the filesystem is not expanded exactly
 * to new_size, set new_size to the new_size of the filesystem.
 **/
static int xfs_expand(logical_volume_t * volume,
		      sector_count_t * new_size)
{
	char * argv[3];
	char * buffer;
	pid_t pidf;
	int fds2[2];
	int status, bytes_read, rc;

	LOG_ENTRY();

	buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN);
	if (!buffer) {
		rc = ENOMEM;
		goto out;
	}

	/* Open pipe, alloc buffer for collecting fsck.xfs output */
	rc = pipe(fds2);
	if (rc) {
		goto out;
	}

	argv[0] = "xfs_growfs";
	SET_STRING(argv[1], volume->mount_point);
	argv[2] = NULL;

	pidf = EngFncs->fork_and_execvp(volume, argv, NULL, fds2, fds2);
	if (pidf != -1) {
		fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
		while (!(waitpid(pidf, &status, WNOHANG))) {
			bytes_read = read(fds2[0], buffer, MAX_USER_MESSAGE_LEN);
			if (bytes_read > 0) {
				MESSAGE("XFS growfs output: \n%s", buffer);
				memset(buffer, 0, bytes_read);
			}
			usleep(10000);
		}
		if (WIFEXITED(status) && (WEXITSTATUS(status) != ENOENT)) {
			do {
				bytes_read = read(fds2[0], buffer, MAX_USER_MESSAGE_LEN);
				if (bytes_read > 0) {
					MESSAGE("XFS growfs output: \n%s", buffer);
				}
			} while (bytes_read == MAX_USER_MESSAGE_LEN);
			rc = WEXITSTATUS(status);
			if (rc == 0) {
				LOG_DETAILS("XFS growfs completed with rc = %d \n", status);
			} else {
				LOG_ERROR("XFS growfs completed with rc = %d \n", status);
			}
		} else {
			rc = EINTR;
		}
	} else {
		rc = EIO;
	}

        EngFncs->engine_free(argv[1]);

	close(fds2[0]);
	close(fds2[1]);

out:
	EngFncs->engine_free(buffer);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_unmkfs_setup
 *
 * unmkfs has been scheduled.  Do any setup work such as
 * releasing another volume that was used for an external log.
 **/
static int xfs_unmkfs_setup(logical_volume_t * volume)
{
	xfs_volume_t * xfs_vol = volume->private_data;

	LOG_ENTRY();

	if (!xfs_vol) {
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* See if the volume to be unmkfs'ed has an external log. If so, free
	 * the private data on the external log and call unassign to
	 * dissacociate XFS from the volume.
	 */
	if (xfs_vol->sb && xfs_vol->log_vol) {
		EngFncs->engine_free(((xfs_volume_t *)xfs_vol->log_vol->private_data)->log_sb);
		EngFncs->engine_free(xfs_vol->log_vol->private_data);
		EngFncs->unassign_fsim_from_volume(xfs_vol->log_vol);
		xfs_vol->log_vol->private_data = NULL;
	}

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * fs_unmkfs
 *
 * Remove the XFS filesystem from this volume.
 **/
static int xfs_unmkfs(logical_volume_t * volume)
{
	xfs_volume_t * xfs_vol = volume->private_data;
	int rc;

	LOG_ENTRY();

	if (!xfs_vol) {
		rc = EINVAL;
	} else if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If mounted, can't unmkfs. */
		rc = EBUSY;
	} else {
		/* unmkfs the volume */
		rc = xfs_remove(volume);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_shrink
 *
 * Shrinking XFS filesystems is not currently supported.
 */
static int xfs_shrink(logical_volume_t * volume,
		      sector_count_t requested_size,
		      sector_count_t * new_size)
{
	LOG_ENTRY();
	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * xfs_mkfs_setup
 *
 * mkfs has been scheduled.  Do any setup work such as claiming another
 * volume for an external log.
 **/
static int xfs_mkfs_setup(logical_volume_t * volume,
			  option_array_t * options)
{
	int i;
	char * ext_log_vol = NULL;
	list_anchor_t avail_ext_logs;
	list_element_t logs_list_iter;
	logical_volume_t * vol;
	xfs_volume_t * xfs_vol;

	LOG_ENTRY();

	for (i = 0; i < options->count; i++) {
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name, "journalvol")) {
				options->option[i].number = MKFS_JOURNAL_VOL_INDEX;
			}
		}

		if (options->option[i].number == MKFS_JOURNAL_VOL_INDEX) {
			/* 'set journal volume' option */
			if (options->option[i].value.s &&
			    strcmp(options->option[i].value.s, NO_SELECTION)) {
				ext_log_vol = options->option[i].value.s;
			}
		}
	}

	xfs_vol = EngFncs->engine_alloc(sizeof(xfs_volume_t));
	if (!xfs_vol) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	xfs_vol->sb = EngFncs->engine_alloc(SIZE_OF_SUPER);
	if (!xfs_vol->sb) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	volume->private_data = xfs_vol;

	if (ext_log_vol) {
		/* Search through the volumes to find and validate the one
		 * selected for an external log, and if valid, claim it.
		 */
		EngFncs->get_volume_list(NULL, NULL, 0, &avail_ext_logs);
		LIST_FOR_EACH(avail_ext_logs, logs_list_iter, vol) {
			/* TODO: When a min log volume size is put into mkfs.xfs,
			 * add that requirement here for validation of a volume to
			 * be considered for a XFS external log.
			 */
			if (vol->file_system_manager == NULL &&
			    !EngFncs->is_mounted(vol->dev_node, NULL) &&
			    !strcmp(vol->dev_node, ext_log_vol)) {
				EngFncs->assign_fsim_to_volume(my_plugin_record, vol);

				xfs_vol = EngFncs->engine_alloc(sizeof(xfs_volume_t));
				if (!xfs_vol) {
					LOG_EXIT_INT(ENOMEM);
					return ENOMEM;
				}

				xfs_vol->log_sb = EngFncs->engine_alloc(sizeof(xlog_rec_header_t));
				if (!xfs_vol->log_sb) {
					LOG_EXIT_INT(ENOMEM);
					return ENOMEM;
				}

				xfs_vol->fs_vol = volume;
				vol->private_data = xfs_vol;
				((xfs_volume_t*)volume->private_data)->log_vol = vol;
			}
		}
		EngFncs->destroy_list(avail_ext_logs);
	}

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * xfs_mkfs
 *
 * Format the volume.
 **/
static int xfs_mkfs(logical_volume_t * volume, option_array_t * options)
{
	int rc;

	LOG_ENTRY();

	if (EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* Don't format if mounted */
		rc = EBUSY;
	} else {
		rc = xfs_create(volume, options);
		if (!rc) {
			/* Re-probe to set up private data */
			rc = xfs_probe(volume);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_discard
 *
 * Forget about this volume.  Don't remove the file system.  Just clean
 * up any data structures you may have associated with it.
 **/
static int xfs_discard(logical_volume_t * volume)
{
	LOG_ENTRY();

	if (volume->private_data) {
		EngFncs->engine_free(volume->private_data);
		volume->private_data = NULL;
	}

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * xfs_fsck
 *
 * Check the filesystem on this volume.
 **/
static int xfs_fsck(logical_volume_t * volume, option_array_t * options)
{
	int rc;

	LOG_ENTRY();

	rc = xfs_check(volume, options);

	/* If fsck.xfs returns FSCK_CORRECTED, the
	 * file system is clean, so return FSCK_OK.
	 */
	if (rc == FSCK_CORRECTED) {
		rc = FSCK_OK;
	} else if (rc == -1) {
		/* The value of FSCK_CORRECTED is the same as
		 * EPERM, so xfs_check will return -1 for EPERM.
		 */
		rc = EPERM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_get_option_count
 *
 * Return the total number of supported options for the specified task.
 **/
static int xfs_get_option_count(task_context_t * context)
{
	int count;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_mkfs:
		count = MKFS_XFS_OPTIONS_COUNT;
		break;
	case EVMS_Task_fsck:
		count = FSCK_XFS_OPTIONS_COUNT;
		break;
	case EVMS_Task_Expand:
		count = 0;
		break;
	default:
		count = -1;
		break;
	}

	LOG_EXIT_INT(count);
	return count;
}

/**
 * get_volume_list
 *
 * Get the list of volumes on the system that we can use.
 **/
static int get_volume_list(value_list_t ** value_list,
			   list_anchor_t selected_volumes)
{
	logical_volume_t * vol;
	list_anchor_t global_volumes;
	list_element_t vol_list_iter;
	int rc, i, count;

	LOG_ENTRY();

	if (!EngFncs->list_empty(selected_volumes)) {
		vol = EngFncs->first_thing(selected_volumes, NULL);
		if (vol->disk_group != NULL) {
			rc = EngFncs->get_volume_list(NULL, vol->disk_group, 0, &global_volumes);
		} else {
			rc = EngFncs->get_volume_list(NULL, NULL, VOL_NO_DISK_GROUP, &global_volumes);
		}
	} else {
		rc = EngFncs->get_volume_list(NULL, NULL, 0, &global_volumes);
	}

	if (rc) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* Loop through 'selected' volumes, remove them from temp list. */
	LIST_FOR_EACH(selected_volumes, vol_list_iter, vol) {
		LOG_DETAILS("Volume %s selected, removing from 'available' list.\n", vol->name);
		EngFncs->remove_thing(global_volumes, vol);
	}

	/* if 'available volumes' list already exists, free it and start over */
	if (*value_list) {
		for (i = 0; i < (*value_list)->count; i++) {
			if ((*value_list)->value[i].s) {
				EngFncs->engine_free((*value_list)->value[i].s);
			}
		}
		EngFncs->engine_free(*value_list);
	}
	*value_list = NULL;

	count = EngFncs->list_count(global_volumes);
	/* Increment count to hold the 'None' selection. */
	count++;
	/* allocate and populate 'available volumes' list */
	*value_list = EngFncs->engine_alloc(count * sizeof(value_t) +
					    sizeof(value_list_t));
	i = 0;
	SET_STRING((*value_list)->value[i].s, NO_SELECTION);
	i++;
	LIST_FOR_EACH(global_volumes, vol_list_iter, vol) {
		/* add proper volumes to 'available volumes' list */
		/*************************************************************
		 *  TODO: When a min log volume size is put into mkfs.xfs,   *
		 *  add that requirement here for validation of a volume to  *
		 *  be considered for a XFS external log.                    *
		 *************************************************************/
		if (vol->file_system_manager == NULL &&
		    !EngFncs->is_mounted(vol->dev_node, NULL)) {
			(*value_list)->value[i].s = EngFncs->engine_alloc(strlen(vol->name) + 1);
			strcpy((*value_list)->value[i].s, vol->name);
			i++;
		}
	}
	(*value_list)->count = i;
	EngFncs->destroy_list(global_volumes);

	LOG_EXIT_INT(0);
	return 0;
}

/*
 * Initialize mkfs task acceptable objects by enumerating volumes, finding
 * those that have no FSIM claiming them and are of the proper size and
 * adding them to the acceptable objects list.
 */
static int fs_init_mkfs_acceptable_objects(task_context_t * context)
{
	int rc;
	list_anchor_t global_volumes;
	list_element_t vol_list_iter;
	logical_volume_t * volume;

	LOG_ENTRY();

	if (context->volume != NULL) {
		if (context->volume->disk_group != NULL) {
			rc = EngFncs->get_volume_list(NULL, context->volume->disk_group, 0, &global_volumes);
		} else {
			rc = EngFncs->get_volume_list(NULL, NULL, VOL_NO_DISK_GROUP, &global_volumes);
		}
	} else {
		rc = EngFncs->get_volume_list(NULL, NULL, 0, &global_volumes);
	}

	if (!rc) {
		LIST_FOR_EACH(global_volumes, vol_list_iter, volume) {
			/* Only mkfs unformatted, unmounted, 'large enough' volumes */
			if (!volume->file_system_manager &&
			    volume->vol_size * EVMS_VSECTOR_SIZE >= MINXFS &&
			    !EngFncs->is_mounted(volume->dev_node, NULL)) {
				EngFncs->insert_thing(context->acceptable_objects,
						      volume,
						      INSERT_BEFORE,
						      NULL);
			}
		}
		EngFncs->destroy_list(global_volumes);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_init_task
 *
 * Fill in the initial list of acceptable objects.  Fill in the minimum and
 * maximum nuber of objects that must/can be selected.  Set up all initial
 * values in the option_descriptors in the context record for the given
 * task.  Some fields in the option_descriptor may be dependent on a
 * selected object.  Leave such fields blank for now, and fill in during the
 * set_objects call.
 **/
static int xfs_init_task(task_context_t * context)
{
	option_descriptor_t * option = context->option_descriptors->option;
	int rc;

	LOG_ENTRY();

	context->min_selected_objects = 0;
	context->max_selected_objects = 0;
	context->option_descriptors->count = 0;

	switch (context->action) {
	
	case EVMS_Task_mkfs:

		rc = fs_init_mkfs_acceptable_objects(context);
		if (rc) {
			goto out;
		}

		if (!xfsutils_support) {
			int answer = 0;
			char * choice_text[2] = {"Ok", NULL};
			EngFncs->user_message(my_plugin_record, &answer, choice_text,
				"xfsutils were either not found or not at the "
				"proper version. xfsutils must be version 2.0.0 "
				"or later to function properly with this FSIM. "
				"xfsutils must also reside in the search path "
				"specified by PATH. Please get/install the "
				"current version of xfsutils from "
				"http://oss.sgi.com/projects/xfs\n");
			rc = E_CANCELED;
			goto out;
		}

		context->option_descriptors->count = MKFS_XFS_OPTIONS_COUNT;

		/* Set Volume Label option */
		SET_STRING(option[MKFS_SETVOL_INDEX].name, "vollabel");
		SET_STRING(option[MKFS_SETVOL_INDEX].title, "Volume Label");
		SET_STRING(option[MKFS_SETVOL_INDEX].tip, "Set the volume label for the file system.");
		option[MKFS_SETVOL_INDEX].type = EVMS_Type_String;
		option[MKFS_SETVOL_INDEX].min_len = 1;
		option[MKFS_SETVOL_INDEX].max_len = LV_NAME_SIZE;
		option[MKFS_SETVOL_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
						  EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		option[MKFS_SETVOL_INDEX].value.s = EngFncs->engine_alloc(LV_NAME_SIZE + 1);
		if (!option[MKFS_SETVOL_INDEX].value.s) {
			rc = ENOMEM;
			goto out;
		}

		/* Set Journal Volume option */
		SET_STRING(option[MKFS_JOURNAL_VOL_INDEX].name, "journalvol");
		SET_STRING(option[MKFS_JOURNAL_VOL_INDEX].title, "External Journal Volume");
		SET_STRING(option[MKFS_JOURNAL_VOL_INDEX].tip, "Set the volume for the journal if other than the XFS file system volume.");
		option[MKFS_JOURNAL_VOL_INDEX].type = EVMS_Type_String;
		option[MKFS_JOURNAL_VOL_INDEX].max_len = EVMS_VOLUME_NAME_SIZE + 1;
		option[MKFS_JOURNAL_VOL_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED |
						       EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
		option[MKFS_JOURNAL_VOL_INDEX].constraint_type = EVMS_Collection_List;
		get_volume_list(&option[MKFS_JOURNAL_VOL_INDEX].constraint.list,
				context->selected_objects);
		option[MKFS_JOURNAL_VOL_INDEX].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE + 1);
		if (!option[MKFS_JOURNAL_VOL_INDEX].value.s) {
			rc = ENOMEM;
			goto out;
		}

		/* Log size option */
		SET_STRING(option[MKFS_SETLOGSIZE_INDEX].name, "logsize");
		SET_STRING(option[MKFS_SETLOGSIZE_INDEX].title, "Log Size");
		SET_STRING(option[MKFS_SETLOGSIZE_INDEX].tip, "Set log size (in megabytes). Default log size is 0.4% of volume size. Used only if no external journal volume is specified.");
		option[MKFS_SETLOGSIZE_INDEX].type = EVMS_Type_Real32;
		option[MKFS_SETLOGSIZE_INDEX].unit = EVMS_Unit_Megabytes;
		option[MKFS_SETLOGSIZE_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
		option[MKFS_SETLOGSIZE_INDEX].constraint_type = EVMS_Collection_Range;
		option[MKFS_SETLOGSIZE_INDEX].constraint.range = EngFncs->engine_alloc(sizeof(value_range_t));
		if (!option[MKFS_SETLOGSIZE_INDEX].constraint.range) {
			rc = ENOMEM;
			goto out;
		}
		option[MKFS_SETLOGSIZE_INDEX].constraint.range->min.r32 = 2.0;
		option[MKFS_SETLOGSIZE_INDEX].constraint.range->max.r32 = 256.0;
		option[MKFS_SETLOGSIZE_INDEX].constraint.range->increment.r32 = 0.1;
		option[MKFS_SETLOGSIZE_INDEX].value.r32 = 4.6;

		/* Force option */
		SET_STRING(option[MKFS_FORCE_INDEX].name, "force");
		SET_STRING(option[MKFS_FORCE_INDEX].title, "Force");
		SET_STRING(option[MKFS_FORCE_INDEX].tip, "Force the mkfs even if XFS thinks the volume has an existing filesystem");
		option[MKFS_FORCE_INDEX].type = EVMS_Type_Boolean;
		option[MKFS_FORCE_INDEX].unit = EVMS_Unit_None;
		option[MKFS_FORCE_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
		option[MKFS_FORCE_INDEX].constraint_type = EVMS_Collection_None;
		option[MKFS_FORCE_INDEX].value.b = FALSE;

		context->min_selected_objects = 1;
		context->max_selected_objects = 1;
		break;

	case EVMS_Task_fsck:
		if (!xfsutils_support) {
			int answer = 0;
			char * choice_text[2] = {"Ok", NULL};
			EngFncs->user_message(my_plugin_record, &answer, choice_text,
				"xfsutils were either not found or not at the "
				"proper version. xfsutils must be version 2.0.0 "
				"or later to function properly with this FSIM. "
				"xfsutils must also reside in the search path "
				"specified by PATH. Please get/install the "
				"current version of xfsutils from "
				"http://oss.sgi.com/projects/xfs\n");
			rc = E_CANCELED;
			goto out;
		}

		context->option_descriptors->count = FSCK_XFS_OPTIONS_COUNT;

		/* Read-only check option */
		SET_STRING(option[FSCK_READONLY_INDEX].name, "readonly");
		SET_STRING(option[FSCK_READONLY_INDEX].title, "Check Read-Only");
		SET_STRING(option[FSCK_READONLY_INDEX].tip, "Check the file system READ ONLY.  Report but do not correct errors.");
		option[FSCK_READONLY_INDEX].type = EVMS_Type_Boolean;
		option[FSCK_READONLY_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
		option[FSCK_READONLY_INDEX].value.b = TRUE;

		/* Verbose output option */
		SET_STRING(option[FSCK_VERBOSE_INDEX].name, "verbose");
		SET_STRING(option[FSCK_VERBOSE_INDEX].title, "Verbose Output");
		SET_STRING(option[FSCK_VERBOSE_INDEX].tip, "Log fsck.xfs details and debug information.");
		option[FSCK_VERBOSE_INDEX].type = EVMS_Type_Boolean;
		option[FSCK_VERBOSE_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
		option[FSCK_VERBOSE_INDEX].value.b = FALSE;

		context->min_selected_objects = 0;
		context->max_selected_objects = 0;

		break;

	case EVMS_Task_Expand:
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		break;

	default:
		rc = EINVAL;
		goto out;
	}

	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_set_option
 *
 * Examine the specified value, and determine if it is valid for the task
 * and option_descriptor index. If it is acceptable, set that value in the
 * appropriate entry in the option_descriptor. The value may be adjusted
 * if necessary/allowed. If so, set the effect return value accordingly.
 **/
static int xfs_set_option(task_context_t * context,
			  u_int32_t index,
			  value_t * value,
			  task_effect_t * effect)
{
	option_descriptor_t * option = context->option_descriptors->option;
	int rc = 0;

	LOG_ENTRY();

	switch (context->action) {
	
	case EVMS_Task_mkfs:
		switch (index) {
		
		case MKFS_SETVOL_INDEX:
			strncpy(option[index].value.s, value->s, LV_NAME_SIZE);
			break;

		case MKFS_JOURNAL_VOL_INDEX:
			strncpy(option[index].value.s, value->s, EVMS_VOLUME_NAME_SIZE);
			break;

		case MKFS_SETLOGSIZE_INDEX:
			option[index].value.r32 = value->r32;
			break;

		case MKFS_FORCE_INDEX:
			option[index].value.b = value->b;
			break;

		default:
			break;
		}
		break;

	case EVMS_Task_fsck:
		switch (index) {
		
		case FSCK_READONLY_INDEX:
			option[index].value.b = value->b;
			break;

		case FSCK_VERBOSE_INDEX:
			option[index].value.b = value->b;
			break;

		default:
			break;
		}
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_set_volumes
 *
 * Validate the volumes in the selected_objects list in the task context.
 * Remove from the selected objects lists any volumes which are not
 * acceptable. For unacceptable volumes, create a declined_handle_t
 * structure with the reason why it is not acceptable, and add it to the
 * declined_volumes list. Modify the acceptable_objects list in the task
 * context as necessary based on the selected objects and the current
 * settings of the options. Modify any option settings as necessary based
 * on the selected objects. Return the appropriate task_effect_t settings
 * if the object list(s), minimum or maximum objects selected, or option
 * settings have changed.
 **/
static int xfs_set_volumes(task_context_t * context,
			   list_anchor_t declined_volumes,
			   task_effect_t * effect)
{
	logical_volume_t * vol;
	double max_log_size;
	int64_t log_size;
	int rc = 0;

	LOG_ENTRY();

	if (context->action != EVMS_Task_mkfs) {
		goto out;
	}

	/* get the selected volume */
	vol = EngFncs->first_thing(context->selected_objects, NULL);
	if (!vol) {
		rc = ENODATA;
		goto out;
	}

	if (EngFncs->is_mounted(vol->dev_node, NULL)) {
		/* If mounted, can't mkfs.xfs. */
		rc = EBUSY;
		goto out;
	}

	if (vol->vol_size * EVMS_VSECTOR_SIZE < MINXFS) {
		/***************************************************
		 *  TODO - move this volume to unacceptable list   *
		 ***************************************************/
		MESSAGE("The size of volume %s is %"PRIu64" bytes. mkfs.xfs "
			"requires a minimum of %u bytes to build an XFS file "
			"system.", vol->dev_node,
			vol->vol_size * EVMS_VSECTOR_SIZE, MINXFS );
		rc = EPERM;
		goto out;
	}

	log_size = vol->vol_size / 8196;
	log_size = log_size / 2; // = K
	log_size = log_size / 1024; // = M
	log_size = max(log_size, XFS_MIN_LOG_SIZE);
	log_size = max(log_size, XFS_DEFAULT_LOG_SIZE);

	/* calculate and set upper log size bound */
	max_log_size = min(XFS_MAX_LOG_SIZE, (vol->vol_size/8)/2048); // 1/8th max /2048 = convert to MB
	max_log_size = min(max_log_size, (vol->vol_size - XFS_MIN_DATA_SECTORS)/2048);
	max_log_size = max(max_log_size, XFS_MIN_LOG_SIZE); // max can't be less than min
	context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].constraint.range->max.r32 = max_log_size;

	/* make sure log size is not greater than max log size */
	if (log_size > max_log_size) {
		log_size = max_log_size;
	}
	/* initialize log size option */
	context->option_descriptors->option[MKFS_SETLOGSIZE_INDEX].value.r32 = (unsigned int)log_size;
	*effect |= EVMS_Effect_Reload_Options;

	/* Remove selected volume from external log selection list. */
	get_volume_list(&(context->option_descriptors->option[MKFS_JOURNAL_VOL_INDEX].constraint.list),
			context->selected_objects);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_get_volume_info
 *
 * Return any additional information that you wish to provide about the
 * volume.  The Engine privides an external API to get the information
 * stored in the logical_volume_t.  This call is to get any other
 * information about the volume that is not specified in the
 * logical_volume_t.  Any piece of information you wish to provide must be
 * in an extended_info_t structure.  Use the Engine's engine_alloc() to
 * allocate the memory for the extended_info_t.  Also use engine_alloc() to
 * allocate any strings that may go into the extended_info_t.  Then use
 * engine_alloc() to allocate an extended_info_array_t with enough entries
 * for the number of exteneded_info_t structures you are returning.  Fill
 * in the array and return it in *info.
 *
 * If you have extended_info_t descriptors that themselves may have more
 * extended information, set the EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag
 * in the extended_info_t flags field.  If the caller wants more information
 * about a particular extended_info_t item, this API will be called with a
 * pointer to the sotrage_object_t and with a pointer to the name of the
 * extended_info_t item.  In that case, return an extended_info_array_t with
 * further information about the item.  Each of those items may have the
 * EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag set if you desire.  It is your
 * resposibility to give the items unique names so that you know which item
 * the caller is asking additional information for.  If info_name is NULL,
 * the caller just wants top level information about the object.
 */
static int xfs_get_volume_info(logical_volume_t * volume,
			       char * info_name,
			       extended_info_array_t ** info)
{
	xfs_volume_t * xfs_vol = volume->private_data;
	extended_info_array_t * Info;
	char magic[12];
	int rc;

	LOG_ENTRY();

	if (!xfs_vol) {
		rc = EINVAL;
		goto out;
	}

	if (info_name) {
		/* No extra information available for any of the descriptors. */
		rc = EINVAL;
		goto out;
	}

	if (xfs_vol->sb) {
		/* Get and validate current XFS superblock */
		rc = xfs_get_superblock(volume, xfs_vol->sb);
		if (rc) {
			goto out;
		}

		xfs_get_fs_limits(volume, &volume->min_fs_size,
				  &volume->max_vol_size, &volume->max_fs_size);

		Info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
					     10 * sizeof(extended_info_t));
		if (!Info) {
			rc = ENOMEM;
			goto out;
		}

		Info->count = 10;

		SET_STRING(Info->info[0].name, "MagicNumber");
		SET_STRING(Info->info[0].title, "XFS Magic Number");
		SET_STRING(Info->info[0].desc, "XFS Magic Number.");
		Info->info[0].type = EVMS_Type_String;
		sprintf(magic, "0x%x", xfs_vol->sb->sb_magicnum);
		SET_STRING(Info->info[0].value.s, magic);

		SET_STRING(Info->info[1].name, "Version");
		SET_STRING(Info->info[1].title, "XFS SB Version Number");
		SET_STRING(Info->info[1].desc, "XFS Super Block Version Number.");
		Info->info[1].type = EVMS_Type_Unsigned_Int32;
		Info->info[1].value.ui32 = xfs_vol->sb->sb_versionnum & XFS_SB_VERSION_NUMBITS;

		SET_STRING(Info->info[2].name, "State");
		SET_STRING(Info->info[2].title, "XFS State");
		SET_STRING(Info->info[2].desc, "The state of XFS.");
		Info->info[2].type = EVMS_Type_String;
		SET_STRING(Info->info[2].value.s,
			   (xfs_vol->sb->sb_flags == XFS_SBF_READONLY) ?
			   "ReadOnly" : "None");

		SET_STRING(Info->info[3].name, "VolLabel");
		SET_STRING(Info->info[3].title, "Volume Label");
		SET_STRING(Info->info[3].desc, "File system volume label chosen via mkfs.xfs.");
		Info->info[3].type = EVMS_Type_String;
		SET_STRING(Info->info[3].value.s, xfs_vol->sb->sb_fname);

		if (xfs_vol->log_vol) {
			SET_STRING(Info->info[4].name, "ExtLog");
			SET_STRING(Info->info[4].title, "External Log Volume");
			SET_STRING(Info->info[4].desc, "Location of exteral journal");
			Info->info[4].type = EVMS_Type_String;
			SET_STRING(Info->info[4].value.s, xfs_vol->log_vol->name);
		} else {
			SET_STRING(Info->info[4].name, "LogSize");
			SET_STRING(Info->info[4].title, "Inline Log Size");
			SET_STRING(Info->info[4].desc, "Size of the inline file system log (included in the file system size).");
			Info->info[4].type = EVMS_Type_Unsigned_Int64;
			Info->info[4].unit = EVMS_Unit_Sectors;
			Info->info[4].value.ui64 = xfs_vol->sb->sb_logblocks *
						   (xfs_vol->sb->sb_blocksize / EVMS_VSECTOR_SIZE);
		}

		SET_STRING(Info->info[5].name, "xfsutils");
		SET_STRING(Info->info[5].title, "XFS Utilities Version");
		SET_STRING(Info->info[5].desc, "XFS utilities version number.");
		Info->info[5].type = EVMS_Type_String;
		SET_STRING(Info->info[5].value.s, xfsutils_version);

		SET_STRING(Info->info[6].name, "agcount");
		SET_STRING(Info->info[6].title, "AG Count");
		SET_STRING(Info->info[6].desc, "Number of allocation groups");
		Info->info[6].type = EVMS_Type_Unsigned_Int32;
		Info->info[6].value.ui32 = xfs_vol->sb->sb_agcount;

		SET_STRING(Info->info[7].name, "agsize");
		SET_STRING(Info->info[7].title, "AG Size");
		SET_STRING(Info->info[7].desc, "Size of allocation groups");
		Info->info[7].type = EVMS_Type_Unsigned_Int64;
		Info->info[7].unit = EVMS_Unit_Kilobytes;
		Info->info[7].value.ui64 = xfs_vol->sb->sb_agblocks * 4;

		SET_STRING(Info->info[8].name, "sunit");
		SET_STRING(Info->info[8].title, "Stripe Unit");
		SET_STRING(Info->info[8].desc, "Size of stripe unit");
		Info->info[8].type = EVMS_Type_Unsigned_Int32;
		Info->info[8].unit = EVMS_Unit_Sectors;
		Info->info[8].value.ui32 = xfs_vol->sb->sb_unit;

		SET_STRING(Info->info[9].name, "swidth");
		SET_STRING(Info->info[9].title, "Stripe Width");
		SET_STRING(Info->info[9].desc, "Width of stripe");
		Info->info[9].type = EVMS_Type_Unsigned_Int32;
		Info->info[9].unit = EVMS_Unit_Sectors;
		Info->info[9].value.ui32 = xfs_vol->sb->sb_width;

		*info = Info;
	} else {
		/* Get and validate current XFS external log superblock */
		rc = xfs_get_log_superblock(volume, xfs_vol->log_sb);
		if (rc) {
			goto out;
		}

		Info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
					     6 * sizeof(extended_info_t));
		if (!Info) {
			rc = ENOMEM;
			goto out;
		}

		Info->count = 5;

		SET_STRING(Info->info[0].name, "LogMagicNumber");
		SET_STRING(Info->info[0].title, "XFS External Log Magic Number");
		SET_STRING(Info->info[0].desc, "XFS External Log Magic Number.");
		Info->info[0].type = EVMS_Type_String;
		sprintf(magic, "0x%x", xfs_vol->log_sb->h_magicno);
		SET_STRING(Info->info[0].value.s, magic);

		SET_STRING(Info->info[1].name, "LogVersion");
		SET_STRING(Info->info[1].title, "XFS External Log Version Number");
		SET_STRING(Info->info[1].desc, "XFS External Log Version Number.");
		Info->info[1].type = EVMS_Type_Unsigned_Int32;
		Info->info[1].value.i32 = xfs_vol->log_sb->h_version;

		SET_STRING(Info->info[2].name, "fsvol");
		SET_STRING(Info->info[2].title, "Journal for:");
		SET_STRING(Info->info[2].desc, "Volume for which this is an external log volume");
		Info->info[2].type = EVMS_Type_String;
		SET_STRING(Info->info[2].value.s,
			   xfs_vol->fs_vol ? xfs_vol->fs_vol->name : "UNKNOWN");

		SET_STRING(Info->info[3].name, "LogSize");
		SET_STRING(Info->info[3].title, "Log Size");
		SET_STRING(Info->info[3].desc, "Size of the file system log (included in the file system size).");
		Info->info[3].type = EVMS_Type_Unsigned_Int64;
		Info->info[3].unit = EVMS_Unit_Sectors;
		if (xfs_vol->fs_vol) {
			Info->info[3].value.ui64 = ((xfs_volume_t *)(xfs_vol->fs_vol->private_data))->sb->sb_logblocks *
						   (((xfs_volume_t *)(xfs_vol->fs_vol->private_data))->sb->sb_blocksize /
						    EVMS_VSECTOR_SIZE);
		}

		SET_STRING(Info->info[4].name, "xfsutils");
		SET_STRING(Info->info[4].title, "XFS Utilities Version");
		SET_STRING(Info->info[4].desc, "XFS utilities version number.");
		Info->info[4].type = EVMS_Type_String;
		SET_STRING(Info->info[4].value.s, xfsutils_version);

		*info = Info;
	}
	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_get_plugin_info
 *
 * Returns Plugin specific information.
 **/
static int xfs_get_plugin_info(char * descriptor_name,
			       extended_info_array_t ** info)
{
	extended_info_array_t * Info;
	char string[64];
	int rc = 0;

	LOG_ENTRY();

	Info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     6 * sizeof(extended_info_t));
	if (!Info) {
		*info = NULL;
		rc = ENOMEM;
		goto out;
	}

	Info->count = 6;

	SET_STRING(Info->info[0].name, "Short Name");
	SET_STRING(Info->info[0].title, "Short Name");
	SET_STRING(Info->info[0].desc, "A short name given to this plugin.");
	Info->info[0].type = EVMS_Type_String;
	SET_STRING(Info->info[0].value.s, my_plugin_record->short_name);

	SET_STRING(Info->info[1].name, "Long Name");
	SET_STRING(Info->info[1].title, "Long Name");
	SET_STRING(Info->info[1].desc, "A long name given to this plugin.");
	Info->info[1].type = EVMS_Type_String;
	SET_STRING(Info->info[1].value.s, my_plugin_record->long_name);

	SET_STRING(Info->info[2].name, "Type");
	SET_STRING(Info->info[2].title, "Plugin Type");
	SET_STRING(Info->info[2].desc, "There are various types of plugins; each responsible for some kind of storage object.");
	Info->info[2].type = EVMS_Type_String;
	SET_STRING(Info->info[2].value.s, "File System Interface Module");

	sprintf(string, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);

	SET_STRING(Info->info[3].name, "Version");
	SET_STRING(Info->info[3].title, "Plugin Version");
	SET_STRING(Info->info[3].desc, "This is the version number of the plugin.");
	Info->info[3].type = EVMS_Type_String;
	SET_STRING(Info->info[3].value.s, string);

	sprintf(string, "%d.%d.%d",
		my_plugin_record->required_engine_api_version.major,
		my_plugin_record->required_engine_api_version.minor,
		my_plugin_record->required_engine_api_version.patchlevel);

	SET_STRING(Info->info[4].name, "Required Engine Services API Version");
	SET_STRING(Info->info[4].title, "Required Engine Services API Version");
	SET_STRING(Info->info[4].desc, "This is the version of the Engine services that this plug-in requires. It will not run on older versions of the Engine services.");
	Info->info[4].type = EVMS_Type_String;
	SET_STRING(Info->info[4].value.s, string);

	sprintf(string, "%d.%d.%d",
		my_plugin_record->required_plugin_api_version.fsim.major,
		my_plugin_record->required_plugin_api_version.fsim.minor,
		my_plugin_record->required_plugin_api_version.fsim.patchlevel);

	SET_STRING(Info->info[5].name, "Required Engine FSIM API Version");
	SET_STRING(Info->info[5].title, "Required Engine FSIM API Version");
	SET_STRING(Info->info[5].desc, "This is the version of the Engine FSIM API that this plug-in requires. It will not run on older versions of the Engine FSIM API.");
	Info->info[5].type = EVMS_Type_String;
	SET_STRING(Info->info[5].value.s, string);

	*info = Info;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_can_expand_by
 *
 * How much can file system expand?
 **/
static int xfs_can_expand_by(logical_volume_t * volume,
			     sector_count_t * delta)
{
	int rc = 0;

	LOG_ENTRY();

	if (!xfsutils_support) {
		rc = EINVAL;
	} else if (!EngFncs->is_mounted(volume->dev_node, NULL)) {
		/* If not mounted, can't expand. online only. */
		rc = EBUSY;
	} else if (volume->fs_size + *delta > volume->max_fs_size) {
		*delta = volume->max_fs_size - volume->fs_size;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_can_shrink_by
 *
 * How much can file system shrink? Currently unsupported in XFS.
 **/
static int xfs_can_shrink_by(logical_volume_t * volume,
			     sector_count_t * delta)
{
	LOG_ENTRY();
	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

static fsim_functions_t fsim_ops = {
	.setup_evms_plugin	= xfs_setup,
	.cleanup_evms_plugin	= xfs_cleanup,
	.probe			= xfs_probe,
	.can_mkfs		= xfs_can_mkfs,
	.can_unmkfs		= xfs_can_unmkfs,
	.can_fsck		= xfs_can_fsck,
	.get_fs_size		= xfs_get_fs_size,
	.get_fs_limits		= xfs_get_fs_limits,
	.can_expand_by		= xfs_can_expand_by,
	.can_shrink_by		= xfs_can_shrink_by,
	.expand			= xfs_expand,
	.shrink			= xfs_shrink,
	.mkfs			= xfs_mkfs,
	.mkfs_setup		= xfs_mkfs_setup,
	.discard		= xfs_discard,
	.fsck			= xfs_fsck,
	.unmkfs			= xfs_unmkfs,
	.unmkfs_setup		= xfs_unmkfs_setup,
	.get_option_count	= xfs_get_option_count,
	.init_task		= xfs_init_task,
	.set_option		= xfs_set_option,
	.set_volumes		= xfs_set_volumes,
	.get_volume_info	= xfs_get_volume_info,
	.get_plugin_info	= xfs_get_plugin_info
};

plugin_record_t xfs_plugin_record = {
	.id = SetPluginID(EVMS_OEM_IBM,
			  EVMS_FILESYSTEM_INTERFACE_MODULE,
			  FS_TYPE_XFS),
	.version = {
		.major		= MAJOR_VERSION,
		.minor		= MINOR_VERSION,
		.patchlevel	= PATCH_LEVEL
	},
	.required_engine_api_version = {
		.major		= 13,
		.minor		= 0,
		.patchlevel	= 0
	},
	.required_plugin_api_version = {
		.fsim = {
			.major		= 10,
			.minor		= 0,
			.patchlevel	= 0
		}
	},
	.short_name = "XFS",
	.long_name = "XFS File System Interface Module",
	.oem_name = "IBM",
	.functions = {
		.fsim = &fsim_ops
	},
	.container_functions = NULL
};

plugin_record_t *evms_plugin_records[] = {
	&xfs_plugin_record,
	NULL
};

