/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: snap_activate.c
 *
 * Helper functions for activating and deactivating snapshots.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <plugin.h>
#include "snapshot.h"

/**
 * get_origin_child_targets
 *
 * Get the list of DM targets for the origin child. Incredibly, this works
 * for both activation and deactivation.
 **/
static int get_origin_child_targets(snapshot_volume_t * org_volume,
				    dm_target_t ** origin_targets)
{
	int rc;

	LOG_ENTRY();

	rc = EngFncs->dm_get_targets(org_volume->child, origin_targets);
	if (rc) {
		LOG_ERROR("Error getting target list for origin child %s.\n",
			  org_volume->child->name);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * rename_origin_child
 *
 * Tell DM to rename the origin child. If the flag parameter is TRUE, it will
 * rename from the child name to the parent name. If the flag parameter is
 * FALSE, it will rename from the parent name to the child name.
 **/
static int rename_origin_child(snapshot_volume_t * org_volume, int flag)
{
	char * old_name, * new_name;
	int rc;

	LOG_ENTRY();

	/* Change the name. */
	old_name = flag ? org_volume->child->name : org_volume->parent->name;
	new_name = flag ? org_volume->parent->name : org_volume->child->name;

	rc = EngFncs->dm_rename(org_volume->child, old_name, new_name);
	if (rc) {
		LOG_ERROR("Error renaming origin child name from %s to %s.\n",
			  old_name, new_name);
		goto out;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * switch_origin_names
 *
 * Switch the names between the origin child and parent storage objects.
 * This operation will only be temporary, during snapshot activation, and
 * will be switched back. The engine will not notice the name switch.
 **/
static void switch_origin_names(snapshot_volume_t * org_volume)
{
	char tmp_name[EVMS_NAME_SIZE+1];

	LOG_ENTRY();

	strncpy(tmp_name, org_volume->parent->name, EVMS_NAME_SIZE);
	strncpy(org_volume->parent->name, org_volume->child->name, EVMS_NAME_SIZE);
	strncpy(org_volume->child->name, tmp_name, EVMS_NAME_SIZE);

	LOG_EXIT_VOID();
}

/**
 * activate_new_origin_child
 *
 * Activate a new DM device that looks just like the origin child. This
 * activation will use the origin parent object, but will have the origin
 * child's name and target list.
 **/
static int activate_new_origin_child(snapshot_volume_t * org_volume,
				     dm_target_t * origin_targets)
{
	int rc;

	LOG_ENTRY();

	switch_origin_names(org_volume);

	rc = EngFncs->dm_activate(org_volume->parent, origin_targets);
	if (rc) {
		LOG_ERROR("Error activating new origin child %s.\n",
			  org_volume->parent->name);
	}

	switch_origin_names(org_volume);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * deactivate_new_origin_child
 **/
static int deactivate_new_origin_child(snapshot_volume_t * org_volume)
{
	int rc;

	LOG_ENTRY();

	switch_origin_names(org_volume);

	rc = EngFncs->dm_deactivate(org_volume->parent);
	if (rc) {
		LOG_ERROR("Error deactivating new origin child %s.\n",
			  org_volume->parent->name);
	}

	switch_origin_names(org_volume);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * switch_origin_device_numbers
 *
 * Just like switch_origin_names, but now we're switching the major and minor
 * numbers between the origin child and parent. Unlike switch_origin_names,
 * this switch will be permanent.
 **/
static void switch_origin_device_numbers(snapshot_volume_t * org_volume)
{
	u_int32_t tmp_dev;

	LOG_ENTRY();

	tmp_dev = org_volume->child->dev_major;
	org_volume->child->dev_major = org_volume->parent->dev_major;
	org_volume->parent->dev_major = tmp_dev;

	tmp_dev = org_volume->child->dev_minor;
	org_volume->child->dev_minor = org_volume->parent->dev_minor;
	org_volume->parent->dev_minor = tmp_dev;

	LOG_EXIT_VOID();
}

/**
 * load_snapshot_module
 *
 * Attempt to load the "dm-snapshot" kernel module, which contains both the
 * snapshot and the snapshot-origin targets.
 **/
static int load_snapshot_module(void)
{
	char *argv[] = {"modprobe", "dm-snapshot", NULL};
	pid_t pid;
	int status, rc;

	LOG_ENTRY();

	pid = EngFncs->fork_and_execvp(NULL, argv, NULL, NULL, NULL);
	if (pid < 0) {
		rc = errno;
		LOG_ERROR("Error running modprobe to load the snapshot "
			  "kernel module: %s.\n", strerror(rc));
		goto out;
	}

	rc = waitpid(pid, &status, 0);
	if (rc < 0) {
		rc = errno;
		LOG_ERROR("Error waiting for modprobe to complete: %s.\n",
			  strerror(rc));
		goto out;
	}

	if (!WIFEXITED(status)) {
		LOG_ERROR("Error: modprobe completed abnormally.\n");
		rc = EINTR;
		goto out;
	}

	rc = WEXITSTATUS(status);
	LOG_DEFAULT("modprobe completed with rc = %d \n", rc);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * activate_origin_parent
 *
 * Activate the origin parent with a snapshot-origin mapping. This is actually
 * a re-activation of the origin child's DM device with a new mapping.
 **/
static int activate_origin_parent(snapshot_volume_t * org_volume)
{
	dm_target_t target;
	dm_device_t origin;
	int rc;

	LOG_ENTRY();

	target.start = 0;
	target.length = org_volume->parent->size;
	target.type = DM_TARGET_SNAPSHOT_ORG;
	target.data.linear = &origin;
	target.data.linear->major = org_volume->child->dev_major;
	target.data.linear->minor = org_volume->child->dev_minor;
	target.data.linear->start = 0;
	target.params = NULL;
	target.next = NULL;

	rc = EngFncs->dm_activate(org_volume->parent, &target);
	if (rc) {
		/* The snapshot-origin target is in the dm-snapshot kernel
		 * module, so it won't get auto-loaded by device-mapper. If
		 * the activation fails, try loading the snapshot module and
		 * try the activation again.
		 */
		rc = load_snapshot_module();
		if (!rc) {
			rc = EngFncs->dm_activate(org_volume->parent, &target);
		}
		if (rc) {
			LOG_ERROR("Error activating origin %s.\n",
				  org_volume->parent->name);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * deactivate_origin_parent
 *
 * Deactivate the origin parent. This is actually a reactivation of the
 * origin child's DM device with the child's original DM mapping.
 **/
static int deactivate_origin_parent(snapshot_volume_t * org_volume,
				    dm_target_t * origin_targets)
{
	int rc;

	LOG_ENTRY();

	rc = EngFncs->dm_activate(org_volume->parent, origin_targets);
	if (rc) {
		LOG_ERROR("Error activating new origin child %s.\n",
			  org_volume->parent->name);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * activate_origin
 *
 * Perform all steps necessary to activate the snapshot origin. The
 * basic procedure is:
 * 1. Get the origin child's dm target list.
 * 2. DM-rename the origin child to the origin parent's name.
 * 3. Activate the origin parent with the origin child's target list.
 *    This includes temporarily switching the origin child and
 *    parent names.
 * 4. Switch the origin child and parent objects' device numbers.
 * 5. Reactivate the origin parent with an origin mapping.
 **/
int activate_origin(snapshot_volume_t * org_volume)
{
	dm_target_t * origin_targets = NULL;
	int rc;

	LOG_ENTRY();

	rc = get_origin_child_targets(org_volume, &origin_targets);
	if (rc) {
		goto out;
	}

	rc = rename_origin_child(org_volume, TRUE);
	if (rc) {
		goto out;
	}

	rc = activate_new_origin_child(org_volume, origin_targets);
	if (rc) {
		rename_origin_child(org_volume, FALSE);
		goto out;
	}

	switch_origin_device_numbers(org_volume);

	rc = activate_origin_parent(org_volume);
	if (rc) {
		switch_origin_device_numbers(org_volume);
		deactivate_new_origin_child(org_volume);
		rename_origin_child(org_volume, FALSE);
		goto out;
	}

out:
	if (rc) {
		LOG_ERROR("Error activating snapshot origin %s.\n",
			  org_volume->parent->name);
	}
	EngFncs->dm_deallocate_targets(origin_targets);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * deactivate_origin
 *
 * Deactivating the origin is almost the reverse of the activation process.
 * 1. Get the origin child's dm target list.
 * 2. Reactivate the origin parent with this target list, which is actually
 *    a reactivation of the origin child's DM device with its original
 *    mapping.
 * 3. Switch the origin child and parent objects' device numbers.
 * 4. Deactivate the origin parent, which includes temporarily switching the
 *    origin child and parent names.
 * 5. DM-rename the origin child from the parent's to the child's name.
 * 6. Copy the child object's device numbers to the parent.
 **/
int deactivate_origin(snapshot_volume_t * org_volume)
{
	dm_target_t * origin_targets = NULL;
	int rc;

	LOG_ENTRY();

	rc = get_origin_child_targets(org_volume, &origin_targets);
	if (rc) {
		goto out;
	}

	rc = deactivate_origin_parent(org_volume, origin_targets);
	if (rc) {
		goto out;
	}

	switch_origin_device_numbers(org_volume);

	rc = deactivate_new_origin_child(org_volume);
	if (rc) {
		goto out;
	}

	rc = rename_origin_child(org_volume, FALSE);
	if (rc) {
		goto out;
	}

out:
	EngFncs->dm_deallocate_targets(origin_targets);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * activate_snapshot_sibling
 *
 * Activate a linear device that will sit between the actual snapshot child and
 * the new snapshot device. Since Device-Mapper tries to use the entire child
 * device, we need the sibling to prevent the EVMS feature headers and snapshot
 * metadata from being overwritten.
 **/
int activate_snapshot_sibling(snapshot_volume_t * snap_volume)
{
	dm_target_t target;
	dm_device_t linear;
	int rc;

	LOG_ENTRY();

	target.start = 0;
	target.length = snap_volume->sibling->size;
	target.type = DM_TARGET_LINEAR;
	target.data.linear = &linear;
	target.data.linear->major = snap_volume->child->dev_major;
	target.data.linear->minor = snap_volume->child->dev_minor;
	target.data.linear->start = 0;
	target.params = NULL;
	target.next = NULL;

	rc = EngFncs->dm_activate(snap_volume->sibling, &target);
	if (rc) {
		LOG_ERROR("Error activating snapshot sibling %s\n",
			  snap_volume->sibling->name);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * deactivate_snapshot_sibling
 *
 * Deactivate the snapshot sibling. The only reason this should fail is
 * either if we've run out of memory to process the ioctl, or if DM simply
 * can't find the sibling device.
 **/
static int deactivate_snapshot_sibling(snapshot_volume_t * snap_volume)
{
	int rc;

	LOG_ENTRY();

	rc = EngFncs->dm_deactivate(snap_volume->sibling);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * suspend_origin_volume
 *
 * Suspend/resume the device for the volume on top of the origin-parent, but
 * only if it's not the same device as the origin parent.
 **/
static int suspend_origin_volume(snapshot_volume_t * org_volume, int suspend)
{
	storage_object_t *parent = org_volume->parent;
	logical_volume_t *volume = parent->volume;
	int rc = 0;

	LOG_ENTRY();

	if (volume && volume->dev_major &&
	    volume->dev_minor != volume->object->dev_minor) {
		rc = EngFncs->dm_suspend_volume(volume, suspend);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * suspend_origin_parent
 *
 * Suspend/resume the device for the origin-parent.
 **/
static int suspend_origin_parent(snapshot_volume_t * org_volume, int suspend)
{
	int rc;

	LOG_ENTRY();

	rc = EngFncs->dm_suspend(org_volume->parent, suspend);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * suspend_origin
 *
 * Suspend both the origin parent and volume. The parent must be suspended so
 * it can later be resumed, which is when the kernel will update the origin's
 * chunk-size. The volume must be suspended because that is when the kernel
 * will lock the VFS. If the parent and the volume are the same device, we'll
 * obviously only suspend once.
 **/
static int suspend_origin(snapshot_volume_t * org_volume)
{
	int rc;

	LOG_ENTRY();

	EngFncs->dm_set_suspended_flag(TRUE);

	rc = suspend_origin_volume(org_volume, TRUE);
	if (rc) {
		goto out;
	}

	rc = suspend_origin_parent(org_volume, TRUE);
	if (rc) {
		suspend_origin_volume(org_volume, FALSE);
		goto out;
	}

out:
	if (rc) {
		EngFncs->dm_set_suspended_flag(FALSE);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * resume_origin
 *
 * Resume both the origin parent and volume. The parent must be resumed because
 * that is when the kernel will update the origin's chunk-size. The volume must
 * be resumed because that is when the kernel will unlock the VFS. If the parent
 * and the volume are the same device, we'll obviously only resume once.
 *
 * The ordering of resume is the opposite of suspend. Since we suspend the
 * volume first, and then the parent, all pending I/Os are queued on the volume
 * (and none on the parent). If we resumed the volume first, all those pending
 * I/Os would temporarily queue up on the parent before we have a chance to
 * resume it, which would just a waste of time.
 **/
static int resume_origin(snapshot_volume_t * org_volume)
{
	int rc = 0;

	LOG_ENTRY();

	rc = suspend_origin_parent(org_volume, FALSE);
	if (rc) {
		goto out;
	}

	rc = suspend_origin_volume(org_volume, FALSE);
	if (rc) {
		goto out;
	}

	EngFncs->dm_set_suspended_flag(FALSE);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * activate_snapshot_parent_v3
 *
 * Snapshot activation sequence for DM ioctl version 3. Just call dm_activate().
 **/
static int activate_snapshot_parent_v3(snapshot_volume_t * snap_volume,
				       dm_target_t * target)
{
	int rc;
	LOG_ENTRY();
	rc = EngFncs->dm_activate(snap_volume->parent, target);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * activate_snapshot_parent_v4
 *
 * Snapshot activation sequence for DM ioctl version 4.
 * 1) Create the snapshot device (no initial mapping).
 * 2) Suspend the origin device.
 * 3) Load the mapping into the snapshot device.
 * 4) Resume the snapshot device.
 * 5) Resume the origin device.
 **/
static int activate_snapshot_parent_v4(snapshot_volume_t * snap_volume,
				       dm_target_t * target)
{
	int rc, suspended = FALSE;

	LOG_ENTRY();

	rc = EngFncs->dm_create(snap_volume->parent);
	if (rc) {
		goto out;
	}

	rc = suspend_origin(snap_volume->origin);
	if (rc) {
		goto out;
	}
	suspended = TRUE;

	rc = EngFncs->dm_load_targets(snap_volume->parent, target);
	if (rc) {
		goto out;
	}

	rc = EngFncs->dm_suspend(snap_volume->parent, FALSE);

out:
	if (suspended) {
		resume_origin(snap_volume->origin);
	}
	if (rc) {
		EngFncs->dm_deactivate(snap_volume->parent);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * activate_snapshot_parent
 *
 * Activate the snapshot parent object.
 **/
static int activate_snapshot_parent(snapshot_volume_t * snap_volume)
{
	dm_target_t target;
	dm_target_snapshot_t snapshot;
	snapshot_volume_t * org_volume = snap_volume->origin;
	int rc;

	LOG_ENTRY();

	target.start = 0;
	target.length = snap_volume->origin->parent->size;
	target.type = DM_TARGET_SNAPSHOT;
	target.data.snapshot = &snapshot;
	target.data.snapshot->origin.major = org_volume->child->dev_major;
	target.data.snapshot->origin.minor = org_volume->child->dev_minor;
	target.data.snapshot->origin.start = 0;
	target.data.snapshot->origin_parent.major = org_volume->parent->volume->dev_major ?
						    org_volume->parent->volume->dev_major :
						    org_volume->parent->dev_major;
	target.data.snapshot->origin_parent.minor = org_volume->parent->volume->dev_minor ?
						    org_volume->parent->volume->dev_minor :
						    org_volume->parent->dev_minor;
	target.data.snapshot->origin_parent.start = 0;
	target.data.snapshot->snapshot.major = snap_volume->sibling->dev_major;
	target.data.snapshot->snapshot.minor = snap_volume->sibling->dev_minor;
	target.data.snapshot->snapshot.start = 0;
	target.data.snapshot->persistent = TRUE;
	target.data.snapshot->chunk_size = snap_volume->metadata->chunk_size;
	target.params = NULL;
	target.next = NULL;

	if (EngFncs->dm_get_version() == 3) {
		rc = activate_snapshot_parent_v3(snap_volume, &target);
	} else {
		rc = activate_snapshot_parent_v4(snap_volume, &target);
	}

	if (rc) {
		LOG_ERROR("Error activating snapshot %s\n",
			  snap_volume->parent->name);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * reload_snapshot_parent
 *
 * When deactivating the snapshot device, we can't delete the device while the
 * origin is suspended. This is because deactivating a device can cause a
 * hotplug event, which needs to run /sbin/hotplug. If the origin volume is
 * the root filesystem, this will cause the system to hang. To prevent this,
 * simply reload the snapshot device with an error map while the origin is
 * suspended. Then we can deactivate the device after the origin is resumed.
 **/
static int reload_snapshot_parent(snapshot_volume_t * snap_volume)
{
	dm_target_t target;
	int rc;

	LOG_ENTRY();

	target.start = 0;
	target.length = snap_volume->origin->parent->size;
	target.type = DM_TARGET_ERROR;
	target.data.linear = NULL;
	target.params = NULL;
	target.next = NULL;

	rc = EngFncs->dm_activate(snap_volume->parent, &target);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * deactivate_snapshot_parent
 *
 * Deactivate the snapshot parent object. The only reason this should fail is
 * either if we've run out of memory to process the ioctl, or if DM simply
 * can't find the snapshot device.
 **/
static int deactivate_snapshot_parent(snapshot_volume_t * snap_volume)
{
	int rc;

	LOG_ENTRY();

	if (EngFncs->dm_get_version() != 3) {
		suspend_origin(snap_volume->origin);
	}

	reload_snapshot_parent(snap_volume);

	if (EngFncs->dm_get_version() != 3) {
		resume_origin(snap_volume->origin);
	}

	rc = EngFncs->dm_deactivate(snap_volume->parent);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * activate_snapshot
 *
 * Activating a snapshot requires first activating the snapshot sibling
 * object, and then the snapshot parent object.
 **/
int activate_snapshot(snapshot_volume_t * snap_volume)
{
	int rc;

	LOG_ENTRY();

	rc = activate_snapshot_sibling(snap_volume);
	if (rc) {
		goto out;
	}

	rc = activate_snapshot_parent(snap_volume);
	if (rc) {
		deactivate_snapshot_sibling(snap_volume);
	} else {
		snap_volume->origin->active_count++;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * deactivate_snapshot
 *
 * Deactivate the snapshot by deactivating the snapshot parent, followed,
 * by the snapshot sibling.
 **/
void deactivate_snapshot(snapshot_volume_t * snap_volume)
{
	LOG_ENTRY();

	deactivate_snapshot_parent(snap_volume);
	deactivate_snapshot_sibling(snap_volume);
 	snap_volume->origin->active_count--;

	LOG_EXIT_VOID();
}

