/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: raid5_mgr.c
 *
 * Description: This file contains all of the required engine-plugin APIs
 *              for the Raid5 MD region manager.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include "md.h"
#include "xor.h"
#include "raid5_mgr.h"
#include "raid5_discover.h"

#define my_plugin_record raid5_plugin

/* Global variables */

kill_sectors_t * kill_sector_list_head = NULL;


/* Forward references */

static int raid5_write( storage_object_t * region,
			lsn_t              lsn,
			sector_count_t     count,
			void             * buffer );


/* Function: raid5_setup_evms_plugin
 *
 *  This function gets called shortly after the plugin is loaded by the
 *  Engine. It performs all tasks that are necessary before the initial
 *  discovery pass.
 */
static int raid5_setup_evms_plugin(engine_functions_t * functions) {
	int rc = 0;

	/* Parameter check */
	if (!functions) {
		return EINVAL;
	}

	EngFncs = functions;

	my_plugin = raid5_plugin;
	LOG_ENTRY();
	rc = md_register_name_space();

	if (rc != 0) {
		LOG_SERIOUS("Failed to register the MD name space.\n");
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/****** Region Checking Functions ******/


/* All of the following md_can_ functions return 0 if they are able to
 * perform the specified action, or non-zero if they cannot.
 */


/* Function: raid5_can_delete
 *
 *  Can we remove the specified MD logical volume
 */
static int raid5_can_delete( storage_object_t * region ) {

	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}


static int raid5_can_replace_child(storage_object_t *region,
				   storage_object_t *child,
				   storage_object_t *new_child)
{
	int rc;
	my_plugin = raid5_plugin;
	LOG_ENTRY();
	rc = md_can_replace_child(region, child, new_child);
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid5_discover
 *
 *  Examine all disk segments and find MD PVs. Assemble volume groups
 *  and export all MD logical volumes as EVMS regions.
 *
 *  All newly created regions must be added to the output list, and all
 *  segments from the input list must either be claimed or moved to the
 *  output list.
 */
static int raid5_discover( list_anchor_t input_list,
			   list_anchor_t output_list,
			   boolean final_call ) {
	int count = 0;

	my_plugin = raid5_plugin;
	LOG_ENTRY();

	/* Parameter check */
	if (!input_list || !output_list) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (final_call) {
		md_discover_final_call(input_list, output_list, &count);
	} else {
		md_discover_volumes(input_list, output_list);
		LOG_DETAILS("PV discovery complete.\n");

		/* LV discovery and exporting */
		raid5_discover_regions(output_list, &count, final_call);
		LOG_DETAILS("RAID4/5 volume discovery complete.\n");
	}

	LOG_EXIT_INT(count);
	return count;
}


/*
 * Input: a 'big' sector number,
 * Output: index of the data and parity disk, and the sector # in them.
 */
static lsn_t raid5_compute_sector(lsn_t vol_sector,
				  unsigned int raid_disks, unsigned int data_disks,
				  unsigned int * dd_idx, unsigned int * pd_idx,
				  raid5_conf_t * conf) {

	unsigned long long stripe;
	unsigned long long chunk_number;
	lsn_t              chunk_offset;
	lsn_t              new_sector;
	unsigned long long sectors_per_chunk = (unsigned long long) (conf->chunk_size >> 9);

	LOG_ENTRY();

	/*
	 * Compute the chunk number and the sector offset inside the chunk
	 */
	chunk_number = (unsigned long long) (vol_sector / sectors_per_chunk);
	chunk_offset = (lsn_t) (vol_sector % sectors_per_chunk);

	/*
	 * Compute the stripe number
	 */
	stripe = (unsigned long long) (chunk_number / data_disks);

	/*
	 * Compute the data disk and parity disk indexes inside the stripe
	 */
	*dd_idx = chunk_number % data_disks;

	/*
	 * Select the parity disk based on the user selected algorithm.
	 */
	if (conf->level == 4)
		*pd_idx = data_disks;
	else switch (conf->algorithm) {
		case ALGORITHM_LEFT_ASYMMETRIC:
			*pd_idx = data_disks - stripe % raid_disks;
			if (*dd_idx >= *pd_idx)
				(*dd_idx)++;
			break;
		case ALGORITHM_RIGHT_ASYMMETRIC:
			*pd_idx = stripe % raid_disks;
			if (*dd_idx >= *pd_idx)
				(*dd_idx)++;
			break;
		case ALGORITHM_LEFT_SYMMETRIC:
			*pd_idx = data_disks - stripe % raid_disks;
			*dd_idx = (*pd_idx + 1 + *dd_idx) % raid_disks;
			break;
		case ALGORITHM_RIGHT_SYMMETRIC:
			*pd_idx = stripe % raid_disks;
			*dd_idx = (*pd_idx + 1 + *dd_idx) % raid_disks;
			break;
		default:
			LOG_WARNING("raid5: unsupported algorithm %d\n", conf->algorithm);
			break;
		}

	/*
	 * Finally, compute the new sector number.
	 */
	new_sector = (lsn_t) (stripe * sectors_per_chunk + chunk_offset);
	LOG_DEBUG("new sector is %"PRIu64".\n", new_sector);
	LOG_EXIT_INT((int)new_sector);
	return new_sector;
}


static int get_child_run( md_volume_t       * volume,
			  lsn_t               lsn,
			  sector_count_t      count,
			  storage_object_t ** child_object,
			  lsn_t             * child_lsn,
			  sector_count_t    * child_count) {

	raid5_conf_t * conf = mdvol_to_conf(volume);
	unsigned int data_disk_index;
	unsigned int parity_disk_index;
	sector_count_t sectors_per_chunk = conf->chunk_size >> EVMS_VSECTOR_SIZE_SHIFT;

	LOG_ENTRY();

	*child_lsn = raid5_compute_sector(lsn,
					  conf->raid_disks, conf->raid_disks - 1,
					  &data_disk_index, &parity_disk_index,
					  conf);

	*child_object = conf->disks[data_disk_index].dev;

	*child_count = min(count, sectors_per_chunk - (*child_lsn & (sectors_per_chunk - 1)));

	LOG_EXIT_INT(0);
	return 0;
}



/****** Region Functions ******/


static int raid5_get_create_options( option_array_t * options,
				     char          ** spare_disk,
				     unsigned int   * chunk_size,
				     unsigned int   * raid_level,
				     unsigned int   * parity_algorithm ) {
	int i;
	int rc = 0;

	LOG_ENTRY();

	for (i = 0; i < options->count; i++) {

		if (options->option[i].is_number_based) {

			switch (options->option[i].number) {
			
			case MD_OPTION_SPARE_DISK_INDEX:
				/*
				 * Not worth validation, will catch errors when
				 * we try to find the original.
				 */
				*spare_disk = options->option[i].value.s;
				break;

			case MD_OPTION_CHUNK_SIZE_INDEX:
				*chunk_size = options->option[i].value.ui32;
				break;

			case MD_OPTION_RAID_LEVEL_INDEX:
				if (strcmp(options->option[i].value.s, RAID4_LEVEL_NAME) == 0) {
					*raid_level = 4;
				} else if (strcmp(options->option[i].value.s, RAID5_LEVEL_NAME) == 0) {
					*raid_level = 5;
				}
				break;

			case MD_OPTION_PARITY_ALGORITHM_INDEX:
				if (strcmp(options->option[i].value.s, ALGORITHM_LEFT_ASYMMETRIC_NAME) == 0) {
					*parity_algorithm = ALGORITHM_LEFT_ASYMMETRIC;
				} else if (strcmp(options->option[i].value.s, ALGORITHM_RIGHT_ASYMMETRIC_NAME) == 0) {
					*parity_algorithm = ALGORITHM_RIGHT_ASYMMETRIC;
				} else if (strcmp(options->option[i].value.s, ALGORITHM_LEFT_SYMMETRIC_NAME) == 0) {
					*parity_algorithm = ALGORITHM_LEFT_SYMMETRIC;
				} else if (strcmp(options->option[i].value.s, ALGORITHM_RIGHT_SYMMETRIC_NAME) == 0) {
					*parity_algorithm = ALGORITHM_RIGHT_SYMMETRIC;
				}
				break;

			default:
				break;

			}

		} else {
			if (strcmp(options->option[i].name, MD_OPTION_SPARE_DISK_NAME) == 0) {
				*spare_disk = options->option[i].value.s;
			} else if (strcmp(options->option[i].name, MD_OPTION_CHUNK_SIZE_NAME) == 0) {
				*chunk_size = options->option[i].value.ui32;

			} else if (strcmp(options->option[i].name, MD_OPTION_RAID_LEVEL_NAME) == 0) {
				if (strcmp(options->option[i].value.s, RAID4_LEVEL_NAME) == 0) {
					*raid_level = 4;
				} else if (strcmp(options->option[i].value.s, RAID5_LEVEL_NAME) == 0) {
					*raid_level = 5;
				}

			} else if (strcmp(options->option[i].name, MD_OPTION_PARITY_ALGORITHM_NAME) == 0) {
				if (strcmp(options->option[i].value.s, ALGORITHM_LEFT_ASYMMETRIC_NAME) == 0) {
					*parity_algorithm = ALGORITHM_LEFT_ASYMMETRIC;
				} else if (strcmp(options->option[i].value.s, ALGORITHM_RIGHT_ASYMMETRIC_NAME) == 0) {
					*parity_algorithm = ALGORITHM_RIGHT_ASYMMETRIC;
				} else if (strcmp(options->option[i].value.s, ALGORITHM_LEFT_SYMMETRIC_NAME) == 0) {
					*parity_algorithm = ALGORITHM_LEFT_SYMMETRIC;
				} else if (strcmp(options->option[i].value.s, ALGORITHM_RIGHT_SYMMETRIC_NAME) == 0) {
					*parity_algorithm = ALGORITHM_RIGHT_SYMMETRIC;
				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* get the list of objects, search for the one requested.  If found, return the top object
 otherwise return NULL */
static storage_object_t * find_object( char * name ) {

	int rc = 0;
	storage_object_t * object = NULL;
	storage_object_t * obj;
	list_anchor_t objects;
	list_element_t li;

	LOG_ENTRY();
	if (!name) {
		LOG_EXIT_PTR(NULL);
		return NULL;
	}

	/* Gget the list of objects. */
	rc = EngFncs->get_object_list(DISK | SEGMENT | REGION,
				      DATA_TYPE,
				      NULL,
					  NULL,
				      VALID_INPUT_OBJECT,
				      &objects);
	if (!rc) {

		LIST_FOR_EACH(objects, li, obj) {
			if (!strncmp(obj->name, name, EVMS_VOLUME_NAME_SIZE + 1)) {
				object = obj;
				break;
			}
		}

		EngFncs->destroy_list(objects);
	} else {
		LOG_ERROR("Error getting object list = %d....\n",rc);
	}

	LOG_EXIT_PTR(object);
	return object;
}


static int create_region_post_activate(md_volume_t *volume, md_ioctl_pkg_t *pkg)
{
	volume->region_mgr_flags &= ~MD_RAID5_REGION_NEW;
	return 0;
}


/* Function: raid5_create
 *
 *  Create a new MD volume.
 */
static int raid5_create( list_anchor_t          objects,
			 option_array_t * options,
			 list_anchor_t          new_region_list ) {
	md_volume_t * volume = NULL;
	storage_object_t * object;
	int nr_disks;
	unsigned long size = -1;
	int i, spare_disks=0, spare_index = 0, index = 0;
	int rc = 0;
	mdp_disk_t disk;
	storage_object_t * spare=NULL;
	char * spare_disk = NULL;
	int chunk_size = MD_DEFAULT_CHUNK_SIZE;
	unsigned int raid_level = 5;
	unsigned int parity_algorithm = ALGORITHM_LEFT_SYMMETRIC;
	list_element_t iter1, iter2;

	my_plugin = raid5_plugin;
	LOG_ENTRY();

	/* Parameter check */
	if (!objects || !options || !new_region_list) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	nr_disks = EngFncs->list_count(objects);

	if (nr_disks > MAX_MD_DEVICES) {
		LOG_ERROR("Too many objects (%d) given. Maximum is %d.\n", nr_disks, MAX_MD_DEVICES);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (!(volume = EngFncs->engine_alloc(sizeof(md_volume_t) ))) {
		LOG_CRITICAL("Memory error new volume structure.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}
	LIST_FOR_EACH_SAFE(objects, iter1, iter2, object) {
		size = min(size, object->size);	 /* Track smallest object for super block */
		volume->child_object[index] = object;
		index ++;
		EngFncs->remove_element(iter1);
	}

	raid5_get_create_options(options, &spare_disk, &chunk_size, &raid_level, &parity_algorithm);

	if (spare_disk) {
		spare = find_object( spare_disk );
	}
	if (spare) {
		size = min(size, spare->size);	/* Track smallest object for super block */
		volume->child_object[index] = spare;
		nr_disks++;
		spare_disks = 1;
		spare_index = index;
	}
	disk.number = 0;
	disk.raid_disk = 0;
	disk.state = (1 << MD_DISK_ACTIVE) | (1 << MD_DISK_SYNC);

	size = MD_CHUNK_ALIGN_NEW_SIZE_BLOCKS(chunk_size, VSECTORS_TO_BLOCKS(size));

	rc = md_create_first_superblock(volume, disk, raid_level, chunk_size, size, nr_disks, spare_disks, 0);
	if (rc) {
		EngFncs->engine_free(volume);
		LOG_EXIT_INT(rc);
		return rc;
	}

	volume->super_block->level = raid_level;
	if (raid_level == 5) {
		volume->super_block->layout = parity_algorithm;
	} else {
		volume->super_block->layout = 0;
	}

	if (spare) {
		/* Set the state to inactive for the spare disk. */
		volume->super_block->disks[spare_index].state = 0;
	}

	for (i = 0; i < nr_disks; i++) {
		rc = md_clone_superblock(volume, i);
		if (rc) {
			for (i--; i>=0; i--) {
				EngFncs->engine_free(volume->super_array[i]);
			}
			EngFncs->engine_free(volume->super_block);
			EngFncs->engine_free(volume);
			LOG_EXIT_INT(rc);
			return rc;
		}
	}

	volume->personality = RAID5;
	volume->nr_disks = nr_disks;
	volume->next = volume_list_head;
	volume_list_head = volume;
	volume->setup_funcs = EngFncs->allocate_list();
	volume->ioctl_pkgs = EngFncs->allocate_list();
	volume->ioctl_cleanup = EngFncs->allocate_list();

	rc = raid5_create_region(volume, new_region_list, TRUE);
	if (!rc) {
		volume->flags |= MD_DIRTY;
		volume->region_mgr_flags |= MD_RAID5_REGION_NEW;
		schedule_md_ioctl_pkg(volume, EVMS_MD_INVOKE_CALLBACK, NULL, create_region_post_activate);
	}
	LOG_EXIT_INT(rc);
	return rc;
}


static int forward_kill_sectors() {

	int rc = 0;
	kill_sectors_t * ks = kill_sector_list_head;
	storage_object_t * child_object;
	lsn_t              child_lsn;
	sector_count_t     child_count;

	LOG_ENTRY();

	while ((rc == 0) && (ks != NULL)) {
		md_volume_t * volume = (md_volume_t *) ks->region->private_data;

		while ((rc == 0) && (ks->count > 0)) {
			get_child_run(volume, ks->lsn, ks->count,
				      &child_object, &child_lsn, &child_count);

			if (child_object != NULL) {
				rc = KILL_SECTORS(child_object, child_lsn, child_count);
			}

			if (rc == 0) {
				ks->count -= child_count;
				ks->lsn += child_count;
			}
		}

		kill_sector_list_head = ks->next;
		free(ks);
		ks = kill_sector_list_head;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: w_delete
 *
 * Worker function for raid5_delete and raid5_discard
 */
static int w_delete(storage_object_t *region, list_anchor_t children, boolean tear_down)
{
	int     rc;
	md_volume_t * volume = (md_volume_t *) region->private_data;
	raid5_conf_t * conf = mdvol_to_conf(volume);
	
	LOG_ENTRY();

	/* Check that this region can be removed. */
	if ((rc = raid5_can_delete(region))) {
		LOG_EXIT_INT(rc);
		return rc;
	}
	volume = region->private_data;

	rc = forward_kill_sectors();

	if (rc == 0) {
		/* Remove the parent/child associations with the PVs. */
		md_clear_child_list(region, children);

		EngFncs->engine_free(conf);
		/* Delete the volume. */
		md_delete_volume(volume, tear_down);
		region->private_data = NULL;
		EngFncs->free_region(region);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid5_delete
 *
 *  Remove the specified region and consolidate all of its space into
 *  the appropriate freespace region.
 */
static int raid5_delete(storage_object_t * region, list_anchor_t children)
{
	int     rc;
	
	my_plugin = raid5_plugin;
	LOG_ENTRY();
	rc = w_delete(region, children, TRUE);
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Function: raid5_discard
 * 
 * This function is similar to delete.  Just call delete to free all
 * data structures related to the regions.
 */
static int raid5_discard(list_anchor_t regions)
{
	storage_object_t * region;
	list_element_t le;

	LOG_ENTRY();

	LIST_FOR_EACH(regions, le, region) {
		w_delete(region, NULL, FALSE);
	}

	LOG_EXIT_INT(0);
	return 0;
}


static int raid5_replace_child(storage_object_t *region,
			       storage_object_t *child,
			       storage_object_t *new_child)
{
	int rc;
	my_plugin = raid5_plugin;
	LOG_ENTRY();
	rc = md_replace_child(region, child, new_child);
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid5_add_sectors_to_kill_list
 *
 *  The kill_sectors list contains a list of sectors that need to be zeroed
 *  during the next commit. This function is very similar to read/write.
 */
static int raid5_add_sectors_to_kill_list( storage_object_t * region,
					   lsn_t              lsn,
					   sector_count_t     count ) {

	int              rc = 0;
	md_volume_t    * volume = (md_volume_t *)region->private_data;
	kill_sectors_t * ks;

	my_plugin = raid5_plugin;
	LOG_ENTRY();

	if (volume->flags & MD_CORRUPT) {
		MESSAGE("MD Object %s is corrupt, writing data is not allowed.\n",volume->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}
	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to write past end of region %s sector=%"PRIu64"\n",volume->name,lsn+count);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	ks = malloc(sizeof(kill_sectors_t));

	if (ks != NULL) {

		ks->region = region;
		ks->lsn    = lsn;
		ks->count  = count;

		ks->next = kill_sector_list_head;
		kill_sector_list_head = ks;

		/*
		 * Mark the region dirty so that it will get called to commit
		 * the kill sectors.
		 */
		region->flags |= SOFLAG_DIRTY;

	} else {
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Process the kill sectors list.
 */
static int kill_sectors(void) {

	int rc = 0;
	kill_sectors_t * ks;
	unsigned char  * buffer = NULL;
	sector_count_t   buffer_size = 0;

	LOG_ENTRY();

	/*
	 * Copy the kill sector list head and NULL out the gloabal variable.
	 * This function uses raid5_write() to write out the kill sectors,
	 * but raid5_write() has a check to write kill sectors before it does
	 * any writing.  We could end up in infinite recursion between
	 * kill_sectors() and raid5_write().  raid5_write() has a check to
	 * see if there are any kill sectors on the global list.  By having
	 * this function remove the kill sectors from the global list the
	 * recursion is stopped.
	 */
	ks = kill_sector_list_head;
	kill_sector_list_head = NULL;

	while ((rc == 0) && (ks != NULL)) {
		if (buffer_size < ks->count) {
			if (buffer != NULL) {
				free(buffer);
			}
			buffer = calloc(1, EVMS_VSECTOR_SIZE * ks->count);

			if (buffer != NULL) {
				buffer_size = ks->count;
			} else {
				buffer_size = 0;
				rc = ENOMEM;
			}
		}

		if (rc == 0) {
			kill_sectors_t * ks_prev = ks;

			LOG_DEBUG("Killing %"PRIu64" sectors on %s at sector offset %"PRIu64".\n", ks->count, ks->region->name, ks->lsn);
			rc = raid5_write(ks->region, ks->lsn, ks->count, buffer);

			ks = ks->next;
			free(ks_prev);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid5_commit_changes
 *
 * The phases:
 *	SETUP
 *	  - for actions required prior to superblocks update
 *	FIRST_METADATA_WRITE
 *	  - write MD superblocks, must have enabled MD_DIRTY flag
 *	SECOND_METADATA_WRITE
 *	  - not used
 *	POST_ACTIVATE
 *	  -  for actions required after superblocks update, _or_
 *	     for queued IOCTLs
 *	  -  reload superblocks via raid1_rediscover_region
 *	
 *	NOTE:  In order to get invoked for all phases of commit process,
 *		leave SOFLAG_DIRTY until the last phase (POST_ACTIVATE)
 */
static int raid5_commit_changes( storage_object_t * region,
				 uint               phase )
{
	md_volume_t * volume;
	int         rc = 0;
	int saved_rc;

	my_plugin = raid5_plugin;
	LOG_ENTRY();

	if (!region) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (region->plugin != raid5_plugin) {
		LOG_ERROR("Region %s does not belong to MD.\n", region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}
	if (!(region->flags & SOFLAG_DIRTY)) {
		LOG_WARNING("Region %s is not dirty - not committing.\n", region->name);
		LOG_EXIT_INT(0);
		return 0;
	}
	
	volume = (md_volume_t *)region->private_data;
	if (!volume) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}
	
	switch (phase) {
	case SETUP:
		rc = process_setup_funcs(region);
		break;
	case FIRST_METADATA_WRITE:
		kill_sectors();
		if (volume->flags & MD_DIRTY) {
			rc = md_write_sbs_to_disk(volume);
		}
		break;
	case SECOND_METADATA_WRITE:
		break;
	case POST_ACTIVATE:
		rc = process_md_ioctl_pkgs(region);

		/* Despite rc, we will rediscover the MD array, save the return code */
		saved_rc = rc;
		rc = raid5_rediscover_region(region);
		if (saved_rc != 0)
			rc = saved_rc;
		free_ioctl_pkgs(volume);
		region->flags &= ~SOFLAG_DIRTY;
		break;
	default	:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;

}

static int raid5_activate_region(storage_object_t * region)
{
	int rc;
	md_volume_t *volume;
	my_plugin = raid5_plugin;
	LOG_ENTRY();
	
	if (!region) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	volume = (md_volume_t *)region->private_data;
	if (!volume) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	raid5_verify_and_fix_array(volume, 0);	

	rc = md_activate_region (region);

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid5_deactivate_region(storage_object_t * region)
{
	int rc=0;
	my_plugin = raid5_plugin;
	LOG_ENTRY();

	rc = md_deactivate_region (region);

	LOG_EXIT_INT(rc);
	return rc;
}



/* Function: raid5_get_option_count
 *
 *  Determine the type of Task that is being performed, and return
 *  the number of options that are available for that Task.
 */
static int raid5_get_option_count( task_context_t * task ) {
	int count = 0;

	my_plugin = raid5_plugin;
	LOG_ENTRY();

	switch (task->action) {
	case EVMS_Task_Create:
		count = MD_CREATE_OPTIONS_COUNT;
		break;

	case MD_RAID5_FUNCTION_FIX:
	case MD_RAID5_FUNCTION_ADD_SPARE:
	case MD_RAID5_FUNCTION_REMOVE_SPARE:
	case MD_RAID5_FUNCTION_REMOVE_FAULTY:
	case MD_RAID5_FUNCTION_MARK_FAULTY:
	case MD_RAID5_FUNCTION_REMOVE_STALE:
	case MD_RAID5_RESTORE_SUPERBLOCK:
		count = 0;
		break;

	default:
		count = -1;
		break;
	}

	LOG_EXIT_INT(count);
	return count;
}


/* Get the list of objects on the system that we can use. */
static int get_object_list( value_list_t ** value_list,
			    list_anchor_t   selected_objects,
			    sector_count_t  min_size) {

	int rc = 0;
	storage_object_t * object;
	list_anchor_t tmp_list;
	list_element_t li;
	int count, i;

	LOG_ENTRY();

	rc = EngFncs->get_object_list(DISK | SEGMENT | REGION,
				      DATA_TYPE,
				      NULL,
				      NULL,
				      VALID_INPUT_OBJECT,
				      &tmp_list);
	
	if (rc) {
		LOG_ERROR("Could not get available objects.\n");
		LOG_EXIT_INT(rc);
		return rc;
	}

	/*
	 * Loop through the selected objects, removing those objects from
	 * tmp_list.
	 */

	LIST_FOR_EACH(selected_objects, li, object) {
		LOG_DETAILS("Object %s selected, removing from spare list\n",object->name);
		EngFncs->remove_thing(tmp_list, object);
	}
	
	if (*value_list) {
		for (i = 0; i < (*value_list)->count; i++) {
			if ((*value_list)->value[i].s) {
				EngFncs->engine_free((*value_list)->value[i].s);
			}
		}
		EngFncs->engine_free(*value_list);
	}

	count = EngFncs->list_count(tmp_list);
	/* Increment count to hold the 'None' selection. */
	count++;
	*value_list = EngFncs->engine_alloc(count * sizeof(value_t) + sizeof(value_list_t));  /* yeah it's too big, but so what */
	
	if (*value_list == NULL) {
		LOG_ERROR("No memory\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}


	if (!rc) {
		i = 0;
		SET_STRING((*value_list)->value[i].s, MD_NO_SELECTION);
		i++;
		LIST_FOR_EACH(tmp_list, li, object) {
			if (object->size >= min_size) {
				(*value_list)->value[i].s = EngFncs->engine_alloc(strlen(object->name) + 1);
				strcpy((*value_list)->value[i].s, object->name);
				i++;
			}
		}
		(*value_list)->count = i;
	}
	EngFncs->destroy_list(tmp_list);

	LOG_EXIT_INT(rc);
	return rc;
}


static int get_raid_level_list(value_list_t * * raid_level_list) {

	int rc = 0;

	LOG_ENTRY();

	*raid_level_list = EngFncs->engine_alloc(sizeof(value_list_t) + sizeof(value_t));

	if (*raid_level_list != NULL) {
		(*raid_level_list)->count = 2;

		SET_STRING((*raid_level_list)->value[0].s, RAID4_LEVEL_NAME);
		SET_STRING((*raid_level_list)->value[1].s, RAID5_LEVEL_NAME);

	} else {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int get_algorithm_list(value_list_t * * algorithm_list) {

	int rc = 0;

	LOG_ENTRY();

	*algorithm_list = EngFncs->engine_alloc(sizeof(value_list_t) + 3 * sizeof(value_t));

	if (*algorithm_list != NULL) {
		(*algorithm_list)->count = 4;

		SET_STRING((*algorithm_list)->value[ALGORITHM_LEFT_ASYMMETRIC].s,  ALGORITHM_LEFT_ASYMMETRIC_NAME);
		SET_STRING((*algorithm_list)->value[ALGORITHM_RIGHT_ASYMMETRIC].s, ALGORITHM_RIGHT_ASYMMETRIC_NAME);
		SET_STRING((*algorithm_list)->value[ALGORITHM_LEFT_SYMMETRIC].s,   ALGORITHM_LEFT_SYMMETRIC_NAME);
		SET_STRING((*algorithm_list)->value[ALGORITHM_RIGHT_SYMMETRIC].s,  ALGORITHM_RIGHT_SYMMETRIC_NAME);

	} else {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}



static int get_spare_disks(md_volume_t * volume, list_anchor_t spare_disks)
{
	int rc = 0;
	int i;
	list_element_t li;

	LOG_ENTRY();

	/* Clear out the spare_disks list. */
	EngFncs->delete_all_elements(spare_disks);

	/*
	 * If there is only one spare and the array is in degrade mode, then
	 * the spare cannot be removed.
	 */
	if (volume->super_block->spare_disks <= 1) {
		if (volume->super_block->active_disks != volume->super_block->raid_disks) {
			LOG_EXIT_INT(0);
			return 0;
		}
	}

	for (i = 0; (rc == 0) && (i < MAX_MD_DEVICES); i++ ) {
		/* Check for null object, if missing, skip. */
		if (volume->child_object[i]) {
			if ((volume->super_block->disks[i].state & ~(1 << MD_DISK_NEW)) == 0) {

				li = EngFncs->insert_thing(spare_disks,
							   volume->child_object[i],
							   INSERT_AFTER,
							   NULL);
				if (!li) {
					LOG_ERROR("Could not insert object into spare list.\n");
					rc = ENOMEM;
				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int get_faulty_disks(md_volume_t * volume, list_anchor_t faulty_disks)
{
	int rc = 0;
	list_element_t li;
	int i;

	LOG_ENTRY();

	EngFncs->delete_all_elements(faulty_disks);

	for (i = 0; (rc == 0) && (i < MAX_MD_DEVICES); i++ ) {
		/* Check for null object, if missing, skip. */
		if (volume->child_object[i]) {
			if ( (volume->super_block->disks[i].state & (1 << MD_DISK_FAULTY)) ||
			     ((volume->super_block->disks[i].state & (1 << MD_DISK_ACTIVE)) &&
			      !(volume->super_block->disks[i].state & (1 << MD_DISK_SYNC) ))) {

				li = EngFncs->insert_thing(faulty_disks,
							   volume->child_object[i],
							   INSERT_AFTER,
							   NULL);
				if (!li) {
					LOG_ERROR("Could not insert object into faulty list.\n");
					rc = ENOMEM;
				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int get_stale_disks(md_volume_t * volume, list_anchor_t stale_disks)
{

	int rc = 0;
	int i, found;
	list_element_t li;

	LOG_ENTRY();

	EngFncs->delete_all_elements(stale_disks);

	for (i=0, found=0;
	     (rc == 0) && (found < volume->stale_disks) && (i < MAX_MD_DEVICES); i++ ) {

		if (volume->stale_object[i])
		{
			found++;
			li = EngFncs->insert_thing(stale_disks,
						   volume->stale_object[i],
						   INSERT_AFTER,
						   NULL);
			if (!li) {
				LOG_ERROR("Could not insert object into stale list.\n");
				rc = ENOMEM;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int get_active_disks(md_volume_t * volume, list_anchor_t active_disks)
{

	int rc = 0;
	int i;
	list_element_t li;
	boolean can_remove = FALSE;

	LOG_ENTRY();

	EngFncs->delete_all_elements(active_disks);


	/* Region must not be new. */
	/* Must not be running in degrade mode. */
	if (!(volume->region->flags & SOFLAG_NEW) &&
	    (volume->super_block->active_disks == volume->super_block->raid_disks)) {

		/* Must have a spare already in the array, i.e., not new */
		for (i = 0, can_remove = FALSE; !can_remove && (i < MAX_MD_DEVICES); i++ ) {
			/* Check for null object, if missing, skip. */
			if (volume->child_object[i]) {
				if (volume->super_block->disks[i].state == 0) {
					can_remove = TRUE;
				}
			}
		}
	}

	if (can_remove) {
		for (i = 0; (rc == 0) && (i < MAX_MD_DEVICES); i++ ) {
			/* Check for null object, if missing, skip. */
			if (volume->child_object[i]) {
				if (volume->super_block->disks[i].state & (1 << MD_DISK_ACTIVE)) {
					li = EngFncs->insert_thing(active_disks,
								   volume->child_object[i],
								   INSERT_AFTER,
								   NULL);
					if (!li) {
						LOG_ERROR("Could not insert object into active list.\n");
						rc = ENOMEM;
					}
				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static void prune_small_objects(list_anchor_t objects,
				mdp_super_t *sb)
{
	list_element_t iter1, iter2;
	storage_object_t * obj;
	sector_count_t size;

	LOG_ENTRY();


	LIST_FOR_EACH_SAFE(objects, iter1, iter2, obj) {
		size = MD_CHUNK_ALIGN_NEW_SIZE_SECTORS((sb->chunk_size / EVMS_VSECTOR_SIZE), obj->size);
		if (size < (sb->size * (BLOCK_SIZE / EVMS_VSECTOR_SIZE))) {
			EngFncs->remove_element(iter1);
		}
	}

	LOG_EXIT_VOID();
}


/* Function: raid5_init_task
 *
 *  Determine the type of Task that is being performed, and set up the
 *  context structure with the appropriate initial values.
 */
static int raid5_init_task( task_context_t * context ) {

	int                  rc = 0;
	list_anchor_t              tmp_list;
	md_volume_t        * volume;
	raid5_conf_t       * conf;

	my_plugin = raid5_plugin;
	LOG_ENTRY();

	/* Parameter check */
	if (!context) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	switch (context->action) {
	
	case EVMS_Task_Create:

		context->option_descriptors->count = MD_CREATE_OPTIONS_COUNT;

		/* Spare disk option */
		context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
		/* Get the list of disks that can be spares. */
		get_object_list((value_list_t **)&context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].constraint.list,
				context->selected_objects,
				0);
		context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].help = NULL;
		SET_STRING(context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].name, MD_OPTION_SPARE_DISK_NAME );
		context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].min_len = 1;
		context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].max_len = EVMS_VOLUME_NAME_SIZE + 1;
		SET_STRING(context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].tip, "Object to use as a spare disk in the array" );
		SET_STRING(context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].title, "Spare Disk" );
		context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].type = EVMS_Type_String;
		context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].unit = EVMS_Unit_None;
		context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE + 1);

		/* Chunk size option */
		context->option_descriptors->option[MD_OPTION_CHUNK_SIZE_INDEX].flags = 0;
		SET_POWER2_LIST(context->option_descriptors->option[MD_OPTION_CHUNK_SIZE_INDEX].constraint.list, MD_MIN_CHUNK_SIZE, MD_MAX_CHUNK_SIZE);
		context->option_descriptors->option[MD_OPTION_CHUNK_SIZE_INDEX].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[MD_OPTION_CHUNK_SIZE_INDEX].help = NULL;
		SET_STRING(context->option_descriptors->option[MD_OPTION_CHUNK_SIZE_INDEX].name, MD_OPTION_CHUNK_SIZE_NAME );
		SET_STRING(context->option_descriptors->option[MD_OPTION_CHUNK_SIZE_INDEX].tip, "Size of the chunks in the RAID array" );
		SET_STRING(context->option_descriptors->option[MD_OPTION_CHUNK_SIZE_INDEX].title, "Chunk size" );
		context->option_descriptors->option[MD_OPTION_CHUNK_SIZE_INDEX].type = EVMS_Type_Unsigned_Int32;
		context->option_descriptors->option[MD_OPTION_CHUNK_SIZE_INDEX].unit = EVMS_Unit_Kilobytes;
		context->option_descriptors->option[MD_OPTION_CHUNK_SIZE_INDEX].value.ui32 = MD_DEFAULT_CHUNK_SIZE;

		/* RAID level option */
		context->option_descriptors->option[MD_OPTION_RAID_LEVEL_INDEX].flags = 0;
		get_raid_level_list(&context->option_descriptors->option[MD_OPTION_RAID_LEVEL_INDEX].constraint.list);
		context->option_descriptors->option[MD_OPTION_RAID_LEVEL_INDEX].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[MD_OPTION_RAID_LEVEL_INDEX].help = NULL;
		SET_STRING(context->option_descriptors->option[MD_OPTION_RAID_LEVEL_INDEX].name, MD_OPTION_RAID_LEVEL_NAME);
		context->option_descriptors->option[MD_OPTION_RAID_LEVEL_INDEX].min_len = 5;
		context->option_descriptors->option[MD_OPTION_RAID_LEVEL_INDEX].max_len = 19;
		SET_STRING(context->option_descriptors->option[MD_OPTION_RAID_LEVEL_INDEX].tip, "RAID Level -- RAID4 or RAID5" );
		SET_STRING(context->option_descriptors->option[MD_OPTION_RAID_LEVEL_INDEX].title, "RAID level" );
		context->option_descriptors->option[MD_OPTION_RAID_LEVEL_INDEX].type = EVMS_Type_String;
		context->option_descriptors->option[MD_OPTION_RAID_LEVEL_INDEX].unit = EVMS_Unit_None;
		SET_STRING(context->option_descriptors->option[MD_OPTION_RAID_LEVEL_INDEX].value.s, RAID5_LEVEL_NAME);

		/* Parity algorithm option */
		context->option_descriptors->option[MD_OPTION_PARITY_ALGORITHM_INDEX].flags = 0;
		get_algorithm_list(&context->option_descriptors->option[MD_OPTION_PARITY_ALGORITHM_INDEX].constraint.list);
		context->option_descriptors->option[MD_OPTION_PARITY_ALGORITHM_INDEX].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[MD_OPTION_PARITY_ALGORITHM_INDEX].help = NULL;
		SET_STRING(context->option_descriptors->option[MD_OPTION_PARITY_ALGORITHM_INDEX].name, MD_OPTION_PARITY_ALGORITHM_NAME);
		context->option_descriptors->option[MD_OPTION_PARITY_ALGORITHM_INDEX].min_len = 5;
		context->option_descriptors->option[MD_OPTION_PARITY_ALGORITHM_INDEX].max_len = 19;
		SET_STRING(context->option_descriptors->option[MD_OPTION_PARITY_ALGORITHM_INDEX].tip, "RAID5 Parity algorithm" );
		SET_STRING(context->option_descriptors->option[MD_OPTION_PARITY_ALGORITHM_INDEX].title, "RAID5 Algorithm" );
		context->option_descriptors->option[MD_OPTION_PARITY_ALGORITHM_INDEX].type = EVMS_Type_String;
		context->option_descriptors->option[MD_OPTION_PARITY_ALGORITHM_INDEX].unit = EVMS_Unit_None;
		context->option_descriptors->option[MD_OPTION_PARITY_ALGORITHM_INDEX].value.s = EngFncs->engine_alloc(20);
		if (context->option_descriptors->option[MD_OPTION_PARITY_ALGORITHM_INDEX].value.s != NULL) {
			strcpy(context->option_descriptors->option[MD_OPTION_PARITY_ALGORITHM_INDEX].value.s, ALGORITHM_LEFT_SYMMETRIC_NAME);
		} else {
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}

		/* Get a list of all valid input disks, segments, and regions. */
		EngFncs->get_object_list(DISK | SEGMENT | REGION,
					 DATA_TYPE,
					 NULL,
					 NULL,
					 VALID_INPUT_OBJECT,
					 &tmp_list);

		/* Move these items to the acceptable objects list. */
		md_transfer_list(tmp_list, context->acceptable_objects);
		EngFncs->destroy_list(tmp_list);

		context->min_selected_objects = 2;
		context->max_selected_objects = MAX_MD_DEVICES;
		break;

	case MD_RAID5_FUNCTION_FIX:
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		context->option_descriptors->count = 0;
		break;

	case MD_RAID5_FUNCTION_ADD_SPARE:
		{
			list_anchor_t acceptable_objects;
                        volume = (md_volume_t *) context->object->private_data;
			conf = mdvol_to_conf(volume);

			context->min_selected_objects = 1;
			context->max_selected_objects = MAX_MD_DEVICES - volume->super_block->nr_disks;
			context->option_descriptors->count = 0;

			rc = EngFncs->get_object_list(DISK | SEGMENT | REGION,
						      DATA_TYPE,
						      NULL,
							  NULL,
						      VALID_INPUT_OBJECT,
						      &acceptable_objects);

			if (rc == 0) {
				//prune_small_object_parms_t parms;

				/*
				 * If this MD region is available, it will
				 * appear in the list.  Bad things happen if
				 * this region is made a child of itself.
				 * Remove this MD region if it is in the list.
				 */
				EngFncs->remove_thing(acceptable_objects, context->object);

				prune_small_objects(acceptable_objects, volume->super_block);
				if (context->acceptable_objects != NULL) {
					EngFncs->destroy_list(context->acceptable_objects);
				}
				context->acceptable_objects = acceptable_objects;
			}
		}
		break;

	case MD_RAID5_FUNCTION_REMOVE_SPARE:
		volume = (md_volume_t *) context->object->private_data;
		conf = mdvol_to_conf(volume);

		context->min_selected_objects = 1;
		context->max_selected_objects = -1;
		context->option_descriptors->count = 0;

		rc = get_spare_disks(volume, context->acceptable_objects);
		break;

	case MD_RAID5_FUNCTION_REMOVE_FAULTY:
		volume = (md_volume_t *) context->object->private_data;
		conf = mdvol_to_conf(volume);

		context->min_selected_objects = 1;
		context->max_selected_objects = -1;
		context->option_descriptors->count = 0;

		rc = get_faulty_disks(volume, context->acceptable_objects);
		break;

	case MD_RAID5_FUNCTION_REMOVE_STALE:
		volume = (md_volume_t *) context->object->private_data;

			context->min_selected_objects = 1;
			context->max_selected_objects = -1;	
			context->option_descriptors->count = 0;

			rc = get_stale_disks(volume, context->acceptable_objects);
		break;

	case MD_RAID5_FUNCTION_MARK_FAULTY:
		volume = (md_volume_t *) context->object->private_data;
		conf = mdvol_to_conf(volume);

		context->min_selected_objects = 1;
		context->max_selected_objects = 1;
		context->option_descriptors->count = 0;

		rc = get_active_disks(volume, context->acceptable_objects);
		break;

	case MD_RAID5_RESTORE_SUPERBLOCK:
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		context->option_descriptors->count = 0;
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


#define PERCENT_WARN_THRESHOLD  5

static void warn_if_big_objects( task_context_t * context )
{
	storage_object_t * obj = NULL;
	storage_object_t * spare = NULL;
	u_int64_t smallest_size = 0;
	list_element_t li;

	LOG_ENTRY();

	/* Find the smallest object. */
	LIST_FOR_EACH(context->selected_objects, li, obj) {
		smallest_size = min(smallest_size, MD_NEW_SIZE_SECTORS(obj->size));
	}

	/*
	 * If we got a smallest size, then check the size of the spare, if one
	 * is specified and see if it is the smallest.
	 */
	if (smallest_size != 0) {
		if (context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].value.s != NULL) {
			spare = find_object(context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].value.s);

			if (spare != NULL) {
				smallest_size = min(smallest_size, MD_NEW_SIZE_SECTORS(spare->size));
			}
		}
	}

	/*
	 * Now go through the objects again and issue a warnign message for
	 * any object whose size exceeds the threshold over the smallest
	 * object size.
	 */
	if (smallest_size != 0) {
		u_int64_t diffsize;

		LIST_FOR_EACH(context->selected_objects, li, obj) {
			diffsize = MD_NEW_SIZE_SECTORS(obj->size) - smallest_size;

			if (diffsize > (smallest_size * PERCENT_WARN_THRESHOLD) / 100) {
				MESSAGE("The %s object is %"PRIu64" KB larger than the smallest object in the RAID array.  "
					"The extra space will not be used.\n",
					obj->name, diffsize * EVMS_VSECTOR_SIZE / 1024);
			}
		}

		/*
		 * If we have a spare, check its size too.
		 */
		if (spare != NULL) {
			diffsize = MD_NEW_SIZE_SECTORS(spare->size) - smallest_size;

			if (diffsize > (smallest_size * PERCENT_WARN_THRESHOLD) / 100) {
				MESSAGE("The %s object is %"PRIu64" KB larger than the smallest object in the RAID array.  "
					"The extra space will not be used.\n",
					spare->name, diffsize * EVMS_VSECTOR_SIZE / 1024);
			}
		}
	}

	LOG_EXIT_VOID();
}

/* Function: raid5_set_option
 *
 *  Determine the type of Task that is being performed. Then examine the
 *  desired option (using the index), and verify that the given value is
 *  appropriate. Reset the value if necessary and possible. Adjust other
 *  options as appropriate.
 */
static int raid5_set_option( task_context_t * context,
			     u_int32_t        index,
			     value_t        * value,
			     task_effect_t  * effect ) {
	int rc = 0;

	my_plugin = raid5_plugin;
	LOG_ENTRY();

	/* Parameter check */
	if (!context || !value || !effect) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	switch (context->action) {
	
	case EVMS_Task_Create:
		switch (index) {
		
		case MD_OPTION_SPARE_DISK_INDEX:
			/*
			 * Not worth validation, will catch when we try to find
			 * the original.
			 */
			strcpy(context->option_descriptors->option[index].value.s, value->s);
			warn_if_big_objects(context);
			break;

		case MD_OPTION_CHUNK_SIZE_INDEX:
			if ((value->ui32 < MD_MIN_CHUNK_SIZE) ||
			    (value->ui32 > MD_MAX_CHUNK_SIZE)) {
				/* Chunk size is out of bounds. */
				rc = EINVAL;

			} else {
				/*
				 * Chunk size must be a power of 2.
				 * calc_log2 returns -1 if the number is not a
				 * power of 2.
				 */
				if (calc_log2((long) value->ui32) == -1) {
					rc = EINVAL;
				}
			}
			if (rc == 0) {
				context->option_descriptors->option[index].value.ui32 = value->ui32;
			}
			break;

		case MD_OPTION_RAID_LEVEL_INDEX:
			if (strcmp(value->s, RAID4_LEVEL_NAME) == 0) {
				strcpy(context->option_descriptors->option[index].value.s, value->s);
				/*
				 * RAID4 does not have a parity algorithm.
				 * Disable the algorithm option.
				 */
				context->option_descriptors->option[MD_OPTION_PARITY_ALGORITHM_INDEX].flags |= (EVMS_OPTION_FLAGS_NOT_REQUIRED | EVMS_OPTION_FLAGS_INACTIVE);

				*effect |= EVMS_Effect_Reload_Options;

			} else if (strcmp(value->s, RAID5_LEVEL_NAME) == 0) {
				strcpy(context->option_descriptors->option[index].value.s, value->s);
				/*
				 * RAID5 does have a parity algorithm.
				 * Make sure the algorthm option is active and required.
				 */
				context->option_descriptors->option[MD_OPTION_PARITY_ALGORITHM_INDEX].flags &= ~(EVMS_OPTION_FLAGS_NOT_REQUIRED | EVMS_OPTION_FLAGS_INACTIVE);

				*effect |= EVMS_Effect_Reload_Options;

			} else {
				rc = EINVAL;
			}
			break;

		case MD_OPTION_PARITY_ALGORITHM_INDEX:
			if ((strcmp(value->s,ALGORITHM_LEFT_ASYMMETRIC_NAME) == 0)  ||
			    (strcmp(value->s,ALGORITHM_RIGHT_ASYMMETRIC_NAME) == 0) ||
			    (strcmp(value->s,ALGORITHM_LEFT_SYMMETRIC_NAME) == 0)   ||
			    (strcmp(value->s,ALGORITHM_RIGHT_SYMMETRIC_NAME) == 0) ) {
				strcpy(context->option_descriptors->option[index].value.s, value->s);

			} else {
				rc = EINVAL;
			}

			break;

		default:
			break;
		}

		break;

	default:
		break;
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid5_set_objects
 *
 *  Determine the type of task, and then validate that the objects on the
 *  "selected" list are valid for that task. If so, adjust the option
 *  descriptor as appropriate.
 */
static int raid5_set_objects( task_context_t * context,
			      list_anchor_t          declined_objects,
			      task_effect_t  * effect ) {
	int rc = 0;
	uint count = 0;
	storage_object_t * obj;
	list_element_t li;
	declined_object_t * dec_obj;
	md_volume_t * volume = NULL;

	my_plugin = raid5_plugin;
	LOG_ENTRY();

	/* Parameter check */
	if (!context || !declined_objects || !effect) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	count = EngFncs->list_count(context->selected_objects);
	if (context->object != NULL) {
		volume = (md_volume_t *) context->object->private_data;
	}

	switch (context->action) {
	
	case EVMS_Task_Create:
		get_object_list((value_list_t **)&context->option_descriptors->option[0].constraint.list,
				context->selected_objects,
				0);
		warn_if_big_objects(context);
		*effect |= EVMS_Effect_Reload_Options;
		break;

	case MD_RAID5_FUNCTION_FIX:
		break;

		/*
		 * The Engine makes sure that only available objects appear in the
		 * selected_objects list.
		 */
	case MD_RAID5_FUNCTION_ADD_SPARE:
		/* Make sure too many objects are not selected. */
		if (count > (MAX_MD_DEVICES - volume->super_block->nr_disks)) {
			LOG_ERROR("Can only specify up to %d object(s) to add as spare(s).\n", MAX_MD_DEVICES - volume->super_block->nr_disks);
			rc = EINVAL;
		}
		break;

	case MD_RAID5_FUNCTION_REMOVE_SPARE:

		/*
		 * If the array is running in degrade mode, make sure at
		 * least one spare is available.
		 */
		if (count &&
		    (volume->super_block->active_disks < volume->super_block->raid_disks)) {

			MESSAGE("At least one spare object must be left for recovering degraded array %s.\n", volume->region->name);

			/*
			 * One spare must be left.
			 * Reject the last object(s) in
			 * the list.
			 */

			obj = EngFncs->last_thing(context->selected_objects, &li);
			EngFncs->remove_element(li);
				
			dec_obj = EngFncs->engine_alloc(sizeof(declined_object_t));

			if (dec_obj != NULL) {

				dec_obj->object = obj;
				dec_obj->reason = EPERM;

				li = EngFncs->insert_thing(declined_objects,
							   dec_obj,
							   INSERT_AFTER,
							   NULL);

				if (!li) {
					LOG_CRITICAL("Could not insert into declined object list.\n");
					rc = ENOMEM;
				}
			} else {
				LOG_CRITICAL("Error allocating memory for a declined object.\n");
				rc = ENOMEM;
			}
		}
		break;

	case MD_RAID5_FUNCTION_REMOVE_FAULTY:
	case MD_RAID5_FUNCTION_REMOVE_STALE:
		/* No additional checks needed. */
		break;

	case MD_RAID5_FUNCTION_MARK_FAULTY:
		/* Verify that only one object is selected. */
		if (count > 1) {
			LOG_ERROR("Must select only one object to be marked faulty.\n");
			rc = EINVAL;
		}
		break;

	default:
		LOG_ERROR("%d is not a valid task action.\n", context->action);
		rc = EINVAL;
		break;
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid5_get_info
 *
 *  Return MD-specific information about the specified region. If the
 *  name field is set, only return the "extra" information pertaining
 *  to that name.
 */
static int raid5_get_info( storage_object_t       * region,
			   char                   * name,
			   extended_info_array_t ** info_array ) {

	md_volume_t * volume = NULL;
	int           rc= 0;

	my_plugin = raid5_plugin;
	LOG_ENTRY();

	/* Parameter check */
	if (!info_array) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	/* Make sure this is an MD RAID5 region */
	if (region->plugin != raid5_plugin) {
		LOG_ERROR("Region %s is not owned by MD RAID5\n", region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	volume = region->private_data;

	rc = md_get_info(volume, name, info_array);

	LOG_EXIT_INT(rc);
	return rc;
}



/* Function: raid5_get_plugin_info
 *
 *  Return information about the MD plugin. There is no "extra"
 *  information about MD, so "name" should always be NULL.
 */
static int raid5_get_plugin_info( char                     * name,
				  extended_info_array_t   ** info_array ) {

	extended_info_array_t   * info = NULL;
	char buffer[50] = {0};
	int i = 0;

	my_plugin = raid5_plugin;
	LOG_ENTRY();

	/* Parameter check */
	if (!info_array) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (!name) {
		/* Get memory for the info array. */
		if (!(info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*5))) {
			LOG_ERROR("Error allocating memory for info array\n");
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}

		SET_STRING(info->info[i].name, "ShortName");
		SET_STRING(info->info[i].title, "Short Name");
		SET_STRING(info->info[i].desc, "A short name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, raid5_plugin->short_name);
		i++;

		SET_STRING(info->info[i].name, "LongName");
		SET_STRING(info->info[i].title, "Long Name");
		SET_STRING(info->info[i].desc, "A long name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, raid5_plugin->long_name);
		i++;

		SET_STRING(info->info[i].name, "Type");
		SET_STRING(info->info[i].title, "Plugin Type");
		SET_STRING(info->info[i].desc, "There are various types of plugins; each responsible for some kind of storage object.");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, "Region Manager");
		i++;

		SET_STRING(info->info[i].name, "Version");
		SET_STRING(info->info[i].title, "Plugin Version");
		SET_STRING(info->info[i].desc, "This is the version number of the plugin.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		// Required Engine Services Version
		SET_STRING(info->info[i].name, "Required_Engine_Version");
		SET_STRING(info->info[i].title, "Required Engine Services Version");
		SET_STRING(info->info[i].desc, "This is the version of the Engine services that this plug-in requires.  It will not run on older versions of the Engine services.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", raid5_plugin->required_engine_api_version.major, raid5_plugin->required_engine_api_version.minor, raid5_plugin->required_engine_api_version.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		// Required Plug-in API Version
		SET_STRING(info->info[i].name, "Required_Plugin_Version");
		SET_STRING(info->info[i].title, "Required Plug-in API Version");
		SET_STRING(info->info[i].desc, "This is the version of the Engine plug-in API that this plug-in requires.  It will not run on older versions of the Engine plug-in API.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", raid5_plugin->required_plugin_api_version.plugin.major, raid5_plugin->required_plugin_api_version.plugin.minor, raid5_plugin->required_plugin_api_version.plugin.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

	} else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	info->count = i;
	*info_array = info;
	LOG_EXIT_INT(0);
	return 0;
}


static int reconstruct_chunk(raid5_conf_t * conf, stripe_t * stripe, unsigned int dev_index) {

	int rc = 0;
	int i;
	int count;
	xorblock_t xorblock;

	LOG_ENTRY();

	xorblock.buf[0] = stripe->chunk[dev_index].data;

	for (i = 0, count = 1; i < conf->raid_disks; i++) {
		if (i == dev_index) {
			continue;
		}

		xorblock.buf[count] = stripe->chunk[i].data;

		count++;
		if (count == MAX_XOR_BLOCKS) {
			xor_block(count, &xorblock, conf->chunk_size);

			count = 1;
		}
	}

	if (count > 1) {
		xor_block(count, &xorblock, conf->chunk_size);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int free_stripe(stripe_t * stripe) {

	int rc = 0;
	int i;

	LOG_ENTRY();

	if (stripe->data_size != 0) {

		/* Free up all the buffers in the stripe. */
		for (i = 0; i < MD_SB_DISKS; i++) {
			if (stripe->chunk[i].data != NULL) {
				free(stripe->chunk[i].data);
			}
		}

		memset(stripe, 0, sizeof(stripe_t));
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int get_stripe(md_volume_t * volume, lsn_t lsn, stripe_t * stripe) {

	int rc = 0;
	int i;
	raid5_conf_t * conf = mdvol_to_conf(volume);
	lsn_t curr_lsn;
	sector_count_t data_stripe_size = (conf->chunk_size >> EVMS_VSECTOR_SIZE_SHIFT) * (conf->raid_disks - 1);

	LOG_ENTRY();

	memset(stripe, 0, sizeof(stripe_t));

	/* Set the volume to which the stripe belongs. */
	stripe->volume = volume;

	/* Calculate stripe number and starting sector. */
	stripe->number = lsn / data_stripe_size;
	stripe->start_lsn = stripe->number * data_stripe_size;
	stripe->data_size = data_stripe_size;

	/* Read in all the chunks for the stripe. */
	for (i = 0, curr_lsn = lsn; (rc == 0) && (i < conf->raid_disks); i++) {
		stripe->chunk[i].data = calloc(1, conf->chunk_size);
		if (stripe->chunk[i].data != NULL) {
			/* If the disk is operational, fill in the child device. */
			if (conf->disks[i].operational) {
				stripe->chunk[i].dev = conf->disks[i].dev;
			} else {
				stripe->chunk[i].dev = NULL;
			}
			stripe->chunk[i].lsn_on_dev = stripe->number * (conf->chunk_size >> EVMS_VSECTOR_SIZE_SHIFT);

		} else {
			/* Memory allocation failure */
			rc = ENOMEM;
		}
	}

	if (rc == 0) {
		for (i = 0; (rc == 0) && (i < conf->raid_disks); i++) {
			if (stripe->chunk[i].dev != NULL) {
				LOG_DEBUG("Reading %d sectors from %s at sector offset %"PRIu64".\n", conf->chunk_size >> EVMS_VSECTOR_SIZE_SHIFT, stripe->chunk[i].dev->name, stripe->chunk[i].lsn_on_dev);
				rc = READ(stripe->chunk[i].dev,stripe->chunk[i].lsn_on_dev,conf->chunk_size >> EVMS_VSECTOR_SIZE_SHIFT,stripe->chunk[i].data);
			}
		}

		if (rc == 0) {
			if (conf->failed_raid_disks != 0) {
				LOG_DEBUG("Reconstructing data for failed disk %d\n", conf->failed_disk_index);
				reconstruct_chunk(conf, stripe, conf->failed_disk_index);
			}
		}
	}

	if (rc != 0) {
		/* Something went wrong.  Clean up the stripe. */
		free_stripe(stripe);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


typedef enum {
	STRIPE_IO_READ,
	STRIPE_IO_WRITE
} stripe_io_cmd_t;


static int stripe_io(stripe_io_cmd_t  cmd,
		     md_volume_t    * volume,
		     stripe_t       * stripe,
		     lsn_t            lsn,
		     sector_count_t   sectors,
		     unsigned char  * buffer,
		     sector_count_t * sectors_read) {

	lsn_t dev_offset;
	lsn_t sector_offset_in_chunk;
	unsigned int byte_offset_in_chunk;
	unsigned int chunk_index;
	unsigned int parity_index;
	sector_count_t stripe_end_lsn = stripe->start_lsn + stripe->data_size - 1;
	raid5_conf_t * conf = mdvol_to_conf(volume);
	sector_count_t chunk_size_in_sectors = conf->chunk_size >> EVMS_VSECTOR_SIZE_SHIFT;
	sector_count_t sectors_to_copy;
	unsigned int bytes_to_copy;

	LOG_ENTRY();

	if (cmd > STRIPE_IO_WRITE) {
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (sectors == 0) {
		/* Nothing to read.  We're finished. */
		LOG_EXIT_INT(0);
		return 0;
	}

	if ((lsn < stripe->start_lsn) ||
	    (lsn > stripe_end_lsn)) {
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	*sectors_read = 0;

	do {
		dev_offset = raid5_compute_sector(lsn,
						  conf->raid_disks, conf->raid_disks - 1,
						  &chunk_index, &parity_index,
						  conf);
		sector_offset_in_chunk = dev_offset & (chunk_size_in_sectors - 1);
		byte_offset_in_chunk = (unsigned int) (sector_offset_in_chunk << EVMS_VSECTOR_SIZE_SHIFT);

		sectors_to_copy = min(sectors, chunk_size_in_sectors - (sector_offset_in_chunk & (chunk_size_in_sectors - 1)));
		bytes_to_copy = (unsigned int) (sectors_to_copy << EVMS_VSECTOR_SIZE_SHIFT);

		if (cmd == STRIPE_IO_READ) {
			LOG_DEBUG("Reading %d bytes from stripe %ld, chunk %d, offset %d.\n", bytes_to_copy, stripe->number, chunk_index, byte_offset_in_chunk);
			LOG_DEBUG("AKA: Reading %"PRIu64" sectors from (%s) at sector offset %"PRIu64".\n", sectors_to_copy, (conf->disks[chunk_index].dev != NULL) ? conf->disks[chunk_index].dev->name : "nul", dev_offset);
			memcpy(buffer,
			       stripe->chunk[chunk_index].data + byte_offset_in_chunk,
			       bytes_to_copy);

		} else if (cmd == STRIPE_IO_WRITE) {
			LOG_DEBUG("Writing %d bytes to stripe %ld, chunk %d, offset %d.\n", bytes_to_copy, stripe->number, chunk_index, byte_offset_in_chunk);
			LOG_DEBUG("AKA: Writing %"PRIu64" sectors from (%s) at sector offset %"PRIu64".\n", sectors_to_copy, (conf->disks[chunk_index].dev != NULL) ? conf->disks[chunk_index].dev->name : "nul", dev_offset);
			memcpy(stripe->chunk[chunk_index].data + byte_offset_in_chunk,
			       buffer,
			       bytes_to_copy);
		}

		sectors -= sectors_to_copy;
		*sectors_read += sectors_to_copy;
		lsn += sectors_to_copy;
		buffer += bytes_to_copy;



	} while ((sectors != 0) && (lsn <= stripe_end_lsn));

	LOG_EXIT_INT(0);
	return 0;
}


/* Function: raid5_read
 *
 *  Perform a logical-to-physical remapping, and send the read down to
 *  the next plugin.
 */
static int raid5_read( storage_object_t * region,
		       lsn_t              lsn,
		       sector_count_t     count,
		       void             * buffer ) {

	int                rc = 0;
	md_volume_t *      volume = (md_volume_t *)region->private_data;
	stripe_t           stripe = {0};
	unsigned long      buffer_offset;
	storage_object_t * child_object;
	lsn_t              child_lsn;
	sector_count_t     child_count;

	my_plugin = raid5_plugin;
	LOG_ENTRY();

	LOG_DEBUG("Request to read %"PRIu64" sectors from %s at sector offset %"PRIu64".\n", count, region->name, lsn);

	/* Parameter check */
	if (!buffer) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (volume->flags & MD_CORRUPT) {
		memset(buffer, 0x0, count * EVMS_VSECTOR_SIZE);
		LOG_ERROR("MD Object %s is corrupt, data is suspect \n",volume->name);
		LOG_EXIT_INT(0);
		return 0;
	}

	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to read past end of region %s sector=%"PRIu64"\n",volume->name,lsn+count);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	rc = md_region_rw(region, lsn, count, buffer, 0);

	if (rc) {

		rc = 0;
		buffer_offset = 0;
		if (!(volume->flags & MD_DEGRADED)) {
			while ((count != 0) && (rc == 0)) {
				rc = get_child_run(volume, lsn, count,
						   &child_object, &child_lsn, &child_count);
				if (rc == 0) {
					LOG_DEBUG("Reading %"PRIu64" sectors from %s at sector offset %"PRIu64".\n", child_count, child_object->name, child_lsn);
					rc = READ(child_object, child_lsn, child_count, buffer + buffer_offset);
					count -= child_count;
					lsn += child_count;
					buffer_offset += child_count << EVMS_VSECTOR_SIZE_SHIFT;
				}
			}
	
		} else {
			while ((count != 0) && (rc == 0)) {
				if ((lsn >= stripe.start_lsn) &&
				    (lsn < stripe.start_lsn + stripe.data_size)) {
					rc = stripe_io(STRIPE_IO_READ,
						       volume, &stripe,
						       lsn, count,
						       buffer + buffer_offset, &child_count);
					count -= child_count;
					lsn += child_count;
					buffer_offset += child_count << EVMS_VSECTOR_SIZE_SHIFT;
	
				} else {
					free_stripe(&stripe);
	
					/* Read a new stripe and try again. */
					rc = get_stripe(volume, lsn, &stripe);
				}
			}
		}
	
		if (stripe.data_size != 0) {
			free_stripe(&stripe);
		}

	}
	LOG_EXIT_INT(rc);
	return rc;
}


static int write_stripe(md_volume_t * volume, stripe_t * stripe) {

	int rc = 0;
	int i;
	unsigned int data_index;
	unsigned int parity_index;
	raid5_conf_t * conf = mdvol_to_conf(volume);

	LOG_ENTRY();

	/* Find which one is the parity disk. */
	raid5_compute_sector(stripe->start_lsn,
			     conf->raid_disks, conf->raid_disks - 1,
			     &data_index, &parity_index,
			     conf);

	/* Recalcluate the parity. */
	LOG_DEBUG("Reconstructing parity on disk %d.\n", parity_index);
	memset(stripe->chunk[parity_index].data, 0, conf->chunk_size);
	reconstruct_chunk(conf, stripe, parity_index);

	/* Write the stripe to the disks. */
	for (i = 0; (rc == 0) && (i < conf->raid_disks); i++) {
		chunk_t * chunk = &stripe->chunk[i];

		/*
		 * One of the devices may be missing or faulty. If so,
		 * its dev field won't be filled in in the chunk
		 * structure.  Skip it since there is no device to
		 * write to.
		 */
		if (chunk->dev != NULL) {
			LOG_DEBUG("Writing %d sectors to %s at sector offset %"PRIu64".\n", conf->chunk_size >> EVMS_VSECTOR_SIZE_SHIFT, chunk->dev->name, chunk->lsn_on_dev);
			rc = WRITE(chunk->dev, chunk->lsn_on_dev, conf->chunk_size >> EVMS_VSECTOR_SIZE_SHIFT, chunk->data);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*   Function reads or writes sectors from a disk.
 *   depending on flag 0=READ 1=WRITE
 *   Returns: (positive) amount read if successful,
 *            (negative) else rc=errno of cause of failure.
static int seek_and_io( int fd, lsn_t offset, sector_count_t sectors,
		 void *buf, int flag )
{
    ssize_t n = 0;
    u_int32_t count = sectors << EVMS_VSECTOR_SIZE_SHIFT;
    u_int32_t tot = 0;

    if ( lseek64(fd, offset<<EVMS_VSECTOR_SIZE_SHIFT, SEEK_SET) == -1 ) {
	return -1;
    }

    while (tot < count) {
	do
	    if (flag)
		n = write(fd, buf, count - tot);
	    else
		n = read(fd, buf, count - tot);
	while ((n < 0) && ((errno == EINTR) || (errno == EAGAIN)));

	if (n <= 0)
	    return tot ? tot : n;

	tot += n;
	buf += n;
    }

    return tot;
} */
/* Function: raid5_write
 *
 *  Perform a logical-to-physical remapping, and send the write down to
 *  the next plugin.
 */
static int raid5_write( storage_object_t * region,
			lsn_t              lsn,
			sector_count_t     count,
			void             * buffer ) {

	int                     rc = 0;
	md_volume_t             * volume = (md_volume_t *)region->private_data;
	stripe_t                stripe = {0};
	sector_count_t          sectors_written;

	my_plugin = raid5_plugin;
	LOG_ENTRY();

	LOG_DEBUG("Request to write %"PRIu64" sectors to %s at sector offset %"PRIu64".\n", count, region->name, lsn);

	/* Parameter check */
	if (!buffer) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (volume->flags & MD_CORRUPT) {
		LOG_ERROR("MD Object %s is corrupt, writing data is not allowed\n",volume->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}
	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to write past end of region %s sector=%"PRIu64"\n",volume->name,lsn+count);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (kill_sector_list_head != NULL) {
		kill_sectors();
	}

	rc = md_region_rw(region, lsn, count, buffer, 1);

	/* If the target RAID5 array does not exist, we can try do to the write by hand. */
	if (rc) {
		rc = 0;
		while ((count != 0) && (rc == 0)) {
			if ((lsn >= stripe.start_lsn) &&
			    (lsn < stripe.start_lsn + stripe.data_size)) {
				rc = stripe_io(STRIPE_IO_WRITE,
					       volume, &stripe,
					       lsn, count,
					       buffer, &sectors_written);
				count -= sectors_written;
				lsn += sectors_written;
				buffer += sectors_written << EVMS_VSECTOR_SIZE_SHIFT;

			} else {
				if (stripe.data_size != 0) {
					write_stripe(volume, &stripe);
					free_stripe(&stripe);
				}

				/* Read a new stripe and try again. */
				rc = get_stripe(volume, lsn, &stripe);
			}
		}

		if (stripe.data_size != 0) {
			write_stripe(volume, &stripe);
			free_stripe(&stripe);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid5_enable_fix_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;

	LOG_ENTRY();

	raid5_verify_and_fix_array(volume, 0);

	if (volume->flags & (MD_DEGRADED | MD_ARRAY_SYNCING)) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	if (md_can_stop_array(volume->region) == FALSE) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	if (volume->flags & (MD_CORRUPT | MD_PROBLEMATIC_SB)) {

		fia->info[*function_count].function = MD_RAID5_FUNCTION_FIX;
		SET_STRING(fia->info[*function_count].name, "fix");
		SET_STRING(fia->info[*function_count].title, "Fix");
		SET_STRING(fia->info[*function_count].verb, "Fix");
		SET_STRING(fia->info[*function_count].help,
			"The RAID array has inconsistent metadata.  Use this function to fix the metadata.");
		if (volume->flags & MD_ARRAY_SYNCING) {
			fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
		}

		++*function_count;
		enable = 1;
	}

	LOG_EXIT_INT(enable);
	return enable;
}

static int raid5_enable_restore_major_minor_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;

	LOG_ENTRY();
	
	if (volume->flags & (MD_CORRUPT | MD_PROBLEMATIC_SB)) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	//if (volume->region_mgr_flags & MD_RAID5_CONFIG_CHANGE_PENDING) {
	//	LOG_EXIT_INT(enable);
	//	return enable;
	//}


	if (md_can_restore_saved_sb(volume->region)) {
		fia->info[*function_count].function = MD_RAID5_RESTORE_SUPERBLOCK;
		SET_STRING(fia->info[*function_count].name, "ressuperblock");
		SET_STRING(fia->info[*function_count].title, "Restore orginal major/minor");
		SET_STRING(fia->info[*function_count].verb, "Restore");
		SET_STRING(fia->info[*function_count].help,
			"Use this function to restore the original major and minor of all devices made up the MD array.");
		if ((volume->flags & MD_ARRAY_SYNCING) ||
		    (volume->region_mgr_flags & MD_RAID5_CONFIG_CHANGE_PENDING)) {
			fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
		}

		++*function_count;
		enable = 1;
	}

	LOG_EXIT_INT(enable);
	return enable;
}

static int raid5_enable_remove_spare_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;
	LOG_ENTRY();

	if ((volume->super_block->spare_disks > 1) ||
	    ((volume->super_block->spare_disks == 1) &&
	     volume->super_block->active_disks == volume->super_block->raid_disks)) {

		fia->info[*function_count].function = MD_RAID5_FUNCTION_REMOVE_SPARE;
		SET_STRING(fia->info[*function_count].name, "remspare");
		SET_STRING(fia->info[*function_count].title, "Remove spare object");
		SET_STRING(fia->info[*function_count].verb, "Remove");
		SET_STRING(fia->info[*function_count].help, "Use this function to remove a spare object from this RAID array.");
		if ((volume->flags & MD_ARRAY_SYNCING) ||
		    (volume->region_mgr_flags & MD_RAID5_CONFIG_CHANGE_PENDING) ) {
			fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
		}
		
		++*function_count;
		enable = 1;
	}
	LOG_EXIT_INT(enable);
	return enable;
	
}

static int raid5_enable_add_spare_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;
	int rc = 0;
	list_anchor_t available_objects = NULL;
	uint count;
	LOG_ENTRY();

	/*
	 * If there are available top level objects in
	 * the system that are at least as big as the
	 * smallest child object, then the Add Spare
	 * action is avalable.
	 */
	rc = EngFncs->get_object_list(DISK | SEGMENT| REGION,
				      DATA_TYPE,
				      NULL,
					  NULL,
				      VALID_INPUT_OBJECT,
				      &available_objects);

	if (rc == 0) {
		/*
		 * If this MD region is available, it
		 * will appear in the list.  Bad things
		 * happen if this region is made a
		 * child of itself.  Remove this MD
		 * region if it is in the list.
		 */
		EngFncs->remove_thing(available_objects, volume->region);

		prune_small_objects(available_objects, volume->super_block );

		count = EngFncs->list_count(available_objects);

		if (count > 0) {

			fia->info[*function_count].function = MD_RAID5_FUNCTION_ADD_SPARE;
			SET_STRING(fia->info[*function_count].name, "addspare");
			SET_STRING(fia->info[*function_count].verb, "Add spare");
			if (volume->flags & MD_DEGRADED && (volume->nr_disks < volume->super_block->raid_disks)) {
				SET_STRING(fia->info[*function_count].title,
					   "Add spare to fix degraded array");
				SET_STRING(fia->info[*function_count].help,
					   "Use this function to add a spare object to replace a missing or faulty entry of this degraded region.");
			} else {
				SET_STRING(fia->info[*function_count].title,
					   "Add spare object");
				SET_STRING(fia->info[*function_count].help,
					   "Use this function to add an object as a spare object for this RAID array.");
			}
			if ((volume->flags & MD_ARRAY_SYNCING) ||
			    (volume->region_mgr_flags & MD_RAID5_CONFIG_CHANGE_PENDING) ){
				fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
			}
			
			++*function_count;
			enable = 1;
		}

		EngFncs->destroy_list(available_objects);
	}
	
	LOG_EXIT_INT(enable);
	return enable;
	
}

static int raid5_enable_remove_faulty_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;
	int i;
	storage_object_t *obj;

	LOG_ENTRY();

	/*
	 * If the RAID array has a faulty disk then
	 * Remove Faulty is available.
	 */
	for (i = 0; i < MAX_MD_DEVICES; i++) {

		obj = volume->child_object[i];

		if ( obj && disk_faulty(&volume->super_block->disks[i]) ) {

			fia->info[*function_count].function = MD_RAID5_FUNCTION_REMOVE_FAULTY;
			SET_STRING(fia->info[*function_count].name, "remfaulty");
			SET_STRING(fia->info[*function_count].title, "Remove a faulty object");
			SET_STRING(fia->info[*function_count].verb, "Remove");
			SET_STRING(fia->info[*function_count].help,
				"Use this function to permanently remove a faulty object from this RAID array.");
			if (volume->flags & MD_ARRAY_SYNCING) {
				fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
			}

			++*function_count;
			enable = 1;
			break;
		}
	}
	LOG_EXIT_INT(enable);
	return enable;
}

static int raid5_enable_remove_stale_disk_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;
	int i;

	LOG_ENTRY();

	/*
	 * If the RAID array has a stale disk then
	 * Remove Stale is available.
	 */
	if (!volume->stale_disks) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	for (i = 0; i < MAX_MD_DEVICES; i++) {

		if ( volume->stale_object[i] )
		{

			fia->info[*function_count].function = MD_RAID5_FUNCTION_REMOVE_STALE;
			SET_STRING(fia->info[*function_count].name, "remstale");
			SET_STRING(fia->info[*function_count].title, "Remove a stale object");
			SET_STRING(fia->info[*function_count].verb, "Remove");
			SET_STRING(fia->info[*function_count].help,
				"Use this function to permanently remove a stale (possibly faulty) object from this RAID array.");
			if (volume->flags & MD_ARRAY_SYNCING) {
				fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
			}

			++*function_count;
			enable = 1;
			break;
		}
	}
	LOG_EXIT_INT(enable);
	return enable;
}

static int raid5_enable_mark_disk_faulty_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;
	int i;

	LOG_ENTRY();

	if ((volume->region_mgr_flags & MD_RAID5_REGION_NEW) ||
	    !md_is_region_active(volume->region) ) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	/*
	 * We have been experiencing problems with SET_DISK_FAULTY ioctl on 2.6 kernel.
	 * The kernel segfaults in drivers/md/raid5.c.
	 * For now, disable mark_disk_faulty if we are running on 2.6 kernel
	 */
	{
		mdu_array_info_t array_info;
		if (md_ioctl_get_array_info(volume->region, &array_info) == 0)
		if (array_info.patch_version > 0) {
			LOG_EXIT_INT(enable);
			return enable;
		}
	}


	/*
	 * If the RAID array is not running in degrade mode and
	 * there is a spare available that is not new then Mark
	 * Faulty is available.
	 */
	if (volume->super_block->active_disks == volume->super_block->raid_disks) {

		/* Array is not degraded.  Check for a spare. */
		for (i = 0; i < MAX_MD_DEVICES; i++) {
			if ( ( volume->child_object[i] != NULL) &&
			     ( disk_spare(&volume->super_block->disks[i])) ) {

				fia->info[*function_count].function = MD_RAID5_FUNCTION_MARK_FAULTY;
				SET_STRING(fia->info[*function_count].name, "markfaulty");
				SET_STRING(fia->info[*function_count].title, "Mark object faulty");
				SET_STRING(fia->info[*function_count].verb, "Mark faulty");
				SET_STRING(fia->info[*function_count].help, "Use this function to mark an object faulty in this RAID array.  If the RAID array has a spare object, the spare object will be brought on-line to replace the faulty object.");
				if ((volume->flags & MD_ARRAY_SYNCING) ||
				    (volume->region_mgr_flags & MD_RAID5_CONFIG_CHANGE_PENDING)) {
					fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
				}

				++*function_count;
				enable = 1;
				break;
			}
		}
	}
	LOG_EXIT_INT(enable);
	return enable;

}


/* Function:  raid5_get_plugin_functions
 */
static int raid5_get_plugin_functions(storage_object_t        * region,
				      function_info_array_t * * functions)
{
	function_info_array_t * fia;
	int rc, function_count = 0;
	md_volume_t * volume;

	my_plugin = raid5_plugin;
	LOG_ENTRY();

	/*
	 * If region is NULL, that means the user is asking for plug-in
	 * functions on the plug-in.  We don't have any plug-in functions that
	 * are global for the plug-in.
	 */
	if (region == NULL) {
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	volume = (md_volume_t *) region->private_data;

	fia = EngFncs->engine_alloc(sizeof(function_info_array_t) + sizeof(function_info_t) * (MD_RAID5_FUNCTION_COUNT - 1));
	if (!fia) {
		LOG_CRITICAL("Error allocating memory for an action info array.\n");
		LOG_EXIT_INT (ENOMEM);
		return ENOMEM;
	}

	/*
	 * Our functions are only allowed if the Engine is opened for
	 * writing.
	 */
	if (!(EngFncs->get_engine_mode() & ENGINE_WRITE)) {
		fia->count = function_count;
		*functions = fia;
		LOG_EXIT_INT(0);
		return 0;
	}

	if (md_is_recovery_running(region)) {
		volume->flags |= MD_ARRAY_SYNCING;
		LOG_DEBUG("%s : Resync/recovery is running\n", region->name);
	} else {
		if (volume->flags & MD_ARRAY_SYNCING) {
			rc = raid5_rediscover_region(region);
		}
		volume->flags &= ~MD_ARRAY_SYNCING;
	}

	/*
	 * If the array needs fixing, then that is the only function available and maybe remove stale disks
	 */
	if ( raid5_enable_fix_function(volume, fia, &function_count) && (volume->flags & MD_CORRUPT)) {

		raid5_enable_remove_stale_disk_function(volume, fia, &function_count);

		fia->count = function_count;
		*functions = fia;
		LOG_EXIT_INT(0);
		return (0);
	}

	raid5_enable_restore_major_minor_function(volume, fia, &function_count);
	raid5_enable_add_spare_function(volume, fia, &function_count);
	raid5_enable_remove_spare_function(volume, fia, &function_count);
	raid5_enable_remove_faulty_function(volume, fia, &function_count);
	raid5_enable_remove_stale_disk_function(volume, fia, &function_count);
	raid5_enable_mark_disk_faulty_function(volume, fia, &function_count);

	fia->count = function_count;
	*functions = fia;

	LOG_EXIT_INT(0);
	return 0;
}


static int can_be_added(md_volume_t * volume, storage_object_t * spare_candidate)
{
	mdp_super_t *sb = volume->super_block;
	sector_count_t size;

	LOG_ENTRY();

	/* The spare must be a disk, segment, or region. */
	if ((spare_candidate->object_type != DISK) &&
	    (spare_candidate->object_type != SEGMENT) &&
	    (spare_candidate->object_type != REGION)) {
		LOG_ERROR("The type of object %s is not data.\n", spare_candidate->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* The spare must not be too small. */
	size = MD_CHUNK_ALIGN_NEW_SIZE_SECTORS((sb->chunk_size / EVMS_VSECTOR_SIZE), spare_candidate->size);
	if (size < (sb->size / EVMS_VSECTOR_SIZE)) {
		LOG_ERROR("Object %s is too small to be a spare object for array %s.\n",
			  spare_candidate->name, volume->region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* The region cannot be a spare of itself. */
	if (spare_candidate == volume->region) {
		LOG_ERROR("Region %s cannot be a spare object for itself.\n", spare_candidate->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	LOG_EXIT_INT(0);
	return 0;
}


static int is_spare(md_volume_t * volume, storage_object_t * spare_disk)
{
	int i, j;

	LOG_ENTRY();

	j = 0;
	for (i = 0; (i < MAX_MD_DEVICES) && (j < volume->nr_disks); i++) {
		if (volume->child_object[i] != NULL) {
			if (volume->child_object[i] == spare_disk) {
				if ((volume->super_block->disks[i].state & ~(1 << MD_DISK_NEW)) == 0) {
					LOG_EXIT_INT(0);
					return 0;
				} else {
					LOG_ERROR("Object %s is in array %s but is not a spare object.\n", spare_disk->name, volume->region->name);
					LOG_EXIT_INT(EINVAL);
					return EINVAL;
				}
			} else {
				j++;
			}
		}
	}

	/* The object was not found in the array. */
	LOG_ERROR("Object %s is not part of array %s.\n", spare_disk->name, volume->region->name);
	LOG_EXIT_INT(EINVAL);
	return EINVAL;
}


static int is_faulty(md_volume_t *volume, storage_object_t * faulty_disk)
{
	int i, j;

	LOG_ENTRY();

	j = 0;
	for (i = 0; (i < MAX_MD_DEVICES) && (j < volume->nr_disks); i++) {
		if (volume->child_object[i] != NULL) {
			if (volume->child_object[i] == faulty_disk) {
				if (volume->super_block->disks[i].state & (1 << MD_DISK_FAULTY)) {
					LOG_EXIT_INT(0);
					return 0;
				} else {
					LOG_ERROR("Object %s is in array %s but is not faulty.\n", faulty_disk->name, volume->region->name);
					LOG_EXIT_INT(EINVAL);
					return EINVAL;
				}
			} else {
				j++;
			}
		}
	}

	/* The object was not found in the array. */
	LOG_ERROR("Object %s is not part of array %s.\n", faulty_disk->name, volume->region->name);
	LOG_EXIT_INT(EINVAL);
	return EINVAL;
}

static int is_stale(md_volume_t *volume, storage_object_t *stale_disk)
{
	int i;
	storage_object_t * obj;

	LOG_ENTRY();

	if (!volume || !stale_disk) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	for (i = 0; i < MAX_MD_DEVICES; i++) {
		obj = volume->stale_object[i];
		if (obj) {
			if (obj == stale_disk) {
				LOG_EXIT_INT(0);
				return 0;
			}
		}
	}

	/* The object was not found in the array. */
	LOG_ERROR("Object %s is not part of array %s.\n", stale_disk->name, volume->region->name);
	LOG_EXIT_INT(EINVAL);
	return EINVAL;
}


static int is_active(md_volume_t * volume, storage_object_t * active_disk)
{
	int i, j;

	LOG_ENTRY();

	j = 0;
	for (i = 0; (i < MAX_MD_DEVICES) && (j < volume->nr_disks); i++) {
		if (volume->child_object[i] != NULL) {
			if (volume->child_object[i] == active_disk) {
				if (volume->super_block->disks[i].state == ((1 << MD_DISK_ACTIVE) | (1 << MD_DISK_SYNC))) {
					LOG_EXIT_INT(0);
					return 0;
				} else {
					LOG_ERROR("Object %s is in array %s but is not active.\n", active_disk->name, volume->region->name);
					LOG_EXIT_INT(EINVAL);
					return EINVAL;
				}
			} else {
				j++;
			}
		}
	}

	/* The object was not found in the array. */
	LOG_ERROR("Object %s is not part of array %s.\n", active_disk->name, volume->region->name);
	LOG_EXIT_INT(EINVAL);
	return EINVAL;
}


/* Function:  raid5_plugin_function
 */
static int raid5_plugin_function(storage_object_t * region,
				 task_action_t      action,
				 list_anchor_t            objects,
				 option_array_t   * options) {

	int rc = 0;
	md_volume_t * volume = (md_volume_t *) region->private_data;
	storage_object_t * object;
	list_element_t li;
	uint count;

	my_plugin = raid5_plugin;
	LOG_ENTRY();

	if ((action < EVMS_Task_Plugin_Function) ||
	    (action >= EVMS_Task_Plugin_Function + MD_RAID5_FUNCTION_COUNT)) {
		LOG_ERROR("Action code 0x%x is out of range.\n", action);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (action == MD_RAID5_FUNCTION_FIX) {
		raid5_fix_array(volume);
	} else if (action == MD_RAID5_RESTORE_SUPERBLOCK) {
		rc = md_restore_saved_sb(volume);
	} else {

		count = EngFncs->list_count(objects);

		switch (action) {
		case MD_RAID5_FUNCTION_ADD_SPARE:
			if (count == 0) {
				LOG_ERROR("Must specify at least one spare object to be added.\n");
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
			if (count > (MAX_MD_DEVICES - volume->super_block->nr_disks)) {
				LOG_ERROR("Can only specify up to %d object(s) to add as spare(s).\n", MAX_MD_DEVICES - volume->super_block->nr_disks);
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
			LIST_FOR_EACH(objects, li, object) {
				rc |= can_be_added(volume, object);
			}
			if (rc != 0) {
				LOG_EXIT_INT(rc);
				return rc;
			}
			break;

		case MD_RAID5_FUNCTION_REMOVE_SPARE:
			if (count == 0) {
				LOG_ERROR("Must specify at least one spare object to be removed.\n");
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
			LIST_FOR_EACH(objects, li, object) {
				rc |= is_spare(volume, object);
			}
			if (rc != 0) {
				LOG_EXIT_INT(rc);
				return rc;
			}
			if ((volume->super_block->active_disks < volume->super_block->raid_disks) &&
			    (count >= volume->super_block->spare_disks)) {
				LOG_ERROR("Array %s is running in degrade mode.  "
					  "At least one spare must be left for the array to recover.\n",
					  volume->region->name);
			}
			break;

		case MD_RAID5_FUNCTION_REMOVE_FAULTY:
			if (count == 0) {
				LOG_ERROR("Must specify at least one faulty object to be removed.\n");
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
			LIST_FOR_EACH(objects, li, object) {
				rc |= is_faulty(volume, object);
			}
			if (rc != 0) {
				LOG_EXIT_INT(rc);
				return rc;
			}
			break;
		
		case MD_RAID5_FUNCTION_REMOVE_STALE:
			if (count == 0) {
				LOG_ERROR("Must specify at least one stale object to be removed.\n");
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
			LIST_FOR_EACH(objects, li, object) {
				rc |= is_stale(volume, object);
			}
			if (rc != 0) {
				LOG_EXIT_INT(rc);
				return rc;
			}
			break;

		case MD_RAID5_FUNCTION_MARK_FAULTY:
			if (count != 1) {
				LOG_ERROR("Must specify only one object to be marked faulty.\n");
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
			LIST_FOR_EACH(objects, li, object) {
				rc |= is_active(volume, object);
			}
			if (rc != 0) {
				LOG_EXIT_INT(rc);
				return rc;
			}
			break;

		default:
			LOG_ERROR("0x%x is not a valid action code.\n", action);
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
			break;
		}

		LIST_FOR_EACH(objects, li, object) {
			switch (action) {
			case MD_RAID5_FUNCTION_ADD_SPARE:
				rc = raid5_add_spare_disk(volume, object);
				break;

			case MD_RAID5_FUNCTION_REMOVE_SPARE:
				rc = raid5_remove_spare_disk(volume, object);
				break;

			case MD_RAID5_FUNCTION_REMOVE_FAULTY:
				rc = raid5_remove_faulty_disk(volume, object);
				break;

			case MD_RAID5_FUNCTION_MARK_FAULTY:
				rc = raid5_mark_faulty_disk(volume, object);
				if (!rc) {
					if (follow_up_mark_faulty(volume, object) == TRUE) {
						rc = raid5_remove_faulty_disk(volume, object);
					}
				}

				break;

			case MD_RAID5_FUNCTION_REMOVE_STALE:
				rc = raid5_remove_stale_disk(volume, object);
				break;

			default:
				/*
				 * Shouldn't get here if the validiation
				 * code above did its job.
				 */
				LOG_WARNING("Action code 0x%x slipped past validation.\n", action);
				rc = EINVAL;
				break;
			}

			if (rc) {
				break;
			}

		}
	}

	if (rc == 0) {
		volume->region->flags |= SOFLAG_DIRTY;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static void free_region (storage_object_t *region)
{
	md_volume_t * volume = (md_volume_t *)region->private_data;
	raid5_conf_t * conf = mdvol_to_conf(volume);

	EngFncs->engine_free(conf);

	md_free_volume(volume);
	LOG_EXIT_VOID();
}



static void raid5_plugin_cleanup(void) {

	int rc;
	list_anchor_t raid5_regions = NULL;
	list_element_t li;
	storage_object_t *region;

	my_plugin = raid5_plugin;
	LOG_ENTRY();

	rc = EngFncs->get_object_list(REGION, DATA_TYPE, raid5_plugin, NULL, 0, &raid5_regions);

	if (rc == 0) {
		LIST_FOR_EACH(raid5_regions, li, region) {
			free_region(region);
		}

		EngFncs->destroy_list(raid5_regions);
	}

	LOG_EXIT_VOID();
}



/* Function tables for the MD Region Manager */
static plugin_functions_t raid5_functions = {
	setup_evms_plugin       : raid5_setup_evms_plugin,
	cleanup_evms_plugin     : raid5_plugin_cleanup,
	can_delete              : raid5_can_delete,
	can_replace_child	: raid5_can_replace_child,
	discover                : raid5_discover,
	create                  : raid5_create,
	delete                  : raid5_delete,
	discard                 : raid5_discard,
	replace_child		: raid5_replace_child,
	add_sectors_to_kill_list: raid5_add_sectors_to_kill_list,
	commit_changes          : raid5_commit_changes,
	activate		: raid5_activate_region,
	deactivate		: raid5_deactivate_region,
	get_option_count        : raid5_get_option_count,
	init_task               : raid5_init_task,
	set_option              : raid5_set_option,
	set_objects             : raid5_set_objects,
	get_info                : raid5_get_info,
	get_plugin_info         : raid5_get_plugin_info,
	read                    : raid5_read,
	write                   : raid5_write,
	get_plugin_functions    : raid5_get_plugin_functions,
	plugin_function         : raid5_plugin_function,
};



/* Function: PluginInit
 *
 *  Initializes the local plugin record
 */

plugin_record_t raid5_plugin_record = {
	id:                     SetPluginID(EVMS_OEM_IBM, EVMS_REGION_MANAGER, 7),

	version:                {major:      MAJOR_VERSION,
		minor:      MINOR_VERSION,
		patchlevel: PATCH_LEVEL},

	required_engine_api_version: {major:      13,
		minor:      0,
		patchlevel: 0},
	required_plugin_api_version: {plugin: {major:      12,
			minor:      0,
			patchlevel: 0}},

	short_name:             "MDRaid5RegMgr",
	long_name:              "MD RAID 4/5 Region Manager",
	oem_name:               "IBM",
	functions:              {plugin: &raid5_functions},
	container_functions:    NULL
};
