/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: raid1_mgr.c
 *
 * Description: This file contains all of the required engine-plugin APIs
 *              for the Raid1 MD region manager.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include <sys/ioctl.h>

#include "md.h"
#include "raid1_mgr.h"
#include "raid1_discover.h"

#define my_plugin_record raid1_plugin

static boolean raid1_can_change_region_configuration( storage_object_t *region )
{
	boolean rc = TRUE;
	md_volume_t * volume = (md_volume_t *)region->private_data;

	if (!volume)
		rc = FALSE;
	else if (raid1_region_new(volume))
		rc = FALSE;
	else if (raid1_region_config_change_pending(volume))
		rc = FALSE;
	else if ((volume->flags & (MD_CORRUPT | MD_DEGRADED | MD_PROBLEMATIC_SB)))
		rc = FALSE;
	else if (!md_can_stop_array(region))
		rc = FALSE;

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid1_can_children_expand_shrink
 *
 * Call can_expand()/can_shrink() on each child to find out if they can expanded/shrunk.
 * The region can be expanded/shrunk if all children can.
 */
static int raid1_can_children_expand_shrink( int cmd,
					     storage_object_t *region,
					     u_int64_t        * max_size)
{
	int i, rc = 0;
	list_anchor_t temp_list;
	md_volume_t * volume = (md_volume_t *)region->private_data;
	expand_object_info_t * expand_object;
	shrink_object_info_t * shrink_object;
	void *temp_obj;
	u_int64_t size = -1;
	u_int64_t minimum_size;
	list_element_t li;

	LOG_ENTRY();

	switch (cmd) {
	case RAID1_EXPAND:
		minimum_size = RAID1_MINIMUM_EXPAND_SIZE;
		break;
	case RAID1_SHRINK:
		minimum_size = RAID1_MINIMUM_SHRINK_SIZE;
		break;
	default:
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	temp_list  = EngFncs->allocate_list();
	if (temp_list == NULL) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	for (i = 0; !rc && i < MAX_MD_DEVICES; i++ ) {
		if (volume->child_object[i]) {
			u_int64_t max_delta_size = -1;
			storage_object_t *child = volume->child_object[i];
			if (cmd == RAID1_EXPAND)
				rc = child->plugin->functions.plugin->can_expand(child, &max_delta_size, temp_list);
			else
				rc = child->plugin->functions.plugin->can_shrink(child, &max_delta_size, temp_list);
			if (rc)
				LOG_DEBUG(" %s can not %s!\n", child->name, (cmd == RAID1_EXPAND) ? "expand" : "shrink");
		}
	}


	if (!rc) {
		
		LIST_FOR_EACH(temp_list, li, temp_obj) {
			if (cmd == RAID1_EXPAND) {
				expand_object = (expand_object_info_t *)temp_obj;
				LOG_DEBUG(" object %s said its max expand size is %"PRIu64"\n",
						  expand_object->object->name, expand_object->max_expand_size);
				size = min(size, expand_object->max_expand_size);
			} else {
				shrink_object = (shrink_object_info_t *)temp_obj;
				LOG_DEBUG(" object %s said its max shrink size is %"PRIu64"\n",
						  shrink_object->object->name, shrink_object->max_shrink_size);
				size = min(size, shrink_object->max_shrink_size);
			}
		}
	} else {
		/* one of the children can not handle expand and shrink */
		*max_size = 0;
		EngFncs->destroy_list(temp_list);
		LOG_EXIT_INT(rc);
		return rc;
	}

	rc = 0;

	if (size < minimum_size) {
		LOG_WARNING(" %s size is too small (%"PRIu64" sectors)\n",
					(cmd == RAID1_EXPAND) ? "expandable" : "shrinkable",
					size);
		rc = EINVAL;
	} else {
		*max_size = size;
		if (cmd == RAID1_SHRINK) {
			if (*max_size > (region->size * RAID1_PERCENT_SHRINK_THRESHOLD) / 100) {
				*max_size = (region->size * RAID1_PERCENT_SHRINK_THRESHOLD) / 100;
				LOG_WARNING("Adjust max shrink size down to %"PRIu64" (%d%% threshold)\n",
							*max_size, RAID1_PERCENT_SHRINK_THRESHOLD);
			}
		}
	}

	EngFncs->destroy_list(temp_list);

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid1_setup_evms_plugin
 *
 *  This function gets called shortly after the plugin is loaded by the
 *  Engine. It performs all tasks that are necessary before the initial
 *  discovery pass.
 */
static int raid1_setup_evms_plugin(engine_functions_t * functions)
{
	int rc = 0;

	// Parameter check
	if (!functions) {
		return EINVAL;
	}

	EngFncs = functions;

	my_plugin = raid1_plugin;
	LOG_ENTRY();
	rc = md_register_name_space();

	if (rc != 0) {
		LOG_SERIOUS("Failed to register the MD name space.\n");
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/****** Region Checking Functions ******/


/* All of the following md_can_ functions return 0 if they are able to
 * perform the specified action, or non-zero if they cannot.
 */


/* Function: raid1_can_delete
 *
 *  Can we remove the specified MD logical volume
 */
static int raid1_can_delete( storage_object_t * region )
{
	my_plugin = raid1_plugin;
	LOG_ENTRY();

	// Parameter check
	if (! region) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	// The region must be owned by MD,
	if (region->plugin != raid1_plugin) {
		LOG_ERROR("Region %s is not owned by MD.\n", region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}
	LOG_EXIT_INT(0);
	return 0;
}

/*
 *  Called by raid1_init_task() for Task_Expand to allocate and initialize
 *  the expand option descriptor array.
 */
static int raid1_init_expand_option_descriptors( task_context_t * context )
{
	int rc = EINVAL;
	storage_object_t *region;

	LOG_ENTRY();

	context->option_descriptors->count = 0;

	region = context->object;
	if ( region ) {

		if (region->object_type == REGION) {

			if (region->data_type == DATA_TYPE) {

				u_int64_t max_expand_size = 0;

				rc = raid1_can_children_expand_shrink(RAID1_EXPAND, region, &max_expand_size);
				if (!rc) {

					context->option_descriptors->count = RAID1_EXPAND_OPTION_COUNT;

					// Expanded RAID1 region size delta
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint.list = NULL;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
					if (context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint.range==NULL) {
						LOG_EXIT_INT(ENOMEM);
						return ENOMEM;
					}
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint_type = EVMS_Collection_Range;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].flags = 0;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].help = NULL;
					SET_STRING( context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].name, RAID1_EXPAND_OPTION_SIZE_NAME );
					SET_STRING( context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].tip,
									  "Use this option to specify how much space to add to the region.");
					SET_STRING( context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].title, "Additional Size");
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].type = EVMS_Type_Unsigned_Int64;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].unit = EVMS_Unit_Sectors;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint.range->min.ui64 = RAID1_MINIMUM_EXPAND_SIZE;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint.range->max.ui64 = max_expand_size;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = 1;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].value.ui64 = max_expand_size;

				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 *  Called by raid1_init_task() for Task_Expand to allocate and initialize
 *  the shrink option descriptor array.
 */
static int raid1_init_shrink_option_descriptors( task_context_t * context )
{
	int rc = EINVAL;
	storage_object_t *region;

	LOG_ENTRY();

	context->option_descriptors->count = 0;

	region = context->object;
	if ( region ) {

		if (region->object_type == REGION) {

			if (region->data_type == DATA_TYPE) {

				u_int64_t max_shrink_size = 0;

				rc = raid1_can_children_expand_shrink(RAID1_SHRINK, region, &max_shrink_size);
				if (!rc) {

					context->option_descriptors->count = RAID1_SHRINK_OPTION_COUNT;

					// Expanded RAID1 region size delta
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint.list = NULL;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
					if (context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint.range==NULL) {
						LOG_EXIT_INT(ENOMEM);
						return ENOMEM;
					}
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint_type = EVMS_Collection_Range;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].flags = 0;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].help = NULL;
					SET_STRING( context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].name, RAID1_SHRINK_OPTION_SIZE_NAME );
					SET_STRING( context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].tip,
									  "Use this option to specify how much space to reduce from the region.");
					SET_STRING( context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].title, "Shrink by Size");
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].type = EVMS_Type_Unsigned_Int64;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].unit = EVMS_Unit_Sectors;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint.range->min.ui64 = RAID1_MINIMUM_SHRINK_SIZE;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint.range->max.ui64 = max_shrink_size;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = 1;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].value.ui64 = max_shrink_size;

				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: set_expand_option_descriptors
 *
 * Called by raid1_set_option() for Task_Expand
 *
 */
static int raid1_set_expand_option( task_context_t * context,
				    u_int32_t        index,
				    value_t        * value,
				    task_effect_t  * effect )

{
	int rc = EINVAL;
	storage_object_t *region;
	u_int64_t expand_sectors = value->ui64;

	LOG_ENTRY();

	region = context->object;
	if ( region ) {

		if (region->object_type == REGION) {

			if (region->data_type == DATA_TYPE) {

				u_int64_t max_expand_size = 0;

				rc = raid1_can_children_expand_shrink(RAID1_EXPAND, region, &max_expand_size);
				if (!rc) {
					if (expand_sectors > max_expand_size) {
						expand_sectors = max_expand_size;
						*effect |= EVMS_Effect_Inexact;
					} else if (expand_sectors < RAID1_MINIMUM_EXPAND_SIZE) {
						expand_sectors = RAID1_MINIMUM_EXPAND_SIZE;
						*effect |= EVMS_Effect_Inexact;
					}
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint.range->min.ui64 = RAID1_MINIMUM_EXPAND_SIZE;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint.range->max.ui64 = max_expand_size;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].value.ui64 = expand_sectors;
				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: set_shrink_option_descriptors
 *
 * Called by raid1_set_option() for Task_Expand
 *
 */
static int raid1_set_shrink_option( task_context_t * context,
				    u_int32_t        index,
				    value_t        * value,
				    task_effect_t  * effect )

{
	int rc = EINVAL;
	storage_object_t *region;
	u_int64_t shrink_sectors = value->ui64;

	LOG_ENTRY();

	region = context->object;
	if ( region ) {

		if (region->object_type == REGION) {

			if (region->data_type == DATA_TYPE) {

				u_int64_t max_shrink_size = 0;

				rc = raid1_can_children_expand_shrink(RAID1_SHRINK, region, &max_shrink_size);
				if (!rc) {
					if (shrink_sectors > max_shrink_size) {
						shrink_sectors = max_shrink_size;
						*effect |= EVMS_Effect_Inexact;
					} else if (shrink_sectors < RAID1_MINIMUM_SHRINK_SIZE) {
						shrink_sectors = RAID1_MINIMUM_SHRINK_SIZE;
						*effect |= EVMS_Effect_Inexact;
					}
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint.range->min.ui64 = RAID1_MINIMUM_SHRINK_SIZE;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint.range->max.ui64 = max_shrink_size;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].value.ui64 = shrink_sectors;
				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static void raid1_get_expand_options( option_array_t * options, sector_count_t  *size)
{
    int i;

    LOG_ENTRY();

    for (i = 0; i < options->count; i++) {

        if (options->option[i].is_number_based) {

            if (options->option[i].number == RAID1_EXPAND_OPTION_SIZE_INDEX) {
                *size = options->option[i].value.ui64;
            }

        }
        else {

            if (strcmp(options->option[i].name, RAID1_EXPAND_OPTION_SIZE_NAME) == 0) {
                *size = options->option[i].value.ui64;
            }

        }
    }
	LOG_DEBUG(" expand size %"PRIu64" sectors\n", *size);

    LOG_EXIT_VOID();
}

static void raid1_get_shrink_options( option_array_t * options, sector_count_t  *size)
{
    int i;

    LOG_ENTRY();

    for (i = 0; i < options->count; i++) {

        if (options->option[i].is_number_based) {

            if (options->option[i].number == RAID1_SHRINK_OPTION_SIZE_INDEX) {
                *size = options->option[i].value.ui64;
            }

        }
        else {

            if (strcmp(options->option[i].name, RAID1_SHRINK_OPTION_SIZE_NAME) == 0) {
                *size = options->option[i].value.ui64;
            }

        }
    }
	LOG_DEBUG(" shrink size %"PRIu64" sectors\n", *size);

    LOG_EXIT_VOID();
}

/* Function: raid1_can_expand
 *
 *	Can this region be expanded? If so,
 *	add the region to the expansion_points output list. else, just
 *      return 0 to allow those above us to do as they please
 */
static int raid1_can_expand( storage_object_t * region,
			     u_int64_t        * expand_limit,
			     list_anchor_t            expansion_points )
{
	int rc = 0;
	expand_object_info_t * expand_object;
	u_int64_t max_expand_size = 0;
	list_element_t li=NULL;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	if (!raid1_can_change_region_configuration(region)) {
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	/* This region can expand if all children can, else return 0 */
	rc = raid1_can_children_expand_shrink(RAID1_EXPAND, region, &max_expand_size);
	if (rc) {
		LOG_EXIT_INT(0);
		return 0;
	}

	expand_object = (expand_object_info_t *) EngFncs->engine_alloc( sizeof(expand_object_info_t) );
	if (expand_object) {
		expand_object->object          = region;
		expand_object->max_expand_size = max_expand_size;

		li = EngFncs->insert_thing(expansion_points,
					   expand_object,
					   INSERT_AFTER,
					   NULL);

		if (!li) {
			EngFncs->engine_free( expand_object );
			rc = ENOMEM;
		}
	} else {
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_can_expand_by
 *
 *  No plans for raid1 expand.
 *
 */

static int raid1_can_expand_by( storage_object_t * child_object,
				u_int64_t        * size )
{
	my_plugin = raid1_plugin;
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}


/* Function: raid1_can_shrink
 *
 *	Just like can_expand, but in the opposite direction.
 */
static int raid1_can_shrink( storage_object_t * region,
			     u_int64_t        * shrink_limit,
			     list_anchor_t            shrink_points )
{
	int rc = 0;
	shrink_object_info_t * shrink_object;
	u_int64_t max_shrink_size = 0;
	list_element_t li = NULL;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	if (!raid1_can_change_region_configuration(region)) {
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	/* This region can be shrunk if all children can, else return 0 */
	rc = raid1_can_children_expand_shrink(RAID1_SHRINK, region, &max_shrink_size);
	if (rc) {
		LOG_EXIT_INT(0);
		return 0;
	}

	shrink_object = (shrink_object_info_t *) EngFncs->engine_alloc( sizeof(shrink_object_info_t) );
	if (shrink_object) {
		shrink_object->object          = region;
		shrink_object->max_shrink_size = max_shrink_size;

		li = EngFncs->insert_thing(shrink_points, shrink_object, INSERT_AFTER, NULL);

		if (!li) {
			EngFncs->engine_free( shrink_object );
			rc = ENOMEM;
		}
	} else {
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_can_shrink_by
 *
 *  Can we shrink the specified region? If it can be shrunk, but not to
 *  exactly new_size, reset new_size to the next lower possible size.
 */
static int raid1_can_shrink_by( storage_object_t * region,
				u_int64_t        * size )
{
	my_plugin = raid1_plugin;
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}


static int raid1_can_replace_child(storage_object_t *region,
				   storage_object_t *child,
				   storage_object_t *new_child)
{
	int rc;
	my_plugin = raid1_plugin;
	LOG_ENTRY();
	rc = md_can_replace_child(region, child, new_child);
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_discover
 *
 *  Examine all disk segments and find MD PVs. Assemble volume groups
 *  and export all MD logical volumes as EVMS regions.
 *
 *  All newly created regions must be added to the output list, and all
 *  segments from the input list must either be claimed or moved to the
 *  output list.
 */
static int raid1_discover( list_anchor_t input_list,
			   list_anchor_t output_list,
			   boolean final_call )
{
	int count = 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	if (final_call) {
		md_discover_final_call(input_list, output_list, &count);
	} else {
		md_discover_volumes(input_list, output_list);
		LOG_DETAILS("PV discovery complete.\n");

		// LV discovery and exporting
		raid1_discover_regions(output_list, &count, final_call);
		LOG_DETAILS("RAID1 volume discovery complete.\n");
	}

	LOG_EXIT_INT(count);
	return count;
}



/****** Region Functions ******/


static int raid1_get_create_options( option_array_t * options,
				     char          ** spare_disk)
{
	int i;
	int rc = 0;

	LOG_ENTRY();

	for (i = 0; i < options->count; i++) {

		if (options->option[i].is_number_based) {

			switch (options->option[i].number) {
			
			case MD_OPTION_SPARE_DISK_INDEX:
				// Not worth validation, will catch when we try to find the original
				*spare_disk = options->option[i].value.s;
				break;

			default:
				break;

			}

		} else {

			if (strcmp(options->option[i].name, MD_OPTION_SPARE_DISK_NAME) == 0) {
				*spare_disk = options->option[i].value.s;
			}
		}						
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* get the list of objects, search for the one requested.  If found, return the top object
 otherwise return NULL */
static storage_object_t * find_object( char * name )
{

	int rc = 0;
	storage_object_t * object = NULL;
	storage_object_t *obj;
	list_anchor_t objects;
	list_element_t li;

	LOG_ENTRY();
	if (!name) {
		LOG_EXIT_PTR(NULL);
		return NULL;
	}

	// get the list of objects
	rc = EngFncs->get_object_list(DISK | SEGMENT | REGION,		// non-evms objects
				 DATA_TYPE,
				 NULL,
				 NULL,
				 VALID_INPUT_OBJECT,
				 &objects);
	if (!rc) {

		LIST_FOR_EACH(objects, li, obj) {
			if (!strncmp(obj->name ,name,128)) {
				object = obj;
				break;
			}
		}
		EngFncs->destroy_list(objects);
	} else {
		LOG_ERROR("Error getting object list = %d....\n",rc);
	}
	LOG_EXIT_PTR(object);
	return object;
}

static int create_region_post_activate(md_volume_t *volume, md_ioctl_pkg_t *pkg)
{
	volume->region_mgr_flags &= ~MD_RAID1_REGION_NEW;
	return 0;
}

/* Function: raid1_create
 *
 *  Create a new MD volume using the specified freespace.
 */
static int raid1_create( list_anchor_t          objects,
			 option_array_t * options,
			 list_anchor_t          new_region_list )
{
	md_volume_t * volume = NULL;
	storage_object_t * object;
	int nr_disks;
	unsigned long size = -1;
	int i, spare_disks=0, spare_index = 0, index = 0;
	int rc = 0;
	mdp_disk_t disk;
	storage_object_t * spare=NULL;
	char * spare_disk = NULL;
	int chunk_size = MD_DEFAULT_CHUNK_SIZE;
	list_element_t iter1, iter2;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	// Parameter check
	if (!objects || !options || !new_region_list) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	nr_disks = EngFncs->list_count(objects);

	// allow 1 or more disks
	if (nr_disks < 1) {
		LOG_CRITICAL("Error, must have at least 2 objects selected for RAID 1 array. %d selected.\n",nr_disks);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (!(volume = EngFncs->engine_alloc(sizeof(md_volume_t) ))) {
		LOG_CRITICAL("Memory error new volume structure.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}
	LIST_FOR_EACH_SAFE(objects, iter1, iter2, object) {
		size = min(size, object->size);	 // track smallest object for super block
		volume->child_object[index] = object;
		index ++;
		EngFncs->remove_element(iter1);
	}

	raid1_get_create_options(options, &spare_disk);

	if (spare_disk) {
		spare = find_object( spare_disk);
	}
	if (spare) {
		size = min(size, spare->size);	// track smallest object for super block
		volume->child_object[index] = spare;
		nr_disks++;
		spare_disks = 1;
		spare_index = index;
	}
	disk.number = 0;
	disk.raid_disk = 0;
	disk.state = (1 << MD_DISK_ACTIVE) | (1 << MD_DISK_SYNC);

	size = MD_NEW_SIZE_BLOCKS(size/2); // first convert sectors to blocks

	rc = md_create_first_superblock(volume, disk, pers_to_level(RAID1), chunk_size, size, nr_disks, spare_disks, 0);

	if (rc) {
		EngFncs->engine_free(volume);
		LOG_EXIT_INT(rc);
		return rc;
	}

	if (spare) {
		volume->super_block->disks[spare_index].state = 0;  // set the state to inactive for the spare disk.
	}

	for (i = 0; i < nr_disks; i++) {
		rc = md_clone_superblock(volume, i);
		if (rc) {
			for (i--; i>=0; i--) {
				EngFncs->engine_free(volume->super_array[i]);
			}
			EngFncs->engine_free(volume->super_block);
			EngFncs->engine_free(volume);
			LOG_EXIT_INT(rc);
			return rc;
		}
	}

	volume->personality = RAID1;
	volume->nr_disks = nr_disks;
	volume->next = volume_list_head;
	volume_list_head = volume;
	volume->setup_funcs = EngFncs->allocate_list();
	volume->ioctl_pkgs = EngFncs->allocate_list();
	volume->ioctl_cleanup = EngFncs->allocate_list();

	rc = raid1_create_region(volume, new_region_list, TRUE);
	if (!rc) {
		volume->flags |= MD_DIRTY;
		volume->region_mgr_flags |= MD_RAID1_REGION_NEW;
		schedule_md_ioctl_pkg(volume, EVMS_MD_INVOKE_CALLBACK, NULL, create_region_post_activate);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: w_delete
 *
 * Worker function for raid1_delete and raid1_discard
 */
static int w_delete(storage_object_t *region, list_anchor_t children, boolean tear_down)
{
	md_volume_t   * volume;
	int     rc;

	LOG_ENTRY();

	// Check that this region can be removed.
	if ((rc = raid1_can_delete(region))) {
		LOG_EXIT_INT(rc);
		return rc;
	}
	volume = region->private_data;

	// Remove the parent/child associations with the PVs
	md_clear_child_list(region, children);

	// Delete the volume.
	md_delete_volume(volume, tear_down);
	region->private_data = NULL;
	EngFncs->free_region(region);

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid1_delete
 *
 *  Remove the specified region and consolidate all of its space into
 *  the appropriate freespace region.
 */
static int raid1_delete(storage_object_t * region, list_anchor_t children)
{
	int rc;
	
	my_plugin = raid1_plugin;
	LOG_ENTRY();
	rc = w_delete(region, children, TRUE);
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: raid1_discard
 * 
 * This function is similar to delete.  Just call delete to free all
 * data structures related to the regions.
 */
static int raid1_discard(list_anchor_t regions)
{
	storage_object_t * region;
	list_element_t le;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	LIST_FOR_EACH(regions, le, region) {
		w_delete(region, NULL, FALSE);
	}

	LOG_EXIT_INT(0);
	return 0;
}

/* Function: raid1_expand_shrink_children
 *
 * Tell all children to expand
 */
static int raid1_expand_shrink_children(int cmd, storage_object_t *region, u_int64_t sectors)
{
	int rc = 0;
	int i;
	md_volume_t *volume = (md_volume_t *)region->private_data;
	option_array_t option_array;
	u_int64_t size = -1;

	LOG_ENTRY();

	option_array.count = 1;
	option_array.option[0].is_number_based = FALSE;
	option_array.option[0].name = RAID1_EXPAND_OPTION_SIZE_NAME;
	option_array.option[0].value.ui64 = sectors;

	LOG_DEBUG(" %s region %s. current size = %"PRIu64" sectors\n",
			  (cmd == RAID1_EXPAND) ? "expanding" : "shrinking", region->name, region->size);

	for (i = 0; !rc && i < MAX_MD_DEVICES; i++ ) {
		if (volume->child_object[i]) {
			storage_object_t *child = volume->child_object[i];

			KILL_SECTORS(child, MD_NEW_SIZE_SECTORS(child->size), MD_RESERVED_SECTORS);
			LOG_DEBUG(" %s (current size=%"PRIu64")\n", child->name, child->size);
			if (cmd == RAID1_EXPAND)
				rc = child->plugin->functions.plugin->expand(child, child, NULL, &option_array);
			else
				rc = child->plugin->functions.plugin->shrink(child, child, NULL, &option_array);
			if (!rc) {
				LOG_DEBUG(" %s (new size=%"PRIu64")\n", child->name, child->size);
				size = min(size, child->size);
			}
		}
	}

	if (!rc) {
		size = MD_NEW_SIZE_BLOCKS(size/2);
		volume->super_block->size = size;
		region->size = size * 2;
		LOG_DEBUG(" region %s now has new size = %"PRIu64" sectors\n", region->name, region->size);
	} else {
		LOG_ERROR(" [%s] Error! setting back to orginal size = %"PRIu64" sectors\n", region->name, region->size);
		size = volume->super_block->size;
	}

	for (i=0; i < MAX_MD_DEVICES; i++) {
		if (volume->child_object[i] && volume->super_array[i]) {
			volume->super_array[i]->size = size;
		}
	}

	volume->flags |= MD_DIRTY;
	region->flags |= (SOFLAG_DIRTY | SOFLAG_NEEDS_DEACTIVATE);

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid1_expand
 */
static int raid1_expand( storage_object_t    * region,
			 storage_object_t    * expand_object,
			 list_anchor_t               input_objects,
			 option_array_t      * options )
{
	int rc = 0;
	u_int64_t expand_size = 0;
	u_int64_t max_expand_size = 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	/* check to make sure */
	if (region &&
		region == expand_object &&
		region->object_type == REGION &&
		region->data_type == DATA_TYPE) {

		raid1_get_expand_options(options, &expand_size);

		/* confirm one more time */
		rc = raid1_can_children_expand_shrink(RAID1_EXPAND, region, &max_expand_size);
		if (!rc) {
			if (expand_size > max_expand_size) {
				LOG_WARNING(" requested expand_size=%"PRIu64" max_expand_size=%"PRIu64"\n", expand_size, max_expand_size);
				expand_size = max_expand_size;
			}
			rc = raid1_expand_shrink_children(RAID1_EXPAND, region, expand_size);
		}


	} else {
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_shrink
 */
static int raid1_shrink( storage_object_t * region,
			 storage_object_t * shrink_object,
			 list_anchor_t            input_objects,
			 option_array_t   * options )
{
	int rc = 0;
	u_int64_t shrink_size = 0;
	u_int64_t max_shrink_size = 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	/* check to make sure */
	if (region &&
		region == shrink_object &&
		region->object_type == REGION &&
		region->data_type == DATA_TYPE) {

		raid1_get_shrink_options(options, &shrink_size);

		/* confirm one more time */
		rc = raid1_can_children_expand_shrink(RAID1_SHRINK, region, &max_shrink_size);
		if (!rc) {
			if (shrink_size > max_shrink_size) {
				LOG_WARNING(" requested shrink_size=%"PRIu64" max_shrink_size=%"PRIu64"\n", shrink_size, max_shrink_size);
				shrink_size = max_shrink_size;
			}
			rc = raid1_expand_shrink_children(RAID1_SHRINK, region, shrink_size);
		}


	} else {
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}




static int raid1_replace_child(storage_object_t *region,
			       storage_object_t *child,
			       storage_object_t *new_child)
{
	int rc;
	my_plugin = raid1_plugin;
	LOG_ENTRY();
	rc = md_replace_child(region, child, new_child);
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_add_sectors_to_kill_list
 *
 *  The kill_sectors list contains a list of sectors that need to be zeroed
 *  during the next commit. This function is very similar to read/write.
 */
static int raid1_add_sectors_to_kill_list( storage_object_t * region,
					   lsn_t              lsn,
					   sector_count_t     count )
{

	md_volume_t   * volume = (md_volume_t *)region->private_data;
	int     rc = 0;
	int i, found=0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!region) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (volume->flags & MD_CORRUPT) {
		LOG_ERROR("MD Object %s is corrupt, writing data is not allowed.\n",volume->name);
		MESSAGE("MD Object %s is corrupt, writing data is not allowed.\n",volume->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}
	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to write past end of region %s sector=%"PRIu64"\n",volume->name,lsn+count);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	for (i = 0; (i < MAX_MD_DEVICES) && (found < volume->nr_disks); i++ ) {
		// check for null object, if missing, skip
		if (volume->child_object[i]) {
			found++;
			// only kill sectors on Active disks, not the spares.
			if (volume->super_block->disks[i].state & (1 << MD_DISK_ACTIVE)) {
				rc = KILL_SECTORS(volume->child_object[i], lsn, count);
				if (rc) {
					//BUGBUG disable mirror
				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid1_commit_changes
 *
 * The phases:
 *	SETUP
 *	  - for actions required prior to superblocks update
 *	FIRST_METADATA_WRITE
 *	  - write MD superblocks, must have enabled MD_DIRTY flag
 *	SECOND_METADATA_WRITE
 *	  - not used
 *	POST_ACTIVATE
 *	  -  for actions required after superblocks update, _or_
 *	     for queued IOCTLs
 *	  -  reload superblocks via raid1_rediscover_region
 *	
 *	NOTE:  In order to get invoked for all phases of commit process,
 *		leave SOFLAG_DIRTY until the last phase (POST_ACTIVATE)
 */
static int raid1_commit_changes( storage_object_t * region,
				 uint               phase )
{
	md_volume_t * volume;
	int         rc = 0;
	int saved_rc;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	if (! region) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (region->plugin != raid1_plugin) {
		LOG_ERROR("Region %s does not belong to MD.\n", region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}
	if (! (region->flags & SOFLAG_DIRTY)) {
		LOG_WARNING("Region %s is not dirty - not committing.\n", region->name);
		LOG_EXIT_INT(0);
		return 0;
	}
	
	volume = (md_volume_t *)region->private_data;
	if (!volume) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}


	switch (phase) {
	case SETUP:
		rc = process_setup_funcs(region);
		break;
	case FIRST_METADATA_WRITE:
		if (volume->flags & MD_DIRTY) {
			rc = md_write_sbs_to_disk(volume);
		}
		break;
	case SECOND_METADATA_WRITE:
		break;
	case POST_ACTIVATE:
		rc = process_md_ioctl_pkgs(region);

		/* Despite rc, we will rediscover the MD array, save the return code */
		saved_rc = rc;
		rc = raid1_rediscover_region(region);
		if (saved_rc != 0)
			rc = saved_rc;
		free_ioctl_pkgs(volume);
		region->flags &= ~SOFLAG_DIRTY;
		break;
	default	:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid1_activate_region(storage_object_t * region)
{
	int rc;
	md_volume_t *volume;
	my_plugin = raid1_plugin;
	LOG_ENTRY();
	
	if (!region || !(volume = (md_volume_t *)region->private_data)) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	raid1_verify_and_fix_array(volume, 0);	

	rc = md_activate_region (region);

	LOG_EXIT_INT(rc);
	return rc;

}

static int raid1_deactivate_region(storage_object_t * region)
{
	int rc=0;
	my_plugin = raid1_plugin;
	LOG_ENTRY();

	rc = md_deactivate_region (region);

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_get_option_count
 *
 *  Determine the type of Task that is being performed, and return
 *  the number of options that are available for that Task.
 */
static int raid1_get_option_count( task_context_t * task )
{
	int count = 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	switch (task->action) {
	case EVMS_Task_Create:
		count = 1;
		break;
	case EVMS_Task_Expand:
		count = RAID1_EXPAND_OPTION_COUNT;
		break;
	case EVMS_Task_Shrink:
		count = RAID1_SHRINK_OPTION_COUNT;
		break;
	
	case MD_RAID1_FUNCTION_FIX:
	case MD_RAID1_FUNCTION_ADD_SPARE:
	case MD_RAID1_FUNCTION_ADD_ACTIVE:
	case MD_RAID1_FUNCTION_ACTIVATE_SPARE:
	case MD_RAID1_FUNCTION_REMOVE_SPARE:
	case MD_RAID1_FUNCTION_DEACTIVATE_ACTIVE:
	case MD_RAID1_FUNCTION_REMOVE_ACTIVE:
	case MD_RAID1_FUNCTION_REMOVE_FAULTY:
	case MD_RAID1_FUNCTION_MARK_FAULTY:
	case MD_RAID1_FUNCTION_REMOVE_STALE:
	case MD_RAID1_RESTORE_SUPERBLOCK:
		count = 0;
		break;
	default:
		count = -1;
		break;
	}

	LOG_EXIT_INT(count);
	return count;
}


// Get the list of volumes on the system that we can use as spares
static int get_object_list( value_list_t **value_list,
			    list_anchor_t selected_objects,
			    u_int64_t min_size)
{
	int rc = 0;
	storage_object_t * object;
	list_anchor_t tmp_list;
	int count, i;
	list_element_t li;

	LOG_ENTRY();

	rc = EngFncs->get_object_list(DISK | SEGMENT | REGION, // all types
				      DATA_TYPE,
				      NULL,
					  NULL,
				      VALID_INPUT_OBJECT,
				      &tmp_list);

	if (rc) {
		LOG_ERROR("Could not get available objects.\n");
		LOG_EXIT_INT(rc);
		return rc;
	}

	// loop through selected object, removing objects from tmp_list
	LIST_FOR_EACH(selected_objects, li, object) {
		LOG_DETAILS("Object %s selected, removing from spare list\n",object->name);
		EngFncs->remove_thing(tmp_list, object);
	}

	if (*value_list) {
		for (i = 0; i < (*value_list)->count; i++) {
			if ((*value_list)->value[i].s) {
				EngFncs->engine_free((*value_list)->value[i].s);
			}
		}
		EngFncs->engine_free(*value_list);
	}

	count = EngFncs->list_count(tmp_list);
	count++; // increment count to holed the 'None' selection
	
	*value_list = EngFncs->engine_alloc(count * sizeof(value_t) + sizeof(value_list_t));

	if (*value_list == NULL) {
		LOG_ERROR("No memory\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	(*value_list)->count = count;
	i = 0;
	SET_STRING((*value_list)->value[i].s,MD_NO_SELECTION);
	i++;
	LIST_FOR_EACH(tmp_list, li, object) {
		if (object->size >= min_size) {
			(*value_list)->value[i].s = EngFncs->engine_alloc(strlen(object->name) + 1);
			strcpy((*value_list)->value[i].s, object->name);
			i++;
		} else {
			(*value_list)->count--;
		}
	}
	EngFncs->destroy_list(tmp_list);

	LOG_EXIT_INT(0);
	return 0;
}


static int get_spare_disks(md_volume_t * volume, list_anchor_t spare_disks) {

	int rc = 0;
	int i, found;
	list_element_t li=NULL;

	LOG_ENTRY();

	/* Clear out the spare_disks list. */
	EngFncs->delete_all_elements(spare_disks);

	for (i=0, found=0;
	     (rc == 0) && (found < volume->nr_disks) && (i < MAX_MD_DEVICES); i++ ) {
		/* Check for null object, if missing, skip. */
		if (volume->child_object[i]) {
			found++;
			if (disk_spare(&volume->super_block->disks[i])) {

				li = EngFncs->insert_thing(spare_disks,
							   volume->child_object[i],
							   INSERT_AFTER,
							   NULL);
				if (!li) {
					LOG_ERROR("Could not insert object to spare disk list.\n");
					rc = ENOMEM;
				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int get_faulty_disks(md_volume_t * volume, list_anchor_t faulty_disks) {

	int rc = 0;
	int i, found;
	storage_object_t *obj;
	list_element_t li=NULL;

	LOG_ENTRY();

	EngFncs->delete_all_elements(faulty_disks);

	for (i=0, found=0;
	     (rc == 0) && (found < volume->nr_disks) && (i < MAX_MD_DEVICES); i++ ) {

		obj = volume->child_object[i];
		
		if ( obj ) {
			found++;
			if (disk_faulty(&volume->super_block->disks[i]) ) {

				li = EngFncs->insert_thing(faulty_disks, obj, INSERT_AFTER, NULL);
				if (!li) {
					LOG_ERROR("Could not insert object into faulty disk list.\n");
					rc = ENOMEM;
				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int get_stale_disks(md_volume_t * volume, list_anchor_t stale_disks) {

	int rc = 0;
	int i, found;
	list_element_t li=NULL;

	LOG_ENTRY();

	EngFncs->delete_all_elements(stale_disks);

	for (i=0, found=0;
	     (rc == 0) && (found < volume->stale_disks) && (i < MAX_MD_DEVICES); i++ ) {

		if (volume->stale_object[i])
		{
			found++;
			li = EngFncs->insert_thing(stale_disks,
						   volume->stale_object[i],
						   INSERT_AFTER,
						   NULL);
			if (!li) {
				LOG_ERROR("Could not insert object into stale disk list.\n");
				rc = ENOMEM;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int get_active_disks(md_volume_t * volume, list_anchor_t active_disks) {

	int rc = 0;
	int i, found;
	list_element_t li = NULL;

	LOG_ENTRY();

	/* Clear out the spare_disks list. */
	EngFncs->delete_all_elements(active_disks);

	/*
	 * Region must have more than one active disk in order for one to be
	 * removed.
	 */
	if (volume->super_block->active_disks > 1) {
		for (i=0, found=0;
		     (rc == 0) && (found < volume->nr_disks) && (i < MAX_MD_DEVICES); i++ ) {
			/* Check for null object, if missing, skip. */
			if (volume->child_object[i]) {
				found++;
				if (volume->super_block->disks[i].state & (1 << MD_DISK_ACTIVE)) {

					li = EngFncs->insert_thing(active_disks,
								   volume->child_object[i],
								   INSERT_AFTER,
								   NULL);
					if (!li) {
						LOG_ERROR("Could not insert object into active disk list.\n");
						rc = ENOMEM;
					}
				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


typedef struct prune_small_object_parms_s {
	sector_count_t min_size;
	sector_count_t chunk_size;
} prune_small_object_parms_t;

static void prune_small_objects(list_anchor_t objects,
				prune_small_object_parms_t * parms)
{
	list_element_t iter1, iter2;
	storage_object_t * obj;

	LOG_ENTRY();

	LIST_FOR_EACH_SAFE(objects, iter1, iter2, obj) {
		if (MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(parms->chunk_size, obj->size) < parms->min_size) {
			EngFncs->remove_element(iter1);
		}
	}

	LOG_EXIT_VOID();
}


/* Function: raid1_init_task
 *
 *  Determine the type of Task that is being performed, and set up the
 *  context structure with the appropriate initial values.
 */
static int raid1_init_task( task_context_t * context )
{
	int rc = 0;
	list_anchor_t tmp_list;
	md_volume_t * volume;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	switch (context->action) {
	
	case EVMS_Task_Create:

		context->option_descriptors->count = 1;

		// option 0 is the spare disk
		context->option_descriptors->option[0].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
		// get the list of disks that can be spares.
		get_object_list((value_list_t **)&context->option_descriptors->option[0].constraint.list,context->selected_objects,0);
		context->option_descriptors->option[0].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[0].help = NULL;
		SET_STRING(context->option_descriptors->option[0].name, MD_OPTION_SPARE_DISK_NAME );
		context->option_descriptors->option[0].min_len = 1;
		context->option_descriptors->option[0].max_len = EVMS_VOLUME_NAME_SIZE;
		SET_STRING(context->option_descriptors->option[0].tip, "Object to use as a spare disk in the array" );
		SET_STRING(context->option_descriptors->option[0].title, "Spare Disk" );
		context->option_descriptors->option[0].type = EVMS_Type_String;
		context->option_descriptors->option[0].unit = EVMS_Unit_None;
		context->option_descriptors->option[0].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE+1);
		strcpy(context->option_descriptors->option[0].value.s, MD_NO_SELECTION);

		// get a list of all valid input  disks, segments, and regions.
		EngFncs->get_object_list(DISK | SEGMENT | REGION,
					 DATA_TYPE,
					 NULL,
					 NULL,
					 VALID_INPUT_OBJECT,
					 &tmp_list);

		// move these items to the acceptable objects list.
		md_transfer_list(tmp_list, context->acceptable_objects);
		EngFncs->destroy_list(tmp_list);

		context->min_selected_objects = 1;
		context->max_selected_objects = 27;
		break;

	case EVMS_Task_Expand:

		volume = (md_volume_t *) context->object->private_data;
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		rc = raid1_init_expand_option_descriptors(context);
		EngFncs->delete_all_elements(context->acceptable_objects);
		break;

	case EVMS_Task_Shrink:

		volume = (md_volume_t *) context->object->private_data;
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		rc = raid1_init_shrink_option_descriptors(context);
		EngFncs->delete_all_elements(context->acceptable_objects);
		break;


	case MD_RAID1_FUNCTION_FIX:
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		context->option_descriptors->count = 0;
		break;

	case MD_RAID1_FUNCTION_ADD_ACTIVE:
	case MD_RAID1_FUNCTION_ADD_SPARE:
		volume = (md_volume_t *) context->object->private_data;
		{
			list_anchor_t acceptable_objects;

			context->min_selected_objects = 1;
			if ((context->action == MD_RAID1_FUNCTION_ADD_SPARE) &&
			    (volume->flags & MD_DEGRADED)) {
				context->max_selected_objects =
					volume->super_block->raid_disks - volume->active_disks;

			} else {
				context->max_selected_objects =
					MAX_MD_DEVICES - volume->super_block->nr_disks;
			}
			context->option_descriptors->count = 0;

			rc = EngFncs->get_object_list(DISK | SEGMENT | REGION,
						      DATA_TYPE,
						      NULL,
							  NULL,
						      VALID_INPUT_OBJECT,
						      &acceptable_objects);

			if (rc == 0) {
				prune_small_object_parms_t parms;

				/*
				 * If this MD region is available, it will
				 * appear in the list.  Bad things happen if
				 * this region is made a child of itself.
				 * Remove this MD region if it is in the list.
				 */
				EngFncs->remove_thing(acceptable_objects, context->object);

				parms.min_size = volume->super_block->size * (BLOCK_SIZE / EVMS_VSECTOR_SIZE);
				parms.chunk_size = 0;

				prune_small_objects(acceptable_objects, &parms);

				if (context->acceptable_objects != NULL) {
					EngFncs->destroy_list(context->acceptable_objects);
				}
				context->acceptable_objects = acceptable_objects;
			}
		}
		break;

	case MD_RAID1_FUNCTION_ACTIVATE_SPARE:
	case MD_RAID1_FUNCTION_REMOVE_SPARE:
		volume = (md_volume_t *) context->object->private_data;

			context->min_selected_objects = 1;
			context->max_selected_objects = -1;
			context->option_descriptors->count = 0;

			rc = get_spare_disks(volume, context->acceptable_objects);
		break;

	case MD_RAID1_FUNCTION_REMOVE_ACTIVE:
	case MD_RAID1_FUNCTION_MARK_FAULTY:
		volume = (md_volume_t *) context->object->private_data;
			context->min_selected_objects = 1;
			context->max_selected_objects = 1;
			context->option_descriptors->count = 0;

			rc = get_active_disks(volume, context->acceptable_objects);
		break;

	case MD_RAID1_FUNCTION_REMOVE_FAULTY:
		volume = (md_volume_t *) context->object->private_data;

			context->min_selected_objects = 1;
			context->max_selected_objects = -1;	
			context->option_descriptors->count = 0;

			rc = get_faulty_disks(volume, context->acceptable_objects);
		break;
	
	case MD_RAID1_FUNCTION_REMOVE_STALE:
		volume = (md_volume_t *) context->object->private_data;

			context->min_selected_objects = 1;
			context->max_selected_objects = -1;	
			context->option_descriptors->count = 0;

			rc = get_stale_disks(volume, context->acceptable_objects);
		break;
	
	case MD_RAID1_RESTORE_SUPERBLOCK:
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		context->option_descriptors->count = 0;
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


#define PERCENT_WARN_THRESHOLD  5

static void issue_warning_big_obj(storage_object_t *obj, u_int64_t diffsize)
{
	MESSAGE("The %s object is %"PRIu64" MB larger than the smallest object in the RAID array.  "
		"The extra space will not be used.\n",
		obj->name, diffsize * EVMS_VSECTOR_SIZE / (1024*1024) );
}

static void warn_if_big_objects( task_context_t * context )
{
	storage_object_t * obj;
	storage_object_t * spare = NULL;
	u_int64_t smallest_size = 0;
	list_element_t li;

	LOG_ENTRY();

	if ( EngFncs->list_empty(context->selected_objects) ) {
		LOG_WARNING("Selected objects list is empty!!!.\n");
		LOG_EXIT_VOID();
	}
	obj = EngFncs->first_thing(context->selected_objects, NULL);
	if (obj) {
		smallest_size = MD_NEW_SIZE_SECTORS(obj->size);
	} else {
		LOG_ERROR("Can not get the first item in selected objects list.\n");
		LOG_EXIT_VOID();
	}

	/* Find the smallest object. */
	LIST_FOR_EACH(context->selected_objects, li, obj) {
		smallest_size = min(smallest_size, MD_NEW_SIZE_SECTORS(obj->size));
	}

	/*
	 * If we got a smallest size, then check the size of the spare, if one
	 * is specified and see if it is the smallest.
	 */
	if (smallest_size != 0) {
		if (context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].value.s != NULL) {
			spare = find_object(context->option_descriptors->option[MD_OPTION_SPARE_DISK_INDEX].value.s);

			if (spare != NULL) {
				smallest_size = min(smallest_size, MD_NEW_SIZE_SECTORS(spare->size));
			}
		}
	}

	/*
	 * Now go through the objects again and issue a warning message for
	 * any object whose size exceeds the threshold over the smallest
	 * object size.
	 */
	if (smallest_size != 0) {
		u_int64_t diffsize;

		LIST_FOR_EACH(context->selected_objects, li, obj) {
			diffsize = MD_NEW_SIZE_SECTORS(obj->size) - smallest_size;

			if (diffsize > (smallest_size * PERCENT_WARN_THRESHOLD) / 100) {
				issue_warning_big_obj(obj, diffsize);
			}
		}

		/*
		 * If we have a spare, check its size too.
		 */
		if (spare != NULL) {
			diffsize = MD_NEW_SIZE_SECTORS(spare->size) - smallest_size;

			if (diffsize > (smallest_size * PERCENT_WARN_THRESHOLD) / 100) {
				issue_warning_big_obj(obj, diffsize);
			}
		}
	}

	LOG_EXIT_VOID();
}

/* Function: raid1_set_option
 *
 *  Determine the type of Task that is being performed. Then examine the
 *  desired option (using the index), and verify that the given value is
 *  appropriate. Reset the value if necessary and possible. Adjust other
 *  options as appropriate.
 */
static int raid1_set_option( task_context_t * context,
			     u_int32_t        index,
			     value_t        * value,
			     task_effect_t  * effect )
{
	int rc = 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!context || !value || !effect) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	

	switch (context->action) {
	
	case EVMS_Task_Create:
		switch (index) {
		
		case MD_OPTION_SPARE_DISK_INDEX:
			// Not worth validation, will catch when we try to find the original
			strcpy(context->option_descriptors->option[index].value.s, value->s);
			warn_if_big_objects(context);
			break;

		default:
			break;

		}
		break;
	case EVMS_Task_Expand:
		rc = raid1_set_expand_option(context, index, value, effect);
		break;
	case EVMS_Task_Shrink:
		rc = raid1_set_shrink_option(context, index, value, effect);
		break;

	default:
		break;
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_set_objects
 *
 *  Determine the type of task, and then validate that the objects on the
 *  "selected" list are valid for that task. If so, adjust the option
 *  descriptor as appropriate.
 */
static int raid1_set_objects( task_context_t * context,
			      list_anchor_t declined_objects,
			      task_effect_t * effect )
{
	int rc = 0;
	uint count = 0;
	md_volume_t * volume = NULL;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!context || !declined_objects || !effect) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	count = EngFncs->list_count(context->selected_objects);
	if (context->object != NULL) {
		volume = (md_volume_t *) context->object->private_data;
	}

	switch (context->action) {
	
	case EVMS_Task_Create:
		get_object_list((value_list_t **)&context->option_descriptors->option[0].constraint.list,context->selected_objects,0);
		warn_if_big_objects(context);
		*effect |= EVMS_Effect_Reload_Options;
		break;
	
	case EVMS_Task_Expand:
		/* Make sure too many objects are not selected. */
		LOG_DEBUG(" Task_Expand: selected objects count = %u\n", count);
		if (count > (volume->super_block->nr_disks)) {
			LOG_ERROR("Can only specify up to %d object(s) to expand\n", volume->super_block->nr_disks);
			rc = EINVAL;
		}
		break;

	case MD_RAID1_FUNCTION_FIX:
		break;

	// The Engine makes sure that only available objects appear in the
	// selected_objects list.  Verify that only one object is selected.
	case MD_RAID1_FUNCTION_ADD_SPARE:
	case MD_RAID1_FUNCTION_ADD_ACTIVE:
		/* Make sure too many objects are not selected. */
		if (count > (MAX_MD_DEVICES - volume->super_block->nr_disks)) {
			LOG_ERROR("Can only specify up to %d object(s) to added.\n", MAX_MD_DEVICES - volume->super_block->nr_disks);
			rc = EINVAL;
		}
		break;

	case MD_RAID1_FUNCTION_ACTIVATE_SPARE:
	case MD_RAID1_FUNCTION_REMOVE_SPARE:
		if (count > volume->super_block->spare_disks) {
			LOG_ERROR("Can only specify at most %d spare object(s).\n", volume->super_block->spare_disks);
			rc = EINVAL;
		}
		break;

	case MD_RAID1_FUNCTION_DEACTIVATE_ACTIVE:
	case MD_RAID1_FUNCTION_REMOVE_ACTIVE:
		if (count > volume->super_block->active_disks) {
			LOG_ERROR("Can only specify at most %d active object(s).\n", volume->super_block->active_disks);
			rc = EINVAL;
		}
		break;

	case MD_RAID1_FUNCTION_REMOVE_FAULTY:
		if (count > volume->super_block->failed_disks) {
			LOG_ERROR("Can only specify at most %d faulty object(s).\n",
				  volume->super_block->failed_disks);
			rc = EINVAL;
		}
		break;
	
	case MD_RAID1_FUNCTION_MARK_FAULTY:
		if (count > 1) {
			LOG_ERROR("Can mark only 1 faulty object.\n");
			rc = EINVAL;
		}
		break;
	
	case MD_RAID1_FUNCTION_REMOVE_STALE:
		if (count > volume->stale_disks) {
			LOG_ERROR("Can only specify at most %d stale object(s).\n", volume->stale_disks);
			rc = EINVAL;
		}
		break;

	default:
		break;
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_get_info
 *
 *  Return MD-specific information about the specified region. If the
 *  name field is set, only return the "extra" information pertaining
 *  to that name.
 */
static int raid1_get_info( storage_object_t       * region,
			   char                   * name,
			   extended_info_array_t ** info_array )
{

	md_volume_t * volume = NULL;
	int           rc= 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	// Parameter check
	if (!region || !info_array) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	// Make sure this is an MD region
	if (region->plugin != raid1_plugin) {
		LOG_ERROR("Region %s is not owned by MD RAID1\n", region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	volume = region->private_data;

	rc = md_get_info(volume, name, info_array);

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_get_plugin_info
 *
 *  Return information about the MD plugin. There is no "extra"
 *  information about MD, so "name" should always be NULL.
 */
static int raid1_get_plugin_info( char                     * name,
				  extended_info_array_t   ** info_array )
{

	extended_info_array_t   * info = NULL;
	char buffer[50] = {0};
	int i = 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	// Parameter check
	if (! info_array) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (! name) {
		// Get memory for the info array
		if (! (info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*5))) {
			LOG_ERROR("Error allocating memory for info array\n");
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}

		// Short Name
		SET_STRING(info->info[i].name, "ShortName");
		SET_STRING(info->info[i].title, "Short Name");
		SET_STRING(info->info[i].desc, "A short name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, raid1_plugin->short_name);
		i++;

		// Long Name
		SET_STRING(info->info[i].name, "LongName");
		SET_STRING(info->info[i].title, "Long Name");
		SET_STRING(info->info[i].desc, "A long name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, raid1_plugin->long_name);
		i++;

		// Plugin Type
		SET_STRING(info->info[i].name, "Type");
		SET_STRING(info->info[i].title, "Plugin Type");
		SET_STRING(info->info[i].desc, "There are various types of plugins; each responsible for some kind of storage object.");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, "Region Manager");
		i++;

		// Plugin Version
		SET_STRING(info->info[i].name, "Version");
		SET_STRING(info->info[i].title, "Plugin Version");
		SET_STRING(info->info[i].desc, "This is the version number of the plugin.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		// Required Engine Services Version
		SET_STRING(info->info[i].name, "Required_Engine_Version");
		SET_STRING(info->info[i].title, "Required Engine Services Version");
		SET_STRING(info->info[i].desc, "This is the version of the Engine services that this plug-in requires.  It will not run on older versions of the Engine services.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", raid1_plugin->required_engine_api_version.major, raid1_plugin->required_engine_api_version.minor, raid1_plugin->required_engine_api_version.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		// Required Plug-in API Version
		SET_STRING(info->info[i].name, "Required_Plugin_Version");
		SET_STRING(info->info[i].title, "Required Plug-in API Version");
		SET_STRING(info->info[i].desc, "This is the version of the Engine plug-in API that this plug-in requires.  It will not run on older versions of the Engine plug-in API.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", raid1_plugin->required_plugin_api_version.plugin.major, raid1_plugin->required_plugin_api_version.plugin.minor, raid1_plugin->required_plugin_api_version.plugin.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;
	} else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	info->count = i;
	*info_array = info;
	LOG_EXIT_INT(0);
	return 0;
}


/* Function: raid1_read
 *
 *  Perform a logical-to-physical remapping, and send the read down to
 *  the next plugin.
 */
static int raid1_read( storage_object_t * region,
		       lsn_t              lsn,
		       sector_count_t     count,
		       void             * buffer )
{

	md_volume_t *   volume = (md_volume_t *)region->private_data;
	int             rc = 0;
	int i;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!region || !buffer) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (volume->flags & MD_CORRUPT) {
		memset(buffer, 0x0, count * EVMS_VSECTOR_SIZE);
		LOG_ERROR("MD Object %s is corrupt, returning zero filled buffer.\n",volume->name);
		LOG_EXIT_INT(0);
		return 0;
	}

	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to read past end of region %s sector=%"PRIu64"\n",volume->name,lsn+count);
	}

	rc = md_region_rw(region, lsn, count, buffer, 0);
	if (rc) {
		rc = 0;
		for (i = 0; i < MAX_MD_DEVICES; i++ ) {
			// check for null object, if missing, skip
			if (volume->child_object[i]) {
				// only read from Active disks, not the spares.
				if (volume->super_block->disks[i].state & ( 1 << MD_DISK_ACTIVE)) {
					rc = READ(volume->child_object[i], lsn, count, buffer);
					if (rc) {
						//BUGBUG disable mirror
						MESSAGE("Error reading from mirror %s of region %s sector=%"PRIu64", Mirror disabled.\n",
							volume->child_object[i]->name, volume->name,lsn+count);
					}else{
						break;
					}
	
				}
			}
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_write
 *
 *  Perform a logical-to-physical remapping, and send the write down to
 *  the next plugin.
 */
static int raid1_write( storage_object_t * region,
			lsn_t              lsn,
			sector_count_t     count,
			void             * buffer )
{

	md_volume_t *   volume = (md_volume_t *)region->private_data;
	int             rc = 0;
	int i, found=0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!region || !buffer) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (volume->flags & MD_CORRUPT) {
		MESSAGE("MD Object %s is corrupt, writing data is not allowed.\n",volume->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}
	
	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to write past end of region %s sector=%"PRIu64"\n",volume->name,lsn+count);
	}

	rc = md_region_rw(region, lsn, count, buffer, 1);

	if (rc) {
		rc = 0;
		for (i = 0; (i < MAX_MD_DEVICES) && (found < volume->nr_disks); i++ ) {
			// check for null object, if missing, skip
			if (volume->child_object[i]) {
				found++;
				// write to all non-faulty disks.
				if (!(volume->super_block->disks[i].state & (1 << MD_DISK_FAULTY))) {
					rc = WRITE(volume->child_object[i], lsn, count, buffer);
					if (rc) {
						//BUGBUG disable mirror
					}
				}
			}
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}

static int raid1_enable_fix_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;

	LOG_ENTRY();

	raid1_verify_and_fix_array(volume, 0);

	if (volume->flags & (MD_DEGRADED | MD_ARRAY_SYNCING)) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	if (md_can_stop_array(volume->region) == FALSE) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	if (volume->flags & (MD_CORRUPT | MD_PROBLEMATIC_SB)) {

		fia->info[*function_count].function = MD_RAID1_FUNCTION_FIX;
		SET_STRING(fia->info[*function_count].name, "fix");
		SET_STRING(fia->info[*function_count].title, "Fix");
		SET_STRING(fia->info[*function_count].verb, "Fix");
		SET_STRING(fia->info[*function_count].help,
			"The RAID array has inconsistent metadata.  Use this function to fix the metadata.");
		if (volume->flags & MD_ARRAY_SYNCING) {
			fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
		}
		++*function_count;
		enable = 1;
	}

	LOG_EXIT_INT(enable);
	return enable;
}

static int raid1_enable_restore_major_minor_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;

	LOG_ENTRY();
	
	if (volume->flags & (MD_CORRUPT | MD_PROBLEMATIC_SB)) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	if (volume->region_mgr_flags & MD_RAID1_CONFIG_CHANGE_PENDING) {
		LOG_EXIT_INT(enable);
		return enable;
	}


	if (md_can_restore_saved_sb(volume->region)) {
		fia->info[*function_count].function = MD_RAID1_RESTORE_SUPERBLOCK;
		SET_STRING(fia->info[*function_count].name, "ressuperblock");
		SET_STRING(fia->info[*function_count].title, "Restore orginal major/minor");
		SET_STRING(fia->info[*function_count].verb, "Restore");
		SET_STRING(fia->info[*function_count].help,
			"Use this function to restore the original major and minor of all devices made up the MD array.");
		if (volume->flags & MD_ARRAY_SYNCING) {
			fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
		}
		++*function_count;
		enable = 1;
	}

	LOG_EXIT_INT(enable);
	return enable;
}

static int raid1_enable_add_disk_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;
	int rc;
	list_anchor_t available_objects = NULL;
	uint count;
	prune_small_object_parms_t parms;
	mdp_super_t *sb = volume->super_block;

	LOG_ENTRY();
	
	if (volume->flags & MD_CORRUPT) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	rc = EngFncs->get_object_list(DISK | SEGMENT| REGION, DATA_TYPE, NULL, NULL, VALID_INPUT_OBJECT, &available_objects);

	if (rc == 0) {

		/*
		 * If this MD region is available, it
		 * will appear in the list.  Bad things
		 * happen if this region is made a
		 * child of itself.  Remove this MD
		 * region if it is in the list.
		 */
		EngFncs->remove_thing(available_objects, volume->region);

		parms.min_size = sb->size * (BLOCK_SIZE / EVMS_VSECTOR_SIZE);
		parms.chunk_size = 0;

		prune_small_objects(available_objects, &parms);

		count = EngFncs->list_count(available_objects);

		if (count > 0) {

			/* Add spare disk */
			fia->info[*function_count].function = MD_RAID1_FUNCTION_ADD_SPARE;
			SET_STRING(fia->info[*function_count].name, "addspare");
			SET_STRING(fia->info[*function_count].verb, "Add");
			if (volume->flags & MD_DEGRADED && (volume->nr_disks < volume->super_block->raid_disks)) {
				SET_STRING(fia->info[*function_count].title, "Add spare to fix degraded array");
				SET_STRING(fia->info[*function_count].help,
					"Use this function to add a spare object to replace a missing or faulty entry of this degraded region.");
			} else {
				SET_STRING(fia->info[*function_count].title, "Add spare object");
				SET_STRING(fia->info[*function_count].help,
					"Use this function to add an object as a spare object for this RAID array.");
			}

			if ((volume->flags & MD_ARRAY_SYNCING) ||
			    (volume->region_mgr_flags & MD_RAID1_CONFIG_CHANGE_PENDING)) {
				/* temporarily disable this function */
				LOG_DEBUG("%s in array %s temporarily disable Add Spare function\n",
					  (volume->flags & MD_ARRAY_SYNCING) ? "syncing" : "configuration pending",
					volume->name);
				fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
			}

			++*function_count;
			enable = 1;

			/* Add active disk option is available only if
			 * - there is no spare
			 * - we can reconfig the array
			 */
			if (!volume->super_block->spare_disks &&
			    raid1_can_change_region_configuration(volume->region) ) {
				fia->info[*function_count].function = MD_RAID1_FUNCTION_ADD_ACTIVE;
				SET_STRING(fia->info[*function_count].name, "addactive");
				SET_STRING(fia->info[*function_count].title, "Add active object");
				SET_STRING(fia->info[*function_count].verb, "Add");
				SET_STRING(fia->info[*function_count].help,
					"Use this function to increase the number of mirrors for this RAID array.");
				if (volume->flags & MD_ARRAY_SYNCING) {
					fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
				}
				++*function_count;
			}
		}
		EngFncs->destroy_list(available_objects);
	}

	LOG_EXIT_INT(enable);
	return enable;
}

static int raid1_enable_remove_spare_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;
	int i;

	LOG_ENTRY();

	if (volume->flags & MD_CORRUPT) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	/*
	 * If the RAID array has a spare disk then
	 * Remove Spare is available.
	 */
	for (i = 0; i < MAX_MD_DEVICES; i++) {
		if ((volume->child_object[i] != NULL) &&
		    (!disk_new(&volume->super_block->disks[i])) &&
		    (disk_spare(&volume->super_block->disks[i])) ) {
			fia->info[*function_count].function = MD_RAID1_FUNCTION_REMOVE_SPARE;
			SET_STRING(fia->info[*function_count].name, "remspare");
			SET_STRING(fia->info[*function_count].title, "Remove spare object");
			SET_STRING(fia->info[*function_count].verb, "Remove");
			SET_STRING(fia->info[*function_count].help,
				"Use this function to remove a spare object from this RAID array.");
			if ((volume->flags & MD_ARRAY_SYNCING) ||
			    (volume->region_mgr_flags & MD_RAID1_CONFIG_CHANGE_PENDING)) {
				fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
			}

			++*function_count;
			enable = 1;
			break;
		}
	}

	LOG_EXIT_INT(enable);
	return enable;
}


static int raid1_enable_activate_spare_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;
	int i;

	LOG_ENTRY();
	
	/*
	 * Activate Spare will reconfigure the array.
	 * Therefore, make sure that we can change the array configuration.
	 */
	if (!raid1_can_change_region_configuration(volume->region)) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	/*
	 * If the RAID array has a spare disk then
	 * Activate Spare is available.
	 */
	for (i = 0; i < MAX_MD_DEVICES; i++) {
		if ((volume->child_object[i] != NULL) &&
		    (!disk_new(&volume->super_block->disks[i])) &&
		    (disk_spare(&volume->super_block->disks[i])) ) {

			fia->info[*function_count].function = MD_RAID1_FUNCTION_ACTIVATE_SPARE;
			SET_STRING(fia->info[*function_count].name, "activatespare");
			SET_STRING(fia->info[*function_count].title, "Activate spare object");
			SET_STRING(fia->info[*function_count].verb, "Activate");
			SET_STRING(fia->info[*function_count].help,
					   "Use this function to activate a spare object to reconfigure this RAID1 region.  "
					   "For example, the current RAID1 region is a 2-way mirror with a spare.  "
					   "You may reconfigure it to become a 3-way mirror with no spare.\n");
			if (volume->flags & MD_ARRAY_SYNCING) {
				fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
			}
			++*function_count;
			enable = 1;
			break;
		}
	}

	LOG_EXIT_INT(enable);
	return enable;
}

static int raid1_enable_remove_active_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;

	LOG_ENTRY();

	/*
	 * Remove active will reconfigure the array.
	 * Therefore, make sure that we can change the array configuration.
	 */
	if (!raid1_can_change_region_configuration(volume->region)) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	/*
	 * If the array has at least two active members
	 * then Remove Active is available.
	 */
	if (volume->super_block->active_disks > 1) {
		fia->info[*function_count].function = MD_RAID1_FUNCTION_REMOVE_ACTIVE;
		SET_STRING(fia->info[*function_count].name, "remactive");
		SET_STRING(fia->info[*function_count].title, "Remove active object");
		SET_STRING(fia->info[*function_count].verb, "Remove");
		SET_STRING(fia->info[*function_count].help,
			"Use this function to remove an active object from this RAID array.");
		if (volume->flags & MD_ARRAY_SYNCING) {
			fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
		}
		++*function_count;
		enable = 1;
	}

	LOG_EXIT_INT(enable);
	return enable;
}


static int raid1_enable_remove_faulty_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;
	int i;
	storage_object_t *obj;

	LOG_ENTRY();

	/*
	 * If the RAID array has a faulty disk then
	 * Remove Faulty is available.
	 */
	for (i = 0; i < MAX_MD_DEVICES; i++) {

		obj = volume->child_object[i];

		if ( obj && disk_faulty(&volume->super_block->disks[i]) ) {

			fia->info[*function_count].function = MD_RAID1_FUNCTION_REMOVE_FAULTY;
			SET_STRING(fia->info[*function_count].name, "remfaulty");
			SET_STRING(fia->info[*function_count].title, "Remove a faulty object");
			SET_STRING(fia->info[*function_count].verb, "Remove");
			SET_STRING(fia->info[*function_count].help,
				"Use this function to permanently remove a faulty object from this RAID array.");
			if (volume->flags & MD_ARRAY_SYNCING) {
				fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
			}
			++*function_count;
			enable = 1;
			break;
		}
	}
	LOG_EXIT_INT(enable);
	return enable;
}

static int raid1_enable_remove_stale_disk_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;
	int i;

	LOG_ENTRY();

	/*
	 * If the RAID array has a stale disk then
	 * Remove Stale is available.
	 */
	if (!volume->stale_disks) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	for (i = 0; i < MAX_MD_DEVICES; i++) {

		if ( volume->stale_object[i] )
		{

			fia->info[*function_count].function = MD_RAID1_FUNCTION_REMOVE_STALE;
			SET_STRING(fia->info[*function_count].name, "remstale");
			SET_STRING(fia->info[*function_count].title, "Remove a stale object");
			SET_STRING(fia->info[*function_count].verb, "Remove");
			SET_STRING(fia->info[*function_count].help,
				"Use this function to permanently remove a stale (possibly faulty) object from this RAID array.");
			if (volume->flags & MD_ARRAY_SYNCING) {
				fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
			}
			++*function_count;
			enable = 1;
			break;
		}
	}
	LOG_EXIT_INT(enable);
	return enable;
}

static int raid1_enable_mark_disk_faulty_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;
	int active_disks;

	LOG_ENTRY();

	if ((volume->region_mgr_flags & MD_RAID1_REGION_NEW) ||
	    !md_is_region_active(volume->region) ) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	if (volume->flags & MD_CORRUPT) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	/*
	 * As long as the array is not 1-way mirror, Mark Faulty is available.
	 */

	active_disks = min(volume->active_disks, volume->super_block->active_disks);
	if (active_disks > 1) {

		fia->info[*function_count].function = MD_RAID1_FUNCTION_MARK_FAULTY;
		SET_STRING(fia->info[*function_count].name, "markfaulty");
		SET_STRING(fia->info[*function_count].title, "Mark object faulty");
		SET_STRING(fia->info[*function_count].verb, "Mark faulty");
		SET_STRING(fia->info[*function_count].help, "Use this function to mark an object faulty in this RAID array.  If the RAID array has a spare object, the spare object will be brought on-line to replace the faulty object.");
		if ((volume->flags & MD_ARRAY_SYNCING) ||
		    (volume->region_mgr_flags & MD_RAID1_CONFIG_CHANGE_PENDING)) {
			fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
		}

		++*function_count;
		enable = 1;
	}
	LOG_EXIT_INT(enable);
	return enable;
}

/* Function:  raid1_get_plugin_functions
 */
static int raid1_get_plugin_functions(storage_object_t        * region,
				      function_info_array_t * * functions) {

	function_info_array_t * fia;
	int rc, function_count = 0;
	md_volume_t * volume;

	my_plugin = raid1_plugin;
	LOG_ENTRY();
	
	/*
	 * If region is NULL, that means the user is asking for plug-in
	 * functions on the plug-in.  We don't have any plug-in functions that
	 * are global for the plug-in.
	 */
	if (region == NULL) {
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	volume = (md_volume_t *) region->private_data;

	fia = EngFncs->engine_alloc(sizeof(function_info_array_t) + sizeof(function_info_t) * (MD_RAID1_FUNCTION_COUNT - 1));
	if (!fia) {
		LOG_CRITICAL("Error allocating memory for an action info array.\n");
		LOG_EXIT_INT (ENOMEM);
		return ENOMEM;
	}

	/*
	 * Our functions are only allowed if the Engine is opened for
	 * writing.
	 */
	if (!(EngFncs->get_engine_mode() & ENGINE_WRITE)) {
		fia->count = function_count;
		*functions = fia;
		LOG_EXIT_INT(0);
		return 0;
	}
	
	if (md_is_recovery_running(region)) {
		volume->flags |= MD_ARRAY_SYNCING;
		LOG_DEBUG("%s : Resync/recovery is running\n", region->name);
	} else {
		if (volume->flags & MD_ARRAY_SYNCING) {
			rc = raid1_rediscover_region(region);
		}
		volume->flags &= ~MD_ARRAY_SYNCING;
	}

	/*
	 * If the array needs fixing, then that is the only function available and maybe remove stale disks
	 */
	if ( raid1_enable_fix_function(volume, fia, &function_count) && (volume->flags & MD_CORRUPT)) {

		raid1_enable_remove_stale_disk_function(volume, fia, &function_count);

		fia->count = function_count;
		*functions = fia;
		LOG_EXIT_INT(0);
		return (0);
	}

	raid1_enable_restore_major_minor_function(volume, fia, &function_count);
	raid1_enable_add_disk_function(volume, fia, &function_count);
	raid1_enable_remove_spare_function(volume, fia, &function_count);
	raid1_enable_activate_spare_function(volume, fia, &function_count);
	raid1_enable_remove_active_function(volume, fia, &function_count);
	raid1_enable_remove_faulty_function(volume, fia, &function_count);
	raid1_enable_remove_stale_disk_function(volume, fia, &function_count);
	raid1_enable_mark_disk_faulty_function(volume, fia, &function_count);

	fia->count = function_count;
	*functions = fia;

	LOG_EXIT_INT(0);
	return 0;
}


static int can_be_added(md_volume_t * volume, storage_object_t * spare_candidate)
{
	mdp_super_t *sb = volume->super_block;

	LOG_ENTRY();

	/* The spare must be a disk, segment, or region. */
	if ((spare_candidate->object_type != DISK) &&
	    (spare_candidate->object_type != SEGMENT) &&
	    (spare_candidate->object_type != REGION)) {
		LOG_ERROR("The type of object %s is not data.\n", spare_candidate->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* The spare must not be too small. */
	//if (MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(sb->chunk_size / EVMS_VSECTOR_SIZE, spare_candidate->size) < sb->size * 2) {
	if (MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(0, spare_candidate->size) < sb->size * 2) {
		LOG_ERROR("Object %s is too small to be a spare object for array %s.\n", spare_candidate->name, volume->region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* The region cannot be a spare of itself. */
	if (spare_candidate == volume->region) {
		LOG_ERROR("Region %s cannot be a spare object for itself.\n", spare_candidate->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	LOG_EXIT_INT(0);
	return 0;
}


static int is_spare(md_volume_t * volume, storage_object_t *spare_disk)
{

	int i, j;

	LOG_ENTRY();

	j = 0;
	for (i = 0; (i < MAX_MD_DEVICES) && (j < volume->nr_disks); i++) {
		if (volume->child_object[i] != NULL) {
			if (volume->child_object[i] == spare_disk) {
				if (disk_spare(&volume->super_block->disks[i])) {
					LOG_EXIT_INT(0);
					return 0;
				} else {
					LOG_ERROR("Object %s is in array %s but is not a spare object.\n", spare_disk->name, volume->region->name);
					LOG_EXIT_INT(EINVAL);
					return EINVAL;
				}
			} else {
				j++;
			}
		}
	}

	/* The object was not found in the array. */
	LOG_ERROR("Object %s is not part of array %s.\n", spare_disk->name, volume->region->name);
	LOG_EXIT_INT(EINVAL);
	return EINVAL;
}


static int is_faulty(md_volume_t * volume, storage_object_t *faulty_disk)
{

	int i;
	storage_object_t * obj;

	LOG_ENTRY();

	if (!volume || !faulty_disk) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	for (i = 0; i < MAX_MD_DEVICES; i++) {
		obj = volume->child_object[i];
		if (obj) {
			if (obj == faulty_disk) {
				if (disk_faulty(&volume->super_block->disks[i])) {
					LOG_EXIT_INT(0);
					return 0;
				} else {
					LOG_ERROR("Object %s is in array %s but is not faulty.\n", faulty_disk->name, volume->region->name);
					LOG_EXIT_INT(EINVAL);
					return EINVAL;
				}
			}
		}
	}

	/* The object was not found in the array. */
	LOG_ERROR("Object %s is not part of array %s.\n", faulty_disk->name, volume->region->name);
	LOG_EXIT_INT(EINVAL);
	return EINVAL;
}

static int is_stale(md_volume_t * volume, storage_object_t * stale_disk)
{

	int i;
	storage_object_t * obj;

	LOG_ENTRY();

	if (!volume || !stale_disk) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	for (i = 0; i < MAX_MD_DEVICES; i++) {
		obj = volume->stale_object[i];
		if (obj) {
			if (obj == stale_disk) {
				LOG_EXIT_INT(0);
				return 0;
			}
		}
	}

	/* The object was not found in the array. */
	LOG_ERROR("Object %s is not part of array %s.\n", stale_disk->name, volume->region->name);
	LOG_EXIT_INT(EINVAL);
	return EINVAL;
}

static int is_active(md_volume_t * volume, storage_object_t * active_disk)
{

	int i, j;

	LOG_ENTRY();

	j = 0;
	for (i = 0; (i < MAX_MD_DEVICES) && (j < volume->nr_disks); i++) {
		if (volume->child_object[i] != NULL) {
			if (volume->child_object[i] == active_disk) {
				if (volume->super_block->disks[i].state == ((1 << MD_DISK_ACTIVE) | (1 << MD_DISK_SYNC))) {
					LOG_EXIT_INT(0);
					return 0;
				} else {
					LOG_ERROR("Object %s is in array %s but is not active.\n", active_disk->name, volume->region->name);
					LOG_EXIT_INT(EINVAL);
					return EINVAL;
				}
			} else {
				j++;
			}
		}
	}

	/* The object was not found in the array. */
	LOG_ERROR("Object %s is not part of array %s.\n", active_disk->name, volume->region->name);
	LOG_EXIT_INT(EINVAL);
	return EINVAL;
}


/* Function:  raid1_plugin_function
 */
static int raid1_plugin_function(storage_object_t * region,
				 task_action_t action,
				 list_anchor_t objects,
				 option_array_t * options)
{

	int rc = 0;
	md_volume_t * volume = (md_volume_t *) region->private_data;
	storage_object_t * object;
	list_element_t li;
	uint count;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	if ((action < EVMS_Task_Plugin_Function) ||
	    (action >= EVMS_Task_Plugin_Function + MD_RAID1_FUNCTION_COUNT)) {
		LOG_ERROR("Action code 0x%x is out of range.\n", action);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (action == MD_RAID1_FUNCTION_FIX) {
		raid1_fix_array(volume);

	} else if (action == MD_RAID1_RESTORE_SUPERBLOCK) {
		rc = md_restore_saved_sb(volume);
	} else {

		count = EngFncs->list_count(objects);

		switch (action) {
		case MD_RAID1_FUNCTION_ADD_SPARE:
		case MD_RAID1_FUNCTION_ADD_ACTIVE:
			if (count == 0) {
				LOG_ERROR("Must specify at least one object to be added.\n");
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
			if (count > (MAX_MD_DEVICES - volume->super_block->nr_disks)) {
				LOG_ERROR("Can only specify up to %d object(s) to added.\n", MAX_MD_DEVICES - volume->super_block->nr_disks);
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}

			LIST_FOR_EACH(objects, li, object) {
				rc |= can_be_added(volume, object);
			}
			if (rc != 0) {
				LOG_EXIT_INT(rc);
				return rc;
			}
			break;

		case MD_RAID1_FUNCTION_ACTIVATE_SPARE:
		case MD_RAID1_FUNCTION_REMOVE_SPARE:
			if (count == 0) {
				LOG_ERROR("Must specify at least one spare object.\n");
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
			LIST_FOR_EACH(objects, li, object) {
				rc |= is_spare(volume, object);
			}
			if (rc != 0) {
				LOG_EXIT_INT(rc);
				return rc;
			}

			break;

		case MD_RAID1_FUNCTION_DEACTIVATE_ACTIVE:
		case MD_RAID1_FUNCTION_REMOVE_ACTIVE:
		case MD_RAID1_FUNCTION_MARK_FAULTY:
			if (count == 0) {
				LOG_ERROR("Must specify at least one active object.\n");
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
			LIST_FOR_EACH(objects, li, object) {
				rc |= is_active(volume, object);
			}
			if (rc != 0) {
				LOG_EXIT_INT(rc);
				return rc;
			}
			break;

		case MD_RAID1_FUNCTION_REMOVE_FAULTY:
			if (count == 0) {
				LOG_ERROR("Must specify at least one faulty object.\n");
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
			LIST_FOR_EACH(objects, li, object) {
				rc |= is_faulty(volume, object);
			}
			if (rc != 0) {
				LOG_EXIT_INT(rc);
				return rc;
			}
			break;
		
		case MD_RAID1_FUNCTION_REMOVE_STALE:
			if (count == 0) {
				LOG_ERROR("Must specify at least one stale object.\n");
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
			LIST_FOR_EACH(objects, li, object) {
				rc |= is_stale(volume, object);
			}
			if (rc != 0) {
				LOG_EXIT_INT(rc);
				return rc;
			}
			break;

		default:
			LOG_ERROR("0x%x is not a valid action code.\n", action);
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}

		LIST_FOR_EACH(objects, li, object) {
			switch (action) {
			case MD_RAID1_FUNCTION_ADD_SPARE:
				rc = add_spare_disk(volume, object);
				break;
			case MD_RAID1_FUNCTION_ADD_ACTIVE:
				rc = add_active_disk(volume, object);
				break;
			case MD_RAID1_FUNCTION_ACTIVATE_SPARE:
				rc = activate_spare_disk(volume, object);
				break;
			case MD_RAID1_FUNCTION_REMOVE_SPARE:
				rc = remove_spare_disk(volume, object);
				break;
			case MD_RAID1_FUNCTION_DEACTIVATE_ACTIVE:
				rc = deactivate_active_disk(volume, object);
				break;
			case MD_RAID1_FUNCTION_REMOVE_ACTIVE:
				rc = remove_active_disk(volume, object);
				break;
			case MD_RAID1_FUNCTION_REMOVE_FAULTY:
				rc = remove_faulty_disk(volume, object);
				break;
			case MD_RAID1_FUNCTION_MARK_FAULTY:
				rc = raid1_mark_disk_faulty(volume, object);
				if (!rc) {
					if (follow_up_mark_faulty(volume, object) == TRUE) {
						rc = remove_faulty_disk(volume, object);
					}
				}

				break;
			case MD_RAID1_FUNCTION_REMOVE_STALE:
				rc = remove_stale_disk(volume, object);
				break;
			
			default:
				/*
				 * Shouldn't get here if the validiation
				 * code above did its job.
				 */
				LOG_WARNING("Action code 0x%x slipped past validation.\n", action);
				rc = EINVAL;
				break;
			}
			
			if (rc) {
				break;
			}
		}
	}


	if (rc == 0) {
		volume->region->flags |= SOFLAG_DIRTY;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static void free_region(storage_object_t *region)
{
	md_volume_t * volume = (md_volume_t *)region->private_data;

	md_free_volume(volume);

	LOG_EXIT_VOID();
}



static void raid1_plugin_cleanup(void) {

	list_anchor_t regions_list;
	list_element_t li;
	storage_object_t *region;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	EngFncs->get_object_list(REGION, DATA_TYPE, raid1_plugin, NULL, 0, &regions_list);

	LIST_FOR_EACH(regions_list, li, region) {
		free_region(region);
	}

	EngFncs->destroy_list(regions_list);

	LOG_EXIT_VOID();
}


/* Function tables for the MD Region Manager */
static plugin_functions_t raid1_functions = {
	setup_evms_plugin       : raid1_setup_evms_plugin,
	cleanup_evms_plugin     : raid1_plugin_cleanup,
	can_delete              : raid1_can_delete,
	can_expand              : raid1_can_expand,
	can_expand_by           : raid1_can_expand_by,
	can_shrink              : raid1_can_shrink,
	can_shrink_by           : raid1_can_shrink_by,
	can_replace_child	: raid1_can_replace_child,
	discover                : raid1_discover,
	create                  : raid1_create,
	delete                  : raid1_delete,
	discard                 : raid1_discard,
	expand                  : raid1_expand,
	shrink                  : raid1_shrink,
	replace_child		: raid1_replace_child,
	add_sectors_to_kill_list: raid1_add_sectors_to_kill_list,
	commit_changes          : raid1_commit_changes,
	activate		: raid1_activate_region,
	deactivate		: raid1_deactivate_region,
	get_option_count        : raid1_get_option_count,
	init_task               : raid1_init_task,
	set_option              : raid1_set_option,
	set_objects             : raid1_set_objects,
	get_info                : raid1_get_info,
	get_plugin_info         : raid1_get_plugin_info,
	read                    : raid1_read,
	write                   : raid1_write,
	get_plugin_functions    : raid1_get_plugin_functions,
	plugin_function         : raid1_plugin_function,
};



/* Function: PluginInit
 *
 *  Initializes the local plugin record
 */

plugin_record_t raid1_plugin_record = {
	id:                     SetPluginID(EVMS_OEM_IBM, EVMS_REGION_MANAGER, 5),

	version:               {major:      MAJOR_VERSION,
				minor:      MINOR_VERSION,
				patchlevel: PATCH_LEVEL},

	required_engine_api_version: {major:      13,
				      minor:      0,
				      patchlevel: 0},
	required_plugin_api_version: {plugin: {major:      12,
					       minor:      0,
					       patchlevel: 0} },

	short_name:             "MDRaid1RegMgr",
	long_name:              "MD Raid 1 Region Manager",
	oem_name:               "IBM",

	functions:              {plugin: &raid1_functions},

	container_functions:    NULL
};
