/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: raid1_funcs.c
 *
 * Description: This file contains all MD RAID1's plugin functions.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include <sys/ioctl.h>

#include "md.h"
#include "raid1_mgr.h"
#include "raid1_discover.h"

#define my_plugin_record raid1_plugin

static int swap_disks(md_volume_t * volume, int index, int index2){
	storage_object_t * tmp_object;
	mdp_disk_t tmp_disk;
	LOG_ENTRY();

	// swap the disk entries in the master SB
	tmp_disk = volume->super_block->disks[index];
	volume->super_block->disks[index] = volume->super_block->disks[index2];
	volume->super_block->disks[index2] = tmp_disk;

	volume->super_block->disks[index].number = index;
	volume->super_block->disks[index].raid_disk = index;
	volume->super_block->disks[index2].number = index2;
	volume->super_block->disks[index2].raid_disk = index2;
	// swap the object pointers
	tmp_object = volume->child_object[index];
	volume->child_object[index] = volume->child_object[index2];
	volume->child_object[index2] = tmp_object;
 /*
	tmp_super = volume->super_array[index];
	volume->super_array[index] = volume->super_array[index2];
	volume->super_array[index2] = tmp_super;

	
	for (i = 0; i < volume->nr_disks; i++) {
		tmp_disk = volume->super_array[i]->disks[index];
		volume->super_array[i]->disks[index] = volume->super_array[i]->disks[index2];
		volume->super_array[i]->disks[index2] = tmp_disk;
	}*/
	LOG_EXIT_INT(0);
	return 0;
}


// Make sure all active disks are before all spare disks.
static int order_disks(md_volume_t * volume, int index){
	int      rc= 0;
	LOG_ENTRY();

	if (volume->super_block->disks[index].state & (1<< MD_DISK_PENDING_ACTIVE)) {
		// disk is active, needs to move up in list
		while (index && !(volume->super_block->disks[index - 1].state & (1<< MD_DISK_ACTIVE))) {
			swap_disks(volume, index, index-1);
			index--;
		}
	} else {
		// disk is spare, needs to move down in list below the Actives
		while ((index < volume->nr_disks - 1)
			&& (volume->super_block->disks[index + 1].state & (1<< MD_DISK_ACTIVE | 1<<MD_DISK_PENDING_ACTIVE))) {
			swap_disks(volume, index, index +1);
			index++;
		}
		// Also, move the spare above any faulty or removed disks
		while (index && (volume->super_block->disks[index - 1].state & (1<<MD_DISK_FAULTY | 1<<MD_DISK_REMOVED))) {
			swap_disks(volume, index, index-1);
			index--;
		}
	}


	LOG_EXIT_INT(rc);
	return rc;
}

static int free_disk_info_post_ioctl(md_volume_t *volume, md_ioctl_pkg_t *pkg)
{
	LOG_ENTRY();
	if (pkg->parm.disk_info) {
		EngFncs->engine_free(pkg->parm.disk_info);
		pkg->parm.disk_info = NULL;
	}
	LOG_EXIT_INT(0);
	return 0;
}

static int free_temp_sb(md_volume_t *volume, md_ioctl_pkg_t *pkg)
{
	if (pkg->parm.sb) {
		EngFncs->engine_free(pkg->parm.sb);
		pkg->parm.sb = NULL;
	}
	return 0;
}

int deactivate_active_disk(md_volume_t * volume, storage_object_t * active_disk)
{
	int      i, rc= 0;
	LOG_ENTRY();

	for (i = 0; i < MAX_MD_DEVICES; i++ ) {
		if (volume->child_object[i] == active_disk) {
			// ok, found the disk to deactivate
			volume->super_block->disks[i].state = 0;
			volume->super_block->spare_disks++;
			volume->super_block->active_disks--;
			volume->super_block->raid_disks--;
			break;
		}
	}
	order_disks(volume, i);

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: activate_spare_post_activate
 *
 * This function should be called during POST_ACTIVATE phase of commit.
 *
 */
static int activate_spare_post_ioctl(md_volume_t *volume, md_ioctl_pkg_t *pkg)
{
	LOG_ENTRY();
	if (pkg->parm.disk_info) {
		EngFncs->engine_free(pkg->parm.disk_info);
		pkg->parm.disk_info = NULL;
	}
	volume->region_mgr_flags &= ~(MD_RAID1_CONFIG_CHANGE_PENDING | MD_RAID1_IGNORE_VERIFY);
	LOG_EXIT_INT(0);
	return 0;
}


/*
 * Function: activate_spare_setup
 *
 * This function should be called during SETUP phase of commit.
 * Update the following fields in the superblock
 *		+ increment failed_disks
 *		+ find the first available and mark it faulty
 * Next phase of commit process is FIRST_METADATA_WRITE
 */
static int activate_spare_setup(md_volume_t * volume, md_setup_func_t *setup)
{
	int rc = 0;
	evms_md_disk_info_t *disk_info;
	storage_object_t *spare_disk;
	evms_md_ioctl_parm_t parm;
	mdp_disk_t *disk;
	mdp_super_t *sb;

	LOG_ENTRY();

	disk_info = setup->disk_info;
	spare_disk = disk_info->object;
	sb = volume->super_block;

	if (setup->proceed == FALSE) {
		LOG_DEBUG("Cancel activate spare (%s) for region %s\n", spare_disk->name, volume->name);
		rc = 0;
		goto undo_activate_spare;
	}

	if (volume->child_object[disk_info->number] != spare_disk) {
		LOG_DEBUG("Could not find the spare disk %s to activate region [%s]\n",
			  spare_disk->name, volume->name);
		rc = EINVAL;
		goto undo_activate_spare;
	}
	
	parm.disk_info = disk_info;
	rc = schedule_md_ioctl_pkg(volume, EVMS_MD_ADD, &parm, activate_spare_post_ioctl);
	if (rc) {
		goto undo_activate_spare;
	}

	sb->failed_disks++;
	sb->working_disks--;
	sb->spare_disks--;

	disk = &sb->disks[disk_info->number];

	disk->major = spare_disk->dev_major;
	disk->minor = spare_disk->dev_minor;
	disk->number = disk_info->number;
	disk->raid_disk = disk_info->number;
	disk->state = (1 << MD_DISK_FAULTY);
	
	volume->flags |= MD_DIRTY;

	LOG_EXIT_INT(rc);
	return rc;

undo_activate_spare:
	mark_disk_spare(&sb->disks[disk_info->number]);
	sb->raid_disks--;
	EngFncs->engine_free(setup->disk_info);
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: activate_spare_disk
 *
 * This function changes the configuration of the RAID1 region
 * from n-way mirror to n+1-way mirror.
 */
int activate_spare_disk(md_volume_t * volume, storage_object_t * spare_disk)
{
	int      i, rc= 0;
	evms_md_disk_info_t *disk_info;
	LOG_ENTRY();

	for (i = 0; i < MAX_MD_DEVICES; i++ ) {
		if (volume->child_object[i] == spare_disk) {
			
			disk_info = EngFncs->engine_alloc(sizeof(evms_md_disk_info_t));
			if ( !disk_info ) {
				LOG_CRITICAL("No Memory\n");
				rc = ENOMEM;
				break;
			}

			disk_info->number = i;
			disk_info->object = spare_disk;

			rc = schedule_setup_func(volume, disk_info, activate_spare_setup);
			if (rc) {
				EngFncs->engine_free(disk_info);
				break;
			}

			volume->super_block->disks[i].state = (1<<MD_DISK_NEW | 1<<MD_DISK_PENDING_ACTIVE);
			volume->super_block->raid_disks++;

			break;
		}
	}

	if (!rc) {
		order_disks(volume, i);
		volume->region->flags |= SOFLAG_NEEDS_DEACTIVATE;
		volume->region_mgr_flags |= (MD_RAID1_CONFIG_CHANGE_PENDING | MD_RAID1_IGNORE_VERIFY);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: remove_active_disk_post_activate
 *
 */
static int remove_active_disk_post_activate(md_volume_t *volume, md_ioctl_pkg_t *pkg)
{
	LOG_ENTRY();
	volume->region_mgr_flags &= ~(MD_RAID1_CONFIG_CHANGE_PENDING | MD_RAID1_IGNORE_VERIFY);
	LOG_EXIT_INT(0);
	return 0;
}

int remove_active_disk(md_volume_t * volume, storage_object_t *active_disk)
{
	int i, k, rc = 0;
	mdp_super_t *sb;
	LOG_ENTRY();

	if (!volume || !active_disk) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	sb = volume->super_block;

	for (i = 0; i < MAX_MD_DEVICES; i++ ) {
		if (volume->child_object[i] == active_disk) {

			rc = schedule_md_ioctl_pkg( volume, EVMS_MD_INVOKE_CALLBACK,
						    NULL, remove_active_disk_post_activate);
			if (rc) {
				MESSAGE("Can not schedule callback function to remove %s from region %s.\n",
					active_disk->name, volume->name);
				break;
			}

			md_remove_region_from_object(volume->region, volume->child_object[i]);
			KILL_SECTORS(volume->child_object[i],
				MD_NEW_SIZE_SECTORS(volume->child_object[i]->size),
				MD_RESERVED_SECTORS);
			EngFncs->engine_free(volume->super_array[i]);
			volume->super_array[i] = NULL;
			volume->child_object[i] = NULL;
			memset(&sb->disks[i], 0, sizeof(mdp_disk_t));
			
			volume->region->flags |= SOFLAG_NEEDS_DEACTIVATE;
			volume->region_mgr_flags |= (MD_RAID1_CONFIG_CHANGE_PENDING | MD_RAID1_IGNORE_VERIFY);

			// collapse super array and object array
			for (k = i; k < MAX_MD_DEVICES - 1; k++) {
				volume->super_array[k]= volume->super_array[k+1];
				volume->child_object[k] = volume->child_object[k+1];
				volume->stale_object[k] = volume->stale_object[k+1];
				sb->disks[k]= sb->disks[k+1];
				sb->disks[k].number = k;
				sb->disks[k].raid_disk = k;
			}
			volume->child_object[MAX_MD_DEVICES-1] = NULL;
			volume->stale_object[MAX_MD_DEVICES-1] = NULL;
			volume->super_array[MAX_MD_DEVICES-1] = NULL;

			memset(&sb->disks[k],0,sizeof(mdp_disk_t));

			sb->active_disks--;
			sb->working_disks--;
			sb->raid_disks--;
			sb->nr_disks--;
			volume->nr_disks--;
			volume->flags |= MD_DIRTY;
			break;
		}
	}

	if (i == MAX_MD_DEVICES) {
		rc = ENODEV;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: add_active_disk_post_ioctl
 *
 * This function should be called during POST_ACTIVATE phase of commit.
 *
 */
static int add_active_disk_post_ioctl(md_volume_t *volume, md_ioctl_pkg_t *pkg)
{
	LOG_ENTRY();
	if (pkg->parm.disk_info) {
		EngFncs->engine_free(pkg->parm.disk_info);
		pkg->parm.disk_info = NULL;
	}
	volume->region_mgr_flags &= ~(MD_RAID1_CONFIG_CHANGE_PENDING | MD_RAID1_IGNORE_VERIFY);
	LOG_EXIT_INT(0);
	return 0;
}

/*
 * Function: add_active_disk_setup
 *
 * This function should be called during SETUP phase of commit.
 *
 * Next phase of commit process is FIRST_METADATA_WRITE
 */
static int add_active_disk_setup(md_volume_t * volume, md_setup_func_t *setup)
{
	int rc = 0;
	evms_md_ioctl_parm_t parm;
	evms_md_disk_info_t *disk_info;
	storage_object_t *new_disk;
	mdp_disk_t *disk;
	mdp_super_t *sb;

	LOG_ENTRY();


	disk_info = setup->disk_info;
	new_disk = disk_info->object;
	sb = volume->super_block;
	
	if (setup->proceed == FALSE) {
		LOG_WARNING("Cancel add new active disk (%s) to region %s\n", new_disk->name, volume->name);
		rc = 0;
		goto undo_add_active_disk;
	}
	
	if (volume->child_object[disk_info->number] != new_disk) {
		LOG_WARNING("Could not find the new disk %s to add to region [%s]\n",
			    new_disk->name, volume->name);
		rc = EINVAL;
		goto undo_add_active_disk;
	}

	parm.disk_info = disk_info;
	rc = schedule_md_ioctl_pkg(volume, EVMS_MD_ADD, &parm, add_active_disk_post_ioctl);
	if (rc) {
		goto undo_add_active_disk;
	}

	sb->failed_disks++;
	sb->working_disks--;
	sb->spare_disks--;

	disk = &sb->disks[disk_info->number];
	
	disk->major = new_disk->dev_major;
	disk->minor = new_disk->dev_minor;
	disk->number = disk_info->number;
	disk->raid_disk = disk_info->number;
	disk->state = (1 << MD_DISK_FAULTY);

	volume->flags |= MD_DIRTY;

	LOG_EXIT_INT(rc);
	return rc;

undo_add_active_disk:

	memset(&sb->disks[disk_info->number], 0, sizeof(mdp_disk_t));

	md_remove_region_from_object(volume->region, volume->child_object[disk_info->number]);
	EngFncs->engine_free(volume->super_array[disk_info->number]);
	volume->super_array[disk_info->number] = NULL;

	EngFncs->engine_free(setup->disk_info);
	
	sb->raid_disks--;
	sb->working_disks--;
	sb->spare_disks--;
	sb->nr_disks--;
	volume->nr_disks--;

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: add_active_disk
 *
 */
int add_active_disk(md_volume_t * volume, storage_object_t * new_disk)
{
	int rc= 0, i;
	evms_md_disk_info_t *disk_info;
	mdp_disk_t *disk;
	
	LOG_ENTRY();

	i = find_empty_slot(volume->super_block);
	if (i==MAX_MD_DEVICES) {
		LOG_EXIT_INT(ENODEV);
		return ENODEV;
	}

	rc = md_clone_superblock(volume, i);

	if (rc) {
		MESSAGE("Error adding object %s to region: no more memory\n", new_disk->name);
		LOG_EXIT_INT(rc);
		return rc;
	}

	disk_info = EngFncs->engine_alloc(sizeof(evms_md_disk_info_t));
	if ( !disk_info ) {
		LOG_CRITICAL("No Memory\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	disk_info->number = i;
	disk_info->object = new_disk;

	rc = schedule_setup_func(volume, disk_info, add_active_disk_setup);
	if (rc) {
		MESSAGE("Error adding object %s to region: Can not shedule setup function\n", new_disk->name);
		EngFncs->engine_free(disk_info);
		LOG_EXIT_INT(rc);
		return rc;
	}
	

	volume->child_object[volume->nr_disks] = new_disk;
	md_append_region_to_object(volume->region, new_disk);

	disk = &volume->super_block->disks[i];

	disk->major = new_disk->dev_major;
	disk->minor = new_disk->dev_minor;
	disk->number = i;
	disk->raid_disk = i;
	disk->state = (1<<MD_DISK_NEW | 1<<MD_DISK_PENDING_ACTIVE);

	volume->super_block->raid_disks++;
	volume->super_block->working_disks++;
	volume->super_block->spare_disks++;
	volume->super_block->nr_disks++;
	volume->nr_disks++;
	
	volume->region->flags |= SOFLAG_NEEDS_DEACTIVATE;
	volume->region_mgr_flags |= (MD_RAID1_CONFIG_CHANGE_PENDING | MD_RAID1_IGNORE_VERIFY);
	LOG_EXIT_INT(rc);
	return rc;

}

static int add_spare_disk_to_active_region(md_volume_t * volume, storage_object_t * spare)
{
	int       rc= 0,i;
	mdp_disk_t disk;
	mdp_super_t *sb = volume->super_block;
	evms_md_ioctl_parm_t parm;
	evms_md_disk_info_t *disk_info;

	LOG_ENTRY();
	
	i = find_empty_slot(volume->super_block);
	if (i==MAX_MD_DEVICES) {
		LOG_EXIT_INT(ENODEV);
		return ENODEV;
	}

	LOG_DEBUG("Slot %d is available.\n", i);

	disk_info = EngFncs->engine_alloc(sizeof(evms_md_disk_info_t));
	if ( !disk_info ) {
		LOG_CRITICAL("No Memory\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	disk_info->number = i;
	disk_info->object = spare;
	parm.disk_info = disk_info;
	rc = schedule_md_ioctl_pkg(volume, EVMS_MD_ADD, &parm, free_disk_info_post_ioctl);
	if (rc) {
		goto error_free_mem;
	}

	rc = md_clone_superblock(volume, i);
	if (rc) {
		goto error_free_mem;
	}

	volume->child_object[i] = spare;
	md_append_region_to_object(volume->region, spare);

	disk.major = spare->dev_major;
	disk.minor = spare->dev_minor;
	disk.number = i;
	disk.raid_disk = disk.number;
	if (volume->flags & MD_DEGRADED) {
		disk.state = (1<<MD_DISK_NEW) | (1<<MD_DISK_PENDING_ACTIVE);
		volume->region_mgr_flags |= MD_RAID1_IGNORE_VERIFY;
	} else {
		disk.state = (1<<MD_DISK_NEW);
	}

	//copy new disk entry to master SB and all SB
       	memcpy(&sb->disks[i], &disk, sizeof(mdp_disk_t));

	sb->spare_disks++;
	sb->working_disks++;
	sb->nr_disks++;
	volume->nr_disks++;
	
	LOG_EXIT_INT(rc);
	return rc;

error_free_mem:
	if (disk_info) {
		EngFncs->engine_free(disk_info);
	}
	LOG_EXIT_INT(rc);
	return rc;

}

static int add_spare_disk_to_inactive_region(md_volume_t * volume, storage_object_t * spare)
{
	int i, rc=0;
	mdp_super_t *sb = volume->super_block;
	evms_md_ioctl_parm_t parm;
	mdp_disk_t *disk;

	LOG_ENTRY();
	
	if (!(volume->flags & (MD_CORRUPT | MD_DEGRADED))) {

		/* The MD array is not active and normal, we can write*/

		i = find_empty_slot(volume->super_block);
		if (i==MAX_MD_DEVICES) {
			LOG_EXIT_INT(ENODEV);
			return ENODEV;
		}
	
		rc = md_clone_superblock(volume, i);
	
		if (rc) {
			MESSAGE("Cannot allocate memory to add object %s to region.\n",
				spare->name);
			LOG_EXIT_INT(rc);
			return rc;
		}
		
		volume->child_object[i] = spare;
		md_append_region_to_object(volume->region, spare);

		disk = &volume->super_block->disks[i];

		disk->major = spare->dev_major;
		disk->minor = spare->dev_minor;
		disk->number = i;
		disk->raid_disk = i;
		disk->state = 0;

		volume->super_block->working_disks++;
		volume->super_block->spare_disks++;
		volume->super_block->nr_disks++;
		volume->nr_disks++;

		volume->flags |= MD_DIRTY;

		LOG_EXIT_INT(rc);
		return rc;
	}

	/*
	 * We are about to add a spare to a degraded array.
	 * We first schedule an activation, then use
	 * add_spare_to_active_region() to hot add the spare later.
	 */

	sb = EngFncs->engine_alloc(sizeof(mdp_super_t));

	if (!sb) {
		LOG_CRITICAL("No Memory\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	memcpy(sb, volume->super_block, sizeof(mdp_super_t));
	parm.sb = sb;

	schedule_md_ioctl_pkg(volume, EVMS_MD_ACTIVATE_REGION, &parm, free_temp_sb);

	rc = add_spare_disk_to_active_region(volume, spare);
	if (rc)
		empty_ioctl_queue(volume);

	LOG_EXIT_INT(rc);
	return rc;
}

int add_spare_disk(md_volume_t * volume, storage_object_t * spare)
{
	if (md_is_region_active(volume->region))
		return add_spare_disk_to_active_region(volume, spare);
	else
		return add_spare_disk_to_inactive_region(volume, spare);
}

int remove_spare_disk(md_volume_t * volume, storage_object_t * spare)
{
	int      i,k, rc= 0;
	mdp_super_t *sb;
	evms_md_disk_info_t *disk_info;
	evms_md_ioctl_parm_t parm;
	mdu_disk_info_t info;
	LOG_ENTRY();

	if (!volume || !spare) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	sb = volume->super_block;

	for (i = 0; i < MAX_MD_DEVICES; i++ ) {
		if (volume->child_object[i] == spare) {
			// ok, found the disk to remove

			// this disk is going away, add it to our list of disks
			// to ioctl to the kernel.
			if (md_is_region_active(volume->region)) {
				/*
				 * IOCTL the kernel
				 * do not update superblocks (ie. do not set DIRTY flag)
				 */
				info.number = i;
				rc = md_ioctl_get_disk_info(volume->region, &info);
				if (rc) {
					LOG_ERROR("(%s) does not exist\n", spare->name);
					rc = ENODEV;
					break;
				} else {
					if ((info.major != spare->dev_major) || (info.minor != spare->dev_minor)) {

						MESSAGE("WARNING: Region:%s, Device:%s: Index:%d\n"
							" There is a mismatch major/minor, Kernel MD driver has (%d:%d), EVMS has (%d:%d)."
							"  However, if %s was created by another MD tool such as mdadm or raidtools,"
							" the operation will succeed.\n",
						  volume->name, spare->name, i,
						  info.major, info.minor,
						  spare->dev_major, spare->dev_minor,
						  volume->name);
						//rc = ENODEV;
						//break;
					}
				}
				
				disk_info = EngFncs->engine_alloc(sizeof(evms_md_disk_info_t));
				if ( !disk_info ) {
					LOG_CRITICAL("No Memory\n");
					rc = ENOMEM;
					break;
				}

				disk_info->number = i;
				//disk_info->major = spare->dev_major;
				//disk_info->minor = spare->dev_minor;
				disk_info->major = info.major;
				disk_info->minor = info.minor;
				disk_info->object = spare;
				parm.disk_info = disk_info;
				schedule_md_ioctl_pkg(volume, EVMS_MD_REMOVE, &parm, free_disk_info_post_ioctl);
				md_remove_region_from_object(volume->region, spare);

				KILL_SECTORS(spare, MD_NEW_SIZE_SECTORS(spare->size), MD_RESERVED_SECTORS);
				EngFncs->engine_free(volume->super_array[i]);

				volume->super_array[i] = NULL;
				volume->child_object[i] = NULL;
				memset(&sb->disks[i],0,sizeof(mdp_disk_t));

			} else {
				
				md_remove_region_from_object(volume->region, spare);
				KILL_SECTORS(spare, MD_NEW_SIZE_SECTORS(spare->size), MD_RESERVED_SECTORS);
				EngFncs->engine_free(volume->super_array[i]);
				volume->super_array[i] = NULL;
				volume->child_object[i] = NULL;
				memset(&sb->disks[i],0,sizeof(mdp_disk_t));

				// collapse super array and object array
				for (k = i; k < MAX_MD_DEVICES -1; k++) {
					volume->super_array[k] = volume->super_array[k+1];
					volume->child_object[k] = volume->child_object[k+1];
					volume->stale_object[k] = volume->stale_object[k+1];
					sb->disks[k]= sb->disks[k+1];
					sb->disks[k].number = k;
					sb->disks[k].raid_disk = k;
				}
				
				volume->child_object[MAX_MD_DEVICES-1] = NULL;
				volume->stale_object[MAX_MD_DEVICES-1] = NULL;
				volume->super_array[MAX_MD_DEVICES-1] = NULL;
				memset(&sb->disks[MAX_MD_DEVICES-1],0,sizeof(mdp_disk_t));
 				
				volume->flags |= MD_DIRTY;
			}

			sb->working_disks--;
			sb->spare_disks--;
			sb->nr_disks--;
			volume->nr_disks--;
			break;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int remove_stale_disk(md_volume_t * volume, storage_object_t * stale_disk)
{
	int i, rc=0;
	evms_md_disk_info_t *disk_info;
	evms_md_ioctl_parm_t parm;
	mdu_disk_info_t info;
	storage_object_t *obj;
	
	LOG_ENTRY();

	if (!volume || !stale_disk) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	for (i = 0; i < MAX_MD_DEVICES; i++ ) {
		
		obj = volume->stale_object[i];
		
		if (obj == stale_disk) {
			break;
		}
	}

	if ((i == MAX_MD_DEVICES) || !obj || !volume->stale_disks) {
		md_log_internal_bug(__FILE__, __FUNCTION__, __LINE__);
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}


	/*
	 * Removing a stale disk which is still in the array.
	 * Make sure to get the index from the kernel MD driver.
	 */
	if (md_is_region_active(volume->region)) {
		int kernel_idx;
		kernel_idx = find_disk_in_active_region(volume->region,
				stale_disk->dev_major,
				stale_disk->dev_minor);
		if (kernel_idx != i) {
			LOG_DEFAULT("The superblock has %s at index=%d,"
				" whereas the kernel said index=%d.\n",
				stale_disk->name, i, kernel_idx);
		}
		if (kernel_idx < MAX_MD_DEVICES) {
			info.number = kernel_idx;
			rc = md_ioctl_get_disk_info(volume->region, &info);
			if (!rc && 
				(info.major == stale_disk->dev_major) &&
			    (info.minor == stale_disk->dev_minor) &&
			    (!(info.state & (1<<MD_DISK_REMOVED)))) {
				disk_info = EngFncs->engine_alloc(sizeof(evms_md_disk_info_t));

				if (!disk_info ) {
					LOG_CRITICAL("No Memory\n");
					LOG_EXIT_INT(ENOMEM);
					return ENOMEM;
				}

				disk_info->number = kernel_idx;
				disk_info->major = stale_disk->dev_major;
				disk_info->minor = stale_disk->dev_minor;
				disk_info->object = stale_disk;
				parm.disk_info = disk_info;
				schedule_md_ioctl_pkg(volume,
						      EVMS_MD_REMOVE,
						      &parm,
						      free_disk_info_post_ioctl);
			}
		}
	}

	memset(&volume->super_block->disks[i],0,sizeof(mdp_disk_t));
	md_remove_region_from_object(volume->region, obj);
	KILL_SECTORS(obj, MD_NEW_SIZE_SECTORS(obj->size), MD_RESERVED_SECTORS);
	EngFncs->engine_free(volume->super_array[i]);

	volume->super_array[i] = NULL;
	volume->stale_object[i] = NULL;
	
	volume->stale_disks--;

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Function: 	remove_faulty_disk
 *
 * If the region is active
 *	- schedule an HOT_REMOVE_DISK ioctl (will be performed during commit process)
 *	- remove faulty disk from region's children list
 *	- set kill sectors to wipe out the MD superblock on the faulty disk
 *	- mark the faulty disk removed
 */
int remove_faulty_disk(md_volume_t * volume, storage_object_t * faulty_disk)
{
	int i, rc = 0;
	mdp_super_t *sb;
	storage_object_t *obj;
	evms_md_disk_info_t *disk_info;
	evms_md_ioctl_parm_t parm;
	mdu_disk_info_t info;
	LOG_ENTRY();

	if (!volume || !faulty_disk) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	sb = volume->super_block;

	for (i = 0; i < MAX_MD_DEVICES; i++ ) {
		obj = volume->child_object[i];
		if (obj == faulty_disk) {
			break;
		}
	}

	if ((i == MAX_MD_DEVICES) || !obj) {
		md_log_internal_bug(__FILE__, __FUNCTION__, __LINE__);
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	LOG_DEBUG("Faulty disk (%s) is found at index:%d.\n", faulty_disk->name, i);

	/*
	 * Removing a faulty disk which is still in the array
	 */
	if (md_is_region_active(volume->region)) {


		info.number = i;
		rc = md_ioctl_get_disk_info(volume->region, &info);
		if (rc) {
			LOG_ERROR("(%s) does not exist\n", faulty_disk->name);
			LOG_EXIT_INT(ENODEV);
			return ENODEV;
		} else {
			if ((info.major != faulty_disk->dev_major) || (info.minor != faulty_disk->dev_minor)) {
				MESSAGE("WARNING: Region:%s, Device:%s: Index:%d\n"
					" There is a mismatch major/minor, Kernel MD driver has (%d:%d), EVMS has (%d:%d)."
					"  However, if %s was created by another MD tool such as mdadm or raidtools,"
					" the operation will succeed.\n",
					volume->name, faulty_disk->name, i,
					info.major, info.minor,
					faulty_disk->dev_major, faulty_disk->dev_minor,
					volume->name);
				//LOG_EXIT_INT(ENODEV);
				//return ENODEV;
			}
		}
		
		disk_info = EngFncs->engine_alloc(sizeof(evms_md_disk_info_t));
		
		if (!disk_info ) {
			LOG_CRITICAL("No Memory\n");
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}

		disk_info->number = i;
		//disk_info->major = faulty_disk->dev_major;
		//disk_info->minor = faulty_disk->dev_minor;
		disk_info->major = info.major;
		disk_info->minor = info.minor;
		disk_info->object = faulty_disk;
		parm.disk_info = disk_info;
		schedule_md_ioctl_pkg(volume, EVMS_MD_REMOVE, &parm, free_disk_info_post_ioctl);
	} else {
		volume->flags |= MD_DIRTY;
	}

	md_remove_region_from_object(volume->region, obj);
	KILL_SECTORS(obj, MD_NEW_SIZE_SECTORS(obj->size), MD_RESERVED_SECTORS);
	EngFncs->engine_free(volume->super_array[i]);
	volume->super_array[i] = NULL;
	volume->child_object[i] = NULL;
	
	memset(&sb->disks[i], 0, sizeof(mdp_disk_t));
	mark_disk_removed(&sb->disks[i]);

	sb->failed_disks--;
	sb->nr_disks--;
	volume->nr_disks--;

	LOG_EXIT_INT(0);
	return 0;
}

static int mark_disk_faulty_post_ioctl(md_volume_t *volume, md_ioctl_pkg_t *pkg)
{
	LOG_ENTRY();
	if (pkg->parm.disk_info) {
		EngFncs->engine_free(pkg->parm.disk_info);
		pkg->parm.disk_info = NULL;
	}
	volume->region_mgr_flags &= ~(MD_RAID1_CONFIG_CHANGE_PENDING | MD_RAID1_IGNORE_VERIFY);
	LOG_EXIT_INT(0);
	return 0;
}

int raid1_mark_disk_faulty(md_volume_t * volume, storage_object_t * active)
{
	int            rc = 0;
	int            i;
	boolean        found;
	mdp_super_t *sb;
	evms_md_disk_info_t *disk_info;
	evms_md_ioctl_parm_t parm;
	mdu_disk_info_t info;

	LOG_ENTRY();

	if (!volume) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	sb = volume->super_block;

	//if (sb->spare_disks == 0) {
	//	LOG_ERROR("RAID array %s had no spare disks available.\n", volume->region->name);
	//	LOG_EXIT_INT(EINVAL);
	//	return EINVAL;
	//}

	for (i = 0, found = FALSE; !found && (i < MAX_MD_DEVICES); i++ ) {
		if (volume->child_object[i] == active) {
			found = TRUE;

			/* Found the disk to remove.  Make sure it is active.*/
			if (!disk_active(&sb->disks[i])) {
				LOG_ERROR("Object %s is not marked active in RAID array %s.\n",
					  active->name, volume->region->name);
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
			break;
		}
	}

	if (!found) {
		LOG_ERROR("Object %s is not part of RAID array %s.\n",
			  active->name, volume->region->name);
		LOG_EXIT_INT(ENODEV);
		return ENODEV;
	}

	info.number = i;
	rc = md_ioctl_get_disk_info(volume->region, &info);
	if (rc) {
		LOG_ERROR("(%s) does not exist\n", active->name);
		LOG_EXIT_INT(ENODEV);
		return ENODEV;
	} else {
		if ((info.major != active->dev_major) || (info.minor != active->dev_minor)) {
			LOG_ERROR("(%s) mismatch major/minor, kernel(%d:%d), active(%d:%d)\n",
				  active->name, info.major, info.minor, active->dev_major, active->dev_minor);
			//LOG_EXIT_INT(EINVAL);
			//return EINVAL;
		}
	}

	disk_info = EngFncs->engine_alloc(sizeof(evms_md_disk_info_t));
	if ( !disk_info ) {
		LOG_CRITICAL("No Memory\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	disk_info->number = i;
	disk_info->major = info.major;
	disk_info->minor = info.minor;
	disk_info->object = active;
	parm.disk_info = disk_info;
	schedule_md_ioctl_pkg(volume, EVMS_MD_DEACTIVATE, &parm, mark_disk_faulty_post_ioctl);

	/*
	 * Mark the disk entry faulty.  That will cause the
	 * MD recovery to swap it out with a spare.
	 */
	sb->disks[i].state = (1 << MD_DISK_FAULTY);

	/* Update disk counts. */
	sb->active_disks--;
	sb->working_disks--;
	sb->failed_disks++;

	volume->region_mgr_flags |= (MD_RAID1_CONFIG_CHANGE_PENDING | MD_RAID1_IGNORE_VERIFY);
	LOG_EXIT_INT(0);
	return 0;
}


static int fix_array_post_activate(md_volume_t *volume, md_ioctl_pkg_t *pkg)
{
	volume->region_mgr_flags &= ~MD_RAID1_CONFIG_CHANGE_PENDING;
	return 0;
}


int raid1_fix_array(md_volume_t *volume)
{
	int rc = 0;
	
	LOG_ENTRY();

	raid1_verify_and_fix_array(volume, 0);
	if (volume->flags & (MD_CORRUPT | MD_PROBLEMATIC_SB)) {
		raid1_verify_and_fix_array(volume, 2);
		if ((volume->flags & MD_CORRUPT) == 0) {
			volume->region_mgr_flags |= MD_RAID1_CONFIG_CHANGE_PENDING;
			schedule_md_ioctl_pkg(volume, EVMS_MD_INVOKE_CALLBACK, NULL, fix_array_post_activate);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

