/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: md_io.c
 *
 * Description: This file contains all functions pertaining to disk I/O within
 *              the MD region manager plugin. This file should be the only
 *              location of calls to READ and WRITE (except the md_read and
 *              md_write API calls).
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <plugin.h>

#include "md.h"

#define my_plugin_record my_plugin

int md_delete_volume(md_volume_t * volume, boolean tear_down){
	int rc = 0;
	int i;
	storage_object_t *obj;
	LOG_ENTRY();

	for (i = 0; i < MAX_MD_DEVICES; i++) {
		obj = volume->child_object[i];
		if (!obj)
			obj = volume->stale_object[i];
		if (obj) {
			if (tear_down)
				KILL_SECTORS(obj, MD_NEW_SIZE_SECTORS(obj->size), MD_RESERVED_SECTORS);
			EngFncs->engine_free(volume->super_array[i]);
		}
	}
	md_remove_volume_from_list(volume);
	if (volume->setup_funcs != NULL) {
		empty_setup_funcs_queue(volume);
		EngFncs->destroy_list(volume->setup_funcs);
		volume->setup_funcs = NULL;
	}
	if (volume->ioctl_pkgs != NULL) {
		empty_ioctl_queue(volume);
		EngFncs->destroy_list(volume->ioctl_pkgs);
		volume->ioctl_pkgs = NULL;
	}
	if (volume->ioctl_cleanup != NULL) {
		free_ioctl_pkgs(volume);
		EngFncs->destroy_list(volume->ioctl_cleanup);
		volume->ioctl_cleanup = NULL;
	}
	if (volume->personality != MULTIPATH || !(volume->region->flags & SOFLAG_ACTIVE)) {
		/* For multipath regions, only delete the volume if the object
		 * is not active. If it is active, we will need this
		 * private-data in order to correctly deactivate the region.
		 */
		EngFncs->engine_free(volume);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: md_region_rw
 *
 * If the region is active in the kernel,
 * perform IO on the region device node.
 * Returns 0 for success, else error.
 *
 */
int md_region_rw(storage_object_t *region,
		 lsn_t lsn,
		 sector_count_t count,
		 void *buffer,
		 int rw)
{
	int rc=0;
	int fd;

	LOG_ENTRY();

	if (!count) {
		LOG_EXIT_INT(0); /* nothing to do */
		return 0;
	}

	if ((region->flags & SOFLAG_ACTIVE) == 0) {
		return ENODEV;
	}

	fd = EngFncs->open_object(region, O_SYNC | O_RDWR);
	if (fd <= 0) {
		LOG_ERROR("Error opening md object %s.\n",region->name);
		rc = ENODEV;
	} else {
		if (rw == 1)
			rc = EngFncs->write_object(region, fd, buffer,
						   count<<EVMS_VSECTOR_SIZE_SHIFT,
						   lsn<<EVMS_VSECTOR_SIZE_SHIFT);
		else
			rc = EngFncs->read_object(region, fd, buffer,
						  count<<EVMS_VSECTOR_SIZE_SHIFT,
						  lsn<<EVMS_VSECTOR_SIZE_SHIFT);
		if (rc > 0) {
			rc = 0;
		} else {
			rc = EIO;
			LOG_WARNING("%s failed, md object %s rc=%d.\n",
				    (rw == 1) ? "WRITE" : "READ",
				    region->name,
				    rc);
		}
		EngFncs->close_object(region, fd);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

