/*
 *
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libmac.so
 *
 *   File: discovery.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "mac_plugin.h"


/*
 *  Function:  display_segment_list
 *
 *  Called by discovery code, after successfully creating mac
 *  segment objects, to dump the information to the engine
 *  log for debugging purposes.
 */
static void  display_segment_list( LOGICALDISK  *ld)
{
        DISKSEG *seg;
        list_element_t iter;

        LOG_DEBUG("\n");
        LOG_DEBUG("  MAC Segment List ... logical disk= %s\n", ld->name);

        LIST_FOR_EACH( ld->parent_objects, iter, seg ){
                LOG_DEBUG("   %"PRIu64" %"PRIu64" %"PRIu64" %s\n",
                          seg->start,
                          seg->start+seg->size-1,
                          seg->size,
                          seg->name );
        }

        LOG_DEBUG("\n");
}


/*
 *  Function:  get_mac_disk_label
 *
 *  Called to read the disk label sector off a specified logical
 *  disk and verify the disk label.
 * 
 *  Returns: mac disk label if successful
 *           NULL ptr otherwise
 */
mac_disklabel_t *  get_mac_disk_label( LOGICALDISK *ld )
{
        mac_disklabel_t  *disklabel=NULL;
        int  rc;

        LOG_ENTRY();

        if (ld) {
                disklabel = (mac_disklabel_t *) malloc(EVMS_VSECTOR_SIZE); 
        }

        if (disklabel) {

                rc = READ( ld,0,1,( void *) disklabel );

                if (!rc) {

                        if ( MAC_DISK_TO_CPU16(disklabel->signature) != MAC_DISK_MAGIC) {
                                LOG_DEBUG("signature (0x%X) wrong ... expected(0x%X)\n",MAC_DISK_TO_CPU16(disklabel->signature), MAC_DISK_MAGIC);
                                rc = ENODATA;
                        }

                }

                if (rc) {
                        free(disklabel);
                        disklabel=NULL;                       
                }

        }

        LOG_EXIT_PTR(disklabel);
        return disklabel;
}


/*
 *  Function:  get_segments
 *
 *  Called to create mac segment storage objects that map the
 *  specified logical disk.  The job of this routine is to build
 *  evms segment storage objects to represent the mac slices.
 *
 */
static int get_segments( LOGICALDISK *ld,  mac_disklabel_t *disklabel )
{
        int                     rc=EINVAL,i;
        DISKSEG                 *seg=NULL;
        int                     seg_count=0;
        u_int32_t               map_count=0;
        u_int32_t               minor=1;
        mac_partition_t         *pmap=NULL;
        mac_partition_t         *p=NULL;
        char                     pname[PMAP_STRING_SIZE+1];
        char                     ptype[PMAP_STRING_SIZE+1];
        disk_private_data_t     *disk_pdata = get_mac_disk_private_data(ld);
        int                     vsectors_per_block=0;
        sector_count_t          pmap_size=0;


        LOG_ENTRY();

        // read the PMAP partition record and get a count of the number
        // of records in the partition table
        if (disklabel) {

                vsectors_per_block = MAC_DISK_TO_CPU16(disklabel->block_size)>>EVMS_VSECTOR_SIZE_SHIFT;         

                LOG_DEBUG("vsectors per block = %d\n", vsectors_per_block );

                vsectors_per_block = 1;

                if (vsectors_per_block) {

                        pmap = malloc(vsectors_per_block * EVMS_VSECTOR_SIZE);
                        if (pmap) {
                                rc = READ(ld, vsectors_per_block, 
                                          vsectors_per_block, pmap);
                                if (!rc) {
                                        if (isa_mac_pmap_partition(pmap) == TRUE) {
                                                LOG_DEBUG("found pmap partition record\n");
                                                map_count = MAC_DISK_TO_CPU32(pmap->map_count);
                                                pmap_size = MAC_DISK_TO_CPU32(pmap->block_count) * vsectors_per_block;
                                                disk_pdata->pcount = map_count;
                                                memcpy(&disk_pdata->pmap, pmap, sizeof(mac_partition_t)); 
                                        }
                                        free(pmap);
                                        pmap=NULL;                                      
                                }
                        }

                }

        }


        LOG_DEBUG("PMAP Info ...\n");
        LOG_DEBUG("    map_count   : %d\n", map_count);
        LOG_DEBUG("    pmap size   : %"PRIu64"\n", pmap_size );

        // read the partition table
        if (map_count) {
                pmap = malloc((map_count * vsectors_per_block)*EVMS_VSECTOR_SIZE);
                if (pmap) {
                        rc = READ(ld, vsectors_per_block, 
                                  vsectors_per_block*map_count, pmap);
                }
                else {
                        rc = ENOMEM;
                }
        }
        else {
                LOG_ERROR("error, problems getting pmap info.\n");                
                LOG_EXIT_INT(0);
                return 0;
        }


        if (rc) {
                if (pmap) free(pmap);
                LOG_ERROR("error, problems reading 1st partition record.\n");                
                prune_mac_seg_objects_from_list( ld->parent_objects );
                LOG_EXIT_INT(0);
                return 0;
        }


        // map the disk label and PMAP partition as a single METADATA segment
        seg = create_mac_metadata_segment( ld,                   // logical device
                                           0,                    // starting offset
                                           vsectors_per_block + pmap_size,   // size in sectors 
                                           0,                    // segment flags
                                           "mac_metadata" );     // segment name

        if (seg) {
                rc = insert_mac_segment_into_list( ld->parent_objects, seg);
                if (!rc) {
                        ++seg_count;
                }
        }
        else {
                rc = ENOMEM;
        }

        if (rc) {
                if (pmap) free(pmap);
                LOG_ERROR("error, problems creating mac disklabel metadata seg.\n");                
                prune_mac_seg_objects_from_list( ld->parent_objects );
                LOG_EXIT_INT(0);
                return 0;
        }


        // map each data segment                
        for (i=0,p=pmap; (i<map_count)&&(rc==0); p++, i++ ) {

                strncpy(pname, p->name, PMAP_STRING_SIZE);
                strncpy(ptype, p->type, PMAP_STRING_SIZE);

                LOG_DEBUG( "   Index (%d): type: %s   name: %s  status: 0x%X\n", 
                           i, 
                           ptype,
                           pname, 
                           MAC_DISK_TO_CPU32(p->status));
                LOG_DEBUG( "               start: %u   size: %u   data_start: %u   data_count: %u\n", 
                           MAC_DISK_TO_CPU32(p->start_block), 
                           MAC_DISK_TO_CPU32(p->block_count),               
                           MAC_DISK_TO_CPU32(p->data_start),
                           MAC_DISK_TO_CPU32(p->data_count) );

                if ( MAC_DISK_TO_CPU16(p->signature) == MAC_PARTITION_MAGIC) {

                        if ( MAC_DISK_TO_CPU32(p->status) & MAC_STATUS_VALID ) {

                                if (isa_mac_data_partition(p)==TRUE) {

                                        seg = create_mac_data_segment(  ld,
                                                                        MAC_DISK_TO_CPU32(p->start_block), 
                                                                        MAC_DISK_TO_CPU32(p->block_count),
                                                                        p,
                                                                        minor,
                                                                        i,
                                                                        0 );
                                }
                                else {
                                        seg = NULL;
                                }

                                if (seg) {
                                        rc = insert_mac_segment_into_list( ld->parent_objects, seg);
                                        if (!rc) {
                                                ++minor;
                                                ++seg_count;
                                        }
                                }
                        }
                        else {
                                LOG_DEBUG("   Index (%d):  unused pmap entry\n",i); 
                        }

                }


        }

        if (rc) {
                LOG_ERROR("error, problems mapping mac partitions for disk %s.", ld->name );
                seg_count = 0;
                prune_mac_seg_objects_from_list( ld->parent_objects );
        }

        if (pmap) free(pmap);

        LOG_EXIT_INT(seg_count);
        return seg_count;
}


/*
 *  Function: get_mac_segment_devmap_info
 *
 *  Called to test if the segment has an active device mapper
 *  node in the kernel and set the object info accordingly.
 */
static int get_mac_segment_devmap_info( DISKSEG *seg )
{
        int rc=0;
        dm_target_t *targets=NULL;
        dm_device_t *dev=NULL;

        LOG_ENTRY();

        if (seg->data_type == DATA_TYPE) {

                rc = EngFncs->dm_update_status(seg);
                if (!rc) {

                        if (seg->flags & SOFLAG_ACTIVE) {

                                rc = EngFncs->dm_get_targets(seg, &targets);
                                if (  (!rc) &&                       // RC==Zero
                                      (targets) &&                  // got a target list
                                      (targets->next == NULL) &&    // with 1 target only
                                      (targets->data.linear) &&     // with device data
                                      (targets->start == 0)) {      // and target starts at Zero

                                        dev = targets->data.linear;

                                        if ( ( seg->start != dev->start )||
                                             ( seg->size  != targets->length )) {
                                                LOG_DEBUG("this segment is being marked needs_activate\n");
                                                seg->flags |= SOFLAG_NEEDS_ACTIVATE;
                                        }
                                }

                                if (targets) EngFncs->dm_deallocate_targets(targets);

                        }
                        else {
                                seg->flags |= SOFLAG_NEEDS_ACTIVATE;
                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Function:  discover_mac_segments
 *
 *  Called by the mac discover api ... once for each storage object
 *  that is found in the discovery list of objects.  Our job is to
 *  probe a storage object, looking for a mac disk label.  If we find
 *  our metadata:
 *
 *      - create mac segment storage objects to map the slices
 *
 *  Return: the number of segment objects created 0-n and place new
 *          segment objects on the output_object list.
 */
int discover_mac_segments( storage_object_t *obj, list_anchor_t output_objects, uint *count )
{
        boolean                     success=FALSE;
        int                         rc=EINVAL;
        uint                        seg_count=0;
        seg_private_data_t         *pdata=NULL;
        disk_private_data_t        *disk_pdata=NULL;
        mac_disklabel_t            *disklabel=NULL;
        boolean                     created_disk_pdata=FALSE;
        list_element_t              iter,e;
	DISKSEG 		   *seg;

        LOG_ENTRY();
        LOG_DEBUG("examining object %s\n", obj->name );

        pdata            = (seg_private_data_t *) obj->private_data;

        // check if object is acceptable and probe for mac disk label
        if ( ( obj->plugin == mac_plugin ) ||
             ( obj->data_type != DATA_TYPE ) ||
             ( obj->object_type == SEGMENT ) ) {

                LOG_DEBUG("object is not acceptable or \n" );                

        }
        else {
                disklabel = get_mac_disk_label( obj );
                if (disklabel) {
                        display_mac_disklabel( disklabel );
                        rc = 0;
                }
        }


        if (!rc) {

                create_mac_disk_private_data( obj );

                disk_pdata = get_mac_disk_private_data(obj);

                if (disk_pdata) {

                        created_disk_pdata=TRUE;

                        seg_count = get_segments(obj,disklabel);

                        if (seg_count > 0) {

                                display_segment_list(obj);

                                rc=EngFncs->concatenate_lists( output_objects, obj->parent_objects );

                                if (!rc) {
                                        LIST_FOR_EACH( obj->parent_objects, iter, seg) {
                                                get_mac_segment_devmap_info(seg);
                                        }                                       
                                        success = TRUE;
                                }

                        }
                        else {
                                LOG_ERROR("error, mac object but no segments produced\n");
                        }

                }
                else {
                        LOG_ERROR("error, unable to create logical disk private data\n");
                }

        }

        if (success == FALSE) {

                // toss any segments we created
                prune_mac_seg_objects_from_list(obj->parent_objects);

                // place the disk storage object back onto the discovery list
                e=EngFncs->insert_thing( output_objects,
                                         obj,
                                         INSERT_AFTER,
                                         NULL );
    
                // toss our disk private data
                if (created_disk_pdata==TRUE) {
                        delete_mac_disk_private_data(obj);
                }

                // return - no objects created
                seg_count = 0;

        }

        LOG_DEBUG( "discovery ...resulted in %d new segments\n", seg_count );

        // return count of objects we created
        *count += seg_count;

        if (disklabel) free(disklabel);

        LOG_EXIT_INT(0);
        return 0;
}

