/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: fsimjfs.h
 *
 */

#include <uuid/uuid.h>

/********************
 ********************
 **                **
 **  EVMS defines  **
 **                **
 ********************
 ********************/

extern plugin_record_t jfs_plugin_record;
extern plugin_record_t *my_plugin_record;
extern engine_functions_t *EngFncs;

// file system type ... used by the SetPluginID macro
#define FS_TYPE_JFS     6

/**********************************
 **********************************
 **                              **
 **  fsck.jfs, mkfs.jfs defines  **
 **                              **
 **********************************
 **********************************/

/* fsck.jfs option array indices */
#define FSCK_FORCE_INDEX        0
#define FSCK_READONLY_INDEX     1
#define FSCK_OMITLOG_INDEX      2
#define FSCK_VERBOSE_INDEX      3
#define FSCK_VERSION_INDEX      4
#define FSCK_JFS_OPTIONS_COUNT  5

/* mkfs.jfs option array indices */
#define MKFS_CHECKBB_INDEX      0
#define MKFS_SETVOL_INDEX       1
#define MKFS_CASEINSENS_INDEX   2
#define MKFS_JOURNAL_VOL_INDEX  3
#define MKFS_SETLOGSIZE_INDEX   4
#define MKFS_JFS_OPTIONS_COUNT  5

/* fsck exit codes */
#define FSCK_OK                    0
#define FSCK_CORRECTED             1
#define FSCK_REBOOT                2
#define FSCK_ERRORS_UNCORRECTED    4
#define FSCK_OP_ERROR              8
#define FSCK_USAGE_ERROR          16

/* jfsutils_support flag values */
#define JFS_UTILS_EXIST   0x0001
#define JFS_EXTERNAL_LOG  0x0002
#define JFS_EXPAND        0x0003

/* FSIM options defines */
#define NO_SELECTION      "None"


/*******************************
 *******************************
 **                           **
 **  JFS defines and structs  **
 **                           **
 *******************************
 *******************************/

#define JFS_MAGIC 	  "JFS1"	/* Magic word: Version 1     */
#define JFS_VERSION	   2	    /* Version number: Version 2 */
#define LV_NAME_SIZE  11	

/* generic defines */
#define VOLUME_NOT_JFS         -1
#define FSIM_SUCCESS            0
#define GET                     0
#define PUT                     1

/*
 *	physical xd (pxd)
 */
typedef struct {
	unsigned  len:24;
	unsigned  addr1:8;
	u_int32_t addr2;
} pxd_t;

/*
 * Almost identical to Linux's timespec, but not quite
 */
struct timestruc_t {
	u_int32_t  tv_sec;
	u_int32_t  tv_nsec;
};

/*
 *	aggregate superblock
 */
struct superblock
{
	char      s_magic[4];	    /* 4: magic number */
	u_int32_t s_version;		/* 4: version number */

	int64_t   s_size;		    /* 8: aggregate size in hardware/LVM blocks;
				                 * VFS: number of blocks
				                 */
	int32_t   s_bsize;		    /* 4: aggregate block size in bytes;
				                 * VFS: fragment size
				                 */
	int16_t   s_l2bsize;		/* 2: log2 of s_bsize */
	int16_t   s_l2bfactor;	    /* 2: log2(s_bsize/hardware block size) */
	int32_t   s_pbsize;		    /* 4: hardware/LVM block size in bytes */
	int16_t   s_l2pbsize;		/* 2: log2 of s_pbsize */
	int16_t   pad;		        /* 2: padding necessary for alignment */

	u_int32_t s_agsize;		    /* 4: allocation group size in aggr. blocks */

	u_int32_t s_flag;		    /* 4: aggregate attributes:
				                 *    see jfs_filsys.h
				                 */
	u_int32_t s_state;		    /* 4: mount/unmount/recovery state:
				                 *    see jfs_filsys.h
				                 */
	int32_t   s_compress;		/* 4: > 0 if data compression */

	pxd_t     s_ait2;		    /* 8: first extent of secondary
				                 *    aggregate inode table
				                 */

	pxd_t     s_aim2;		    /* 8: first extent of secondary
				                 *    aggregate inode map
				                 */
	u_int32_t s_logdev;		    /* 4: device address of log */
	int32_t   s_logserial;	    /* 4: log serial number at aggregate mount */
	pxd_t     s_logpxd;		    /* 8: inline log extent */

	pxd_t     s_fsckpxd;	    /* 8: inline fsck work space extent */

	struct timestruc_t  s_time;	/* 8: time last updated */

	int32_t   s_fsckloglen;	    /* 4: Number of filesystem blocks reserved for
				                 *    the fsck service log.
				                 *    N.B. These blocks are divided among the
				                 *         versions kept.  This is not a per
				                 *         version size.
				                 *    N.B. These blocks are included in the
				                 *         length field of s_fsckpxd.
				                 */
	int8_t    s_fscklog;		/* 1: which fsck service log is most recent
				                 *    0 => no service log data yet
				                 *    1 => the first one
				                 *    2 => the 2nd one
				                 */
	char      s_fpack[11];	    /* 11: file system volume name
				                 *     N.B. This must be 11 bytes to
				                 *          conform with the OS/2 BootSector
				                 *          requirements
				                 */

	/* extendfs() parameter under s_state & FM_EXTENDFS */
	int64_t   s_xsize;		    /* 8: extendfs s_size */
	pxd_t     s_xfsckpxd;	    /* 8: extendfs fsckpxd */
	pxd_t     s_xlogpxd;	    /* 8: extendfs logpxd */
	/* - 128 byte boundary - */

	uuid_t s_uuid;		/* 16: 128-bit uuid for volume */
	char s_label[16];	/* 16: volume label */
	uuid_t s_loguuid;	/* 16: 128-bit uuid for log device */
};


/*
 *	log manager configuration parameters
 */

/* log page size */
#define	LOGPSIZE	4096
#define	L2LOGPSIZE	  12
#define LOGPAGES	  16	/* Log pages per mounted file system */

/*
 *	log superblock (block 1 of logical volume)
 */
#define	LOGSUPER_B	1
#define	LOGSTART_B	2

#define	LOGMAGIC	0x87654321
#define	LOGVERSION	1

#define MAX_ACTIVE	128	/* Max active file systems sharing log */

typedef struct {
	u_int32_t magic;	/* 4: log lv identifier */
	int32_t version;	/* 4: version number */
	int32_t serial;		/* 4: log open/mount counter */
	int32_t size;		/* 4: size in number of LOGPSIZE blocks */
	int32_t bsize;		/* 4: logical block size in byte */
	int32_t l2bsize;	/* 4: log2 of bsize */

	u_int32_t flag;		/* 4: option */
	u_int32_t state;	/* 4: state - see below */

	int32_t end;		/* 4: addr of last log record set by logredo */
	uuid_t uuid;		/* 16: 128-bit journal uuid */
	char label[16];		/* 16: journal label */
	uuid_t active[MAX_ACTIVE];	/* 2048: active file systems list */
} logsuper_t;

/* log flag: commit option (see jfs_filsys.h) */

/* log state */
#define	LOGMOUNT	0	/* log mounted by lmLogInit() */
#define LOGREDONE	1	/* log shutdown by lmLogShutdown().
				         * log redo completed by logredo().
				         */
#define LOGWRAP		2	/* log wrapped */
#define LOGREADERR	3	/* log read error detected in logredo() */

#define L2MEGABYTE      20
#define MEGABYTE        (1 << L2MEGABYTE)
#define MEGABYTE32     (MEGABYTE << 5)
#define MAX_LOG_PERCENTAGE  10              /* Log can be at most 10% of disk */

/*
 *	buffer cache configuration
 */
/* page size */
#ifdef PSIZE
#undef PSIZE
#endif
#define	PSIZE		4096	/* page size (in byte) */

/*
 *	fs fundamental size
 *
 * PSIZE >= file system block size >= PBSIZE >= DISIZE
 */
#define	PBSIZE		512	/* physical block size (in byte) */
#define DISIZE		512	/* on-disk inode size (in byte) */
#define INOSPEREXT	 32	/* number of disk inode per extent */
#define	IXSIZE		(DISIZE * INOSPEREXT)	/* inode extent size */
#define INODE_EXTENT_SIZE	    IXSIZE	    /* inode extent size */

/* Minimum number of bytes supported for a JFS partition */
#define MINJFS			(0x1000000)
#define MINJFSTEXT		"16"

/*
 * SIZE_OF_SUPER defines the total amount of space reserved on disk for the
 * superblock.  This is not the same as the superblock structure, since all of
 * this space is not currently being used.
 */
#define SIZE_OF_SUPER	PSIZE

/*
 * SIZE_OF_MAP_PAGE defines the amount of disk space reserved for each page of
 * the inode allocation map (to hold iag)
 */
#define SIZE_OF_MAP_PAGE	PSIZE

/*
 * fixed byte offset address
 */
#define SUPER1_OFF	0x8000	/* primary superblock */
#define AIMAP_OFF	(SUPER1_OFF + SIZE_OF_SUPER)
					/*
					 * Control page of aggregate inode map
					 * followed by 1st extent of map
					 */
#define AITBL_OFF	(AIMAP_OFF + (SIZE_OF_MAP_PAGE << 1))
					/*
					 * 1st extent of aggregate inode table
					 */
#define SUPER2_OFF	(AITBL_OFF + INODE_EXTENT_SIZE)
					/*
					 * secondary superblock
					 */

/*
 *	directory configuration
 */
#define JFS_NAME_MAX	255
#define JFS_PATH_MAX	BPSIZE

/*
 *	file system state (superblock state)
 */
#define FM_CLEAN 0x00000000	/* file system is unmounted and clean */
#define FM_MOUNT 0x00000001	/* file system is mounted cleanly */
#define FM_DIRTY 0x00000002	/* file system was not unmounted and clean
            				 * when mounted or
			            	 * commit failure occurred while being mounted:
				             * fsck() must be run to repair
				             */
#define	FM_LOGREDO 0x00000004	/* log based recovery (logredo()) failed:
				                 * fsck() must be run to repair
				                 */
#define	FM_EXTENDFS 0x00000008	/* file system extendfs() in progress */


/*******************
 *******************
 **               **
 **  Common code  **
 **               **
 *******************
 *******************/

int fsim_get_jfs_superblock( logical_volume_t *, struct superblock * );
//int fsim_get_log_superblock( char *, logsuper_t * );
int fsim_get_log_superblock( logical_volume_t *, logsuper_t * );
int fsim_unmkfs_jfs( logical_volume_t * );
//int fsim_unmkfs_ext_log( char * );
int fsim_unmkfs_ext_log( logical_volume_t * );
int fsim_mkfs( logical_volume_t *, option_array_t * );
int fsim_fsck( logical_volume_t *, option_array_t * );
int fsim_get_volume_limits( struct superblock *, sector_count_t *,
			                   sector_count_t *, sector_count_t * );
int fsim_test_version( void );
int Is_JFS_Log_Vol( logsuper_t * );
