/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libgpt.so
 *
 *   File: options.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "gpt.h"
#include "helpers.h"
#include "checks.h"
#include "options.h"
#include "move.h"


typedef struct gpt_partition_info_s {
        char              *name;
        partition_type_t   number;
} gpt_partition_info_t;


static const gpt_partition_info_t gpt_partition_info[] =
{
        { BASIC_DATA_PARTITION_STRING, 2},
//    {"legacy mbr", 3},
//    {"swap", 4},
//    {"ms reserved", 5},
//    {"ms ldm data", 6},
//    {"ms ldm metadata", 7},
        {NULL, 0}                    /* end-of-list */
};
#define MAX_PARTITION_INFO_ELEMENTS 3



/*
 *  Called by init_task() to allocate the create option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int allocate_move_option_descriptors( task_context_t * context )
{
        int rc; 

        LOG_ENTRY();
        rc=0;
        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to initialize the MOVE options in the task context
 *  for the specified LOGICALDISK.
 */
static int initialize_move_option_descriptors( DISKSEG *seg, task_context_t * context)
{
        int rc;

        LOG_ENTRY();
        rc=0;
        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Called by init_task() to allocate the assign option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int allocate_assign_option_descriptors( task_context_t * context )
{
        int rc = EINVAL;


        LOG_ENTRY();

        if (context) {

                context->option_descriptors->count = SEG_ASSIGN_OPTION_COUNT ;      // must set count

                // Make System Partition ?
                context->option_descriptors->option[SEG_ASSIGN_OPTION_SYSPARTITION_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_SYSPARTITION_INDEX].constraint.list = NULL;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_SYSPARTITION_INDEX].constraint_type = EVMS_Collection_None;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_SYSPARTITION_INDEX].help = NULL;
                SET_STRING( context->option_descriptors->option[SEG_ASSIGN_OPTION_SYSPARTITION_INDEX].name, SEG_ASSIGN_OPTION_SYSPARTITION_NAME );
                SET_STRING( context->option_descriptors->option[SEG_ASSIGN_OPTION_SYSPARTITION_INDEX].tip,
                            "Choose Yes if an EFI system partition should be created. Probably only useful on IA64 systems.");
                SET_STRING( context->option_descriptors->option[SEG_ASSIGN_OPTION_SYSPARTITION_INDEX].title, "System Partition");
                context->option_descriptors->option[SEG_ASSIGN_OPTION_SYSPARTITION_INDEX].type = EVMS_Type_Boolean;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_SYSPARTITION_INDEX].unit = EVMS_Unit_None;
                context->option_descriptors->option[SEG_ASSIGN_OPTION_SYSPARTITION_INDEX].value.b = FALSE;

                rc = 0;
        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to initialize the ASSIGN options in the task context
 *  for the specified LOGICALDISK.
 */
static int initialize_assign_option_descriptors( LOGICALDISK *ld, task_context_t * context)
{
        int rc=EINVAL;


        LOG_ENTRY();

        if ( ld && context ) {

                // there should be exactly SEG_ASSIGN_OPTION_COUNT options
                if ( context->option_descriptors->count == SEG_ASSIGN_OPTION_COUNT) {

                        // System Partition ?
                        context->option_descriptors->option[SEG_ASSIGN_OPTION_SYSPARTITION_INDEX].flags = 0;
                        context->option_descriptors->option[SEG_ASSIGN_OPTION_SYSPARTITION_INDEX].value.b=FALSE;

                        rc = 0;

                }
                else {
                        LOG_ERROR("error, wrong number of assign options.\n");
                        rc = EINVAL;
                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}



static int allocate_create_option_descriptors( task_context_t *context )
{
        int rc = EINVAL;


        LOG_ENTRY();

        if (context) {

                context->option_descriptors->count = SEG_CREATE_OPTION_COUNT ;      // must set count


                // Segment Size
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.list = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
                if (context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range==NULL) {
                        LOG_EXIT_INT(ENOMEM);
                        return ENOMEM;
                }
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint_type                  = EVMS_Collection_Range;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->min.ui64       = 0;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->max.ui64       = 0xffffffff;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = 1;

                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].help = NULL;
                SET_STRING( context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].name, SEG_CREATE_OPTION_SIZE_NAME );
                SET_STRING( context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].tip,"This option allows you to set the size of the segment that you are creating." );
                SET_STRING( context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].title, "Size" );
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].type = EVMS_Type_Unsigned_Int64;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].unit = EVMS_Unit_Sectors;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].value.ui64 = 0;

                // Sector Offset to start of Partition
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
                if (context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range==NULL) {
                        LOG_EXIT_INT(ENOMEM);
                        return ENOMEM;
                }
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint_type                  = EVMS_Collection_Range;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->min.ui64       = 0;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->max.ui64       = 1;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->increment.ui64 = 1;


                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE | EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].help = NULL;
                SET_STRING( context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].name, SEG_CREATE_OPTION_OFFSET_NAME );
                SET_STRING( context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].tip,
                            "Use this option if you dont want the segment you are creating to start at the beginning of freespace. "
                            "It allows you to specifiy the number of sectors to skip before creating the segment.");
                SET_STRING( context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].title, "Offset");
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].type = EVMS_Type_Unsigned_Int64;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].unit = EVMS_Unit_Sectors;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].value.ui64 = 0;


                // Partition Type
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].constraint.list =
                EngFncs->engine_alloc( sizeof(value_list_t) + sizeof(value_t) * MAX_PARTITION_INFO_ELEMENTS );

                if (context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].constraint.list) {
                        int i=0;

                        while (gpt_partition_info[i].name) {
                                SET_STRING(context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].constraint.list->value[i].s, gpt_partition_info[i].name);
                                i++;
                        }

                        context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].constraint.list->count = i;
                }
                else {
                        rc = ENOMEM;
                        LOG_EXIT_INT(rc);
                        return rc;
                }

                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].constraint_type = EVMS_Collection_List;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].help = NULL;
                SET_STRING( context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].name, SEG_CREATE_OPTION_TYPE_NAME );
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].min_len = 1;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].max_len = MAX_TYPENAME_SIZE;
                SET_STRING( context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].tip, "Select from a list of common partition types");
                SET_STRING( context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].title, "Partition Type");
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].type = EVMS_Type_String;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].unit = EVMS_Unit_None;
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].value.s = EngFncs->engine_alloc (MAX_TYPENAME_SIZE+1);

                if (context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].value.s)
                        strcpy (context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].value.s, "basic data");
                else {
                        rc = ENOMEM;
                        LOG_EXIT_INT(rc);
                        return rc;
                }

                rc = 0;
        }


        LOG_EXIT_INT(rc);
        return rc;

}
static int initialize_create_option_descriptors(LOGICALDISK *ld, DISKSEG *freespace, task_context_t *context)
{
        int                 rc=EINVAL;
        lba_t               Start;
        lba_t               End;
        sector_count_t      MaxSize;
        sector_count_t      CylinderSize=0;
        DISK_PRIVATE_DATA  *disk_pdata;


        LOG_ENTRY();

        disk_pdata = get_gpt_disk_private_data( ld );
        if (disk_pdata) {
                if (disk_pdata->signature == GPT_SEG_MGR_PDATA_SIGNATURE) {

                        // we should be able to get the cylinder size if there are no problems with disk geometry
                        CylinderSize = get_cylinder_size(ld);
                        if ( CylinderSize ) {

                                // there should be exactly SEG_CREATE_OPTION_COUNT options
                                if ( context->option_descriptors->count == SEG_CREATE_OPTION_COUNT) {
                                        rc = 0;
                                }
                                else {
                                        LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
                                }

                        }
                        else {
                                LOG_ERROR("error, failed to get the cylinder size for disk %s\n", ld->name);
                        }

                }
        }

        if (rc == 0) {

                // determine defaults based on geometry
                Start    = freespace->start;

                if ( ends_on_cylinder_boundary( ld, freespace->start + freespace->size - 1 ) == TRUE ) {
                        End = freespace->start + freespace->size - 1;
                        MaxSize  = End - Start + 1;
                }
                else if (freespace->size > CylinderSize) {
                        End = rounddown_to_cylinder_boundary( ld, freespace->start + freespace->size - 1 ) - 1;
                        MaxSize  = End - Start + 1;
                }
                else {
                        MaxSize = 0; // force failure ... cuz we cant round down this seg to cyl bdy
                }

                // if MAX_USEABLE_SIZE is at least 1 cylinder in size ... then things are Ok

                if ( MaxSize >= CylinderSize ) {

                        if (rc == 0) {

                                // Segment Size
                                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].flags = 0;
                                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->min.ui64       = CylinderSize;
                                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->max.ui64       = MaxSize;
                                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = CylinderSize;
                                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].value.ui64 = MaxSize;

                                // Sector Offset to start of Partition
                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].flags = 0;
                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].value.ui64 = 0;
                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->min.ui64       = 0;
                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->max.ui64       = MaxSize-CylinderSize;

                                // Partition Type by name
                                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].flags = 0;
                                strcpy (context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].value.s, "basic data");

                                rc = 0;
                        }

                }
                else {
                        rc = EINVAL;
                        LOG_ERROR("error trying to create a segment out of something smaller than a cylinder.\n");
                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called by init_task() to allocate the expand option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int  allocate_expand_option_descriptors( task_context_t * context )
{
        int rc = EINVAL;
        storage_object_t *obj;
        LOGICALDISK      *ld=NULL;
        DISKSEG          *freespace=NULL;
        sector_count_t    cylinder_size;


        LOG_ENTRY();

        context->option_descriptors->count = 0;

        obj = context->object;
        if ( obj ) {

                if (obj->object_type == SEGMENT) {

                        if (obj->data_type == DATA_TYPE) {

                                ld = get_logical_disk(obj);
                                if (ld) {

                                        // we increment partition sizes ... 1 cylinder at a time ... due to forced cyl allignment
                                        cylinder_size = get_cylinder_size(ld);

                                        // DATA segment better have freespace following it.
                                        freespace = get_freespace_following_gpt_disk_segment( obj );
                                        if (freespace == NULL) {
                                                rc = EINVAL;
                                                LOG_EXIT_INT(rc);
                                                return rc;
                                        }

                                        // make sure we can actually expand the DATA segment by at least 1 cylinder
                                        if ( freespace->size >= cylinder_size ) {

                                                context->option_descriptors->count = SEG_EXPAND_OPTION_COUNT;

                                                // Expanded segment size delta
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.list = NULL;
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
                                                if (context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range==NULL) {
                                                        LOG_EXIT_INT(ENOMEM);
                                                        return ENOMEM;
                                                }
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint_type = EVMS_Collection_Range;
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].flags = 0;
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].help = NULL;
                                                SET_STRING( context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].name, SEG_EXPAND_OPTION_SIZE_NAME );
                                                SET_STRING( context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].tip,
                                                            "Use this option to specify how much space to add to the segment. "
                                                            "Remember that segments grow by cylinder size amounts.");
                                                SET_STRING( context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].title, "Additional Size");
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].type = EVMS_Type_Unsigned_Int64;
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].unit = EVMS_Unit_Sectors;

                                                rc = 0;
                                        }

                                }

                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;

}


/*
 *  Called to initialize the EXPAND segment options in the task context
 *  for the specified FREESPACE segment and LOGICALDISK.
 */
static int initialize_expand_option_descriptors( LOGICALDISK *ld, DISKSEG *freespace, task_context_t * context)
{
        int                rc=EINVAL;
        int64_t            MaxSize=0;
        int64_t            MinSize=0;
        sector_count_t     cylinder_size=0;
        DISK_PRIVATE_DATA *disk_pdata;
        lba_t              freespace_end_lba;


        LOG_ENTRY();

        disk_pdata = get_gpt_disk_private_data( ld );
        if (disk_pdata) {

                if (disk_pdata->signature == GPT_SEG_MGR_PDATA_SIGNATURE) {

                        // we should be able to get the cylinder size if there are no problems with disk geometry
                        cylinder_size = get_cylinder_size(ld);
                        if (cylinder_size ) {

                                // there should be exactly SEG_EXPAND_OPTION_COUNT options
                                if ( context->option_descriptors->count == SEG_EXPAND_OPTION_COUNT) {

                                        rc = 0;

                                }
                                else {
                                        LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
                                }

                        }
                        else {
                                LOG_ERROR("error, failed to get the cylinder size for disk %s\n", ld->name);
                        }

                }
        }

        if (rc == 0) {

                // partitions end on a cylinder boundary. if freespace doesnt end on
                // a cylinder bdy then round it down to a cyl bdy before testing if
                // freespace can handle the ExpandBy.
                if (ends_on_cylinder_boundary(ld,freespace->start+freespace->size-1)) {
                        freespace_end_lba = freespace->start+freespace->size-1;
                }
                else {
                        freespace_end_lba = rounddown_to_cylinder_boundary(ld, freespace->start+freespace->size-1) - 1;
                }

                // make sure we can actually expand the DATA segment by at least 1 cylinder
                if ( (freespace_end_lba - freespace->start + 1) >= cylinder_size ) {

                        LOG_DEBUG("freespace= %s   size= %"PRIu64"\n", freespace->name, freespace->size );

                        // get minimum and maximum sizes for expanding this DATA segment
                        MaxSize  = freespace_end_lba - freespace->start + 1;
                        MinSize  = cylinder_size;

                        // ask engine how much we can expand by
                        while (MaxSize > 0) {

                                sector_count_t delta_size = MaxSize;

                                rc = EngFncs->can_expand_by(context->object, &delta_size);
                                if (rc == 0) {
                                        break;
                                }
                                else {
                                        MaxSize -= cylinder_size;
                                }
                        }

                        if (MaxSize > 0) {

                                context->option_descriptors->count = SEG_EXPAND_OPTION_COUNT;
                                LOG_DEBUG("Setting constraints ... Max= %"PRId64"   Min= %"PRId64"\n", MaxSize, MinSize );
                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range->min.ui64       = MinSize;
                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range->max.ui64       = MaxSize;
                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = cylinder_size;
                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].value.ui64 = MaxSize;
                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].flags = 0;

                                rc = 0;
                        }
                        else {
                                rc = EINVAL;
                                LOG_ERROR("error, engine said segment could not be expanded\n");
                        }

                }
                else {
                        rc = EINVAL;
                        LOG_ERROR("error trying to expand segment into freespace < 1 cylinder.\n");
                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called by init_task() to allocate the shrink option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int allocate_shrink_option_descriptors( task_context_t * context )
{
        int                rc = EINVAL;
        storage_object_t  *obj;
        LOGICALDISK       *ld;
        sector_count_t     cylinder_size;
        DISK_PRIVATE_DATA *disk_pdata;

        LOG_ENTRY();

        context->option_descriptors->count = 0;

        obj = context->object;

        if ( obj ) {

                if (obj->object_type == SEGMENT) {

                        if (obj->data_type == DATA_TYPE) {

                                ld = get_logical_disk( (DISKSEG *) obj );

                                disk_pdata = get_gpt_disk_private_data(ld);

                                if (disk_pdata) {

                                        // we shrink partition sizes ... 1 cylinder at a time ... due to forced cyl allignment
                                        cylinder_size = get_cylinder_size(ld);

                                        // make sure we can actually shrink the DATA segment while falling
                                        // on a cylinder boundary.
                                        if ( obj->size > cylinder_size ) {

                                                context->option_descriptors->count = SEG_SHRINK_OPTION_COUNT ;

                                                // Shrunken segment size
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.list = NULL;
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
                                                if (context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range==NULL) {
                                                        LOG_EXIT_INT(ENOMEM);
                                                        return ENOMEM;
                                                }
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint_type = EVMS_Collection_Range;
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].flags = 0;
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].help = NULL;
                                                SET_STRING( context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].name, SEG_SHRINK_OPTION_SIZE_NAME );
                                                SET_STRING( context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].tip,
                                                            "Use this option to specify how much space to remove from the segment. "
                                                            "Remember that segments shrink by cylinder size amounts.");
                                                SET_STRING( context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].title, "Shrink by Size" );
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].type = EVMS_Type_Unsigned_Int64;
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].unit = EVMS_Unit_Sectors;

                                                rc = 0;
                                        }

                                }

                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to initialize the SHRINK segment options in the task context
 *  for the specified FREESPACE segment and LOGICALDISK.
 */
static int initialize_shrink_option_descriptors( LOGICALDISK *ld, DISKSEG *seg, task_context_t * context)
{
        int                rc=EINVAL;
        sector_count_t     cylinder_size=0;
        DISK_PRIVATE_DATA *disk_pdata;
        sector_count_t     biggest_delta=0;
        sector_count_t     smallest_delta=0;

        LOG_ENTRY();

        disk_pdata = get_gpt_disk_private_data( ld );
        if (disk_pdata) {

                if (disk_pdata->signature == GPT_SEG_MGR_PDATA_SIGNATURE) {

                        // we should be able to get the cylinder size if there are no problems with disk geometry
                        cylinder_size = get_cylinder_size(ld);
                        if (cylinder_size ) {

                                // there should be exactly SEG_SHRINK_OPTION_COUNT options
                                if ( context->option_descriptors->count == SEG_SHRINK_OPTION_COUNT ) {
                                        rc = 0;
                                }
                                else {
                                        LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
                                }

                        }
                        else {
                                LOG_ERROR("error, failed to get the cylinder size for disk %s\n", ld->name);
                        }

                }

        }

        if (rc == 0) {

                // get minimum and maximum sizes for shrinking this DATA segment
                if ( seg->size > cylinder_size ) {

                        biggest_delta  = seg->size - cylinder_size;
                        smallest_delta = cylinder_size;

                        // ask engine how much we can shrink by ... when the
                        // engine says stop ... check if the engine will allow
                        // any shrink at all.
                        do {
                                sector_count_t delta_size = biggest_delta;

                                rc = EngFncs->can_shrink_by( seg, &delta_size );

                                if (rc == 0) {
                                        break;   // engine said Ok
                                }

                                biggest_delta -= cylinder_size;

                        } while ( biggest_delta >= smallest_delta ); // continue till engine says Ok or we cant shrink at all

                }
                else {  // cant shrink this segment !
                        rc = EINVAL;
                }


                // make sure we can actually shrink the DATA segment
                if ( rc == 0 ) {

                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range->min.ui64       = smallest_delta;
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range->max.ui64       = biggest_delta;
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = cylinder_size;
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].flags = 0;
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].value.ui64 = smallest_delta;

                        rc = 0;

                }


        }


        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Called to set and validate an option for a CREATE task.
 */
static int set_assign_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
        int rc = EINVAL;


        LOG_ENTRY();

        switch (index) {
        
        case SEG_ASSIGN_OPTION_SYSPARTITION_INDEX:

                if ( (value->uc == TRUE) || (value->uc==FALSE) ) {
                        context->option_descriptors->option[index].value.b = value->b;
                        rc = 0;
                }
                else {
                        context->option_descriptors->option[index].value.b = FALSE;
                        *effect |= EVMS_Effect_Inexact;
                        rc = 0;
                }
                break;

        default:
                LOG_ERROR("index is unknown or unsupported\n");
                rc = EINVAL;
                break;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to set and validate an option for a CREATE task.
 */
static int set_create_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
        int                rc = EINVAL,i=0;
        storage_object_t  *obj=NULL;
        LOGICALDISK       *ld=NULL;
        DISK_PRIVATE_DATA *disk_pdata=NULL;
        sector_count_t     MaxSize = 0;
        sector_count_t     size = 0;
        lba_t              start_lba = 0;
        lba_t              end_lba = 0;
        sector_count_t     CylinderSize=0;


        LOG_ENTRY();

        obj = EngFncs->first_thing( context->selected_objects, NULL );
        if (obj) {

                if (obj->data_type == META_DATA_TYPE) {
                        rc = ENODEV;
                        LOG_ERROR("error, i do not create segs from metadata type segments\n");
                        LOG_EXIT_INT(rc);
                        return rc;
                }

                // get disk info
                ld = get_logical_disk(obj);
                if (ld == NULL) {
                        rc = ENODEV;
                        LOG_ERROR("error, no disk info available\n");
                        LOG_EXIT_INT(rc);
                        return rc;
                }
                else {
                        disk_pdata = get_gpt_disk_private_data(ld);
                        if (disk_pdata == NULL) {
                                rc = ENODATA;
                                LOG_ERROR("error, no disk private data available\n");
                                LOG_EXIT_INT(rc);
                                return rc;
                        }
                }


                CylinderSize = get_cylinder_size(ld);

                start_lba    = obj->start;
                size         = obj->size;

                if ( ends_on_cylinder_boundary( ld, obj->start + obj->size - 1 ) == TRUE ) {
                        end_lba = obj->start + obj->size - 1;
                        MaxSize = end_lba - start_lba + 1;
                }
                else if (obj->size > CylinderSize) {
                        end_lba = rounddown_to_cylinder_boundary( ld, obj->start + obj->size - 1 ) - 1;
                        MaxSize = end_lba - start_lba + 1;
                }
                else {
                        rc = EINVAL; // force failure ... cuz we cant round down this seg to cyl bdy
                        LOG_ERROR("error, freespace is too small\n");
                        LOG_EXIT_INT(rc);
                        return rc;
                }

        }
        else {
                rc = ENODEV;
                LOG_ERROR("error, no disk segment found in the selected_objects list\n");
                LOG_EXIT_INT(rc);
                return rc;
        }

        switch (index) {
        
        case SEG_CREATE_OPTION_SIZE_INDEX:
                {

                        sector_count_t segoffset = context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].value.ui64;

                        // SIZE == 0 ... ERROR
                        if ( value->ui64 == 0) {
                                *effect |= EVMS_Effect_Inexact ;
                                rc = EINVAL;
                        }
                        // if OFFSET + SIZE < MAX_USEABLE_FREESPACE_SIZE ... then SIZE is OK
                        else if ( value->ui64+segoffset <= MaxSize ) {

                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->max.ui64 = MaxSize - value->ui64;
                                *effect |= EVMS_Effect_Reload_Options;
                                context->option_descriptors->option[index].value.ui64 = value->ui64;
                                rc = 0;

                        }
                        // user is trying to create something that runs off end
                        // of the useable area of the freespace segment.
                        else {

                                if (segoffset > 0) { // OFFSET and SIZE both need to be examined
                                                     // to determine how to fix things up.

                                        if (segoffset > obj->size) {     // OFFSET is toooo big ... ERROR

                                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->max.ui64 = MaxSize - value->ui64;
                                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].value.ui64 = MaxSize - value->ui64;
                                                context->option_descriptors->option[index].value.ui64 = value->ui64;

                                                *effect |= EVMS_Effect_Reload_Options ;
                                                rc = 0;

                                        }
                                        else {   // OFFSET is OK ... perhaps big ... so reduce SIZE
                                                 // and let the user figure things out.

                                                sector_count_t newsize = ((obj->size - segoffset)/CylinderSize)*CylinderSize;

                                                context->option_descriptors->option[index].constraint.range->max.ui64 = newsize;
                                                context->option_descriptors->option[index].value.ui64 = newsize;
                                                *effect |= EVMS_Effect_Reload_Options ;
                                                rc = 0;
                                        }

                                }
                                else {  // OFFSET == 0 ... so SIZE is the only option we need
                                        // to adjust downward for user.
                                        context->option_descriptors->option[index].value.ui64 = MaxSize;
                                        *effect |= EVMS_Effect_Inexact;
                                        rc = 0;
                                }

                        }


                }

                break;

        case SEG_CREATE_OPTION_OFFSET_INDEX:

                {
                        sector_count_t segsize = context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].value.ui64;

                        // if OFFSET + SIZE < MAX_USEABLE_FREESPACE_SIZE ... then OFFSET is OK
                        if ( value->ui64+segsize <= MaxSize ) {

                                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->max.ui64 = MaxSize - value->ui64;
                                context->option_descriptors->option[index].value.ui64 = value->ui64;
                                *effect |= EVMS_Effect_Reload_Options ;
                                rc = 0;
                        }
                        else {

                                context->option_descriptors->option[index].constraint.range->max.ui64 = MaxSize - segsize;
                                context->option_descriptors->option[index].value.ui64 = MaxSize - segsize;
                                *effect |= EVMS_Effect_Reload_Options  ;
                                rc = 0;
                        }

                }

                break;

        case SEG_CREATE_OPTION_TYPE_INDEX:

                while ( gpt_partition_info[i].name ) {
                        if (strcasecmp (gpt_partition_info[i].name, value->s) == 0) {
                                strncpy ( context->option_descriptors->option[index].value.s,
                                          value->s,
                                          MAX_TYPENAME_SIZE);
                                *effect |= EVMS_Effect_Reload_Options ;
                                rc = 0;
                                break;
                        }

                        i++;
                }
                break;

        default:
                LOG_ERROR("error, index is unknown or unsupported\n");
                rc = EINVAL;
                break;
        }


        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Called to set and validate an option for a EXPAND task.
 */
static int set_expand_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
        int               rc=EINVAL;
        storage_object_t *obj=NULL;
        LOGICALDISK      *ld=NULL;
        DISKSEG          *freespace=NULL;
        DISKSEG          *seg=NULL;
        sector_count_t    cylinder_size;
        lba_t             end_lba = 0;
        sector_count_t    expand_sectors=0;

        LOG_ENTRY();

        if (index == SEG_EXPAND_OPTION_SIZE_INDEX ) {

                obj = EngFncs->first_thing( context->selected_objects, NULL );
                if ( obj ) {

                        if (obj->object_type == SEGMENT) {

                                if (obj->data_type == FREE_SPACE_TYPE) {

                                        seg = context->object;

                                        ld = get_logical_disk( (DISKSEG *) seg );

                                        if (ld) {

                                                freespace = get_freespace_following_gpt_disk_segment( seg );

                                                if ( freespace == obj ) {

                                                        // calc the minimum number of sectors we need for a partition
                                                        cylinder_size = get_cylinder_size(ld);

                                                        // get expand sector count
                                                        expand_sectors = value->ui64;

                                                        LOG_DEBUG("Info ...\n");
                                                        LOG_DEBUG("     segment  = %s\n", obj->name );
                                                        LOG_DEBUG("     start    = %"PRIu64"\n", obj->start );
                                                        LOG_DEBUG("     end      = %"PRIu64"\n", obj->start+obj->size-1);
                                                        LOG_DEBUG("     size     = %"PRIu64"\n", obj->size );
                                                        LOG_DEBUG("     expand sectors = %"PRIu64"\n", expand_sectors );
                                                        LOG_DEBUG("     cylinder size  = %"PRIu64"\n", cylinder_size );

                                                        // we can only expand if in cylinder size amounts so the freespace
                                                        // following the expanding segment must be at least 1 cylinder.
                                                        // and caller should be asking for at least a cylinder
                                                        if ( ( freespace->size >= cylinder_size ) &&
                                                             ( expand_sectors >= cylinder_size ) &&
                                                             ( expand_sectors <= freespace->size ) ) {

                                                                // do cylinder allignment
                                                                end_lba = seg->start + seg->size + expand_sectors - 1;
                                                                if (ends_on_cylinder_boundary(ld, end_lba) == FALSE) {
                                                                        end_lba = roundup_to_cylinder_boundary( ld, end_lba );
                                                                }

                                                                // adjust downwards if too big ...
                                                                if ( end_lba > (freespace->start + freespace->size - 1) ) {
                                                                        end_lba = roundup_to_cylinder_boundary(ld, end_lba - cylinder_size );
                                                                }

                                                                // NOW ... test if we can expand the data seg into the freespace area
                                                                if ( ( end_lba > freespace->start ) &&
                                                                     ( end_lba <= freespace->start+freespace->size - 1) ) {

                                                                        // calc actual expand sector count
                                                                        expand_sectors = end_lba - freespace->start + 1;

                                                                        if (value->ui64 != expand_sectors) {
                                                                                *effect |= EVMS_Effect_Inexact;
                                                                        }

                                                                        context->option_descriptors->option[index].value.ui64 = expand_sectors;
                                                                        rc = 0;
                                                                }
                                                                else {
                                                                        LOG_ERROR("error, cannot expand segment ... still passes end of freespace\n");
                                                                }
                                                        }
                                                        else {
                                                                LOG_ERROR("error, invalid expand sector count\n");
                                                        }
                                                }
                                                else {
                                                        LOG_ERROR("error, selected freespace does not immediately follow data segment\n");
                                                }
                                        }
                                        else {
                                                LOG_ERROR("error, no logical drive found for segment\n");
                                        }
                                }
                                else {
                                        LOG_ERROR("error, selected freespace is NOT a freespace segment\n");
                                }
                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Called to set and validate an option for a SHRINK task.
 */
static int set_shrink_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
        int               rc=EINVAL;
        storage_object_t *obj=NULL;
        LOGICALDISK      *ld=NULL;
        sector_count_t    cylinder_size;
        sector_count_t    shrink_sector_count=0;
        lba_t             end_lba=0;


        LOG_ENTRY();

        if (index == SEG_SHRINK_OPTION_SIZE_INDEX ) {

                obj = EngFncs->first_thing( context->selected_objects, NULL );
                if ( obj ) {

                        if (obj->object_type == SEGMENT) {

                                if (obj->data_type == DATA_TYPE) {

                                        ld = get_logical_disk( (DISKSEG *) obj );

                                        if ( ld ) {

                                                // get option value
                                                shrink_sector_count = value->ui64;

                                                // calc the minimum number of sectors we need for a partition
                                                cylinder_size = get_cylinder_size(ld);

                                                LOG_DEBUG("Info ...\n");
                                                LOG_DEBUG("     segment  = %s\n", obj->name );
                                                LOG_DEBUG("     start    = %"PRIu64"\n", obj->start );
                                                LOG_DEBUG("     end      = %"PRIu64"\n", obj->start+obj->size-1);
                                                LOG_DEBUG("     size     = %"PRIu64"\n", obj->size );
                                                LOG_DEBUG("     shrink sectors = %"PRIu64"\n", shrink_sector_count );
                                                LOG_DEBUG("     cylinder size  = %"PRIu64"\n", cylinder_size );

                                                // we can only shrink in cylinder size amounts so the shrink
                                                // sectors must be >= 1 cylinder.
                                                if ( ( shrink_sector_count < obj->size ) &&
                                                     ( shrink_sector_count >= cylinder_size ) ) {

                                                        // do cylinder allignment
                                                        end_lba = obj->start + (obj->size - shrink_sector_count) - 1;
                                                        if (ends_on_cylinder_boundary(ld, end_lba) == FALSE) {
                                                                end_lba = rounddown_to_cylinder_boundary( ld, end_lba-1 ) - 1;
                                                        }

                                                        // adjust upwards if we shrunk too small ...
                                                        if ( end_lba <= obj->start ) {
                                                                end_lba = roundup_to_cylinder_boundary(ld, end_lba + 1 );
                                                        }

                                                        // final test if we can shrink the segment
                                                        if (  ( end_lba > obj->start ) &&
                                                              ( end_lba < (obj->start+obj->size-1)) ) {

                                                                shrink_sector_count = obj->size - (end_lba - obj->start + 1);

                                                                if ( value->ui64 != shrink_sector_count ) {
                                                                        value->ui64 = shrink_sector_count;
                                                                        *effect |= EVMS_Effect_Inexact;
                                                                }

                                                                context->option_descriptors->option[index].value.ui64 = shrink_sector_count;
                                                                rc = 0;
                                                        }
                                                        else {
                                                                LOG_ERROR("error, invalid shrink sector count after cylinder rounding\n");
                                                        }
                                                }
                                                else {
                                                        LOG_ERROR("error, invalid shrink sector count\n");
                                                }
                                        }
                                        else {
                                                LOG_ERROR("error, logical drive is missing for this segment\n");
                                        }
                                }
                                else {
                                        LOG_ERROR("error, segment is NOT a data segment\n");
                                }

                        }
                        else {
                                LOG_ERROR("error, not a SEGMENT object type\n");
                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to set and validate an option for a MOVE task.
 */
static int set_move_option( task_context_t * context,
                            u_int32_t        index,
                            value_t        * value,
                            task_effect_t  * effect )
{
        int               rc=EINVAL;
        DISKSEG          *seg=NULL;     
        LOGICALDISK      *ld=NULL;


        LOG_ENTRY();

        if (index == SEG_MOVE_OPTION_INDEX ) {

                seg = EngFncs->first_thing( context->selected_objects, NULL );
                if ( seg ) {

                        ld = get_logical_disk( seg );

                        if ( ld != NULL && 
                             seg->object_type == SEGMENT &&
                             seg->data_type   == DATA_TYPE ) {

                                if ( strlen(value->s) > 0 ||
                                     strlen(value->s) <= EVMS_NAME_SIZE ) {

                                        strncpy( context->option_descriptors->option[index].value.s, 
                                                 value->s, 
                                                 EVMS_NAME_SIZE );  
                                        rc = 0;

                                }

                        }
                        else {
                                LOG_ERROR("error, segment cannot be moved.\n");
                        }

                }
                else {
                        LOG_ERROR("error, no object in selected object list.\n");
                }                   
        }
        else {
                LOG_ERROR("error, invalid move option index.\n");
        }

        LOG_EXIT_INT(rc);
        return rc;
}




/*
 *  Get the user selected object from the assign context and init the option descriptors
 *  as appropriate for the object.
 *
 *  Note ... if there is more than 1 object in the source object list we will simply
 *           take the first good object and decline the remainder.  That is the reason for
 *           the boolean flag found_good_object.
 */
static int set_assign_object( task_context_t * context,
                              list_anchor_t          declined_objects,
                              task_effect_t  * effect )
{
        int                 rc=EINVAL;
        storage_object_t   *obj;
        uint                segment_count;
        int                 found_good_object = FALSE;
        declined_object_t  *declined_object=NULL;
        list_element_t      iter,e;

        LOG_ENTRY();

        LIST_FOR_EACH(context->selected_objects, iter, obj ) {

                segment_count = EngFncs->list_count(obj->parent_objects);

                if ( ( found_good_object == FALSE) &&
                     ( obj->object_type  == DISK || obj->object_type == SEGMENT ) &&
                     ( obj->volume       == NULL ) &&
                     ( segment_count     == 0 ) ) {

                        found_good_object = TRUE;

                        rc = initialize_assign_option_descriptors(obj, context);

                        if (rc == 0) {
                                *effect |=  EVMS_Effect_Reload_Options;
                        }

                }
                else {

                        declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                                declined_object->object = obj;
                                declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                                e = EngFncs->insert_thing( declined_objects,
                                                           declined_object,
                                                           INSERT_AFTER,
                                                           NULL ); 
                                if (e!=NULL) {
                                        rc = 0;
                                        *effect |=  EVMS_Effect_Reload_Objects;
                                }
                                else {
                                        rc = EPERM;
                                        EngFncs->engine_free(declined_object);
                                }

                        }
                        else {
                                LOG_ERROR("error, unable to malloc a declined object struct\n");
                                rc = ENOMEM;
                        }

                }





        };

        LOG_EXIT_INT(rc);
        return rc;
}







/*
 *  Get the user selected object from the create context and init the option descriptors
 *  as appropriate for the object.
 *
 *  Note ... if there is more than 1 object in the source object list we will simply
 *           take the first good object and decline the remainder.  That is the reason for
 *           the boolean flag found_good_object.
 */
static int set_create_object( task_context_t * context,
                              list_anchor_t          declined_objects,
                              task_effect_t  * effect )
{
        int                 rc=EINVAL;
        storage_object_t   *obj;
        LOGICALDISK        *ld;
        int                 found_good_object = FALSE;
        declined_object_t  *declined_object=NULL;
        list_element_t      iter,e;

        LOG_ENTRY();

        LIST_FOR_EACH(context->selected_objects, iter, obj ) {


                if ( ( obj->data_type == FREE_SPACE_TYPE ) &&
                     ( obj->plugin    == gpt_plugin_record_ptr) &&
                     ( found_good_object == FALSE ) ) {

                        ld = get_logical_disk(obj);

                        if (ld) {

                                found_good_object = TRUE;
                                rc = initialize_create_option_descriptors(ld, obj, context);

                                if (rc == 0) {
                                        *effect |=  EVMS_Effect_Reload_Options;
                                }

                        }
                        else {
                                rc = ENODEV;
                        }

                }
                else {

                        declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                                declined_object->object = obj;
                                declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                                e = EngFncs->insert_thing( declined_objects,
                                                           declined_object,
                                                           INSERT_AFTER,
                                                           NULL ); 
                                if (e!=NULL) {
                                        rc = 0;
                                        *effect |=  EVMS_Effect_Reload_Objects;
                                }
                                else {
                                        rc = EPERM;
                                        EngFncs->engine_free(declined_object);
                                }

                        }
                        else {
                                LOG_ERROR("unable to malloc a declined object struct\n");
                                rc = ENOMEM;
                        }

                }


        }


        LOG_EXIT_INT(rc);
        return rc;
}



/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 *
 * Context->object == expansion point
 * Context->selected_objects->object == freespace segment
 * Context->acceptable_objects->object == freespace segment chosen during inittask time
 *
 */
static int set_expand_object( task_context_t * context,
                              list_anchor_t          declined_objects,
                              task_effect_t  * effect )
{
        int  rc=EINVAL;
        storage_object_t  *obj;
        storage_object_t  *freespace;
        LOGICALDISK       *ld;
        boolean            found_good_object = FALSE;
        declined_object_t *declined_object=NULL;
        list_element_t      iter,e;

        LOG_ENTRY();

        LIST_FOR_EACH(context->selected_objects, iter, obj ) {

                if ( ( obj->data_type == FREE_SPACE_TYPE ) &&
                     ( obj->plugin    == gpt_plugin_record_ptr) &&
                     ( found_good_object == FALSE ) ) {

                        freespace = get_freespace_following_gpt_disk_segment( context->object );

                        ld = get_logical_disk(obj);

                        if (ld && freespace) {

                                found_good_object = TRUE;

                                // is selected object correct?
                                if (freespace == obj) {

                                        rc = initialize_expand_option_descriptors(ld, obj, context);

                                        if (rc == 0) {
                                                *effect |= EVMS_Effect_Reload_Options;
                                        }

                                }
                                else {
                                        LOG_ERROR("error, selected object is not freespace that follows the data segment\n");
                                        rc = EINVAL;
                                }

                        }
                        else {
                                rc = ENODEV;
                        }

                }
                else {

                        declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                                declined_object->object = obj;
                                declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                                e = EngFncs->insert_thing( declined_objects,
                                                           declined_object,
                                                           INSERT_AFTER,
                                                           NULL ); 
                                if (e!=NULL) {
                                        rc = 0;
                                        *effect |=  EVMS_Effect_Reload_Objects;
                                }
                                else {
                                        rc = EPERM;
                                        EngFncs->engine_free(declined_object);
                                }

                        }
                        else {
                                LOG_ERROR("error, unable to malloc a declined object struct\n");
                                rc = ENOMEM;
                        }

                }


        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int set_shrink_object( task_context_t * context,
                              list_anchor_t          declined_objects,
                              task_effect_t  * effect )
{
        int  rc=EINVAL;
        storage_object_t  *obj;
        LOGICALDISK       *ld;
        boolean            found_good_object = FALSE;
        declined_object_t *declined_object=NULL;
        list_element_t      iter,e;

        LOG_ENTRY();

        LIST_FOR_EACH(context->selected_objects, iter, obj ) {

                if ( ( obj->data_type == DATA_TYPE ) &&
                     ( obj->plugin    == gpt_plugin_record_ptr) &&
                     ( found_good_object == FALSE ) ) {

                        ld = get_logical_disk(obj);

                        if (ld) {

                                found_good_object = TRUE;

                                rc = initialize_shrink_option_descriptors(ld, obj, context);

                                if (rc == 0) {
                                        *effect |=  EVMS_Effect_Reload_Options;
                                }

                        }
                        else {
                                rc = ENODEV;
                        }

                }
                else {

                        declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                                declined_object->object = obj;
                                declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                                e = EngFncs->insert_thing( declined_objects,
                                                           declined_object,
                                                           INSERT_AFTER,
                                                           NULL ); 
                                if (e!=NULL) {
                                        rc = 0;
                                        *effect |=  EVMS_Effect_Reload_Objects;
                                }
                                else {
                                        rc = EPERM;
                                        EngFncs->engine_free(declined_object);
                                }

                        }
                        else {
                                LOG_ERROR("unable to malloc a declined object struct\n");
                                rc = ENOMEM;
                        }

                }
        }


        LOG_EXIT_INT(rc);
        return rc;
}




/*
 *  Function: set move object
 *
 *  Called by set_object() to validate the task selected 
 *  object.
 */
static int set_move_object( task_context_t * context,
                            list_anchor_t          declined_objects,
                            task_effect_t  * effect )
{
        int  rc=EINVAL;
        DISKSEG           *seg;
        boolean            found_good_object = FALSE;
        declined_object_t *declined_object=NULL;
        list_element_t      iter,e;

        LOG_ENTRY();

        LIST_FOR_EACH(context->selected_objects, iter, seg ) {

                if ( found_good_object == FALSE ) {

                        found_good_object = TRUE;

                        rc = initialize_move_option_descriptors(seg, context);

                        if (rc == 0) {
                                *effect |=  EVMS_Effect_Reload_Options;                                 
                        }

                }
                else {

                        declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                                declined_object->object = seg;
                                declined_object->reason = EINVAL;

                                e = EngFncs->insert_thing( declined_objects,
                                                           declined_object,
                                                           INSERT_AFTER,
                                                           NULL ); 
                                if (e!=NULL) {
                                        rc = 0;
                                        *effect |=  EVMS_Effect_Reload_Objects;
                                }
                                else {
                                        rc = EPERM;
                                        EngFncs->engine_free(declined_object);
                                }

                        }
                        else {
                                LOG_ERROR("unable to malloc a declined object struct\n");
                                rc = ENOMEM;
                        }

                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}



/*
 * Validate the objects in the selected_objects list in the task context.
 * Remove from the selected objects lists any objects which are not
 * acceptable.
 *
 * For unacceptable objects, create a declined_handle_t structure with the
 * reason why it is not acceptable, and add it to the declined_objects list.
 * Modify the accepatble_objects list in the task context as necessary
 * based on the selected objects and the current settings of the options.
 *
 * Modify any option settings as necessary based on the selected objects.
 * Return the appropriate task_effect_t settings if the object list(s),
 * minimum or maximum objects selected, or option settings have changed.
 */
int GPT_SetObjects( task_context_t * context,
                    list_anchor_t          declined_objects,
                    task_effect_t  * effect )
{

        int rc = EINVAL;

        LOG_ENTRY();

        if (context) {

                switch (context->action) {
                
                case EVMS_Task_Create:

                        rc = set_create_object( context, declined_objects, effect );
                        break;

                case  EVMS_Task_Assign_Plugin:

                        rc = set_assign_object( context, declined_objects, effect );
                        break;

                case EVMS_Task_Expand:

                        rc = set_expand_object( context, declined_objects, effect );
                        break;

                case EVMS_Task_Shrink:

                        rc = set_shrink_object( context, declined_objects, effect );
                        break;

                case EVMS_Task_GPT_Move_Segment:

                        rc = set_move_object( context, declined_objects, effect );
                        break;


                default:

                        LOG_ERROR("context->action is unknown or unsupported\n");
                        break;
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}




/*
 *  Called from get_acceptable_assign_objects, once for each
 *  storage object we got back from the kernel. The purpose
 *  is to weed out any storage object whose hardsector size
 *  is NOT 512 bytes. This will be fixed shortly.
 */
static void prune_acceptable_disks( list_anchor_t list )
{
        storage_object_t  *obj;
        list_element_t iter, iter2;
        SEG_PRIVATE_DATA  *pdata=NULL;
        boolean prune;

        LIST_FOR_EACH_SAFE( list, iter, iter2, obj ) {

                prune = TRUE;

                if ( obj->size > get_cylinder_size(obj) ) {

                        if ( obj->plugin != gpt_plugin_record_ptr ) {

                                if (obj->object_type == DISK) {
                                        prune = FALSE;  
                                }
                                else if (obj->object_type == SEGMENT) {

                                        pdata = (SEG_PRIVATE_DATA *)obj->private_data;

                                        if (pdata && !(pdata->cflags & SEG_CFLAG_TOP_SEGMENT) ) {
                                                prune = FALSE;
                                        }

                                }

                        }

                }

                if (prune==TRUE) EngFncs->delete_element(iter);
        }

}


/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int get_acceptable_assign_objects( task_context_t * context )
{
        int   rc = EINVAL;


        LOG_ENTRY();

        if ( EngFncs->list_count(context->acceptable_objects) == 0) {

                rc = EngFncs->get_object_list( DISK | SEGMENT,
                                               DATA_TYPE,
                                               NULL,
                                               NULL,
                                               TOPMOST,
                                               &context->acceptable_objects );

                if (rc == 0) {

                        if ( EngFncs->list_count(context->acceptable_objects) > 0 ) {
                                prune_acceptable_disks(context->acceptable_objects);
                        }
                        else {
                                rc = EINVAL;
                                LOG_DEBUG("no storage objects returned by get_object_list call\n");
                        }
                }

        }
        else {
                LOG_ERROR("context already has acceptable objects\n");
        }


        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Called from get_acceptable_create_objects, once for each freespace
 *  segment we got back from the kernel. The purpose is to weed out
 *  any FREESPACE segment that we cannot create a DATA segment from.
 *  This is so we only present FREESPACE segments that are truly acceptable
 *  for CREATE.
 */
static void prune_small_segs( list_anchor_t list )
{
        LOGICALDISK       *ld;
        DISK_PRIVATE_DATA *disk_pdata;
        DISKSEG           *seg;
        sector_count_t     CylinderSize;
        lba_t              Start;
        lba_t              End;
        sector_count_t     MaxSize;
        boolean            prune;
        list_element_t     iter,iter2;

        LIST_FOR_EACH_SAFE( list, iter, iter2, seg ) {

                prune=TRUE;

                ld = get_logical_disk(seg);
                if (ld) {

                        disk_pdata = get_gpt_disk_private_data(ld);

                        CylinderSize = get_cylinder_size(ld);

                        Start = seg->start;

                        if ( ends_on_cylinder_boundary( ld, seg->start + seg->size - 1 ) == TRUE ) {
                                End = seg->start + seg->size - 1;
                        }
                        else if (seg->size > CylinderSize) {
                                End = rounddown_to_cylinder_boundary( ld, seg->start + seg->size - 1 ) - 1;
                        }
                        else {
                                End = Start; // force failure ... cuz we cant round down this seg to cyl bdy
                        }

                        if (Start < End) {

                                MaxSize  = End - Start + 1;

                                // if this seg is at least 1 cylinder (minus a track) in size then it is Ok
                                // because they can start on track boundaries.
                                if ( MaxSize >= CylinderSize ) {
                                        prune = FALSE;  
                                }

                        }

                }

                if (prune==TRUE) EngFncs->delete_element(iter);
        }

}



/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int get_acceptable_create_objects( task_context_t * context )
{
        int rc = EINVAL;

        LOG_ENTRY();

        if (EngFncs->list_count(context->acceptable_objects) == 0) {

                rc = EngFncs->get_object_list( SEGMENT,
                                               FREE_SPACE_TYPE,
                                               gpt_plugin_record_ptr,
                                               NULL,
                                               VALID_INPUT_OBJECT,
                                               &context->acceptable_objects );

                if (rc == 0) {
                        prune_small_segs(context->acceptable_objects);
                }

        }
        else {
                LOG_ERROR("context already has acceptable objects\n");
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int get_acceptable_expand_objects( task_context_t * context )
{
        int   rc = EINVAL;
        storage_object_t  *freespace;
        LOGICALDISK       *ld;
        sector_count_t     cylinder_size;
        lba_t              end_lba;
        list_element_t  e;

        LOG_ENTRY();

        if ( ( context ) &&
             ( context->acceptable_objects ) &&
             ( context->selected_objects ) &&
             ( context->object ) ) {

                if ( EngFncs->list_count(context->acceptable_objects) == 0 ) {   // cant start with any

                        freespace = get_freespace_following_gpt_disk_segment( context->object );
                        if ( freespace ) {

                                ld = get_logical_disk(context->object);
                                if (ld) {

                                        cylinder_size = get_cylinder_size(ld);

                                        if (freespace->size >= cylinder_size) {

                                                end_lba = roundup_to_cylinder_boundary(ld, freespace->start + 1);

                                                if ( end_lba <= freespace->start + freespace->size - 1) {

                                                        e = EngFncs->insert_thing( context->acceptable_objects,
                                                                                   freespace,
                                                                                   INSERT_AFTER | EXCLUSIVE_INSERT,
                                                                                   NULL );
                                                        if (e!=NULL) {
                                                                rc = 0;
                                                        }
                                                        else {
                                                                rc = EPERM;
                                                        }
                                                }

                                        }

                                }

                        }

                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}



/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int get_acceptable_shrink_objects( task_context_t * context )
{
        int   rc = EINVAL;
        list_element_t e;

        LOG_ENTRY();

        if ( (context) &&
             (context->object) &&
             (context->acceptable_objects) ) {

                if ( EngFncs->list_count(context->acceptable_objects) == 0 ) {

                        e = EngFncs->insert_thing( context->acceptable_objects,
                                                   context->object,
                                                   INSERT_AFTER | EXCLUSIVE_INSERT,
                                                   NULL );
                        if (e!=NULL) {
                                rc = 0;
                        }
                        else {
                                rc = EPERM;
                        }

                }
                else {
                        LOG_ERROR("error, context already has acceptable objects\n");
                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Function: prune unmoveable segs
 *
 *  Called by get_acceptable_move_objects() to prune
 *  any segment that cannot be moved from a list of 
 *  segments.
 */
static void prune_invalid_move_targets( DISKSEG *src, list_anchor_t list )
{
        LOGICALDISK *ld1=NULL;
        LOGICALDISK *ld2=NULL;
        list_element_t iter, iter2;
        int rc=0;
        boolean prune;
        DISKSEG *trg;

        ld1 = get_logical_disk(src);

        LIST_FOR_EACH_SAFE( list, iter, iter2, trg ) {

                prune = TRUE;

                ld2 = get_logical_disk(trg);

                if ( ld1 != NULL && ld1 == ld2 ) {
                        rc = gpt_validate_move_target(src,trg);
                        if (rc == 0) {
                                prune=FALSE; 
                        }
                }

                if (prune==TRUE) EngFncs->delete_element(iter);
        }
        
}


/*
 *  Function: get acceptable move objects
 *
 *  Called when setting up a move task to go out and find all the
 *  data segments that we could move on their disks. 
 */
static int get_acceptable_move_objects( task_context_t * context )
{
        int   rc = EINVAL;

        LOG_ENTRY();

        if ( (context) &&
             (context->object) &&
             (context->acceptable_objects) ) {

                if ( EngFncs->list_count(context->acceptable_objects) == 0 == 0) {

                        rc = EngFncs->get_object_list( SEGMENT,
                                                       FREE_SPACE_TYPE,
                                                       NULL,
                                                       NULL,
                                                       TOPMOST,
                                                       &context->acceptable_objects );

                        if (rc == 0) {
                                prune_invalid_move_targets( context->object, context->acceptable_objects );
                        }

                }
                else {
                        LOG_ERROR("error, context already has acceptable objects\n");
                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Initialize a new task context by allocating the option descriptor
 *  array and by getting acceptable objects, from the engine, for the
 *  task context, i.e. create, shrink, expand, assign ...
 */
int GPT_InitTask(task_context_t * context)
{
        int rc = EINVAL;

        LOG_ENTRY();

        if (context) {

                switch (context->action) {
                
                case EVMS_Task_Create:

                        context->min_selected_objects = 1;
                        context->max_selected_objects = 1;

                        rc = allocate_create_option_descriptors( context );
                        if (rc == 0) {
                                rc = get_acceptable_create_objects( context );
                        }

                        break;

                case EVMS_Task_Assign_Plugin:

                        context->min_selected_objects = 1;
                        context->max_selected_objects = 1;

                        rc = allocate_assign_option_descriptors( context );
                        if (rc == 0) {
                                rc = get_acceptable_assign_objects( context );
                        }

                        break;


                case EVMS_Task_Expand:

                        context->min_selected_objects = 1;
                        context->max_selected_objects = 1;

                        rc = allocate_expand_option_descriptors( context );
                        if (rc == 0) {
                                rc = get_acceptable_expand_objects( context );
                        }
                        break;

                case EVMS_Task_Shrink:

                        context->min_selected_objects = 1;
                        context->max_selected_objects = 1;

                        rc = allocate_shrink_option_descriptors( context );
                        if (rc == 0) {
                                rc = get_acceptable_shrink_objects( context );
                        }
                        break;

                case EVMS_Task_GPT_Move_Segment:

                        context->min_selected_objects = 1;
                        context->max_selected_objects = 1;

                        rc = allocate_move_option_descriptors( context );
                        if (rc == 0) {
                                rc = get_acceptable_move_objects( context );
                        }
                        break;

                default:
                        LOG_ERROR("error, context->action is unknown or unsupported\n");
                        break;
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Returns count of options for specified task
 */
int GPT_GetOptionCount(task_context_t * task)
{
        int count;

        LOG_ENTRY();

        switch (task->action) {
        
        case EVMS_Task_Create:
                count = SEG_CREATE_OPTION_COUNT;
                break;

        case EVMS_Task_Assign_Plugin:
                count = SEG_ASSIGN_OPTION_COUNT;
                break;

        case EVMS_Task_Expand:
                count = SEG_EXPAND_OPTION_COUNT;
                break;

        case EVMS_Task_Shrink:
                count = SEG_SHRINK_OPTION_COUNT;
                break;

        case EVMS_Task_GPT_Move_Segment:
                count = SEG_MOVE_OPTION_COUNT;
                break;

        default:
                count = 0;
                break;
        }

        LOG_EXIT_INT(count);
        return count;
}



/*
 *
 */
int GPT_SetOption( task_context_t * context,
                   u_int32_t        index,
                   value_t        * value,
                   task_effect_t  * effect )
{
        int rc=EINVAL;

        LOG_ENTRY();


        if (context ) {

                switch (context->action) {
                
                case EVMS_Task_Create:
                        rc = set_create_option( context, index, value, effect );
                        break;

                case EVMS_Task_Assign_Plugin:
                        rc = set_assign_option( context, index, value, effect );
                        break;

                case EVMS_Task_Expand:
                        rc = set_expand_option( context, index, value, effect );
                        break;

                case EVMS_Task_Shrink:
                        rc = set_shrink_option( context, index, value, effect );
                        break;

                case EVMS_Task_GPT_Move_Segment:
                        rc = set_move_option(context, index, value, effect );
                        break;

                default:
                        LOG_ERROR("error, context->action is unknown or unsupported\n");
                        break;
                }
        }

        LOG_EXIT_INT(rc);

        return rc;
}





/*
 * Returns segment specific information
 */
int GPT_GetInfo( storage_object_t  * object, extended_info_array_t * * info)
{
        int rc = EINVAL;
        extended_info_array_t   *Info;
        SEG_PRIVATE_DATA    *pdata=(SEG_PRIVATE_DATA *)object->private_data;
        char *guid_string=NULL;

        LOG_ENTRY();

        rc    = ENOMEM;  // init to failed calloc
        *info = NULL;     // init to no info returned

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + ( SEG_INFO_COUNT * sizeof(extended_info_t) ) );
        if (Info) {

                Info->count = SEG_INFO_COUNT;

                SET_STRING( Info->info[SEG_INFO_NAME_INDEX].name, "Name" );
                SET_STRING( Info->info[SEG_INFO_NAME_INDEX].title, "Name" );
                SET_STRING( Info->info[SEG_INFO_NAME_INDEX].desc, "This is the partition name. It must be unique on the system.");
                Info->info[SEG_INFO_NAME_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_INFO_NAME_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[SEG_INFO_NAME_INDEX].value.s, object->name );
                Info->info[SEG_INFO_NAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_NAME_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[SEG_INFO_SIZE_INDEX].name, "Size" );
                SET_STRING( Info->info[SEG_INFO_SIZE_INDEX].title, "Size" );
                SET_STRING( Info->info[SEG_INFO_SIZE_INDEX].desc, "This is the size of the partition in sectors.");
                Info->info[SEG_INFO_SIZE_INDEX].type               = EVMS_Type_Unsigned_Int64;
                Info->info[SEG_INFO_SIZE_INDEX].unit               = EVMS_Unit_Sectors;
                Info->info[SEG_INFO_SIZE_INDEX].value.ui64         = object->size;
                Info->info[SEG_INFO_SIZE_INDEX].format             = EVMS_Format_Normal;
                Info->info[SEG_INFO_SIZE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_SIZE_INDEX].group, 0, sizeof(group_info_t));
                Info->info[SEG_INFO_SIZE_INDEX].flags |= EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION;

                SET_STRING( Info->info[SEG_INFO_START_INDEX].name, "Start" );
                SET_STRING( Info->info[SEG_INFO_START_INDEX].title, "Start LBA" );
                SET_STRING( Info->info[SEG_INFO_START_INDEX].desc, "This is the sector offset of the partition on the disk, i.e., the logical block address of the first sector of the partition.");
                Info->info[SEG_INFO_START_INDEX].type               = EVMS_Type_Unsigned_Int64;
                Info->info[SEG_INFO_START_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_INFO_START_INDEX].value.ui64         = object->start;
                Info->info[SEG_INFO_START_INDEX].format             = EVMS_Format_Normal;
                Info->info[SEG_INFO_START_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_START_INDEX].group, 0, sizeof(group_info_t));


                // determine type of segment and display appropriate info
                if ( object->data_type == DATA_TYPE ) {

                        char  ptype[24];

                        SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].name, "Type" );
                        SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].title, "Type" );
                        SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].desc, "Partition type - swap, system, data, etc.");
                        Info->info[SEG_INFO_TYPE_INDEX].type               = EVMS_Type_String;
                        Info->info[SEG_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
                        Info->info[SEG_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
                        memset( &Info->info[SEG_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));

                        switch (pdata->type) {
                        case efi_system_partition:
                                strcpy(ptype, "system");
                                break;
                        case basic_data_partition:
                                strcpy(ptype, "basic data");
                                break;
                        case legacy_mbr_partition:
                                strcpy(ptype, "legacy mbr");
                                break;
                        case swap_partition:
                                strcpy(ptype, "swap");
                                break;
                        case ms_reserved_partition:
                                strcpy(ptype, "ms reserved");
                                break;
                        case ms_ldm_data_partition:
                                strcpy(ptype, "ms ldm data");
                                break;
                        case ms_ldm_metadata_partition:
                                strcpy(ptype, "ms ldm metadata");
                                break;
                        default:
                                strcpy(ptype,"unknown");
                                break;
                        }

                        SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].value.s, ptype );

                }
                else if ( object->data_type == META_DATA_TYPE ) {
                        SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].name, "Type" );
                        SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].title, "Type" );
                        SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].desc, "Partition type - swap, system, data, etc.");
                        Info->info[SEG_INFO_TYPE_INDEX].type            = EVMS_Type_String;
                        Info->info[SEG_INFO_TYPE_INDEX].unit            = EVMS_Unit_None;
                        Info->info[SEG_INFO_TYPE_INDEX].collection_type = EVMS_Collection_None;
                        memset( &Info->info[SEG_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));
                        SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].value.s, "metadata" );

                        // a protective mbr will not have additional info
                        if ( strstr( object->name, "pmbr" )==NULL) {
                                Info->info[SEG_INFO_TYPE_INDEX].flags           = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
                        }

                }
                else {
                        SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].name, "Type" );
                        SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].title, "Type" );
                        SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].desc, "Partition type - swap, system, data, etc.");
                        Info->info[SEG_INFO_TYPE_INDEX].type            = EVMS_Type_String;
                        Info->info[SEG_INFO_TYPE_INDEX].unit            = EVMS_Unit_None;
                        Info->info[SEG_INFO_TYPE_INDEX].collection_type = EVMS_Collection_None;
                        memset( &Info->info[SEG_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));
                        SET_STRING( Info->info[SEG_INFO_TYPE_INDEX].value.s, "freespace" );
                }


                guid_string = guid_to_string( &pdata->guid_type );

                if (guid_string) {

                        SET_STRING( Info->info[SEG_INFO_GUIDT_INDEX].name, "Type - guid" );
                        SET_STRING( Info->info[SEG_INFO_GUIDT_INDEX].title, "Type - guid" );
                        SET_STRING( Info->info[SEG_INFO_GUIDT_INDEX].desc, "This is the partition TYPE globally unique identifier.");
                        Info->info[SEG_INFO_GUIDT_INDEX].type               = EVMS_Type_String;
                        Info->info[SEG_INFO_GUIDT_INDEX].unit               = EVMS_Unit_None;
                        SET_STRING( Info->info[SEG_INFO_GUIDT_INDEX].value.s, guid_string );
                        Info->info[SEG_INFO_GUIDT_INDEX].collection_type    = EVMS_Collection_None;
                        memset( &Info->info[SEG_INFO_GUIDT_INDEX].group, 0, sizeof(group_info_t));

                        free(guid_string);

                }
                else {
                        EngFncs->engine_free( Info );
                        rc = ENOMEM;
                        LOG_EXIT_INT(rc);
                        return rc;
                }

                if ( object->data_type == DATA_TYPE ) {

                        guid_string = guid_to_string( &pdata->guid_id );

                        if (guid_string) {

                                SET_STRING( Info->info[SEG_INFO_GUIDI_INDEX].name, "Id - guid" );
                                SET_STRING( Info->info[SEG_INFO_GUIDI_INDEX].title, "Id - guid" );
                                SET_STRING( Info->info[SEG_INFO_GUIDI_INDEX].desc, "This is the partition globally unique identifier.");
                                Info->info[SEG_INFO_GUIDI_INDEX].type               = EVMS_Type_String;
                                Info->info[SEG_INFO_GUIDI_INDEX].unit               = EVMS_Unit_None;
                                SET_STRING( Info->info[SEG_INFO_GUIDI_INDEX].value.s, guid_string );
                                Info->info[SEG_INFO_GUIDI_INDEX].collection_type    = EVMS_Collection_None;
                                memset( &Info->info[SEG_INFO_GUIDI_INDEX].group, 0, sizeof(group_info_t));

                                free(guid_string);

                        }
                        else {
                                EngFncs->engine_free( Info );
                                rc = ENOMEM;
                                LOG_EXIT_INT(rc);
                                return rc;
                        }
                }
                else {
                        --Info->count;
                }

                *info = Info;

                rc = 0;
        }
        else {
                LOG_ERROR("unable to malloc memory for extended info array\n");
        }


        LOG_EXIT_INT(rc);
        return rc;
}




int GPT_GetMetadataInfo(storage_object_t * object, extended_info_array_t **info)
{
        int rc = EINVAL;
        extended_info_array_t   *Info;
        gpt_header              *gh = ((SEG_PRIVATE_DATA *)object->private_data)->gh;
        char                     vstring[16];
        char                    *guid_string=NULL;

        LOG_ENTRY();

        rc    = ENOMEM;  // init to failed calloc
        *info = NULL;     // init to no info returned

        sprintf(vstring,"%d.%d.%d", (gh->version >> 16), ((gh->version>>8)&0x00ff), (gh->version&0xff));

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + ( GPT_HEADER_INFO_COUNT * sizeof(extended_info_t) ) );
        if (Info) {

                Info->count = GPT_HEADER_INFO_COUNT;

                SET_STRING( Info->info[GPT_HEADER_INFO_VERSION_INDEX].name, "Version" );
                SET_STRING( Info->info[GPT_HEADER_INFO_VERSION_INDEX].title, "Version" );
                SET_STRING( Info->info[GPT_HEADER_INFO_VERSION_INDEX].desc, "This is the version number specified in the header.");
                Info->info[GPT_HEADER_INFO_VERSION_INDEX].type               = EVMS_Type_String;
                Info->info[GPT_HEADER_INFO_VERSION_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[GPT_HEADER_INFO_VERSION_INDEX].value.s, vstring );
                Info->info[GPT_HEADER_INFO_VERSION_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[GPT_HEADER_INFO_VERSION_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[GPT_HEADER_INFO_HDRSIZE_INDEX].name, "Size" );
                SET_STRING( Info->info[GPT_HEADER_INFO_HDRSIZE_INDEX].title, "Header Size" );
                SET_STRING( Info->info[GPT_HEADER_INFO_HDRSIZE_INDEX].desc, "This is the size of header in bytes.");
                Info->info[GPT_HEADER_INFO_HDRSIZE_INDEX].type               = EVMS_Type_Unsigned_Int32;
                Info->info[GPT_HEADER_INFO_HDRSIZE_INDEX].unit               = EVMS_Unit_None;
                Info->info[GPT_HEADER_INFO_HDRSIZE_INDEX].value.ui32         = gh->size;
                Info->info[GPT_HEADER_INFO_HDRSIZE_INDEX].format             = EVMS_Format_Normal;
                Info->info[GPT_HEADER_INFO_HDRSIZE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[GPT_HEADER_INFO_HDRSIZE_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[GPT_HEADER_INFO_MYLBA_INDEX].name, "My LBA" );
                SET_STRING( Info->info[GPT_HEADER_INFO_MYLBA_INDEX].title, "My LBA" );
                SET_STRING( Info->info[GPT_HEADER_INFO_MYLBA_INDEX].desc, "Logical Block Address of the primary gpt header.");
                Info->info[GPT_HEADER_INFO_MYLBA_INDEX].type               = EVMS_Type_Unsigned_Int64;
                Info->info[GPT_HEADER_INFO_MYLBA_INDEX].unit               = EVMS_Unit_None;
                Info->info[GPT_HEADER_INFO_MYLBA_INDEX].value.ui64         = gh->my_lba;
                Info->info[GPT_HEADER_INFO_MYLBA_INDEX].format             = EVMS_Format_Normal;
                Info->info[GPT_HEADER_INFO_MYLBA_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[GPT_HEADER_INFO_MYLBA_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[GPT_HEADER_INFO_ALTLBA_INDEX].name, "Alternate LBA" );
                SET_STRING( Info->info[GPT_HEADER_INFO_ALTLBA_INDEX].title, "Alternate LBA" );
                SET_STRING( Info->info[GPT_HEADER_INFO_ALTLBA_INDEX].desc, "Logical Block Address of the alternate gpt header.");
                Info->info[GPT_HEADER_INFO_ALTLBA_INDEX].type               = EVMS_Type_Unsigned_Int64;
                Info->info[GPT_HEADER_INFO_ALTLBA_INDEX].unit               = EVMS_Unit_None;
                Info->info[GPT_HEADER_INFO_ALTLBA_INDEX].value.ui64         = gh->alternate_lba;
                Info->info[GPT_HEADER_INFO_ALTLBA_INDEX].format             = EVMS_Format_Normal;
                Info->info[GPT_HEADER_INFO_ALTLBA_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[GPT_HEADER_INFO_ALTLBA_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[GPT_HEADER_INFO_START_INDEX].name, "Useable Start" );
                SET_STRING( Info->info[GPT_HEADER_INFO_START_INDEX].title, "Useable Start" );
                SET_STRING( Info->info[GPT_HEADER_INFO_START_INDEX].desc, "Logical Block Address of the start of useable space on the disk.");
                Info->info[GPT_HEADER_INFO_START_INDEX].type               = EVMS_Type_Unsigned_Int64;
                Info->info[GPT_HEADER_INFO_START_INDEX].unit               = EVMS_Unit_None;
                Info->info[GPT_HEADER_INFO_START_INDEX].value.ui64         = gh->start_useable;
                Info->info[GPT_HEADER_INFO_START_INDEX].format             = EVMS_Format_Normal;
                Info->info[GPT_HEADER_INFO_START_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[GPT_HEADER_INFO_START_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[GPT_HEADER_INFO_END_INDEX].name, "Useable End" );
                SET_STRING( Info->info[GPT_HEADER_INFO_END_INDEX].title, "Useable End" );
                SET_STRING( Info->info[GPT_HEADER_INFO_END_INDEX].desc, "Logical Block Address of the end of useable space on the disk.");
                Info->info[GPT_HEADER_INFO_END_INDEX].type               = EVMS_Type_Unsigned_Int64;
                Info->info[GPT_HEADER_INFO_END_INDEX].unit               = EVMS_Unit_None;
                Info->info[GPT_HEADER_INFO_END_INDEX].value.ui64         = gh->end_useable;
                Info->info[GPT_HEADER_INFO_END_INDEX].format             = EVMS_Format_Normal;
                Info->info[GPT_HEADER_INFO_END_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[GPT_HEADER_INFO_END_INDEX].group, 0, sizeof(group_info_t));


                guid_string = guid_to_string( &gh->disk_id );

                if (guid_string) {

                        SET_STRING( Info->info[GPT_HEADER_INFO_GUID_INDEX].name, "Disk - guid" );
                        SET_STRING( Info->info[GPT_HEADER_INFO_GUID_INDEX].title, "Disk - guid" );
                        SET_STRING( Info->info[GPT_HEADER_INFO_GUID_INDEX].desc, "This is the disk globally unique identifier.");
                        Info->info[GPT_HEADER_INFO_GUID_INDEX].type               = EVMS_Type_String;
                        Info->info[GPT_HEADER_INFO_GUID_INDEX].unit               = EVMS_Unit_None;
                        SET_STRING( Info->info[GPT_HEADER_INFO_GUID_INDEX].value.s, guid_string );
                        Info->info[GPT_HEADER_INFO_GUID_INDEX].collection_type    = EVMS_Collection_None;
                        memset( &Info->info[GPT_HEADER_INFO_GUID_INDEX].group, 0, sizeof(group_info_t));

                        free(guid_string);

                }
                else {
                        EngFncs->engine_free( Info );
                        rc = ENOMEM;
                        LOG_EXIT_INT(rc);
                        return rc;
                }


                SET_STRING( Info->info[GPT_HEADER_INFO_PTBLLBA_INDEX].name, "Ptable LBA" );
                SET_STRING( Info->info[GPT_HEADER_INFO_PTBLLBA_INDEX].title, "Ptable LBA" );
                SET_STRING( Info->info[GPT_HEADER_INFO_PTBLLBA_INDEX].desc, "Logical Block Address of the partition table.");
                Info->info[GPT_HEADER_INFO_PTBLLBA_INDEX].type               = EVMS_Type_Unsigned_Int64;
                Info->info[GPT_HEADER_INFO_PTBLLBA_INDEX].unit               = EVMS_Unit_None;
                Info->info[GPT_HEADER_INFO_PTBLLBA_INDEX].value.ui64         = gh->ptable_lba;
                Info->info[GPT_HEADER_INFO_PTBLLBA_INDEX].format             = EVMS_Format_Normal;
                Info->info[GPT_HEADER_INFO_PTBLLBA_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[GPT_HEADER_INFO_PTBLLBA_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[GPT_HEADER_INFO_PCOUNT_INDEX].name, "Ptable Size" );
                SET_STRING( Info->info[GPT_HEADER_INFO_PCOUNT_INDEX].title, "Ptable Size" );
                SET_STRING( Info->info[GPT_HEADER_INFO_PCOUNT_INDEX].desc, "This is the size of the partition table, i.e. its capacity. It is a count of partition entries that can be stored in the table.");
                Info->info[GPT_HEADER_INFO_PCOUNT_INDEX].type               = EVMS_Type_Unsigned_Int32;
                Info->info[GPT_HEADER_INFO_PCOUNT_INDEX].unit               = EVMS_Unit_None;
                Info->info[GPT_HEADER_INFO_PCOUNT_INDEX].value.ui32         = gh->ptable_count;
                Info->info[GPT_HEADER_INFO_PCOUNT_INDEX].format             = EVMS_Format_Normal;
                Info->info[GPT_HEADER_INFO_PCOUNT_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[GPT_HEADER_INFO_PCOUNT_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[GPT_HEADER_INFO_PNUM_INDEX].name, "Ptable Entry Size" );
                SET_STRING( Info->info[GPT_HEADER_INFO_PNUM_INDEX].title, "Ptable Entry Size" );
                SET_STRING( Info->info[GPT_HEADER_INFO_PNUM_INDEX].desc, "This is the size of a single partition table entry.");
                Info->info[GPT_HEADER_INFO_PNUM_INDEX].type               = EVMS_Type_Unsigned_Int32;
                Info->info[GPT_HEADER_INFO_PNUM_INDEX].unit               = EVMS_Unit_None;
                Info->info[GPT_HEADER_INFO_PNUM_INDEX].value.ui32         = gh->ptable_entry_size;
                Info->info[GPT_HEADER_INFO_PNUM_INDEX].format             = EVMS_Format_Normal;
                Info->info[GPT_HEADER_INFO_PNUM_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[GPT_HEADER_INFO_PNUM_INDEX].group, 0, sizeof(group_info_t));

                *info = Info;

                rc = 0;

        }
        else {
                LOG_ERROR("unable to malloc memory for extended info array\n");
        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *
 */
int GPT_GetPluginInfo( char * descriptor_name, extended_info_array_t * * info )
{
        int rc = EINVAL;
        extended_info_array_t   *Info;
        char                     version_string[64];
        char                     required_engine_api_version_string[64];
        char                     required_plugin_api_version_string[64];


        LOG_ENTRY();

        // a measure of protection ...
        if (info == NULL) {
                LOG_EXIT_INT(rc);
                return rc;
        }

        rc    = ENOMEM;  // init to failed calloc
        *info = NULL;     // init to no info returned

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + (SEG_PLUGIN_INFO_COUNT*sizeof(extended_info_t))  );
        if (Info) {

                Info->count = SEG_PLUGIN_INFO_COUNT;

                sprintf(version_string, "%d.%d.%d",
                        MAJOR_VERSION,
                        MINOR_VERSION,
                        PATCH_LEVEL );

                sprintf(required_engine_api_version_string, "%d.%d.%d",
                        gpt_plugin_record_ptr->required_engine_api_version.major,
                        gpt_plugin_record_ptr->required_engine_api_version.minor,
                        gpt_plugin_record_ptr->required_engine_api_version.patchlevel );

                sprintf(required_plugin_api_version_string, "%d.%d.%d",
                        gpt_plugin_record_ptr->required_plugin_api_version.plugin.major,
                        gpt_plugin_record_ptr->required_plugin_api_version.plugin.minor,
                        gpt_plugin_record_ptr->required_plugin_api_version.plugin.patchlevel );

                SET_STRING( Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].name, "ShortName" );
                SET_STRING( Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].title, "Short Name" );
                SET_STRING( Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].desc, "A short name given to this plugin.");
                Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].value.s, gpt_plugin_record_ptr->short_name );
                Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].name, "LongName" );
                SET_STRING( Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].title, "Long Name" );
                SET_STRING( Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].desc, "A longer and more descriptive name for this plugin.");
                Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].value.s, gpt_plugin_record_ptr->long_name );
                Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].name, "Type" );
                SET_STRING( Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].title, "Plugin Type" );
                SET_STRING( Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].desc, "There are various types of plugins; each responsible for some kind of storage object.");
                Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].value.s, "Partition Manager" );
                Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].name, "Version" );
                SET_STRING( Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].title, "Plugin Version" );
                SET_STRING( Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].desc, "This is the version number of the plugin.");
                Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].value.s, version_string );
                Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[SEG_PLUGIN_INFO_REQENGSVCVERSION_INDEX].name, "Required Engine Services Version" );
                SET_STRING( Info->info[SEG_PLUGIN_INFO_REQENGSVCVERSION_INDEX].title, "Required Engine Services Version" );
                SET_STRING( Info->info[SEG_PLUGIN_INFO_REQENGSVCVERSION_INDEX].desc, "This is the version of the Engine services that this plugin requires. It will not run on older versions of the Engine services.");
                Info->info[SEG_PLUGIN_INFO_REQENGSVCVERSION_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_PLUGIN_INFO_REQENGSVCVERSION_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[SEG_PLUGIN_INFO_REQENGSVCVERSION_INDEX].value.s, required_engine_api_version_string );
                Info->info[SEG_PLUGIN_INFO_REQENGSVCVERSION_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_PLUGIN_INFO_REQENGSVCVERSION_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[SEG_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].name, "Required Engine Plug-in API Version" );
                SET_STRING( Info->info[SEG_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].title, "Required Engine Plug-in API Version" );
                SET_STRING( Info->info[SEG_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].desc, "This is the version of the Engine plug-in API that this plugin requires. It will not run on older versions of the Engine plug-in API.");
                Info->info[SEG_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[SEG_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].value.s, required_plugin_api_version_string );
                Info->info[SEG_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].group, 0, sizeof(group_info_t));

                *info = Info;

                rc = 0;
        }


        LOG_EXIT_INT(rc);
        return rc;
}


