/*
 *
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdrivelink.so
 *
 *   File: dl_options.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "dl_common.h"



/*
 *   Called to build a reverse ordered child_objects list from a list
 *   of child objects.
 *
 *   Returns 0 if successful and the child_objects
 *   list should contain all the objects from the drive link
 *   in the proper order.
 *
 *   Returns
 */
static int dl_build_reverse_ordered_child_object_list( storage_object_t *drivelink, list_anchor_t list )
{
        int                       i, rc;
        list_anchor_t             reverse_ordered_list;
        list_element_t            iter, iter2;
        drivelink_private_data_t  *pdata;
        uint                      starting_count=0;
        uint                      ending_count=0;
        storage_object_t         *obj;

        LOG_ENTRY();

        REQUIRE(drivelink != NULL);
        REQUIRE(drivelink->private_data != NULL);
        REQUIRE(list != NULL);
        
        pdata = (drivelink_private_data_t  *) drivelink->private_data;

        starting_count = EngFncs->list_count(list);

        reverse_ordered_list = EngFncs->allocate_list();

        REQUIRE(reverse_ordered_list != NULL);

        for(i=pdata->drive_link_count-1; i>=0; i-- ) {

                if ( pdata->drive_link[i].object != NULL ) {

                        LIST_FOR_EACH(list, iter, obj) {
                
                                if (obj == pdata->drive_link[i].object) {

                                        iter2 = EngFncs->insert_thing(reverse_ordered_list, obj,
                                                                      INSERT_AFTER, NULL);
                                        if (iter2) {
                                                ++ending_count;
                                        }

                                        break;
                                }
                            
                        }
                }

        }
 
        // if I finished with a reversed ordered list that contains
        // the same number of objects that I started with ... then
        // it means that the drive link ordering table matches the
        // input list exactly.
        //
        // So, now copy the reversed ordered list back to the callers
        // list so we can return the reordered drive link objects.
        if (starting_count == ending_count) {

                // first delete all objects in the caller list
                EngFncs->delete_all_elements(list);

                // then copy reordered objects to callers list
                rc = EngFncs->merge_lists(list, reverse_ordered_list, NULL);
        }
        else {
                EngFncs->destroy_list(reverse_ordered_list);
                rc = EPERM;
        }

        LOG_EXIT_INT(rc);
        return rc;
}





/*
 *  Returns count of options for specified task
 */
int dl_get_option_count(task_context_t * task)
{
        int count;

        LOG_ENTRY();

        switch (task->action) {

                case EVMS_Task_Create:
                case EVMS_Task_Add_Feature:
                        count = DL_CREATE_OPTION_COUNT;
                        break;

                case EVMS_Task_Expand:
                        count = DL_EXPAND_OPTION_COUNT;
                        break;

                case EVMS_Task_Shrink:
                        count = DL_SHRINK_OPTION_COUNT;
                        break;

                case EVMS_Replace_Missing_DL_Child:
                        count = DL_REPLACE_MISSING_CHILD_OPTION_COUNT;
                        break;

                default:
                        count = 0;
                        break;
                }

        LOG_EXIT_INT(count);
        return count;
}

/*
 *  Called by init_task() to allocate the REPLACE option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int allocate_replace_option_descriptors( task_context_t * context )
{
        int rc; 

        LOG_ENTRY();
        rc=0;
        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to initialize the REPLACE options in the task context.
 */
static int initialize_replace_option_descriptors( storage_object_t  * missing_child, 
                                                  task_context_t    * context )
{
        int rc;

        LOG_ENTRY();
        rc=0;
        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: set replace object
 *
 *  Called by set_object() to validate the task selected 
 *  object.
 */
static int set_replace_object( task_context_t * context,
                               list_anchor_t    declined_objects,
                               task_effect_t  * effect )
{
    int rc = EINVAL;
    storage_object_t  *new_child;

    LOG_ENTRY();

    new_child = EngFncs->first_thing(context->selected_objects, NULL);
    if (new_child) {
        rc = initialize_replace_option_descriptors(new_child, context);
        if (!rc) {
            *effect |=  EVMS_Effect_Reload_Options;
        }
    }

    LOG_EXIT_INT(rc);
    return rc;
}

/*
 *  Function: get acceptable replace objects
 *
 *  Called when setting up a replace task to go out and find all the
 *  objects that we could replace a missing child with. 
 */
static int get_acceptable_replace_objects( task_context_t * context )
{
        int  rc = EINVAL;
        int  count=0;   
        list_element_t iter1, iter2;
        storage_object_t * parent;
        storage_object_t * new_child;
        storage_object_t * missing_child = context->object;

        LOG_ENTRY();

        if ( (context) &&
             (context->object) &&
             (context->acceptable_objects) ) {

                count = EngFncs->list_count(context->acceptable_objects);
                if (count == 0) {

                        rc = EngFncs->get_object_list( 0,
                                                              DATA_TYPE,
                                                              NULL,
                                                              NULL,
                                                              TOPMOST,
                                                              &context->acceptable_objects );

                        if (rc == 0) {
                                LIST_FOR_EACH_SAFE(context->acceptable_objects, iter1, iter2, new_child) {
                                        parent = dl_get_parent(missing_child);
                                        if (parent != new_child) {
                                                rc = dl_validate_missing_child_replace_target(missing_child,
                                                                                              new_child);
                                                if (rc) {
                                                        EngFncs->delete_element(iter1);
                                                }
                                        } else {
                                                EngFncs->delete_element(iter1);
                                        }
                                }
                        }                       
                }
                else {
                        LOG_ERROR("error, context already has acceptable objects\n");
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}





static int dl_allocate_create_option_descriptors( task_context_t * context )
{
        int rc=0;


        LOG_ENTRY();

        REQUIRE(context != NULL);
   
        context->option_descriptors->count = DL_CREATE_OPTION_COUNT ;     

        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE ;
        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].constraint.list = NULL;
        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].constraint_type = EVMS_Collection_None;
        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].help = NULL;
        SET_STRING( context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].name, DL_CREATE_LINKNAME_NAME );
        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].min_len = 1;
        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].max_len = EVMS_VOLUME_NAME_SIZE;
        SET_STRING( context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].tip, "The name to be given to the drivelink object.");
        SET_STRING( context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].title, "Name" );
        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].type = EVMS_Type_String;
        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].unit = EVMS_Unit_None;
        context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].value.s = EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE+1);
        if (context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].value.s) {
                memset( context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].value.s,
                        0, 
                        EVMS_VOLUME_NAME_SIZE);
        }
        else {
                rc = ENOMEM;
                LOG_EXIT_INT(rc);
                return rc;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


static int dl_initialize_create_option_descriptors(  task_context_t * context )
{
        int rc=0;

        LOG_ENTRY();

        REQUIRE(context != NULL);

        // there should be exactly DL_CREATE_OPTION_COUNT options
        if ( context->option_descriptors->count == DL_CREATE_OPTION_COUNT ) {
                context->option_descriptors->option[DL_CREATE_LINKNAME_INDEX].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
        }
        else {
                LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
                rc = EINVAL;
        }
   
        LOG_EXIT_INT(rc);
        return rc;
}


static int dl_allocate_expand_option_descriptors( task_context_t * context )
{
        int rc=0;

        LOG_ENTRY();

        REQUIRE(context != NULL);

        context->option_descriptors->count = 0;  // no options
        
        LOG_EXIT_INT(rc);
        return rc;
}


static int dl_allocate_shrink_option_descriptors( task_context_t * context )
{
        int rc=0;

        LOG_ENTRY();

        REQUIRE(context != NULL);

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Get the user selected object from the create context and init the option descriptors
 *  as appropriate for the object.
 *
 *  Note ... if there is more than 1 object in the source object list we will simply
 *           take the first good object and decline the remainder.  That is the reason for
 *           the boolean flag found_good_object.
 */
static int dl_set_create_object( task_context_t * context,
                                 list_anchor_t    declined_objects,
                                 task_effect_t  * effect )
{
        int   rc;
        uint  linkcount=0;

        LOG_ENTRY();

        REQUIRE(context != NULL);
        REQUIRE(declined_objects != NULL);
        REQUIRE(effect != NULL);

        linkcount = EngFncs->list_count(context->selected_objects);
    
        REQUIRE(linkcount > 0 && linkcount <= EVMS_DRIVELINK_MAX_ENTRIES);

        rc = dl_initialize_create_option_descriptors( context );

        if (rc == 0) {
                *effect |=  EVMS_Effect_Reload_Options;
        }
        else {
                rc = EINVAL;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int dl_set_expand_object( task_context_t * context,
                                 list_anchor_t    declined_objects,
                                 task_effect_t  * effect )
{
        int                      rc=0;
        storage_object_t        *obj=NULL;
        declined_object_t       *declined_object=NULL;
        int                      selected_objects_count=0;
        int                      max_objects_allowed=0;
        drivelink_private_data_t *pdata;
        int                      declined_object_count=0;
        list_element_t           iter, iter2;


        LOG_ENTRY();
        
        REQUIRE(context != NULL);
        REQUIRE(declined_objects != NULL);
        REQUIRE(effect != NULL);
        REQUIRE(dl_isa_drivelink(context->object) == TRUE);

        pdata = ( drivelink_private_data_t * )context->object->private_data;

        max_objects_allowed = EVMS_DRIVELINK_MAX_ENTRIES - pdata->drive_link_count;

        REQUIRE(max_objects_allowed > 0);

        LIST_FOR_EACH(context->selected_objects, iter, obj) {
        
                if ( max_objects_allowed > selected_objects_count ) {
                        ++selected_objects_count;
                        *effect |=  EVMS_Effect_Reload_Options;                 
                }
                else {

                        LOG_ERROR("declining an object found in context->selected_objects, object name= %s\n", obj->name);

                        ++declined_object_count;

                        declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                                declined_object->object = obj;
                                declined_object->reason = EINVAL; // REMINDER ... need to pickup reason codes the UI understands

                                EngFncs->insert_thing(declined_objects,
                                                      declined_object,
                                                      INSERT_AFTER, NULL);

                                *effect |=  EVMS_Effect_Reload_Objects;
                        }
                        else {
                                LOG_ERROR("unable to malloc a declined object struct\n");
                                rc = ENOMEM;
                        }

                }
        }

        if (declined_object_count) {
                *effect |= EVMS_Effect_Reload_Objects;
        }
        else if ( (rc==0) &&                        // didnt decline any of the objects and the result looks
                  (selected_objects_count > 0) ) {  // good so now call the engine to see if we can expand
                                                // by the specified amount.

                sector_count_t   expand_sectors = 0;
                sector_count_t   child_sectors  = 0;

                LIST_FOR_EACH_SAFE(context->selected_objects, iter, iter2, obj) {

                        child_sectors = obj->size - (DRIVELINK_METADATA_SECTOR_COUNT*2) - (FEATURE_HEADER_SECTOR_COUNT*2);
                        expand_sectors += child_sectors;

                        rc = EngFncs->can_expand_by( obj, &expand_sectors );

                        if (rc) {

                                expand_sectors -= child_sectors;

                                LOG_DEBUG("declining object, object name= %s ... engine wont allow expanding\n", obj->name);

                                declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                                if (declined_object) {

                                        declined_object->object = obj;
                                        declined_object->reason = -1; // ENOCANEXPAND

                                        // remove object from selectable/acceptable object lists
                                        EngFncs->delete_element(iter);
                                        EngFncs->remove_thing(context->acceptable_objects, obj);

                                        EngFncs->insert_thing(declined_objects, obj,
                                                              INSERT_AFTER, NULL);
                                }
                                else {
                                        rc = ENOMEM;
                                        LOG_ERROR("error, unable to malloc a declined object.\n");
                                }
                        }
                }
        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int dl_set_shrink_object( task_context_t * context,
                                 list_anchor_t    declined_objects,
                                 task_effect_t  * effect )
{
        int rc = EINVAL;
        storage_object_t  *obj=NULL;
        storage_object_t  *child;
        boolean            invalidate_objects=FALSE;
        declined_object_t *declined_object=NULL;
        uint               selected_objects_count=0;
        int                search_index=0;
        int                declined_object_count=0;
        int                i;
        drivelink_private_data_t *pdata=NULL;
        list_element_t     iter, iter2;


        LOG_ENTRY();

        REQUIRE(context != NULL);
        REQUIRE(declined_objects != NULL);
        REQUIRE(effect != NULL);
        REQUIRE(context->selected_objects != NULL);
        REQUIRE(context->acceptable_objects != NULL);
        REQUIRE(dl_isa_drivelink(context->object) == TRUE);

        pdata = (drivelink_private_data_t *) context->object->private_data;

        // reorder the selected objects
        rc = dl_build_reverse_ordered_child_object_list( context->object, context->selected_objects );
        if ( rc ) {
                LOG_ERROR("error, unable to build ordered list of selected objects\n");
                LOG_EXIT_INT(rc);
                return rc;
        }

        selected_objects_count = EngFncs->list_count(context->selected_objects);

        REQUIRE( selected_objects_count > 0 &&
                 selected_objects_count < pdata->drive_link_count );

        declined_object_count = 0;
        search_index          = pdata->drive_link_count - 1;

        LIST_FOR_EACH_SAFE(context->selected_objects, iter, iter2, obj) {

                if (obj != pdata->drive_link[search_index].object) {
                        invalidate_objects = TRUE;
                }

                --search_index;

                if ( invalidate_objects == TRUE ) {

                        ++declined_object_count;

                        declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                                declined_object->object = obj;
                                declined_object->reason = -1; // EOUTOFORDER;

                                // remove object from selectable/acceptable object lists
                                EngFncs->delete_element(iter);
                                EngFncs->remove_thing(context->acceptable_objects, obj);

                                // place it on the declined object list
                                EngFncs->insert_thing(declined_objects,
                                                      declined_object,
                                                      INSERT_AFTER, NULL);

                        }
                        else {
                                rc = ENOMEM;
                                LOG_EXIT_INT(rc);
                                return rc;
                        }

                }
                
        }
           
        if (declined_object_count == 0) {

                sector_count_t   shrink_sectors=0;

                for( i=pdata->drive_link_count-1; (i>=0)&&(selected_objects_count); i--, selected_objects_count-- ) {

                        shrink_sectors += pdata->drive_link[i].sector_count;

                        rc = EngFncs->can_shrink_by( context->object, &shrink_sectors );
                        if (rc) {

                                shrink_sectors -= pdata->drive_link[i].sector_count;

                                ++declined_object_count;

                                declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                                if (declined_object) {

                                        declined_object->object = pdata->drive_link[i].object;
                                        declined_object->reason = -1; // ENOCANSHRINK

                                        // remove object from selectable/acceptable object lists
                                        EngFncs->remove_thing(context->selected_objects, obj);
                                        EngFncs->remove_thing(context->acceptable_objects, obj);

                                        // place it on the declined object list
                                        EngFncs->insert_thing(declined_objects,
                                                              declined_object,
                                                              INSERT_AFTER, NULL);
                                }
                                else {
                                        rc = ENOMEM;
                                        LOG_EXIT_INT(rc);
                                        return rc;
                                }
                        }
                }
        }

        // if we can shrink the drivelink Ok then add another link
        // to the acceptable objects list so the user can further
        // shrink the drivelink
        if (declined_object_count == 0) {

                if ( selected_objects_count+1 < pdata->drive_link_count ) {

                        child = pdata->drive_link[pdata->drive_link_count-selected_objects_count-1].object;

                        iter = EngFncs->insert_thing(context->acceptable_objects,
                                                     child, INSERT_AFTER, NULL);

                        if (iter) {
                                context->max_selected_objects = EngFncs->list_count(context->acceptable_objects);
                                *effect |= EVMS_Effect_Reload_Objects;
                        }
                }
        }

        // if we declined some objects then we need to fixup the shrink task
        // context so that we only show the good selected objects PLUS the
        // next object that would be Ok to select.

        EngFncs->delete_all_elements(context->acceptable_objects);
        EngFncs->concatenate_lists(context->acceptable_objects,
                                   context->selected_objects);
        context->max_selected_objects = EngFncs->list_count(context->selected_objects);

        if ( context->max_selected_objects+1 < pdata->drive_link_count ) {

                child = pdata->drive_link[pdata->drive_link_count-context->max_selected_objects-1].object;

                iter = EngFncs->insert_thing(context->acceptable_objects, child,
                                             INSERT_AFTER, NULL);

                if (iter) {
                        ++context->max_selected_objects;
                }

        }

        *effect |= EVMS_Effect_Reload_Objects;

        LOG_EXIT_INT(0);
        return 0;
}


/*
 * Validate the objects in the selected_objects list in the task context.
 * Remove from the selected objects lists any objects which are not
 * acceptable.
 *
 * For unacceptable objects, create a declined_handle_t structure with the
 * reason why it is not acceptable, and add it to the declined_objects list.
 * Modify the accepatble_objects list in the task context as necessary
 * based on the selected objects and the current settings of the options.
 *
 * Modify any option settings as necessary based on the selected objects.
 * Return the appropriate task_effect_t settings if the object list(s),
 * minimum or maximum objects selected, or option settings have changed.
 */
int dl_set_objects( task_context_t * context,
                    list_anchor_t    declined_objects,
                    task_effect_t  * effect )
{

        int rc=0;

        LOG_ENTRY();
        
        REQUIRE(context != NULL);
        REQUIRE(declined_objects != NULL);
        REQUIRE(effect != NULL);

        switch (context->action) {

                case EVMS_Task_Create:
                        rc = dl_set_create_object( context, declined_objects, effect );
                        break;

                case EVMS_Task_Expand:
                        rc = dl_set_expand_object( context, declined_objects, effect );
                        break;

                case EVMS_Task_Shrink:
                        rc = dl_set_shrink_object( context, declined_objects, effect );
                        break;
                
                case EVMS_Replace_Missing_DL_Child:
                        rc = set_replace_object( context, declined_objects, effect );
                        break;

                default:
                        rc=EINVAL;
                        LOG_ERROR("error, context->action is unknown or unsupported\n");
                        break;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


static int dl_set_create_option( task_context_t * context,
                                 u_int32_t        index,
                                 value_t        * value,
                                 u_int32_t      * info )
{
        int rc = EINVAL;

        LOG_ENTRY();

        REQUIRE(context != NULL);
        REQUIRE(index == DL_CREATE_LINKNAME_INDEX);
        REQUIRE(value != NULL);
        REQUIRE(value->s != NULL);
        REQUIRE(strlen(value->s) > 0 &&
                strlen(value->s) <= EVMS_VOLUME_NAME_SIZE );

        rc = EngFncs->validate_name( value->s );
        
        if (rc == 0) {
                strcpy( context->option_descriptors->option[index].value.s, value->s);
        }

        LOG_EXIT_INT(rc);
        return rc;
}


int dl_set_option( task_context_t * context,
                   u_int32_t        index,
                   value_t        * value,
                   u_int32_t      * info )
{
        int rc=EINVAL;

        LOG_ENTRY();
        
        REQUIRE(context != NULL);
        REQUIRE(index == DL_CREATE_LINKNAME_INDEX);
        REQUIRE(value != NULL);
        REQUIRE(value->s != NULL);

        switch (context->action) {

                case EVMS_Task_Create:
                case EVMS_Task_Add_Feature:
                        rc = dl_set_create_option( context, index, value, info );
                        break;

                default:
                        rc = ENOSYS;
                        break;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int dl_get_acceptable_create_objects( task_context_t * context )
{
        int   rc = EINVAL;
        uint  count;

        LOG_ENTRY();

        REQUIRE(context != NULL);

        count = EngFncs->list_count(context->acceptable_objects);

        REQUIRE(count == 0); // should not have any acceptable objects yet

        rc = EngFncs->get_object_list( 0,
                                              DATA_TYPE,
                                              NULL,
                                              NULL,
                                              VALID_INPUT_OBJECT,
                                              &context->acceptable_objects );

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int dl_get_acceptable_expand_objects( task_context_t * context )
{
        list_element_t iter1, iter2;
        storage_object_t * expand_object = context->object;
        storage_object_t * obj;
        int   rc = EINVAL;

        LOG_ENTRY();

        REQUIRE(context != NULL);

        LOG_DEBUG("getting expand objects for drivelink %s\n", context->object->name);

        rc = EngFncs->get_object_list( 0,
                                              DATA_TYPE,
                                              NULL,
                                              NULL,
                                              VALID_INPUT_OBJECT,
                                              &context->acceptable_objects );

        if (rc == 0) {
                LIST_FOR_EACH_SAFE(context->acceptable_objects, iter1, iter2, obj) {
                        if ( obj == expand_object ||
                             obj->disk_group != expand_object->disk_group ||
                             dl_isa_parent(obj, expand_object) == TRUE ) {
                                EngFncs->delete_element(iter1);
                        }
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int dl_get_acceptable_shrink_objects( task_context_t * context  )
{
        int                       rc=0;
        drivelink_private_data_t *pdata;
        storage_object_t         *child;
        list_element_t            iter;


        LOG_ENTRY();

        REQUIRE(context != NULL);
        REQUIRE(dl_isa_drivelink(context->object) == TRUE);
        REQUIRE(context->acceptable_objects != NULL);

        LOG_DEBUG("getting shrink object for drivelink %s\n", context->object->name);

        pdata = (drivelink_private_data_t *) context->object->private_data;

        if (pdata->drive_link_count > 1) {

                child = pdata->drive_link[pdata->drive_link_count-1].object;

                iter = EngFncs->insert_thing(context->acceptable_objects,
                                             child, INSERT_AFTER, NULL);
                if (!iter) {
                        rc = ENOMEM;
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
int dl_init_task( task_context_t * context )
{
        int rc = EINVAL;
        drivelink_private_data_t *pdata=NULL;

        LOG_ENTRY();

        REQUIRE(context != NULL);
       
        switch (context->action) {

                case EVMS_Task_Create:

                        context->min_selected_objects = 1;
                        context->max_selected_objects = EVMS_DRIVELINK_MAX_ENTRIES;

                        rc = dl_allocate_create_option_descriptors( context );
                        if (rc == 0) {
                                rc = dl_get_acceptable_create_objects( context );
                        }
                        break;

                case EVMS_Task_Expand:
                        
                        REQUIRE(dl_isa_drivelink(context->object) == TRUE);
                        
                        pdata = (drivelink_private_data_t *) context->object->private_data;

                        context->min_selected_objects = 1;
                        context->max_selected_objects = EVMS_DRIVELINK_MAX_ENTRIES - pdata->drive_link_count;

                        rc = dl_allocate_expand_option_descriptors( context );
                        if (rc == 0) {
                                rc = dl_get_acceptable_expand_objects( context );
                        }
                        break;

                case EVMS_Task_Shrink:

                        REQUIRE(dl_isa_drivelink(context->object) == TRUE);

                        context->min_selected_objects = 1;

                        rc = dl_allocate_shrink_option_descriptors( context );
                        if (rc == 0) {
                                rc = dl_get_acceptable_shrink_objects( context );
                                if (rc ==0) {
                                        context->max_selected_objects = EngFncs->list_count(context->acceptable_objects);
                                }
                        }
                        break;

                case EVMS_Task_Add_Feature:
                        rc = dl_allocate_create_option_descriptors( context );
                        if (rc == 0) {
                                rc = dl_initialize_create_option_descriptors( context );
                        }
                        break;

                case EVMS_Replace_Missing_DL_Child:
                        context->min_selected_objects = 1;
                        context->max_selected_objects = 1;
                        rc = allocate_replace_option_descriptors( context );
                        if (rc == 0) {
                                rc = get_acceptable_replace_objects( context );
                        }
                        break;

                default:
                        LOG_ERROR("context->action is unknown or unsupported\n");
                        break;
        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Returns specific information
 */
int dl_get_info( storage_object_t  * object, char *name,  extended_info_array_t * * info)
{
        int rc = EINVAL;
        extended_info_array_t    *Info;
        drivelink_private_data_t  *pdata;

        LOG_ENTRY();

        REQUIRE(dl_isa_drivelink(object) == TRUE);
        REQUIRE(info != NULL);
        
        pdata = (drivelink_private_data_t *) object->private_data;

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + ( DL_INFO_COUNT * sizeof(extended_info_t) ) );
        if (Info) {

                // the following info isnt available for missing child objects
                if (pdata->signature & MISSING_CHILD_SIGNATURE) {

                        SET_STRING( Info->info[DL_INFO_NAME_INDEX].name, "Name" );
                        SET_STRING( Info->info[DL_INFO_NAME_INDEX].title, "Name" );
                        SET_STRING( Info->info[DL_INFO_NAME_INDEX].desc, "A temporary name given to the missing drivelink object that is used to assist with recovery.");
                        Info->info[DL_INFO_NAME_INDEX].type               = EVMS_Type_String;
                        Info->info[DL_INFO_NAME_INDEX].unit               = EVMS_Unit_None;
                        SET_STRING( Info->info[DL_INFO_NAME_INDEX].value.s, object->name );
                        Info->info[DL_INFO_NAME_INDEX].collection_type    = EVMS_Collection_None;
                        memset( &Info->info[DL_INFO_NAME_INDEX].group, 0, sizeof(group_info_t));

                        Info->count += 1;

                        SET_STRING( Info->info[DL_INFO_SIZE_INDEX].name, "Size" );
                        SET_STRING( Info->info[DL_INFO_SIZE_INDEX].title, "Size" );
                        SET_STRING( Info->info[DL_INFO_SIZE_INDEX].desc, "The size of the object that is missing, i.e., the useable size after subtracting metadata.");
                        Info->info[DL_INFO_SIZE_INDEX].type               = EVMS_Type_Unsigned_Int64;
                        Info->info[DL_INFO_SIZE_INDEX].unit               = EVMS_Unit_Sectors;
                        Info->info[DL_INFO_SIZE_INDEX].value.ui64         = object->size;
                        Info->info[DL_INFO_SIZE_INDEX].collection_type    = EVMS_Collection_None;
                        memset( &Info->info[DL_INFO_SIZE_INDEX].group, 0, sizeof(group_info_t));
                        Info->info[DL_INFO_SIZE_INDEX].flags |= EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION;

                        Info->count += 1;

                }
                else {

                        SET_STRING( Info->info[DL_INFO_NAME_INDEX].name, "Name" );
                        SET_STRING( Info->info[DL_INFO_NAME_INDEX].title, "Name" );
                        SET_STRING( Info->info[DL_INFO_NAME_INDEX].desc, "This is the name given to the storage object. It must be unique on the system.");
                        Info->info[DL_INFO_NAME_INDEX].type               = EVMS_Type_String;
                        Info->info[DL_INFO_NAME_INDEX].unit               = EVMS_Unit_None;
                        SET_STRING( Info->info[DL_INFO_NAME_INDEX].value.s, object->name );
                        Info->info[DL_INFO_NAME_INDEX].collection_type    = EVMS_Collection_None;
                        memset( &Info->info[DL_INFO_NAME_INDEX].group, 0, sizeof(group_info_t));

                        Info->count += 1;

                        SET_STRING( Info->info[DL_INFO_SIZE_INDEX].name, "Size" );
                        SET_STRING( Info->info[DL_INFO_SIZE_INDEX].title, "Size" );
                        SET_STRING( Info->info[DL_INFO_SIZE_INDEX].desc, "This is the size of the entire drivelink, i.e., the sum of all the drivelink child storage objects.");
                        Info->info[DL_INFO_SIZE_INDEX].type               = EVMS_Type_Unsigned_Int64;
                        Info->info[DL_INFO_SIZE_INDEX].unit               = EVMS_Unit_Sectors;
                        Info->info[DL_INFO_SIZE_INDEX].value.ui64         = object->size;
                        Info->info[DL_INFO_SIZE_INDEX].collection_type    = EVMS_Collection_None;
                        memset( &Info->info[DL_INFO_SIZE_INDEX].group, 0, sizeof(group_info_t));
                        Info->info[DL_INFO_SIZE_INDEX].flags |= EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION;

                        Info->count += 1;

                        SET_STRING( Info->info[DL_INFO_LINKS_INDEX].name, "Links" );
                        SET_STRING( Info->info[DL_INFO_LINKS_INDEX].title, "Children" );
                        SET_STRING( Info->info[DL_INFO_LINKS_INDEX].desc, "This is the number of child storage objects being linked by the drivelink feature.");
                        Info->info[DL_INFO_LINKS_INDEX].type               = EVMS_Type_Unsigned_Int64;
                        Info->info[DL_INFO_LINKS_INDEX].unit               = EVMS_Unit_None;
                        Info->info[DL_INFO_LINKS_INDEX].value.ui64         = pdata->drive_link_count;
                        Info->info[DL_INFO_LINKS_INDEX].collection_type    = EVMS_Collection_None;
                        memset( &Info->info[DL_INFO_LINKS_INDEX].group, 0, sizeof(group_info_t));

                        Info->count += 1;

                        SET_STRING( Info->info[DL_INFO_PSN_INDEX].name, "PSN" );
                        SET_STRING( Info->info[DL_INFO_PSN_INDEX].title, "Parent SN" );
                        SET_STRING( Info->info[DL_INFO_PSN_INDEX].desc, "The serial number of the parent drivelink object.");
                        Info->info[DL_INFO_PSN_INDEX].type               = EVMS_Type_Unsigned_Int64;
                        Info->info[DL_INFO_PSN_INDEX].unit               = EVMS_Unit_None;
                        Info->info[DL_INFO_PSN_INDEX].format             = EVMS_Format_Hex;
                        Info->info[DL_INFO_PSN_INDEX].flags              = EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION;
                        Info->info[DL_INFO_PSN_INDEX].value.ui64         = pdata->parent_serial_number;
                        Info->info[DL_INFO_PSN_INDEX].collection_type    = EVMS_Collection_None;
                        memset( &Info->info[DL_INFO_PSN_INDEX].group, 0, sizeof(group_info_t));

                        Info->count += 1;

                }


                *info = Info;
                rc = 0;
        }
        else {
                rc = ENOMEM;
                LOG_ERROR("unable to malloc memory for extended info array\n");
        }
    

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *
 */
int dl_get_plugin_info( char * descriptor_name, extended_info_array_t * * info )
{
        int rc = EINVAL;
        extended_info_array_t   *Info;
        char                     version_string[64];
        char                     required_engine_api_version_string[64];
        char                     required_plugin_api_version_string[64];


        LOG_ENTRY();

        REQUIRE(info != NULL);

        *info = NULL;     // init to no info returned

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + (DL_PLUGIN_INFO_COUNT*sizeof(extended_info_t))  );
        if (Info) {

                Info->count = DL_PLUGIN_INFO_COUNT;

                sprintf(version_string, "%d.%d.%d",
                        MAJOR_VERSION,
                        MINOR_VERSION,
                        PATCH_LEVEL );

                sprintf(required_engine_api_version_string, "%d.%d.%d",
                        dl_plugin_record->required_engine_api_version.major,
                        dl_plugin_record->required_engine_api_version.minor,
                        dl_plugin_record->required_engine_api_version.patchlevel );

                sprintf(required_plugin_api_version_string, "%d.%d.%d",
                        dl_plugin_record->required_plugin_api_version.plugin.major,
                        dl_plugin_record->required_plugin_api_version.plugin.minor,
                        dl_plugin_record->required_plugin_api_version.plugin.patchlevel );

                SET_STRING( Info->info[DL_PLUGIN_INFO_SNAME_INDEX].name, "Short Name" );
                SET_STRING( Info->info[DL_PLUGIN_INFO_SNAME_INDEX].title, "Short Name" );
                SET_STRING( Info->info[DL_PLUGIN_INFO_SNAME_INDEX].desc, "A short name given to this plugin.");
                Info->info[DL_PLUGIN_INFO_SNAME_INDEX].type               = EVMS_Type_String;
                Info->info[DL_PLUGIN_INFO_SNAME_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[DL_PLUGIN_INFO_SNAME_INDEX].value.s, dl_plugin_record->short_name );
                Info->info[DL_PLUGIN_INFO_SNAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[DL_PLUGIN_INFO_SNAME_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[DL_PLUGIN_INFO_LNAME_INDEX].name, "Long Name" );
                SET_STRING( Info->info[DL_PLUGIN_INFO_LNAME_INDEX].title, "Long Name" );
                SET_STRING( Info->info[DL_PLUGIN_INFO_LNAME_INDEX].desc, "A long name given to this plugin.");
                Info->info[DL_PLUGIN_INFO_LNAME_INDEX].type               = EVMS_Type_String;
                Info->info[DL_PLUGIN_INFO_LNAME_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[DL_PLUGIN_INFO_LNAME_INDEX].value.s, dl_plugin_record->long_name );
                Info->info[DL_PLUGIN_INFO_LNAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[DL_PLUGIN_INFO_LNAME_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[DL_PLUGIN_INFO_TYPE_INDEX].name, "Type" );
                SET_STRING( Info->info[DL_PLUGIN_INFO_TYPE_INDEX].title, "Plugin Type" );
                SET_STRING( Info->info[DL_PLUGIN_INFO_TYPE_INDEX].desc, "There are various types of plugins; each responsible for some kind of storage object.");
                Info->info[DL_PLUGIN_INFO_TYPE_INDEX].type               = EVMS_Type_String;
                Info->info[DL_PLUGIN_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[DL_PLUGIN_INFO_TYPE_INDEX].value.s, "Aggregate Class Feature" );
                Info->info[DL_PLUGIN_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[DL_PLUGIN_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[DL_PLUGIN_INFO_VERSION_INDEX].name, "Version" );
                SET_STRING( Info->info[DL_PLUGIN_INFO_VERSION_INDEX].title, "Plugin Version" );
                SET_STRING( Info->info[DL_PLUGIN_INFO_VERSION_INDEX].desc, "This is the version number of the plugin.");
                Info->info[DL_PLUGIN_INFO_VERSION_INDEX].type               = EVMS_Type_String;
                Info->info[DL_PLUGIN_INFO_VERSION_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[DL_PLUGIN_INFO_VERSION_INDEX].value.s, version_string );
                Info->info[DL_PLUGIN_INFO_VERSION_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[DL_PLUGIN_INFO_VERSION_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[DL_PLUGIN_INFO_REQENGSVCVERSION_INDEX].name, "Required Engine Services Version" );
                SET_STRING( Info->info[DL_PLUGIN_INFO_REQENGSVCVERSION_INDEX].title, "Required Engine Services Version" );
                SET_STRING( Info->info[DL_PLUGIN_INFO_REQENGSVCVERSION_INDEX].desc, "This is the version of the Engine services that this plug-in requires. It will not run on older versions of the Engine services.");
                Info->info[DL_PLUGIN_INFO_REQENGSVCVERSION_INDEX].type               = EVMS_Type_String;
                Info->info[DL_PLUGIN_INFO_REQENGSVCVERSION_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[DL_PLUGIN_INFO_REQENGSVCVERSION_INDEX].value.s, required_engine_api_version_string );
                Info->info[DL_PLUGIN_INFO_REQENGSVCVERSION_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[DL_PLUGIN_INFO_REQENGSVCVERSION_INDEX].group, 0, sizeof(group_info_t));

                SET_STRING( Info->info[DL_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].name, "Required Engine Plug-in API Version" );
                SET_STRING( Info->info[DL_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].title, "Required Engine Plug-in API Version" );
                SET_STRING( Info->info[DL_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].desc, "This is the version of the Engine plug-in API that this plug-in requires. It will not run on older versions of the Engine plug-in API.");
                Info->info[DL_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].type               = EVMS_Type_String;
                Info->info[DL_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].unit               = EVMS_Unit_None;
                SET_STRING( Info->info[DL_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].value.s, required_plugin_api_version_string );
                Info->info[DL_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[DL_PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].group, 0, sizeof(group_info_t));

                *info = Info;
                rc=0;
        }


        LOG_EXIT_INT(rc);
        return rc;
}

