/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libcsm.so
 *
 *   File: helpers.h
 */
#ifndef CSM_HELPERS_HEADER
#define CSM_HELPERS_HEADER 1


DISKSEG *           allocate_csm_segment( LOGICALDISK *object );

void                free_csm_segment( DISKSEG *seg);

DISKSEG *           create_csm_metadata_segment( storage_object_t  *object,
                                                 storage_container_t *container,
                                                 lba_t              start,
                                                 sector_count_t     size,
                                                 char              *name,
                                                 u_int32_t          flags );

DISKSEG *           create_csm_data_segment( storage_object_t  *object,
                                             storage_container_t *container,
                                             lba_t              start,
                                             sector_count_t     size,
                                             u_int32_t          flags );

int                 remove_csm_segment_from_list( list_anchor_t  seglist, DISKSEG *seg );

int                 insert_csm_segment_into_list( list_anchor_t  seglist, DISKSEG *seg);

int                 insert_csm_segment_into_ordered_list( list_anchor_t  seglist, DISKSEG *seg);

char *              guid_to_string( guid_t *id );

int                 string_to_guid( char *str, guid_t *id );

boolean             isa_valid_csm_header( storage_object_t *object, csm_header_t *hdr );

void                prune_csm_seg_objects_from_list( list_anchor_t list );
 
char *              csm_nodeid_to_string( ece_nodeid_t *id );
int                 csm_string_to_nodeid( char *str, ece_nodeid_t *nodeid );

boolean             isa_accessible_container( storage_container_t *container);


static inline LOGICALDISK * get_logical_disk( storage_object_t *object )
{
        LOGICALDISK  *ld=NULL;

        if (object) {

                if (object->object_type == DISK) {
                        ld = object;
                }
                else if (object->object_type == SEGMENT && object->plugin == csm_plugin_record_ptr) {

                        if (object->private_data) {

                                if ( ((seg_private_data_t *)object->private_data)->signature == CSM_SEGMENT_PDATA_SIGNATURE ) {

                                        ld = ((seg_private_data_t *)object->private_data)->logical_disk;

                                }

                        }

                }

        }

        return ld;
}


static inline boolean isa_cluster_logical_disk( storage_object_t *ld )
{
        disk_private_data_t *pdata;

        if (ld) {

                pdata = get_csm_disk_private_data(ld);

                if (pdata) {
                        return TRUE;
                }

        }

        return FALSE;
}


static inline boolean isa_cluster_segment( storage_object_t *object )
{
        seg_private_data_t *pdata;

        if (object) {

                pdata = (seg_private_data_t *) object->private_data;

                if (pdata) {

                        if (pdata->signature == CSM_SEGMENT_PDATA_SIGNATURE) {
                                return TRUE;
                        }

                }

        }

        return FALSE;
}


static inline boolean isa_cluster_container( storage_container_t * container )
{
        container_private_data_t *pdata;

        if (container) {

                pdata = (container_private_data_t *) container->private_data;

                if (pdata) {

                        if (pdata->signature == CSM_CONTAINER_PDATA_SIGNATURE) {
                                return TRUE;
                        }

                }

        }

        return FALSE;
}


static inline lsn_t  rounddown_to_hardsect_boundary( storage_object_t *obj, lsn_t lsn )
{
        int vsects_per_hardsect;
        lsn_t hardsector_alligned_lsn = lsn;

        if (obj) {

                vsects_per_hardsect = obj->geometry.bytes_per_sector >> EVMS_VSECTOR_SIZE_SHIFT;

                if ((lsn+obj->start) % vsects_per_hardsect) {
                        hardsector_alligned_lsn -= (lsn+obj->start) % vsects_per_hardsect;
                }

        }

        return  hardsector_alligned_lsn;             
}



static inline void disk_guid_to_cpu( guid_t  *g )
{
        if (g) {
                g->time_low        = DISK_TO_CPU32(g->time_low);
                g->time_mid        = DISK_TO_CPU16(g->time_mid);
                g->time_high       = DISK_TO_CPU16(g->time_high);
                g->clock_seq_high  = DISK_TO_CPU16(g->clock_seq_high);
                g->clock_seq_low   = DISK_TO_CPU16(g->clock_seq_low);
        }

}


static inline void cpu_guid_to_disk( guid_t  *g )
{
        if (g) {
                g->time_low        = CPU_TO_DISK32(g->time_low);
                g->time_mid        = CPU_TO_DISK16(g->time_mid);
                g->time_high       = CPU_TO_DISK16(g->time_high);
                g->clock_seq_high  = CPU_TO_DISK16(g->clock_seq_high);
                g->clock_seq_low   = CPU_TO_DISK16(g->clock_seq_low);
        }
}


static inline boolean matching_guids( guid_t *g1, guid_t *g2)
{
        if ( g1 && g2 ) {

                if ( (DISK_TO_CPU32(g1->time_low)  == g2->time_low) &&
                     (DISK_TO_CPU16(g1->time_mid)  == g2->time_mid) &&
                     (DISK_TO_CPU16(g1->time_high) == g2->time_high) &&
                     (g1->clock_seq_high           == g2->clock_seq_high) &&
                     (g1->clock_seq_low            == g2->clock_seq_low) ) {

                        return TRUE;
                }

        }

        return FALSE;
}


/*
 *  Function:  isa_known_disk
 *
 *  This routine is called to see if we already know about a
 *  particular disk by using the guid from the csm header.
 *  All disks are registered by converting the guid to an
 *  asciiz and calling the engine api. So, by converting the
 *  guid parm to an asciiz string we can ask the engine if it
 *  knows the name, by using the validate name api, and if it
 *  already knows the name then we know the specified disk!
 */
static inline boolean isa_known_disk( guid_t *disk_id )
{
        boolean result=FALSE;
        char *disk_guid_name=NULL;

        if (disk_id) {

                disk_guid_name = guid_to_string( disk_id );

                if (disk_guid_name) {

                        if (EngFncs->validate_name( disk_guid_name )) {
                                result = TRUE; // engine is saying it knows
                        }                      // the name.
                        free(disk_guid_name);
                }

        }

        return result;
}


static inline void cpu_csm_header_to_disk( csm_header_t *hdr )
{
        if (hdr) {
                hdr->signature              = CPU_TO_DISK64(hdr->signature);
                hdr->size                   = CPU_TO_DISK32(hdr->size);
                hdr->crc                    = CPU_TO_DISK32(hdr->crc);
                hdr->alternate_lba          = CPU_TO_DISK64(hdr->alternate_lba);
                hdr->start_useable          = CPU_TO_DISK64(hdr->start_useable);
                hdr->end_useable            = CPU_TO_DISK64(hdr->end_useable);
                hdr->flags                  = CPU_TO_DISK32(hdr->flags);

                hdr->disk_id.time_low       = CPU_TO_DISK32(hdr->disk_id.time_low);
                hdr->disk_id.time_mid       = CPU_TO_DISK16(hdr->disk_id.time_mid);
                hdr->disk_id.time_high      = CPU_TO_DISK16(hdr->disk_id.time_high);
        }
        else {
                LOG_ERROR("cpu_csm_header_to_disk: error, null ptr arg\n");
        }
}


static inline void disk_csm_header_to_cpu( csm_header_t *hdr )
{
        if (hdr) {
                hdr->signature              = DISK_TO_CPU64(hdr->signature);
                hdr->size                   = DISK_TO_CPU32(hdr->size);
                hdr->crc                    = DISK_TO_CPU32(hdr->crc);
                hdr->alternate_lba          = DISK_TO_CPU64(hdr->alternate_lba);
                hdr->start_useable          = DISK_TO_CPU64(hdr->start_useable);
                hdr->end_useable            = DISK_TO_CPU64(hdr->end_useable);
                hdr->flags                  = DISK_TO_CPU32(hdr->flags);

                hdr->disk_id.time_low       = DISK_TO_CPU32(hdr->disk_id.time_low);
                hdr->disk_id.time_mid       = DISK_TO_CPU16(hdr->disk_id.time_mid);
                hdr->disk_id.time_high      = DISK_TO_CPU16(hdr->disk_id.time_high);
        }
        else {
                LOG_ERROR("disk_csm_header_to_cpu: error, null ptr arg\n");
        }
}


static inline void display_csm_header( csm_header_t *hdr )
{
        char *nodeid=NULL;
        char  type[64];


        if (hdr) {

                if (hdr->flags & CSM_CLUSTER_SHARED) {
                        strcpy(type, "shared");
                }
                else if (hdr->flags & CSM_CLUSTER_PRIVATE) {
                        strcpy(type, "private");
                }
                else if (hdr->flags & CSM_CLUSTER_DEPORTED) {
                        strcpy(type, "deported");
                }
                else {
                        strcpy(type, "unknown");
                }

                nodeid = csm_nodeid_to_string(&hdr->nodeid);
                if (nodeid == NULL) {
                        nodeid = strdup("unknown");
                }

                LOG_DEBUG("CSM HEADER INFO ...\n");
                LOG_DEBUG("     Signature         = 0x%"PRIX64"\n", hdr->signature );
                LOG_DEBUG("     Header Size       = %d\n", hdr->size);
                LOG_DEBUG("     CRC               = 0x%X\n", hdr->crc);
                LOG_DEBUG("     FLAGS             = 0x%X\n", hdr->flags);
                LOG_DEBUG("     Alternate         = %"PRIu64"\n", hdr->alternate_lba);
                LOG_DEBUG("     Start Useable     = %"PRIu64"\n", hdr->start_useable);
                LOG_DEBUG("     End Useable       = %"PRIu64"\n", hdr->end_useable);
                LOG_DEBUG("     Node ID           = %s\n", nodeid);
                LOG_DEBUG("     Storage Type      = %s\n", type );

                if (nodeid) free(nodeid);
        }
        else {
                LOG_ERROR("display_csm_header: error, null ptr arg\n");
        }
}

/*
 *  Function: get device name
 *
 *  On a devfs system ... we need to change the name
 *  from ide/..../disc   to ide/..../part before appending
 *  the partition number.
 */
static inline void get_device_name(storage_object_t *ld, char *device_name)
{       
        int i;

        strcpy(device_name, ld->name);

        for (i=strlen(device_name)-1; i>=0; i--) {
                if (device_name[i] == '/') {
                        if (strncmp(&device_name[i],"/disc",5)==0) {
                                strcpy(&device_name[i+1],"");
                                return;
                        }
                }
        }

}

/*
 *  Function: set segment storage type
 *
 *  Called to update the storage type bits in the flag field of the evms
 *  segment storage object ... based on the storage type of the container.
 */
static inline void set_segment_storage_type(storage_object_t *seg, storage_container_t *container)
{
        if (container->flags & SCFLAG_CLUSTER_PRIVATE) {
                seg->flags &= ~SOFLAG_CLUSTER_SHARED;
                seg->flags &= ~SOFLAG_CLUSTER_DEPORTED;
                seg->flags |= SOFLAG_CLUSTER_PRIVATE;                   
        }
        else if (container->flags & SCFLAG_CLUSTER_SHARED) {
                seg->flags &= ~SOFLAG_CLUSTER_PRIVATE;
                seg->flags &= ~SOFLAG_CLUSTER_DEPORTED;
                seg->flags |= SOFLAG_CLUSTER_SHARED;
        }
        else if (container->flags & SCFLAG_CLUSTER_DEPORTED) {
                seg->flags &= ~SOFLAG_CLUSTER_SHARED;
                seg->flags &= ~SOFLAG_CLUSTER_PRIVATE;
                seg->flags |= SOFLAG_CLUSTER_DEPORTED;
        }
        else {
                LOG_DEBUG("container storage type bits are not set ... treating as deported storage\n");
                seg->flags &= ~SOFLAG_CLUSTER_SHARED;
                seg->flags &= ~SOFLAG_CLUSTER_PRIVATE;
                seg->flags |= SOFLAG_CLUSTER_DEPORTED;
        }
}

static inline void string_to_upper_case(char *str)
{
        int i;
        for (i=0;i<strlen(str); i++) {
                if (str[i] >= 'a'  &&  str[i] <= 'z') {
                        str[i] -= 0x20;
                }
        }
}

#endif

