/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libbsd.so
 *
 *   File: discovery.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "bsd_plugin.h"


/*
 *  Function:  display_segment_list
 *
 *  Called by discovery code, after successfully creating bsd
 *  segment objects, to dump the information to the engine
 *  log for debugging purposes.
 */
static void  display_segment_list( LOGICALDISK  *ld)
{
        DISKSEG *seg;
        list_element_t iter;

        LOG_DEBUG("\n");
        LOG_DEBUG("  BSD Segment List ... logical disk= %s\n", ld->name);

        LIST_FOR_EACH( ld->parent_objects, iter, seg ) {
                LOG_DEBUG("   %08"PRIu64" %08"PRIu64" %08"PRIu64" %s\n",
                          seg->start,
                          seg->start+seg->size-1,
                          seg->size,
                          seg->name );
        }

        LOG_DEBUG("\n");
}


/*
 *  Function:  get_bsd_disk_label
 *
 *  Called to read the disk label sector off a specified logical
 *  disk and verify the disk label.
 * 
 *  Returns: bsd disk label if successful
 *           NULL ptr otherwise
 */
bsd_disklabel_t *  get_bsd_disk_label( LOGICALDISK *ld )
{
        bsd_disklabel_t  *disklabel=NULL;
        int  rc;

        LOG_ENTRY();

        if (ld != NULL) {

                disklabel = (bsd_disklabel_t *) malloc(EVMS_VSECTOR_SIZE); 

                if (disklabel) {

                        rc = READ( ld, 0, 1, ( void *) disklabel );

                        if (!rc) {

                                if ( DISK_TO_CPU32(disklabel->d_magic) != BSD_DISKMAGIC) {
                                        rc = ENODATA;
                                }
                        }

                        if (rc) {
                                free(disklabel);
                                disklabel=NULL;
                        }

                }

        }

        LOG_EXIT_PTR(disklabel);
        return disklabel;
}


/*
 *  Function:  get_segments
 *
 *  Called to create bsd segment storage objects that map the
 *  specified logical disk.  The job of this routine is to build
 *  evms segment storage objects to represent the BSD slices.
 *
 */
static int get_segments( LOGICALDISK *ld,  bsd_disklabel_t *disklabel )
{
        struct bsd_partition       *p;
        int                         rc = 0;
        DISKSEG                    *seg;
        int                         seg_count=0;
        u_int32_t                   ptable_index=0;
        u_int32_t                   minor=1;


        LOG_ENTRY();

        // create a metadata segment to map the disk label
        seg = create_bsd_metadata_segment( ld,  // logical device
                                           0,   // starting offset
                                           1,   // size in sectors 
                                           0 ); // segment flags                                           

        if (seg) {

                rc = insert_bsd_segment_into_list( ld->parent_objects, seg);                                

        }
        else {
                rc = ENOMEM;
        }

        for (  p = disklabel->d_partitions; 
            (p - disklabel->d_partitions < DISK_TO_CPU16(disklabel->d_npartitions) )&&(rc==0); 
            p++ ) {

                if (p->p_fstype != BSD_TAG_UNUSED ) {

                        LOG_DEBUG( "  Slice %d: p_size(%u), p_offset(%u), p_fsize(%u), p_fstype(0x%02X)\n", 
                                   ptable_index, 
                                   DISK_TO_CPU32(p->p_size), 
                                   DISK_TO_CPU32(p->p_offset), 
                                   DISK_TO_CPU32(p->p_fsize), 
                                   p->p_fstype );

                        seg = create_bsd_data_segment( ld,
                                                       DISK_TO_CPU32(p->p_offset), 
                                                       DISK_TO_CPU32(p->p_size),
                                                       p,
                                                       minor,
                                                       0 );

                        if (seg) {

                                rc = insert_bsd_segment_into_list( ld->parent_objects, seg);
                                if (!rc) {
                                        ++minor;
                                        ++seg_count;
                                }

                        }
                        else {
                                rc = ENOMEM;
                        }

                }

                ++ptable_index;
        }


        if (rc) {
                LOG_ERROR("error, problems adding bsd partitions for disk %s.", ld->name );
                seg_count = 0;
                prune_bsd_seg_objects_from_list(ld->parent_objects);
        }


        LOG_EXIT_INT(seg_count);
        return seg_count;
}


/*
 *  Function: get_bsd_segment_devmap_info
 *
 *  Called to test if the segment has an active device mapper
 *  node in the kernel and set the object info accordingly.
 */
static int get_bsd_segment_devmap_info( DISKSEG *seg )
{
        int rc=0;
        dm_target_t *targets=NULL;
        dm_device_t *dev=NULL;

        LOG_ENTRY();

        if (seg->data_type == DATA_TYPE) {

                rc = EngFncs->dm_update_status(seg);
                if (!rc) {

                        if (seg->flags & SOFLAG_ACTIVE) {

                                rc = EngFncs->dm_get_targets(seg, &targets);
                                if (  (!rc) &&                       // RC==Zero
                                      (targets) &&                  // got a target list
                                      (targets->next == NULL) &&    // with 1 target only
                                      (targets->data.linear) &&     // with device data
                                      (targets->start == 0)) {      // and target starts at Zero

                                        dev = targets->data.linear;

                                        if ( ( seg->start != dev->start )||
                                             ( seg->size  != targets->length )) {
                                                LOG_DEBUG("this segment is being marked needs_activate\n");
                                                seg->flags |= SOFLAG_NEEDS_ACTIVATE;
                                        }
                                        else {
                                                seg->flags &= ~SOFLAG_NEEDS_ACTIVATE;
                                        }

                                }

                                if (targets) EngFncs->dm_deallocate_targets(targets);

                        }
                        else {
                                seg->flags |= SOFLAG_NEEDS_ACTIVATE;
                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Function:  discover_bsd_segments
 *
 *  Called by the bsd discover api ... once for each storage object
 *  that is found in the discovery list of objects.  Our job is to
 *  probe a storage object, looking for a bsd disk label.  If we find
 *  our metadata:
 *
 *      - create bsd segment storage objects to map the slices
 *
 *  Return: the number of segment objects created 0-n and place new
 *          segment objects on the output_object list.
 */
int discover_bsd_segments( storage_object_t *obj, list_anchor_t output_objects, uint *count )
{
        boolean                     success=FALSE;
        int                         rc=EINVAL;
        uint                        seg_count=0;
        seg_private_data_t         *pdata=NULL;
        disk_private_data_t        *disk_pdata=NULL;
        struct bsd_disklabel       *disklabel=NULL;
        boolean                     created_disk_pdata=FALSE;
        list_element_t              iter, e;
	storage_object_t	   *seg;

        LOG_ENTRY();
        LOG_DEBUG("examining object %s\n", obj->name );

        pdata = (seg_private_data_t *) obj->private_data;

        // check if object is acceptable and probe for BSD disk label
        if ( ( obj->plugin == bsd_plugin ) ||
             ( obj->data_type != DATA_TYPE ) ||
             ( obj->object_type == SEGMENT ) ) {

                LOG_DEBUG("object is not acceptable or \n" );                

        }
        else {
                disklabel = get_bsd_disk_label( obj );
                if (disklabel) {
                        display_bsd_disklabel( disklabel );
                        rc = 0;
                }
        }

        if (rc==0) {

                create_bsd_disk_private_data( obj );

                disk_pdata = get_bsd_disk_private_data(obj);

                if (disk_pdata) {

                        created_disk_pdata=TRUE;

                        seg_count = get_segments(obj,disklabel);

                        if (seg_count > 0) {

                                display_segment_list(obj);

                                rc = EngFncs->concatenate_lists( output_objects, obj->parent_objects );

                                if (!rc) {
                                        LIST_FOR_EACH( obj->parent_objects, iter, seg ) {
                                                get_bsd_segment_devmap_info(seg);
                                        }                                                                          
                                        success = TRUE;
                                }

                        }
                        else {
                                LOG_ERROR("error, BSD object but no segments produced\n");
                        }

                }
                else {
                        LOG_ERROR("error, unable to create logical disk private data\n");
                }

        }

        if (success == FALSE) {

                // toss any segments we created
                prune_bsd_seg_objects_from_list(obj->parent_objects);

                // place the disk storage object back onto the discovery list
                e=EngFncs->insert_thing( output_objects, 
                                         obj,
                                         INSERT_AFTER | EXCLUSIVE_INSERT,
                                         NULL );
                if (e!=NULL) {
                        rc = 0;
                }
                else {
                        rc = EPERM;
                }

                // toss our disk private data
                if (created_disk_pdata==TRUE) {
                        delete_bsd_disk_private_data(obj);
                }

                // return - no objects created
                seg_count = 0;

        }

        LOG_DEBUG( "discovery ...resulted in %d new segments\n", seg_count );

        // return count of objects we created
        *count += seg_count;

        if (disklabel) free(disklabel);

        LOG_EXIT_INT(0);
        return 0;
}

