/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: bbr_seg.c
 *
 *   BBR == bad block relocation
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <uuid/uuid.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>

#include <plugin.h>

#include "bbr_seg.h"
#include "options.h"

static int BBR_Write( storage_object_t * object,
                      lsn_t              lsn,
                      sector_count_t     count,
                      void             * buffer);


/*-------------------------------------------------------------------------+
+                                                                          +
+              PRIVATE DATA AREAS AND SUBROUTINES                          +
+                                                                          +
+--------------------------------------------------------------------------*/

static plugin_record_t bbr_plugin_record;
plugin_record_t *my_plugin_record = &bbr_plugin_record;
engine_functions_t *EngFncs = NULL;

/*
 *  Only called when debugging ... to display current bbr mapping table.
 *
static void  display_bbr_remap_info( storage_object_t *object, BBR_Private_Data *pdata )
{
    int i,j;
    evms_bbr_table_t  *table=pdata->bbr_table;

LOG_DEBUG("BBR REMAP INFO...\n");
LOG_DEBUG("   TABLE INFO ...\n");
LOG_DEBUG("     1st table LSN: %"PRIu64"\n", pdata->bbr_table_lsn1 );
LOG_DEBUG("     2nd table LSN: %"PRIu64"\n", pdata->bbr_table_lsn2 );
LOG_DEBUG("        table size: %"PRIu64"\n", pdata->bbr_table_size_in_sectors );
LOG_DEBUG("         table ptr: %p\n", pdata->bbr_table );

    for(i=0; i<pdata->bbr_table_size_in_sectors; i++) {

LOG_DEBUG("   TABLE SECTOR %d\n", i );
LOG_DEBUG("      in use count: %d\n", table->in_use_cnt );

        if (table->in_use_cnt > 0) {
            for (j=0; j<EVMS_BBR_ENTRIES_PER_SECT; j++) {
                if ( table->entries[j].replacement_sect != 0 ) {

LOG_DEBUG("BAD LSN: %"PRIu64"  REMAP LSN: %"PRIu64"\n", table->entries[j].bad_sect, table->entries[j].replacement_sect );

                }
            }
        }

        ++table;
    }

LOG_DEBUG("   REPL BLOCKS...\n");
LOG_DEBUG("  total replacement sectors = %"PRIu64"\n", pdata->bbr_table_size_in_sectors );
LOG_DEBUG(" 1st replacement sector LSN = %"PRIu64"\n",  pdata->replacement_blocks_lsn );

}
*/

/*
 *  Called to ask the kernel bbr plugin to perform a sector
 *  READ or WRITE for us.
 *
 *  Returns: the status field set by the kernel bbr plugin
 *           errno if the ioctl itself fails
 */
static int kernel_bbr_sector_io( storage_object_t * object,
                                 lsn_t              lsn,
                                 sector_count_t     count,
                                 void             * buffer,
                                 int                rw )
{
        int        rc=EIO, fd;
        u_int64_t  bytes=0;

        LOG_ENTRY();

        fd = EngFncs->open_object(object, O_RDWR | O_SYNC);
        if (fd>=0) {
            if (rw == SECTOR_IO_WRITE){
                bytes = EngFncs->write_object
                    ( object, fd, buffer,
                      count<<EVMS_VSECTOR_SIZE_SHIFT,
                      lsn<<EVMS_VSECTOR_SIZE_SHIFT);
            }
            else {
                bytes =  EngFncs->read_object
                    ( object, fd, buffer,
                      count<<EVMS_VSECTOR_SIZE_SHIFT,
                      lsn<<EVMS_VSECTOR_SIZE_SHIFT );
            }


            if ( bytes == count<<EVMS_VSECTOR_SIZE_SHIFT )
                rc = 0;
            else
                rc = EIO;

            EngFncs->close_object(object, fd );
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to see if the kernel bbr plugin is active on this object.
 *
 *  Returns:  TRUE if BBR storage object exists in the kernel
 *            FALSE otherwise
 */
static boolean isa_kernel_bbr_object( storage_object_t *object )
{
        boolean result = FALSE;

        LOG_ENTRY();

        //do an dm_update_status here?

        if ( object->flags & SOFLAG_ACTIVE ) {
                result = TRUE;
        }

        LOG_EXIT_BOOL(result);
        return result;
}



/*
 *  Called to get a count of bad blocks being remapped in the
 *  bbr mapping table.
 *
 *  Returns:  count of remapped sectors if successful
 *            otherwise ... 0
 */
static sector_count_t get_bad_block_count( evms_bbr_table_t  *bbr_table,
                                           sector_count_t     sector_count )
{
        int i;
        sector_count_t  bad_blocks = 0;
        evms_bbr_table_t   *table  = bbr_table;

        LOG_ENTRY();

        if (bbr_table && sector_count) {

            for( i = 0; i < sector_count; i++ ) {
                bad_blocks += table->in_use_cnt;
                ++table;
            }

        }

        LOG_DEBUG("     returning count= %"PRIu64"\n", bad_blocks );

        LOG_EXIT_INT((int)bad_blocks);
        return bad_blocks;
}



/*
 *  Called to get a count of bad blocks being remapped to good
 *  sectors by the kernel bbr feature.
 *
 *  Returns:  count of remapped sectors if successful
 *            otherwise ... 0
 */
static sector_count_t
get_kernel_bbr_remap_sector_count( storage_object_t *object )
{
        int rc;
        BBR_Private_Data  *pdata = (BBR_Private_Data *)object->private_data;
        sector_count_t     bad_blocks=0, bad_blocks1=0, bad_blocks2=0, sector_count;
        evms_bbr_table_t   *table;

        LOG_ENTRY();

        if (pdata) {

            if (pdata->bbr_table_size_in_sectors) {

                table = malloc( pdata->bbr_table_size_in_sectors
                                 <<EVMS_VSECTOR_SIZE_SHIFT );
                if (table) {

                    sector_count = pdata->bbr_table_size_in_sectors;

                    rc = kernel_bbr_sector_io( object,
                                               pdata->bbr_table_lsn1,
                                               sector_count,
                                               (void *) table,
                                               SECTOR_IO_READ );

                    if (!rc) {
                        bad_blocks1 = get_bad_block_count(table,sector_count);
                    }


                    rc = kernel_bbr_sector_io( object,
                                               pdata->bbr_table_lsn2,
                                               sector_count,
                                               (void *) table,
                                               SECTOR_IO_READ );

                    if (!rc) {
                        bad_blocks2 = get_bad_block_count(table,sector_count);
                    }

                    free(table);
                }

            }

        }

        if (bad_blocks1 == bad_blocks2) {
            bad_blocks = bad_blocks1;
        }
        else if (bad_blocks1 > bad_blocks2) {
            bad_blocks = bad_blocks1;
        }
        else {
            bad_blocks = bad_blocks2;
        }

        LOG_DEBUG("     returning count= %"PRIu64"\n", bad_blocks );
        LOG_EXIT_INT((int)bad_blocks);
        return bad_blocks;
}


/*
 *  Called to get a count of bad blocks being remapped to good
 *  sectors by examining the bbr mapping table.
 *
 *  Returns:  count of remapped sectors if successful
 *            otherwise ... 0
 */
static sector_count_t get_engine_remap_sector_count( storage_object_t *object )
{
        BBR_Private_Data  *pdata = (BBR_Private_Data *)object->private_data;
        sector_count_t     bad_blocks=0;

        LOG_ENTRY();

        if (pdata) {

            bad_blocks = get_bad_block_count
                ( pdata->bbr_table, pdata->bbr_table_size_in_sectors );

        }

        LOG_DEBUG("     returning count= %"PRIu64"\n", bad_blocks );
        LOG_EXIT_INT((int)bad_blocks);
        return bad_blocks;
}


/*
 *  Called to convert LSN to a remapped LSN.
 *
 *  Returns: the LSN of the replacement block if the sector has been remapped
 *           otherwise ... it returns the callers LSN
 */
static lsn_t  get_lsn( BBR_Private_Data *pdata, lsn_t lsn )
{
    int i,j;
    evms_bbr_table_t  *table=pdata->bbr_table;

    LOG_ENTRY();

    for(i=0; i<pdata->bbr_table_size_in_sectors; i++) {

        if (table->in_use_cnt > 0) {
            for (j=0; j<EVMS_BBR_ENTRIES_PER_SECT; j++) {
                if ( ( table->entries[j].bad_sect == lsn ) &&
                     ( table->entries[j].replacement_sect != 0)) {
                    LOG_EXIT_INT((int)(table->entries[j].replacement_sect));
                    return table->entries[j].replacement_sect;
                }
            }
        }

        ++table;
    }

    LOG_EXIT_INT((int)lsn);
    return lsn;
}


/*
 *  Called to get the next available (unused) replacement block LSN
 *
 *  Returns: success: LSN of the replacement block
 *           failure: returns invalid LSN ( LSN==0 )
 */
static lsn_t get_next_avail_repl_block_lsn( BBR_Private_Data *pdata )
{
    int i;
    evms_bbr_table_t  *table = pdata->bbr_table;
    lsn_t              lsn   = 0;
    int                sectors_in_use=0;

    LOG_ENTRY();

    // count replacement sectors being used
    for(i=0; i<pdata->bbr_table_size_in_sectors ; i++) {
        sectors_in_use += table->in_use_cnt;
        ++table;
    }

    // next avail is N+1 but check that 1 is still available.
    if ( sectors_in_use ) {

        if ( sectors_in_use < pdata->replacement_blocks_size_in_sectors ) {

            lsn = pdata->replacement_blocks_lsn + sectors_in_use;

        }
        else {   // error, there are no replacement blocks available
            LOG_ERROR("error, unable to provide remap because all replacement blocks are used\n");
            lsn = 0;
        }

    }
    else { // ok, grab 1st cuz no replacement blocks being used right now
        lsn = pdata->replacement_blocks_lsn;
    }


    LOG_EXIT_INT((int)lsn);
    return lsn;
}

/*
 *  This routine is called to convert a bbr table from cpu format
 *  to disk format
 */
static void CPU_BBR_Table_To_Disk_Table( evms_bbr_table_t *bbr_table,
                                         sector_count_t sector_count )
{
    sector_count_t    i;
    int               j;
    vsector_t        *sector_ptr;
    evms_bbr_table_t *table;

    LOG_ENTRY();

    if (  bbr_table != NULL )  {

        sector_ptr = (vsector_t *) bbr_table;

        //  one sector at a time
        for (i=0; i<sector_count; i++) {

            table = (evms_bbr_table_t *) sector_ptr;

            table->signature       = CPU_TO_DISK32(table->signature);
            table->crc             = CPU_TO_DISK32(table->crc);
            table->sequence_number = CPU_TO_DISK32(table->sequence_number);
            table->in_use_cnt      = CPU_TO_DISK32(table->in_use_cnt);

            for (j=0; j<EVMS_BBR_ENTRIES_PER_SECT; j++) {
                table->entries[j].bad_sect =
                    CPU_TO_DISK64(table->entries[j].bad_sect);
                table->entries[j].replacement_sect =
                    CPU_TO_DISK64(table->entries[j].replacement_sect);
            }

            ++sector_ptr;
        }

    }

    LOG_EXIT_VOID();
}

/*
 *  Called to write the BBR table. The LSN of the table is obtained
 *  from the BBR feature metadata.
 *
 *  Assumes: evms_bbr_metadata_t and evms_bbr_table_t are padded to 512 bytes
 *
 *  Returns: 0 if BBR Table is successfully written to disk
 *
 */
static int WriteBBRTable( storage_object_t       * object,
                          u_int64_t                lsn,
                          u_int64_t                sector_count,
                          vsector_t              * bbr_table )
{
    int           rc=ENOMEM;
    int           i;
    u_int32_t     crc;
    vsector_t    *table=NULL;
    void         *buffer=NULL;


    LOG_ENTRY();

    LOG_DEBUG("Writing BBR mapping table on object %s at lsn %"PRIu64"\n", object->name, lsn );

    // need a buffer to hold endian neutral table
    buffer = (void *) malloc( sector_count * EVMS_VSECTOR_SIZE );
    table = (vsector_t *)buffer;
    if (table != NULL) {

        memcpy(table, bbr_table, (sector_count * EVMS_VSECTOR_SIZE) );

        // convert to disk format
        CPU_BBR_Table_To_Disk_Table
            ( (evms_bbr_table_t*) table, (sector_count_t) sector_count );

        // step through the bbr mapping table and calc the crc for each sector.
        for (i=0; i < sector_count; i++) {

              ((evms_bbr_table_t *)table)->crc = 0;

            crc = EngFncs->calculate_CRC
                ( EVMS_INITIAL_CRC, table, EVMS_VSECTOR_SIZE );

            ((evms_bbr_table_t *)table)->crc = CPU_TO_DISK32(crc);

            ++table;   // advance to next sector in table
        }

        rc = WRITE( object, lsn, sector_count, (void *) buffer );

        free(buffer);
    }

    LOG_EXIT_INT(rc);
    return rc;
}

/*
 *  Called to remap a bad block LSN to a replacement BLOCK LSN.
 *
 *  Returns: success: LSN of the replacement block
 *           failure: invalid LSN (LSN==0)
 */
static lsn_t  remap_lsn( storage_object_t *parent, BBR_Private_Data *pdata,
                         lsn_t lsn )
{
    int i,j;
    evms_bbr_table_t  *table=pdata->bbr_table;
    lsn_t              replacement_sect;

    LOG_ENTRY();

    for(i=0; i<pdata->bbr_table_size_in_sectors; i++) {

        if ( table->in_use_cnt < EVMS_BBR_ENTRIES_PER_SECT ) {

            for (j=0; j<EVMS_BBR_ENTRIES_PER_SECT; j++) {

                if ( ( table->entries[j].bad_sect == 0 ) &&
                     ( table->entries[j].replacement_sect == 0)) {

                    replacement_sect = get_next_avail_repl_block_lsn(pdata);

                    if ( replacement_sect ) {

                        // update bbr table
                        table->entries[j].bad_sect = lsn;
                        table->entries[j].replacement_sect = replacement_sect;
                        ++table->in_use_cnt;

                        // if not a new bbr object then we need to
                        // sync the bbr table with the on-disk copy
                        if ( (parent->flags & SOFLAG_NEW) == 0 ) {

                            WriteBBRTable( pdata->child,
                                           pdata->bbr_table_lsn1,
                                           pdata->bbr_table_size_in_sectors,
                                           (vsector_t *)pdata->bbr_table );

                            WriteBBRTable( pdata->child,
                                           pdata->bbr_table_lsn2,
                                           pdata->bbr_table_size_in_sectors,
                                           (vsector_t *)pdata->bbr_table );
                        }

                    }

                    LOG_EXIT_INT((int)replacement_sect);
                    return replacement_sect;
                }
            }

        }

        ++table;
    }

    LOG_ERROR("error, no repl blks available\n");
    LOG_EXIT_INT(0);
    return 0;
}


/*
 *  Called to test if we own the specified storage object.  This is also
 *  an opportunity to place code here to further inspect an object prior
 *  to making any changes to it.
 */

static boolean i_can_modify_object( storage_object_t *object )
{
    LOG_ENTRY();

    if (object) {

        // check that we claimed this storage object
        if (object->plugin == my_plugin_record ) {

            // check if we have a private data area for it
            if ( object->private_data ) {

                // test for BBR signature
                if ( ((BBR_Private_Data *)object->private_data)->signature==
                     EVMS_BBR_SIGNATURE) {
                    LOG_DEBUG("Can modify\n");
                    LOG_EXIT_BOOL(TRUE);
                    return TRUE;
                }


            }

        }

    }


    LOG_EXIT_BOOL(FALSE);
    return FALSE;
}

/*
 * Get the useable area of a BBR child object.  Round the useable area 
 * down to a block boundary to prevent higher layers from laying down
 * an evms feature header on an unaccessible sector because the i/o is
 * done through the VFS in block size amounts.
 */
static sector_count_t get_child_useable_size( storage_object_t *parent,
                                              storage_object_t *child )
{
    sector_count_t     child_useable_size=0;
    sector_count_t     metadata_sectors=0;
    BBR_Private_Data  *pdata;
    sector_count_t     blocksize;

    if (parent&&child) {

	blocksize = child->geometry.block_size >> EVMS_VSECTOR_SIZE_SHIFT;  // bytes to vsectors

        pdata = (BBR_Private_Data *) parent->private_data;

        if (pdata) {
            metadata_sectors = pdata->replacement_blocks_lsn +
                pdata->replacement_blocks_size_in_sectors;
            child_useable_size  = child->size - metadata_sectors;
	    child_useable_size -= child_useable_size % blocksize;  // round down to block boundary
        }

    }

    return child_useable_size;
}


/*
 *  Called to validate that a run of disk sectors are Ok.
 *
 *  Just a simple read test on the specified run of sectors
 *  on the storage object.
 *
 *  Returns the number of sectors we read successfully,
 *  prior to an error or successful completion.
 *
 */
static int validate_sectors( storage_object_t *object, u_int64_t start,
                             u_int64_t count, u_int64_t *good_sector_count )
{
    int rc=0;
    int i;
    int use_progress;
    u_int64_t  lsn = start;
    char buffer[EVMS_VSECTOR_SIZE];
    progress_t progress;

    LOG_ENTRY();

    *good_sector_count = 0;

    /*
     * Use the progress service if the UI supports it. Otherwise,
     * use the user message service to provide some status.
     */

    progress.id = 0;
    progress.title = "Performing I/O tests on replacements blocks...";
    progress.description = "";
    progress.type = DISPLAY_PERCENT;
    progress.count = 0;
    progress.total_count = count;
    progress.remaining_seconds = 0;
    progress.plugin_private_data = NULL;
    progress.ui_private_data = NULL;

    use_progress = EngFncs->progress(&progress) == 0;

    if (!use_progress) {
        MESSAGE("\nPerformaing I/O tests on replacement blocks for object %s. This will take a moment or two.\n",
              object->name );
    }

    for (i=0; (i<count)&&(rc==0); i++,lsn++) {

    if (use_progress && !(i % 100)) {
            progress.count = i;
        EngFncs->progress(&progress);
    }

        rc = READ(object, lsn, 1, buffer);

        if (rc == 0) {

          ++ *good_sector_count;

        }

    }

    if (use_progress) {
        progress.title = "Finished testing replacement blocks.";
        progress.count = count;

        EngFncs->progress(&progress);
    }
    else {
        MESSAGE("\nFinished testing replacement blocks.\n");
    }

    LOG_EXIT_INT(rc);
    return rc;
}

/*
 *  This routine is called to initialize a new bbr mapping table
 *  that doesnt map any replacement sectors.
 */
static void initialize_bbr_table( evms_bbr_table_t *bbr_table,
                                  u_int64_t  sector_count )
{
    int  i;
    evms_bbr_table_t *table = bbr_table;

    LOG_ENTRY();
    LOG_DEBUG("table addr= %p   sector count= %"PRIu64"\n",
              bbr_table, sector_count);

    if (  table != NULL )  {

        //  one sector at a time
        for (i=0; i<sector_count; i++) {

            table->signature = EVMS_BBR_TABLE_SIGNATURE;

            ++table;
        }

    }


    LOG_EXIT_VOID();
}



/*
 *  Returns the specified LSN rounded up to a device sector boundary
 */
static lsn_t  roundup_to_hardsect_boundary( storage_object_t *obj, lsn_t lsn )
{
        int vsects_per_hardsect;
        lsn_t hardsector_alligned_lsn = lsn;

        if (obj) {
        
                vsects_per_hardsect = obj->geometry.bytes_per_sector >> EVMS_VSECTOR_SIZE_SHIFT;
               
                if ((lsn+obj->start) % vsects_per_hardsect) {
                        hardsector_alligned_lsn = lsn - 
                                                  ((lsn+obj->start) % vsects_per_hardsect) +
                                                  vsects_per_hardsect;
                }
                                
        }

        return  hardsector_alligned_lsn; 
}


/*
 *  Called to fill in BBR metadata info for a BBR object. This
 *  routine is called from the CREATE api code.
 *
 *  The goal is to reserve no more than 1% of the region for
 *  replacement blocks.
 *
 *  The minimum amount we'll reserve is 63 sectors ... about a track
 *  of disk space in most cases.
 *
 *
 *  Returns: rc==0 if successful
 *
 */
static int create_bbr_metadata( storage_object_t    *parent,
                                storage_object_t    *child )
{
    int                   rc=0;
    BBR_Private_Data     *pdata;
    u_int64_t             good_sectors;
    int                   vsects_per_hardsect;

    LOG_ENTRY();

    vsects_per_hardsect = child->geometry.bytes_per_sector >> EVMS_VSECTOR_SIZE_SHIFT;

    if ( parent != NULL &&
         child != NULL  &&         
         ( vsects_per_hardsect == 1 ||    // looking for power of 2
           vsects_per_hardsect == 2 ||
           vsects_per_hardsect == 4 ||
           vsects_per_hardsect == 8 ||
           vsects_per_hardsect == 16 )) {

        // get BBR private data area
        pdata = (BBR_Private_Data *) parent->private_data;

        // size of replacement blocks
        pdata->block_size = EVMS_VSECTOR_SIZE;

        // figure out the number of replacement blocks needed.
        // default amount, trying for 1% of the storage region.
        pdata->replacement_blocks_needed = child->size / 100;

        // enforce MINIMUM and MAXIMUM and HARD SECTOR MULTIPLE rules
        if ( pdata->replacement_blocks_needed <  BBR_MIN_REPLACEMENT_BLOCKS ) {
		pdata->replacement_blocks_needed =  BBR_MIN_REPLACEMENT_BLOCKS;
        }
        else if ( pdata->replacement_blocks_needed > BBR_MAX_REPLACEMENT_BLOCKS ) {
		pdata->replacement_blocks_needed =  BBR_MAX_REPLACEMENT_BLOCKS;
        }
        else if (pdata->replacement_blocks_needed % vsects_per_hardsect) {                
                pdata->replacement_blocks_needed -= pdata->replacement_blocks_needed % vsects_per_hardsect;
		pdata->replacement_blocks_needed += vsects_per_hardsect;
        }

        // total number of sectors we need
	pdata->replacement_blocks_size_in_sectors = pdata->replacement_blocks_needed;

        // sectors being used.
        pdata->bbr_table_size_in_sectors  =
            pdata->replacement_blocks_needed / EVMS_BBR_ENTRIES_PER_SECT;
        if ( pdata->replacement_blocks_needed % EVMS_BBR_ENTRIES_PER_SECT ){
            ++pdata->bbr_table_size_in_sectors;
        }

        // force mapping tables to be a multiple of the hardsector size to prevent 
	// partial sector i/o requirements in the kernel plug-in
        if (pdata->bbr_table_size_in_sectors % vsects_per_hardsect) {
                pdata->bbr_table_size_in_sectors -= pdata->bbr_table_size_in_sectors % vsects_per_hardsect;
		pdata->bbr_table_size_in_sectors += vsects_per_hardsect;
        }

        // set bbr table metadata fields ... skip boot sector and 2 metadata sectors at 
	// start of the storage object.
        pdata->bbr_table_lsn1         = roundup_to_hardsect_boundary(child,3);
        pdata->bbr_table_lsn2         = pdata->bbr_table_lsn1 + pdata->bbr_table_size_in_sectors;        
        pdata->replacement_blocks_lsn = pdata->bbr_table_lsn2 + pdata->bbr_table_size_in_sectors;

        // validate that we are not using BAD replacement sectors
        rc = validate_sectors( child, 
			       pdata->replacement_blocks_lsn , 
			       pdata->replacement_blocks_size_in_sectors, 
			       &good_sectors );
        if (rc) {
            LOG_ERROR("unable to lay down requested number of replacement sectors, only first %"PRIu64" sectors were Ok\n",
                      good_sectors);
            LOG_EXIT_INT(rc);
            return rc;
        }

        // malloc a bbr table
        pdata->bbr_table = EngFncs->engine_alloc
            ( (pdata->bbr_table_size_in_sectors*EVMS_VSECTOR_SIZE) );
        if (pdata->bbr_table == NULL) {
            LOG_ERROR("unable to malloc a new BBR mapping table of %"PRIu64" sectors, rc= ENOMEM\n", pdata->bbr_table_size_in_sectors);
            LOG_EXIT_INT(ENOMEM);
            return ENOMEM;
        }
        else {
            initialize_bbr_table( pdata->bbr_table,
                                  pdata->bbr_table_size_in_sectors );
        }

 
/* BBR_DEBUG
        LOG_DEBUG("\tBBR LSN Calculations...\n");
        LOG_DEBUG("\t              child name: %s\n", child->name );
        LOG_DEBUG("\t              child size: %"PRIu64"\n", child->size );
        LOG_DEBUG("\t     feature header lsn1: %"PRIu64"\n", child->size - 1 );
        LOG_DEBUG("\t     feature header lsn2: %"PRIu64"\n", child->size - 2 );
        LOG_DEBUG("\t        repl_sectors_lsn: %"PRIu64"\n",
        pdata->replacement_blocks_lsn );
        LOG_DEBUG("\t          nr_repl_blocks: %"PRIu64"  (%"PRIu64" sectors)\n",
        pdata->replacement_blocks_needed,
        pdata->replacement_blocks_size_in_sectors);
        LOG_DEBUG("\t           bbr_table_ptr: %p\n", pdata->bbr_table );
        LOG_DEBUG("\t    nr_bbr_table_sectors: %"PRIu64"\n",
        pdata->bbr_table_size_in_sectors );
        LOG_DEBUG("\t  2nd copy bbr table lsn: %"PRIu64"\n", pdata->bbr_table_lsn2 );
*/
    }
    else {
        rc = EINVAL;
    }


    LOG_EXIT_INT(rc);
    return rc;
}

/*
 *  Called to build a BBR feature metadata structure from the BBR private data
 *  info. It will build a copy of the metadata that can be layed down
 *  at location ... 1 and 2
 *
 *  The second copy of the feature data is the same.
 *
 *  The second table location can be figured by adding the size of the table
 *  table 1's lsn.
 *
 *  Layout:
 *
 *       _____________________________________________________________________
 *      |           |          |          |       |       |            |
 *       Boot Sector Metadata 1 Metadata 2 Table 1 Table 2 Replacements Data ...
 *      |___________|__________|__________|_______|_______|____________|_________
 */
static int BuildMetadata( storage_object_t  *obj,
                          evms_bbr_metadata_t  *metadata )
{

    BBR_Private_Data *pdata = (BBR_Private_Data *) obj->private_data;

    LOG_ENTRY();

    memset(metadata, 0, sizeof(evms_bbr_metadata_t) );

    metadata->signature                = EVMS_BBR_SIGNATURE;

    // bbr table info
    metadata->start_sect_bbr_table     = pdata ->bbr_table_lsn1;
    metadata->nr_sects_bbr_table       = pdata ->bbr_table_size_in_sectors;

    // replacement sector info
    metadata->start_replacement_sect   = pdata ->replacement_blocks_lsn;
    metadata->nr_replacement_blks      = pdata ->replacement_blocks_needed;

    // block size
    metadata->block_size               = pdata ->block_size;

    //enabled or disabled setup could be changing
    if ( pdata->bbr_state & BBR_CHANGE_STATE ) {
        if ( pdata->bbr_state & BBR_ACTIVATE ) {
            metadata->bbr_active |= BBR_ENABLED;
        }
        else
            metadata->bbr_active &= ~BBR_ENABLED;
    }
    else
        metadata->bbr_active = pdata->bbr_state;


    strcpy( metadata->uuid, obj->uuid );

    LOG_EXIT_INT(0);
    return 0;
}



/*
 *  This routine is called to convert a bbr table from disk format
 *  to cpu format
 */
static void Disk_BBR_Table_To_CPU_Table( evms_bbr_table_t *bbr_table,
                                         sector_count_t sector_count )
{
    sector_count_t    i;
    int               j;
    vsector_t        *sector_ptr;
    evms_bbr_table_t *table;

    LOG_ENTRY();

    if (  bbr_table != NULL )  {

        sector_ptr = (vsector_t *) bbr_table;

        //  one sector at a time
        for (i=0; i<sector_count; i++) {

            table = (evms_bbr_table_t *) sector_ptr;

            table->signature       = DISK_TO_CPU32(table->signature);
            table->crc             = DISK_TO_CPU32(table->crc);
            table->sequence_number = DISK_TO_CPU32(table->sequence_number);
            table->in_use_cnt      = DISK_TO_CPU32(table->in_use_cnt);

            for (j=0; j<EVMS_BBR_ENTRIES_PER_SECT; j++) {
                table->entries[j].bad_sect =
                    DISK_TO_CPU64(table->entries[j].bad_sect);
                table->entries[j].replacement_sect =
                    DISK_TO_CPU64(table->entries[j].replacement_sect);
            }

            ++sector_ptr;
        }

    }

    LOG_EXIT_VOID();
}

/*
 *  Called to read the BBR table. The LBA of the table is obtained
 *  from the BBR metadata.
 *
 *  Returns: BBR Table in the supplied buffer if RC=0
 *
 */
static int ReadBBRTable( storage_object_t * object, void * bbr_table,
                         u_int64_t lsn, u_int64_t count )
{
    int          rc;
    int          i;
    u_int32_t    crc;
    u_int32_t    calculated_crc;
    vsector_t   *table;


    LOG_ENTRY();

    LOG_DEBUG("\tbbr table lsn= %"PRIu64"   nr_sects= %"PRIu64"\n", lsn, count);

    // read the sectors off of the disk
    rc = READ( object, lsn, count, bbr_table );

    // if read is Ok then ...
    if (rc==0) {

        table = (vsector_t *) bbr_table;

        // check signature and crc of each sector in table
        for (i=0; (i<count) && (rc==0); i++) {

            if ( DISK_TO_CPU32(((evms_bbr_table_t *)table)->signature) ==
                 EVMS_BBR_TABLE_SIGNATURE ) {

                crc = DISK_TO_CPU32(((evms_bbr_table_t *)table)->crc);

                ((evms_bbr_table_t *)table)->crc = 0;

                calculated_crc = EngFncs->calculate_CRC
                    ( EVMS_INITIAL_CRC, table, EVMS_VSECTOR_SIZE );

                ((evms_bbr_table_t *)table)->crc = CPU_TO_DISK32(crc);

                if ( crc == calculated_crc ) {
                    rc = 0;
                }
                else {
                    LOG_ERROR("crc failed on bbr_table[%d] sector, expected %X  calcd %X\n", i, crc, calculated_crc);
                    rc = ENODATA;
                }
            }
            else {
                LOG_ERROR("not our signature\n");
                rc = ENODATA;
            }

            ++table;  // advance to next sector in table
        }

        if (rc == 0) {
            Disk_BBR_Table_To_CPU_Table
                ( (evms_bbr_table_t *) bbr_table, count );
        }

    }

    LOG_EXIT_INT(rc);
    return rc;
}


/*
 *  Called to get a copy of the BBR table. The LBA of the table is obtained
 *  from the BBR private data.
 *
 *  BBR Table memory is allocated by caller!!
 *
 *  Returns: BBR Table in the supplied buffer if RC=0
 *
 */
static int GetBBRTable( storage_object_t       * child,
                        void                   * bbr_table,
                        sector_count_t           count,
                        lba_t                    bbr_table_lsn1,
                        lba_t                    bbr_table_lsn2 )
{
    int i;
    int rc=EINVAL;
    int rc1=ENODATA;
    int rc2=ENODATA;

    char      *table1=NULL;
    char      *table2=NULL;

    vsector_t  *bbrtable1;
    vsector_t  *bbrtable2;

    u_int32_t  sequence1;
    u_int32_t  sequence2;

    u_int32_t  in_use1;
    u_int32_t  in_use2;


    LOG_ENTRY();

    if ( (child == NULL) ||
         (bbr_table == NULL) ||
         (bbr_table_lsn1==0)) {
        LOG_EXIT_INT(rc);
        return rc;
    }

    // read table 1
    table1 = EngFncs->engine_alloc( count * EVMS_VSECTOR_SIZE );
    if (table1) {

        rc1  = ReadBBRTable( child, (void *)table1, bbr_table_lsn1, count );

    }
    else {
        rc1 = ENOMEM;
    }

    // read table 2 ... if we have a second copy
    if ( ( bbr_table_lsn1 != bbr_table_lsn2 ) &&
         ( bbr_table_lsn2 != 0 ) ) {

        table2 = EngFncs->engine_alloc( count * EVMS_VSECTOR_SIZE );
        if (table2) {

            rc2 = ReadBBRTable
                ( child, (void *)table2, bbr_table_lsn2, count  );

        }
        else {
            rc2 = ENOMEM;
        }

    }

    if (( rc1 == 0) && (rc2 == 0)){

        // ------  COMBINE THE TWO BBR MAPPING TABLES INTO ONE ------
        // merge 2 good tables into 1, using the most recent copy of
        // each sector in the BBR table by inspecting the sequence numbers
        // and sector used counters. The most recent copy of a BBR table
        // sector should have a higher sequence number but if the kernel didnt
        // update this field yet ... then the in_use counter should tell us
        // which is the most recent sector
        //---------------------------------------------------------------

        bbrtable1 = (vsector_t *)table1;
        bbrtable2 = (vsector_t *)table2;
        for(i=0; i<count; i++){

            sequence1 = ((evms_bbr_table_t *)bbrtable1)->sequence_number;
            sequence2 = ((evms_bbr_table_t *)bbrtable2)->sequence_number;
            in_use1   = ((evms_bbr_table_t *)table1)->in_use_cnt;
            in_use2   = ((evms_bbr_table_t *)table2)->in_use_cnt;

            if ( sequence2 > sequence1 ) {
                memcpy(bbrtable1, bbrtable2, sizeof(vsector_t));
            }
            else if ( in_use2 > in_use1 ) {
                memcpy(bbrtable1, bbrtable2, sizeof(vsector_t));
            }

            ++bbrtable1;
            ++bbrtable2;
        }

        memcpy( bbr_table, table1, count * EVMS_VSECTOR_SIZE );
        rc = 0;
    }
    else if (rc1 == 0) {
        memcpy( bbr_table, table1, count * EVMS_VSECTOR_SIZE );
        rc = 0;
    }
    else if (rc2 == 0) {
        memcpy( bbr_table, table2, count * EVMS_VSECTOR_SIZE );
        rc = 0;
    }
    else {
        rc = ENODATA;
    }


    if (table1) EngFncs->engine_free( table1 );
    if (table2) EngFncs->engine_free( table2 );

    LOG_EXIT_INT(rc);

    return rc;
}

static void Disk_Metadata_To_CPU( evms_bbr_metadata_t *metadata )
{

    metadata->signature          = DISK_TO_CPU32(metadata->signature);
    metadata->crc                = DISK_TO_CPU32(metadata->crc);
    metadata->sequence_number    = DISK_TO_CPU64(metadata->sequence_number);
    metadata->start_sect_bbr_table =
        DISK_TO_CPU64(metadata->start_sect_bbr_table);
    metadata->nr_sects_bbr_table = DISK_TO_CPU64(metadata->nr_sects_bbr_table);
    metadata->block_size             = DISK_TO_CPU32(metadata->block_size);
    metadata->start_replacement_sect =
        DISK_TO_CPU64(metadata->start_replacement_sect);
    metadata->nr_replacement_blks =
        DISK_TO_CPU64(metadata->nr_replacement_blks);
    metadata->flags               = DISK_TO_CPU32(metadata->flags);

}

static void CPU_Metadata_To_Disk( evms_bbr_metadata_t *metadata )
{

    metadata->signature          = CPU_TO_DISK32(metadata->signature);
    metadata->crc                = CPU_TO_DISK32(metadata->crc);
    metadata->sequence_number    = CPU_TO_DISK64(metadata->sequence_number);
    metadata->start_sect_bbr_table =
        CPU_TO_DISK64(metadata->start_sect_bbr_table);
    metadata->nr_sects_bbr_table = CPU_TO_DISK64(metadata->nr_sects_bbr_table);
    metadata->block_size         = CPU_TO_DISK32(metadata->block_size);
    metadata->start_replacement_sect =
        CPU_TO_DISK64(metadata->start_replacement_sect);
    metadata->nr_replacement_blks =
        CPU_TO_DISK64(metadata->nr_replacement_blks);
    metadata->flags              = CPU_TO_DISK32(metadata->flags);

}

/*
 *  Called to read the meta data for a feature into the specified
 *  buffer. The feature data is located in known places.
 *
 *  Returns: feature meta data in the supplied buffer if RC=0
 *
 */
static int ReadMetaData( storage_object_t       * object,
                         evms_bbr_metadata_t    * metadata)
{
    int                      rc=EINVAL;//=ENODATA;
    int                      rc1, rc2;
    evms_bbr_metadata_t      metadata2;
    u_int32_t                crc;
    u_int32_t                calculated_crc;


    LOG_ENTRY();

    if (object==NULL || metadata==NULL) {
        LOG_ERROR("object or metadata pointers invalid");
        LOG_EXIT_INT(rc);
        return rc;
    }

    rc1 = READ( object, 1, 1, (void *) metadata );

    rc2 = READ( object, 2, 1, (void *) &metadata2 );

    LOG_DEBUG("read metadata, 1st copy rc= %d    2nd copy rc= %d\n",
              rc1, rc2 );
    if (rc1==0) {

        if ( DISK_TO_CPU32(metadata->signature) == EVMS_BBR_SIGNATURE) {

            crc            = DISK_TO_CPU32(metadata->crc);
            metadata->crc  = 0;
            calculated_crc = EngFncs->calculate_CRC
                ( EVMS_INITIAL_CRC, metadata, sizeof(evms_bbr_metadata_t) );
            metadata->crc  = CPU_TO_DISK32(crc);

            if ( (crc!=0) && (crc!=calculated_crc) ) {
                rc1 = ENODATA;
            }
            else {
                Disk_Metadata_To_CPU( metadata );
            }
        }
        else {
            rc1 = ENODATA;
        }
    }

    if (rc2==0) {

        if ( DISK_TO_CPU32(metadata2.signature) == EVMS_BBR_SIGNATURE ) {

            crc               = DISK_TO_CPU32(metadata2.crc);
            metadata2.crc     = 0;
            calculated_crc    = EngFncs->calculate_CRC
                ( EVMS_INITIAL_CRC, &metadata2, sizeof(evms_bbr_metadata_t) );
            metadata2.crc     = CPU_TO_DISK32(crc);

            if ((crc!=0) && (crc != calculated_crc)) {
                rc2 = ENODATA;
            }
            else {
                Disk_Metadata_To_CPU( &metadata2 );
            }

        }
        else {
            rc2 = ENODATA;
        }
    }

    // if both versions of the metadata are Ok then choose between the
    // two copies using the sequence number. This number is incremented
    // every time the kernel makes a change to the metadata.  The highest
    // number will be the most recent version. Normally both numbers will
    // be the same ... but just in case ... choose highest number.
    if (rc1==0 && rc2==0) {

        LOG_DEBUG("both copies of metadata are Ok, seq_number_1= %"PRIu64"  seq_number_2= %"PRIu64"\n",
                  metadata->sequence_number, metadata2.sequence_number );

        if ( metadata->sequence_number >= metadata2.sequence_number) {
            LOG_DEBUG("using 1st copy cuz seq numbers are same or 1st is > 2nd\n");
            rc = 0;
        }
        else {
            LOG_DEBUG("using 2nd copy of metadata cuz of seq numbers\n");

            memcpy(metadata, &metadata2, sizeof(evms_bbr_metadata_t ));
            rc = 0;
        }

    }
    // if only the first copy is good then we dont have a choice to make
    else if (rc1==0) {
        LOG_DEBUG("using 1st copy of metadata cuz 2nd is missing or bad\n");
        rc = 0;
    }
    // if the second copy is the only good one then we need to copy the data
    // to the callers metadata buffer
    else if (rc2==0) {

        LOG_DEBUG("using 2nd copy of metadata cuz 1st is missing or bad\n");

        // copy over metadata sector to callers buffer
        memcpy(metadata, &metadata2, sizeof(evms_bbr_metadata_t ));

        rc = 0;
    }
    // if neither copy is any good then return NO DATA AVAILABLE.
    else {
        LOG_DEBUG("both copies of metadata are missing or bad\n");
        rc = ENODATA;
    }


    LOG_EXIT_INT(rc);
    return rc;
}


/*
 *  Called to write meta data for a feature out to disk.
 *  the feature data is located in known locations.
 *
 *  Returns: RC=0 if successful
 *
 */
static int WriteMetaData( storage_object_t       * parent,
                          evms_bbr_metadata_t    * metadata,
                          uint                     commit_phase )
{
    int                        rc=EINVAL;
    u_int32_t                  crc;
    storage_object_t          *child=NULL;
    BBR_Private_Data          *pdata =
        (BBR_Private_Data *) parent->private_data;


    LOG_ENTRY();

    // get bbr child object
    child = GET_BBR_CHILD(parent);
    if (child==NULL) {

        LOG_ERROR("bbr object doesnt have a child object\n");
        LOG_EXIT_INT(rc);
        return rc;

    }

    if (commit_phase == 1) {

        // 1st copy of metadata points to ... 1st copy of bbr mapping table
        metadata->start_sect_bbr_table = pdata->bbr_table_lsn1;

        // convert metadata to disk format
        CPU_Metadata_To_Disk( metadata );

        // crc the 1st copy of metadata
        metadata->crc = 0;
        crc = EngFncs->calculate_CRC
            ( EVMS_INITIAL_CRC, metadata, EVMS_VSECTOR_SIZE );
        metadata->crc = CPU_TO_DISK32(crc);

        LOG_DEBUG("commit phase 1, writing metadata to LSN %d\n", 1);

        rc  = WRITE( child, 1, 1, (void *) metadata );

        // now write out the first copy of the bbr mapping table
        if (pdata->bbr_table) {
            rc += WriteBBRTable( child,
                                 pdata->bbr_table_lsn1,
                                 pdata->bbr_table_size_in_sectors,
                                 (vsector_t *)pdata->bbr_table );
        }

    }

    else if (commit_phase == 2) {

        // convert metadata to disk format
        CPU_Metadata_To_Disk( metadata );

        // crc the metadata sector
        metadata->crc = 0;
        crc = EngFncs->calculate_CRC
            ( EVMS_INITIAL_CRC, metadata, EVMS_VSECTOR_SIZE );
        metadata->crc = CPU_TO_DISK32(crc);

        LOG_DEBUG("commit phase 2, writing metadata to LSN %d\n", 2);

        rc  = WRITE( child, 2, 1, (void *) metadata );
        // write out the second copy of the bbr table
        if (pdata->bbr_table) {
            rc = WriteBBRTable( child,
                                pdata->bbr_table_lsn2,
                                pdata->bbr_table_size_in_sectors,
                                (vsector_t *)pdata->bbr_table );
        }

    }


    LOG_EXIT_INT(rc);
    return rc;
}



/*
 *  Called to free up a parent BBR storage object. Walks the
 *  object, freeing any private data structs before calling
 *  the engine to free the object itself.
 *
 *  Returns: RC=0 if object memory was freed
 */
static int free_bbr_object( storage_object_t  * bbr )
{
    int               rc = EINVAL;
    BBR_Private_Data *pdata;

    LOG_ENTRY();

    if (bbr) {
        if ( strlen( bbr->name ) )
            rc = EngFncs->unregister_name( bbr->name );
        if ( strlen( bbr->uuid ) )
            rc = EngFncs->unregister_name( bbr->uuid );

        pdata = (BBR_Private_Data *) bbr->private_data;

        if (pdata) {
            if (pdata->bbr_table) EngFncs->engine_free(pdata->bbr_table);
            free(pdata);
        }

        rc = EngFncs->free_segment(bbr);
    }

    LOG_EXIT_INT(rc);
    return rc;
}

/*
 *  Called to obtain memory for a BBR storage object, allocating
 *  any private data areas needed in addition to the storage object
 *  struct itself.
 *
 *  Returns: ptr to a bbr storage object if successful, NULL if not successful
 */
static storage_object_t * malloc_bbr_object( void )
{
    int                      rc;
    storage_object_t        *bbr;
    BBR_Private_Data        *pdata;

    LOG_ENTRY();

    rc = EngFncs->allocate_segment(NULL, &bbr);
    if ( rc == 0 ) {

        pdata = calloc(1,sizeof(BBR_Private_Data));
        if ( pdata == NULL ) {
            EngFncs->free_evms_object(bbr);
            bbr = NULL;
        }
        else {
            bbr->plugin              = my_plugin_record;
            bbr->private_data        = pdata;
            pdata->signature         = EVMS_BBR_SIGNATURE;

            //new
            bbr->object_type         = SEGMENT;
            bbr->data_type           = DATA_TYPE;
        }
    }

    LOG_EXIT_PTR(bbr);
    return bbr;
}

/*
 * Process the kill sectors list.
 */
static int kill_sectors(storage_object_t *obj)
{
    int rc = 0;
    kill_sectors_t * ks;
    unsigned char  * buffer = NULL;
    sector_count_t   buffer_size = 0;
    BBR_Private_Data * pdata = NULL;

    LOG_ENTRY();

    /*
     * Copy the kill sector list head and NULL out the gloabal variable.
     * This function uses BBR_write() to write out the kill sectors,
     * but BBR_write() has a check to write kill sectors before it does
     * any writing.  We could end up in infinite recursion between
     * kill_sectors() and BBR_write(). By having
     * this function remove the kill sectors from the list the
     * recursion is stopped.
     */

    pdata = (BBR_Private_Data*)obj->private_data;
    ks = pdata->kill_sector_list_head;
    pdata->kill_sector_list_head = NULL;

    while ((rc == 0) && (ks != NULL)) {
        if (buffer_size < ks->count) {
            if (buffer != NULL) {
                free(buffer);
            }
            buffer = calloc(1, EVMS_VSECTOR_SIZE * ks->count);

            if (buffer != NULL) {
                buffer_size = ks->count;
            } else {
                buffer_size = 0;
                rc = ENOMEM;
            }
        }

        if (rc == 0) {
            kill_sectors_t * ks_prev = ks;

            LOG_DEBUG("Killing %"PRIu64" sectors on %s at sector offset %"PRIu64".\n",
                      ks->count, obj->name, ks->lsn);
            rc = BBR_Write( obj, ks->lsn, ks->count, buffer );

            ks = ks->next;
            free(ks_prev);
        }
    }
    LOG_EXIT_INT(rc);
    return(rc);
}

/*
 *  Called to commmit metadata for the specified bbr object.
 *
 *  NOTHING MUST STOP THE COMMIT!!
 *
 *  We may be resizing and other plugins may depend on us.
 *
 *  Returns: RC=0 if feature metadata was committed to the storage object
 *                by writing it out to disk.
 */
static int Commit_BBR_Object( storage_object_t *parent, uint commit_phase )
{
    int                        rc=0;
    evms_bbr_metadata_t        metadata;
    storage_object_t          *child=NULL;
    BBR_Private_Data          *pdata=NULL;

    LOG_ENTRY();

    // catch possible case when object isnt dirty
    if ((parent->flags & SOFLAG_DIRTY)==0) {
        LOG_EXIT_INT(0);
        return 0;
    }

    //get private_data
    pdata = (BBR_Private_Data*)parent->private_data;
    if ( pdata->kill_sector_list_head )
        kill_sectors( parent );

    if ( parent->flags & SOFLAG_NEW || pdata->bbr_state & BBR_CHANGE_STATE ) {
        // get child object
        child  = GET_BBR_CHILD( parent );

        if ( child ) {

            memset(&metadata, 0, sizeof(evms_bbr_metadata_t) );

            rc = BuildMetadata( parent, &metadata);

            if (rc==0)
                rc = WriteMetaData( parent, &metadata,commit_phase );
        }

        else {
            rc = EINVAL;
        }
    }

    if ( rc == 0 && commit_phase == 2) {
        parent->flags &= ~SOFLAG_DIRTY;
        parent->flags &= ~SOFLAG_NEW;
        if ( pdata ) {
            pdata->bbr_state &= ~BBR_CHANGE_STATE;
            pdata->bbr_state &= ~BBR_ACTIVATE;
        }
    }

    LOG_EXIT_INT(rc);
    return rc;
}

/*
 *  This is how a feature applies itself to a storage object, either
 *  during discovery or during create.
 *
 *  It inserts the child object in the parent storage object list.
 *  It then updates fields in the two objects to denote this relationship.
 *
 *  Note: the parent BBR object must have valid metadata in its private
 *        data area.
 *
 */
static int consume_storage_object( storage_object_t  * parent,
                                   storage_object_t  * child  )
{
    int               rc = 0;
    u_int64_t         child_useable_size;
    list_element_t    itr1, itr2;


    LOG_ENTRY();

    // Parent must have BBR metadata for use to calculate the useable area of the BBR object
    if ( (parent->private_data == NULL) ||
         (((BBR_Private_Data *)parent->private_data)->signature!=EVMS_BBR_SIGNATURE)) {

        rc = EINVAL;
        LOG_EXIT_INT(rc);
        return rc;
    }

    // calculate the useable size of the child object
    child_useable_size = get_child_useable_size(parent, child);

    if (child_useable_size > 0) {

        // consumed object is inserted into our child list.
        itr1 = EngFncs->insert_thing(child->parent_objects, parent,
                                     INSERT_BEFORE, NULL);

        if (itr1) {

            // new bbr object is inserted into childs parent list.
            itr2 = EngFncs->insert_thing(parent->child_objects, child,
                                         INSERT_BEFORE, NULL);
            if (itr2) {
                // update fields to show association and to reduce the useable
                // size of the PARENT storage object by the amount of sectors
                // we are using for metadata.
                parent->size         = child_useable_size;
                //new
                parent->start        = child->size - child_useable_size;
            } else {
                EngFncs->delete_element(itr1);
                rc = ENOMEM;
            }
        } else {
            rc = ENOMEM;
        }
    } else {
        rc = EINVAL;
    }


    LOG_EXIT_INT(rc);
    return rc;
}



/*
 *  This routine expects to be called only from DISCOVERY with a child
 *  storage object that has BBR installed on top of it. Meaning that
 *  the child object has a feature_header that names BBR as the
 *  current top most feature and that BBR metadata is layed down
 *  on the disk.
 *
 *  This routine is called to create a BBR storage object by consuming
 *  a child storage object that contains BBR metadata and by filling
 *  in the BBR private data from the feature header and feature metadata.
 *
 *  Returns: RC=0 if successful and the new bbr object is built Ok
 *
 */
static int Build_New_BBR_Object( storage_object_t    *child,
                                 evms_bbr_metadata_t *metadata,
                                 list_anchor_t        new_bbr_objects )
{
    int                   rc;
    storage_object_t     *parent;
    BBR_Private_Data     *pdata=NULL;
    sector_count_t        bad_blocks;
    list_element_t        itr;



    // allocate memory for the new storage object
    parent = malloc_bbr_object();

    if (parent) {

        pdata = (BBR_Private_Data *) parent->private_data;

        pdata ->child                               = child;

        strcpy( parent->name, child->name );
        strcat( parent->name, "_bbr" );

        strcpy( parent->uuid, metadata->uuid );
        pdata ->bbr_state                           = metadata->bbr_active;
        LOG_DEBUG("%s is in state %d\n", parent->name, pdata->bbr_state);
        pdata ->replacement_blocks_lsn              = metadata->start_replacement_sect;
        pdata ->replacement_blocks_needed           = metadata->nr_replacement_blks;
        pdata ->replacement_blocks_size_in_sectors  = metadata->nr_replacement_blks *
            ( metadata->block_size >> EVMS_VSECTOR_SIZE_SHIFT );

        pdata ->block_size                          = metadata->block_size;

        pdata ->bbr_table_lsn1                      = metadata->start_sect_bbr_table;
        pdata ->bbr_table_size_in_sectors           = metadata->nr_sects_bbr_table;
        pdata ->bbr_table_lsn2                      = pdata->bbr_table_lsn1 + pdata->bbr_table_size_in_sectors;

        // malloc a bbr table
        pdata->bbr_table = EngFncs->engine_alloc( (pdata->bbr_table_size_in_sectors*EVMS_VSECTOR_SIZE) );
        if (pdata->bbr_table == NULL) {
            free_bbr_object(parent);
            LOG_ERROR("unable to malloc a new BBR mapping table, rc= ENOMEM\n");
            LOG_EXIT_INT(ENOMEM);
            return ENOMEM;
        }
        else {
            rc = GetBBRTable( child,
                              pdata->bbr_table,
                              pdata->bbr_table_size_in_sectors,
                              pdata->bbr_table_lsn1,
                              pdata->bbr_table_lsn2 );

            if (rc) {
                free_bbr_object(parent);
                LOG_ERROR("failed to read a good BBR mapping table, rc= ENODATA\n");
                LOG_EXIT_INT(ENODATA);
                return ENODATA;
            }
        }

        pdata ->sequence_number = metadata->sequence_number;

        // register the new evms object name
        rc = EngFncs->register_name( parent->name );
        rc = EngFncs->register_name( parent->uuid );

        if (rc == 0) {

            // insert the new BBR object into a discovery list.
            itr = EngFncs->insert_thing(new_bbr_objects, parent,
                                        INSERT_BEFORE, NULL);
            if (itr) {

                // final step is to consume the child storage object
                rc = consume_storage_object( parent, child  );

                if (rc) {
                    EngFncs->delete_element(itr);
                    free_bbr_object(parent);
                }

            }
            else {
                free_bbr_object(parent);
                rc = EPERM;
            }

        }
        else {
            free_bbr_object(parent);
        }

    }
    else {
        rc = ENOMEM;
    }


    // if successful then see if kernel has remapped sectors
    // and warn evms user.
    if (rc == 0) {

        // copy geometry to new storage object
        memcpy(&parent->geometry, &child->geometry, sizeof(geometry_t));

        if ( isa_kernel_bbr_object( parent ) == TRUE ) {
            bad_blocks = get_kernel_bbr_remap_sector_count( parent );
        }
        else {
            bad_blocks = get_engine_remap_sector_count( parent );
        }

        if (bad_blocks > 0) {
              MESSAGE("\nWarning, the kernel bbr feature is reporting that %"PRIu64" bad sector(s) were "
                      "discovered on object %s. Though replacement sectors are being used to remedy the "
                      "problem, you would be well advised to take corrective actions by replacing "
                      "the storage object.\n", bad_blocks, parent->name );
        }

        // BBR_DEBUG if (pdata) display_bbr_remap_info( parent, pdata );
    }

    return rc;
}

/*
 * Called from Create or Assign either can be used to do the
 * same thing for this plugin.
 */
static int make_bbr( storage_object_t * child )
{
    int rc=0;
    storage_object_t *parent;
    BBR_Private_Data *pdata=NULL;
    guid_t id;

    parent = malloc_bbr_object();
    if ( parent ) {

        memcpy(&parent->geometry, &child->geometry, sizeof(geometry_t));

        //name
        strcpy( parent->name, child->name );
        strcat( parent->name, "_bbr" );

        uuid_generate_random( (char*) &id );
        sprintf( parent->uuid,
                 "%08X-%04X-%04X-%02X%02X-%02X%02X%02X%02X%02X%02X",
                 id.time_low,
                 id.time_mid,
                 id.time_high,
                 id.clock_seq_high,
                 id.clock_seq_low,
                 id.node[0],
                 id.node[1],
                 id.node[2],
                 id.node[3],
                 id.node[4],
                 id.node[5] );
        rc = EngFncs->register_name( parent->uuid );

        pdata = (BBR_Private_Data *) parent->private_data;
        if (pdata) {
            pdata->child = child;
        }
        rc = EngFncs->register_name( parent->name );
        if (rc == 0) {

            rc = create_bbr_metadata( parent,
                                      child );
            if (rc == 0) {

                rc = consume_storage_object( parent, child  );

                if ( rc == 0 ) {
                    if (pdata) pdata->bbr_state = BBR_ENABLED;
                    parent->flags |= SOFLAG_DIRTY | SOFLAG_NEW |
                                     SOFLAG_NEEDS_ACTIVATE;
                }
                else
                    rc = EPERM;
            }
        }
    }
    else {
	    rc = ENOMEM;
    }

    if ( rc )
        free_bbr_object( parent );
    return rc;

}

static int forward_kill_sectors( storage_object_t * obj )
{

    int i, rc = 0;
    kill_sectors_t   * ks;
    storage_object_t * child;
    lsn_t              bbr_lsn=0;
    BBR_Private_Data * pdata;
    struct plugin_functions_s *fncs = NULL;
    LOG_ENTRY();

    pdata = (BBR_Private_Data*)obj->private_data;
    ks = pdata->kill_sector_list_head;
    child = GET_BBR_CHILD(obj);
    fncs = child->plugin->functions.plugin;

    while ((rc == 0) && (ks != NULL)) {
        ks->lsn += obj->start;

        for (i=0,rc=0; i<ks->count && rc==0; i++) {
            bbr_lsn = remap_lsn(obj, pdata, bbr_lsn);
            if ( bbr_lsn ) {
                bbr_lsn = get_lsn( pdata, ks->lsn+i );
                rc = fncs->add_sectors_to_kill_list( child, bbr_lsn,
                                                     ks->count );
            }
            else
                rc = EINVAL;
        }
        pdata->kill_sector_list_head = ks->next;
        free(ks);
        ks = pdata->kill_sector_list_head;
    }

    LOG_EXIT_INT(rc);
    return rc;
}

/*
 * Called from Delete and Unassign.  To remove BBR.
 */
static int unmake_bbr( storage_object_t * ld, boolean destroy )
{
	int rc = EINVAL;
	list_element_t itr = NULL;
	storage_object_t * seg;
	BBR_Private_Data * pdata;

	seg = EngFncs->first_thing(ld->parent_objects, &itr);
	EngFncs->delete_element(itr);

	if ( ( seg ) &&
		( seg->object_type == SEGMENT ) &&
		( i_can_modify_object( seg ) == TRUE ) ) {
			pdata = (BBR_Private_Data*)seg->private_data;
			if ( pdata->kill_sector_list_head != NULL ) {
				forward_kill_sectors( seg );
			}

		if (destroy) {
			//metadata
			KILL_SECTORS( ld, 0, 3);

			//first table sectors (necessary?)
			KILL_SECTORS( ld, pdata->bbr_table_lsn1, 1 );
			KILL_SECTORS( ld, pdata->bbr_table_lsn2, 1 );
		}   

		//free engine memory
		free_bbr_object( seg );

		rc = 0;
    }

	return rc;
}


/*
 *  Function: bbr_get_devmap_info
 *
 *  Called to test if the bbr object has an active device mapper
 *  node in the kernel and set the object info accordingly.
 */
static void bbr_get_devmap_info( storage_object_t * bbr_object )
{
        storage_object_t * child=NULL;
        int rc=0;
        dm_target_t * trgts;
        BBR_Private_Data* pdata=NULL;

        LOG_ENTRY();

        EngFncs->dm_update_status(bbr_object);

        if ( bbr_object->flags & SOFLAG_ACTIVE ) {

		pdata = (BBR_Private_Data*)bbr_object->private_data;
		child = GET_BBR_CHILD(bbr_object);

		rc = EngFncs->dm_get_targets( bbr_object, &trgts );

                if ( (rc) ||
                     (trgts->next != NULL) ||
                     (trgts->start != 0) ||
                     (trgts->length != bbr_object->size) ) {

			LOG_DEBUG("On %s the basic table info in the kernel is incorrect\n", bbr_object->name);

                        bbr_object->flags |= SOFLAG_NEEDS_ACTIVATE;

		}
		else if ( pdata->bbr_state & BBR_ENABLED ) {

			if  ( (trgts->type != DM_TARGET_BBR) ||
			      (trgts->data.bbr->device.major != child->dev_major) ||
			      (trgts->data.bbr->device.minor != child->dev_minor) ||
                              (trgts->data.bbr->device.start != bbr_object->start) ||
                              (trgts->data.bbr->table1_lba != pdata->bbr_table_lsn1) ||
                              (trgts->data.bbr->table2_lba != pdata->bbr_table_lsn2) ||
                              (trgts->data.bbr->replacement_blocks_lba!= pdata->replacement_blocks_lsn) ||
                              (trgts->data.bbr->table_size != pdata->bbr_table_size_in_sectors) ||
                              (trgts->data.bbr->num_replacement_blocks != pdata->replacement_blocks_needed) ||
                              (trgts->data.bbr->block_size != pdata->block_size) ) {


				LOG_DEBUG("%s should be a bbr kernel object some table info is incorrect\n",
				           bbr_object->name );

				bbr_object->flags |= SOFLAG_NEEDS_ACTIVATE;

			}

                }
                else if ( (trgts->type != DM_TARGET_LINEAR) ||
                          (trgts->data.linear->major != child->dev_major) ||
                          (trgts->data.linear->minor != child->dev_minor) ||
                          (trgts->data.linear->start != bbr_object->start) ) {

 			LOG_DEBUG("%s should be a linear kernel object some table info is in correct\n",
                                   bbr_object->name );

			bbr_object->flags |= SOFLAG_NEEDS_ACTIVATE;

                }

                EngFncs->dm_deallocate_targets( trgts );
        }
	else {
		bbr_object->flags |= SOFLAG_NEEDS_ACTIVATE;
	}

        LOG_EXIT_VOID();
}



/*-------------------------------------------------------------------------+
+                                                                          +
+                   Start Of EVMS Plugin Functions                         +
+              (exported to engine via function table)                     +
+                                                                          +
+--------------------------------------------------------------------------*/

/*
 *  Called by EVMS, after inspecting and validating our plugin record.
 *  This is the final step when loading a plugin and means we have been
 *  validated by EVMS and will be used.  Most important item here is to
 *  get the Engine's function table so we can call engine API(s) later.
 */
static int BBR_SetupEVMSPlugin( engine_functions_t * engine_functions)
{
    int rc = 0;
    EngFncs = engine_functions;
    LOG_ENTRY();
    LOG_EXIT_INT(rc);
    return rc;
}


static void BBR_Cleanup( void )
{
    LOG_ENTRY();
    // NOP
    LOG_EXIT_VOID();
}




/*
 *  I can allow an object to be a volume if:
 *
 *  - I own the object
 *
 */
static int BBR_CanSetVolume( storage_object_t * object, boolean  flag )
{
    int rc = EINVAL;

    LOG_ENTRY();

    if ( i_can_modify_object(object)==TRUE ) {


        rc = 0;

    }

    LOG_EXIT_INT(rc);
    return rc;
}



/*
 *  I can expand the bbr object if:
 *
 *  - I own the parent object
 *  - I have a child object
 *  - the bbr child says OK to expand
 *
 */
static int BBR_CanExpand( storage_object_t    *object,           // bbr object
                          sector_count_t      *expand_limit,       // ?
                          list_anchor_t        expansion_points )  // list to place expand object on

{
    int                        rc = EINVAL;
    storage_object_t          *child;
    struct plugin_functions_s *fncs;

    LOG_ENTRY();

    if ( ( i_can_modify_object(object)==TRUE ) ) {

        child = GET_BBR_CHILD( object );

        if (child) {

            // get child plugin function table
            fncs = (struct plugin_functions_s *)child->plugin->functions.plugin;

            // pass cmd down feature stack
            rc = fncs->can_expand(child, expand_limit, expansion_points );

        }

    }

    LOG_EXIT_INT(rc);
    return rc;
}


/*
 *  I can remove bbr from this object if:
 *
 *  - I own it
 *
 */
static int BBR_CanDelete( storage_object_t * object)
{
    int    rc = EINVAL;

    if ( i_can_modify_object(object)==TRUE ) {

       rc = 0;

    }

    return rc;
}


/*
 * I can allow a child object to expand if:
 *
 * - I own the parent object
 * - A child exists for the bbr object
 *
 */
static int BBR_CanExpandBy(storage_object_t * object, sector_count_t *size)
{
    int                        rc = EINVAL;
    storage_object_t          *child;

    LOG_ENTRY();

    if ( ( i_can_modify_object(object) == TRUE ) ) {

        child = GET_BBR_CHILD( object );

        if ( child ) {

            rc = 0;

        }

    }

    LOG_EXIT_INT(rc);
    return rc;
}


/*
 *  I can shrink this bbr object if:
 *
 *  - I own it
 *  - lower layer plugins say Ok to shrink
 *
 */

static int BBR_CanShrink( storage_object_t * object,
                          sector_count_t   * shrink_limit,
                          list_anchor_t      shrink_points )
{
    int                        rc = EINVAL;
    storage_object_t          *child;
    struct plugin_functions_s *Fncs;

    LOG_ENTRY();

    if ( ( i_can_modify_object(object) == TRUE ) ) {

        child = GET_BBR_CHILD( object );

        if (child) {

            // get child plugin function table
            Fncs = child->plugin->functions.plugin;

            // pass cmd down feature stack
            rc = Fncs->can_shrink(child, shrink_limit, shrink_points);

        }

    }

    LOG_EXIT_INT(rc);
    return rc;
}

/*
 * I can shrink the bbr object by size sectors if:
 *
 * - I own the object
 * - Some plugin below me says Ok
 */
static int BBR_CanShrinkBy( storage_object_t * object, sector_count_t *size)
{
    int                        rc = EINVAL;
    storage_object_t          *child;
    struct plugin_functions_s *fncs;

    LOG_ENTRY();

    if ( ( i_can_modify_object(object)==TRUE ) ) {

        child = GET_BBR_CHILD( object );

        if ( child ){

            // get child plugin function table
            fncs = child->plugin->functions.plugin;

            // pass cmd down feature stack
            rc = fncs->can_shrink_by(child, size);

        }

    }

    LOG_EXIT_INT(rc);
    return rc;
}


/*
 *  Called by the engine with a list of storage objects that it believes
 *  are BBR storage objects.  We are suppose to examine the objects and
 *  apply the BBR feature to the feature stack on this object once we are
 *  convinced that it is a valid BBR object.  BBR is applied as follows:
 *
 *  - Validate the object by looking for BBR metadata. If we can get a
 *    copy of the metadata and the BBR table then we have a valid object.
 *    Otherwise, we will fail this API call.
 *
 *  - Create a new BBR storage object and put the storage object from
 *    the engine list in the BBR object's child_object list. I call
 *    this consuming the child object.  Fill in BBR fields as appropriate,
 *    e.g. the useable area after we adjust down for our metadata sectors.
 *
 *  - Set the SOFLAG_VOLUME in the new BBR object ... if the feature header
 *    has EVMS_FEATURE_VOLUME_COMPLETE flag bit set.
 */
static int BBR_Feature_Discovery( list_anchor_t input_objects,
                                  list_anchor_t output_objects,
                                  boolean FinalCall )
{
    list_anchor_t         bbr_object_list;
    list_element_t        itr;
    storage_object_t    * object;
    evms_bbr_metadata_t   metadata;
    int                   bbr_object_count;
    int                   rc;

    LOG_ENTRY();

    bbr_object_list = EngFncs->allocate_list();
    if ( !bbr_object_list ) {
        EngFncs->concatenate_lists(output_objects, input_objects);
        LOG_EXIT_INT(0);
        return 0;
    }

    // validate storage objects found in our discovery list.
    // any valid BBR child object will be pruned from the engine list
    // new BBR parent objects will be placed in the BBR_object_list
    LIST_FOR_EACH(input_objects, itr, object) {
        LOG_DEBUG("\tExamining storage object %s\n", object->name);

        if ( object->plugin == my_plugin_record ||
             object->data_type != DATA_TYPE ) {
            EngFncs->insert_thing(output_objects, object, INSERT_AFTER, NULL);
            continue;
        }

        // read and validate bbr feature metadata
        rc = ReadMetaData(object, &metadata);
        if (rc) {
            LOG_DEBUG("BBR metadata not found on storage object\n");
            EngFncs->insert_thing(output_objects, object, INSERT_AFTER, NULL);
            continue;
        }

        rc = Build_New_BBR_Object(object, &metadata, bbr_object_list);
        if (rc) {
            EngFncs->insert_thing(output_objects, object, INSERT_AFTER, NULL);
            continue;
        }
    }


    LIST_FOR_EACH(bbr_object_list, itr, object) {
        bbr_get_devmap_info(object);
    }

    bbr_object_count = EngFncs->list_count(bbr_object_list);
    EngFncs->merge_lists(output_objects, bbr_object_list, NULL);

    LOG_EXIT_INT(bbr_object_count);
    return bbr_object_count;
}

/*
 *  To create a BBR object I check that:
 *
 *  - there is at least 1 storage object in the list that sits
 *    entirely on a single disk
 *
 *  I then create a parent BBR storage object and consume the child object
 *
 *  Returns: the address of the new BBR object if successful
 *
 */
static int BBR_Create( list_anchor_t        input_objects,
                       option_array_t     * options,
                       list_anchor_t        output_objects )
{
    int                   rc = EINVAL;
    storage_object_t     *parent;
    storage_object_t     *child=NULL;
    uint                  count;

    LOG_ENTRY();

    // we should only be called with a single input object
    count = EngFncs->list_count(input_objects);
    if ( count == 1 ) {

        // get the storage object from the list
        child = EngFncs->first_thing(input_objects, NULL);
        rc = 0;

    }

    if (!rc) {

        rc = make_bbr( child );
        if (rc == 0) {
            parent = EngFncs->first_thing(child->parent_objects, NULL);
            EngFncs->insert_thing(output_objects, parent, INSERT_BEFORE, NULL);
        }
    }
    LOG_EXIT_INT(rc);
    return rc;
}

static int BBR_w_delete( storage_object_t *object, list_anchor_t child_objects, boolean destroy)
{
    int                        rc = EINVAL;
    storage_object_t          *child;


	LOG_ENTRY();

	child = GET_BBR_CHILD(object);

	if (child) {

		rc = unmake_bbr(child, destroy);
		if ( rc == 0 ) {
			child->flags &= ~SOFLAG_HAS_STOP_DATA;
            if (child_objects) {
            	EngFncs->insert_thing(child_objects, child, INSERT_BEFORE, NULL);
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 *  This is a non-destructive remove.  Essentially, we do all the same work
 *  as a destroy only we dont tell lower features to also destroy.
 *
 *  - Remove the feature from the object.
 *  - Free any privately allocated data.
 *  - Remove your parent pointer from your child objects.
 *  - Put BBR child onto the list provided in the second parameter.
 *
 *  Returns: RC=0 if successful
 *
 */
static int BBR_Delete( storage_object_t * object, list_anchor_t child_objects )
{
    int rc;
    LOG_ENTRY();
    rc = BBR_w_delete(object, child_objects, TRUE);
    LOG_EXIT_INT(rc);
    return rc;
}

/*
 * Function: BBR_discard
 * 
 * This function is similar to delete.  Just call delete to free all
 * data structures related to the BBR objects.
 */
int BBR_Discard(list_anchor_t objects)
{
	storage_object_t * object;
	list_element_t le;

	LOG_ENTRY();

	LIST_FOR_EACH(objects, le, object) {
		BBR_w_delete(object, NULL, FALSE);
	}

	LOG_EXIT_INT(0);
	return 0;
}



/*
 *  A BBR object is expanded by:
 *
 *  - calling down the plugin stack to expand the child object
 *  - recalculating metadata
 *  - remapping the bbr table
 *  - moving the replacement blocks
 *
 *  I check that:
 *
 *  - I own the object
 *  - the plugin stack can expand the BBR child object
 *  - the expanded child is actually larger in size
 *  - kernel is not actively remapping bad blocks
 *
 *  Returns: RC=0 if we successfully expanded the BBR object
 *
 */
static int BBR_Expand( storage_object_t * object,
                       storage_object_t * expand_object,
                       list_anchor_t      objects,
                       option_array_t   * options )
{
    int                        rc = EINVAL;
    storage_object_t          *child;
    u_int64_t                  starting_size;
    struct plugin_functions_s *Fncs;

    LOG_ENTRY();

    if ( ( i_can_modify_object(object) == TRUE ) ) {

        child = GET_BBR_CHILD(object);

        if (child) {

            // get his plugin function table
            Fncs = child->plugin->functions.plugin;

            // get starting size
            starting_size = child->size;

            // pass expand down the feature stack
            rc = Fncs->expand( child, expand_object, objects, options );

            // check feature RC
            if ( rc==0 ) {

                // make sure that the feature actually expanded the object
                if (child->size > starting_size) {

                    // resize bbr object
                    object->size = get_child_useable_size(object, child);

                    // set object flags
                    object->flags |= SOFLAG_NEEDS_ACTIVATE;

                }
            }
        }
    }
    LOG_EXIT_INT(rc);
    return rc;
}



/*
 *  A BBR object is shrunk by:
 *
 *  - calling down the plugin stack to shrink the child object
 *  - recalculating metadata
 *  - remapping the bbr table
 *  - moving the replacement blocks
 *
 *  I check that:
 *
 *  - I own the object
 *  - the plugin stack can shrink the BBR child object
 *  - the child object is actually smaller in size
 *
 *  Returns: RC=0 if we successfully shrunk the BBR object
 *
 */
static int BBR_Shrink( storage_object_t * object,
                       storage_object_t * shrink_object,
                       list_anchor_t      objects,
                       option_array_t   * options )
{
   int                        rc = EINVAL;
   storage_object_t          *child;
   u_int64_t                  starting_size;
   struct plugin_functions_s *fncs;

   LOG_ENTRY();


    if ( ( i_can_modify_object(object) == TRUE ) ) {

       child = GET_BBR_CHILD(object);

       if (child) {

           // get his plugin function table
           fncs = child->plugin->functions.plugin;

           // starting size
           starting_size = child->size;

           // pass shrink cmd down the feature stack
           rc = fncs->shrink( child, shrink_object, objects, options );

           if ( rc==0 ) {

               // make sure that the feature actually shrunk the object
               if (child->size < starting_size) {

                   // resize bbr object
                   object->size = get_child_useable_size(object, child);

                   // set object flags
                   object->flags |= SOFLAG_NEEDS_ACTIVATE;

               }

           }

       }
    }

    LOG_EXIT_INT(rc);
    return rc;

}


/*
 *  Passes the API call down to the BBR child object.
 *
 *  I check that:
 *
 *  - I own the object
 *  - the logical sectors fall on the useable area of the BBR object
 *
 *  Returns: RC=0 if sectors were added to the kill list successfully
 *
 */
static int BBR_AddSectorsToKillList( storage_object_t * object,
                                     lsn_t              lsn,
                                     sector_count_t     count)
{
    int                        rc=0;
    BBR_Private_Data          *pdata;
    kill_sectors_t * ks;

    LOG_ENTRY();
    if ( ( i_can_modify_object(object)==TRUE ) &&
         ( lsn+count <= object->size ) ) {

        pdata = (BBR_Private_Data *) object->private_data;
        ks = malloc(sizeof(kill_sectors_t));
        if (ks != NULL) {

            ks->lsn    = lsn;
            ks->count  = count;

            ks->next = pdata->kill_sector_list_head;
            pdata->kill_sector_list_head = ks;

            /*
             * Mark the region dirty so that it will get called to commit
             * the kill sectors.
             */
            object->flags |= SOFLAG_DIRTY;

        } else {
            rc = ENOMEM;
        }
    }
    else {
        rc = EINVAL;
    }

    LOG_EXIT_INT(rc);
    return rc;
}



/*
 *  Called to commit changes to a BBR storage object.
 *
 *  This is done by committing the feature metadata
 *
 *  I check that:
 *
 *  - I own the object
 *
 *  I do it by calling Commit_BBR_Object() on the BBR object.
 *
 *  Returns: RC=0 if all objects commit feature data successfully
 *
 */
static int BBR_CommitChanges( storage_object_t * object, uint commit_phase )
{
    int rc=0;

    LOG_ENTRY();
    LOG_DEBUG("bbr object: name= %s \n", object->name );


    if ( i_can_modify_object(object)==TRUE ) {

        // we only work during metadata commit phases
        // phase 1 = commit 1st copy of metadata & 1st bbr table
        // phase 2 = commit 2nd copy of metadata & 2nd bbr table
        switch (commit_phase) {

            case 0:
                break;

            case 1:
            case 2:
                rc = Commit_BBR_Object( object, commit_phase );

                // BBR_DEBUG
                //display_bbr_remap_info
                //    ( object, (BBR_Private_Data *)object->private_data );

                break;

            default:
                rc = 0;
                break;
        }


    }
    else {
        rc = EINVAL;
    }


    LOG_EXIT_INT(rc);
    return rc;
}


static int BBR_Read( storage_object_t * object,
                     lsn_t              lsn,
                     sector_count_t     count,
                     void             * buffer)
{
    int                        rc = EINVAL;
    storage_object_t          *child;
    BBR_Private_Data          *pdata;
    vsector_t                 *sector_ptr = (vsector_t *)buffer;
    int                        i;
    lsn_t                      bbr_lsn;
    struct plugin_functions_s *Fncs;

    LOG_ENTRY();
    LOG_DEBUG("lsn = %"PRIu64"\n", lsn);

    if ( ( lsn+count <= object->size ) &&
         ( i_can_modify_object(object)==TRUE ) ) {

        pdata = (BBR_Private_Data *) object->private_data;

        child = GET_BBR_CHILD( object );

        if (child) {

            if ( isa_kernel_bbr_object(object) == TRUE ) {

                // build bbr i/o ioctl and let kernel handle it.
                rc = kernel_bbr_sector_io( object, lsn, count,
                                           buffer, SECTOR_IO_READ );

            }
            else {
                //why not use READ -- performance (in a loop)
                Fncs = child->plugin->functions.plugin;

                lsn += object->start;

                for (i=0,rc=0; i<count && rc==0; i++) {

                    bbr_lsn = get_lsn( pdata, lsn+i );

                    rc = Fncs->read( child, bbr_lsn, 1, sector_ptr );

                    if ( rc ) {
                        LOG_EXIT_INT(rc);
                        return rc;
                    }
                    ++sector_ptr;
                }

            }

        }

    }


    LOG_EXIT_INT(rc);
    return rc;
}

static int BBR_Write( storage_object_t * object,
                      lsn_t              lsn,
                      sector_count_t     count,
                      void             * buffer)
{
    int                        rc = 0;
    storage_object_t          *child;
    struct plugin_functions_s *Fncs;
    BBR_Private_Data          *pdata;
    lsn_t                      bbr_lsn;
    vsector_t                 *sector_ptr = (vsector_t *)buffer;
    int                        i;

    LOG_ENTRY();
    LOG_DEBUG("lsn = %"PRIu64"\n", lsn);

    if ( ( lsn+count <= object->size ) &&
         ( i_can_modify_object(object)==TRUE )) {

        pdata = (BBR_Private_Data *) object->private_data;

        if ( pdata->kill_sector_list_head != NULL )
            rc = kill_sectors(object);

        child = GET_BBR_CHILD(object);

        if (rc == 0 && child) {

            if ( isa_kernel_bbr_object(object) == TRUE ) {

                // build bbr i/o ioctl and let kernel handle it.
                rc = kernel_bbr_sector_io( object, lsn, count,
                                           buffer, SECTOR_IO_WRITE );

            }
            else {

                Fncs = child->plugin->functions.plugin;

                lsn += object->start;

                for (i=0,rc=0; i<count && rc==0; i++) {

                    bbr_lsn = get_lsn( pdata, lsn+i );

                    rc = Fncs->write(child, bbr_lsn, 1, sector_ptr);

                    while ( rc ) {

                        bbr_lsn = remap_lsn(object, pdata, bbr_lsn);

                        if ( bbr_lsn ) {
                            rc = Fncs->write(child, bbr_lsn, 1, sector_ptr);
                        }
                        else {
                            break;
                        }

                    }

                    ++sector_ptr;
                }

            }

        }

    }


    LOG_EXIT_INT(rc);
    return rc;

}



/*
 *  I can allow an object to be a volume if:
 *
 *  - I own the object
 *
 */
static void BBR_SetVolume( storage_object_t * object, boolean  flag )
{

    LOG_ENTRY();
    // Nothing to do yet.
    LOG_EXIT_VOID();
}

//New Stuff
static int BBR_Assign( storage_object_t * child,
                       option_array_t * options )
{
    int rc;
    LOG_ENTRY();
    rc = make_bbr( child );
    LOG_EXIT_INT(rc);
    return rc;
}

static int BBR_CanUnAssign( storage_object_t * ld )
{
    int rc = EINVAL;
    storage_object_t *seg;

    LOG_ENTRY();

    seg = EngFncs->first_thing(ld->parent_objects, NULL);

    if ( ( seg ) &&
         ( seg->object_type == SEGMENT ) &&
         ( i_can_modify_object( seg ) == TRUE ) ) {
        rc = 0;
    }

    LOG_EXIT_INT(rc);
    return rc;
}

static int BBR_UnAssign( storage_object_t * ld )
{
    int rc;
    LOG_ENTRY();

    rc = unmake_bbr( ld, TRUE );

    LOG_EXIT_INT(rc);
    return rc;
}


/*
 *  Function:  get_DM_info
 *
 *  Called when a DM activate fails.  The job here is to see if the 
 *  segment is active in the kernel.  If so ... compare the target 
 *  mapping against the engine segment object.  If the segments starting
 *  LBA and SIZE match ... then we dont need to activate the segment
 *  because it is already active and has a correct mapping ... return 0.
 */
static int get_DM_info( storage_object_t *seg )
{	
	dm_target_t *targets=NULL;
	dm_device_t *dev;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("seg= %s\n", seg->name);
		
        rc = EngFncs->dm_update_status(seg);
	if (!rc) {
		if (seg->flags & SOFLAG_ACTIVE) {  
			LOG_DEBUG("segment IS active in the kernel\n");
			rc = EngFncs->dm_get_targets(seg, &targets);
			if ( !rc && targets != NULL) {
				dev = targets->data.linear;
				if ( seg->start == dev->start &&
				     seg->size  == targets->length) {
					LOG_DEBUG("kernel object matches ... marking segment active\n");
					rc = 0;
				}
				else {
					LOG_ERROR("error, got a DM object using our segment name but the metadata differs. dont know what to do!\n");
					rc = ENODEV; // dont know how to handle this.
				}
 			}
			else {
				rc = ENODEV;
			}
			if (targets) EngFncs->dm_deallocate_targets(targets);
		}
		else {
			LOG_DEBUG("segment is NOT active in the kernel\n");
			rc=ENODEV;
		}
	}

	if (rc) {
		seg->flags |= SOFLAG_NEEDS_ACTIVATE;
	}
 	
	LOG_EXIT_INT(rc);
	return rc;
}

static int BBR_Activate( storage_object_t * seg )
{
    int rc = EINVAL;
    dm_target_t* trgt;
    dm_target_bbr_t *bbr;
    dm_device_t *lin;
    storage_object_t *child = NULL;
    BBR_Private_Data *pdata;

    LOG_ENTRY();

    if ( !seg ) {
        LOG_EXIT_INT(rc);
        return rc;
    }

    child = GET_BBR_CHILD(seg);
    if ( !child ) {
        LOG_EXIT_INT(rc);
        return rc;
    }

    pdata = (BBR_Private_Data*)seg->private_data;

    if ( pdata->bbr_state & BBR_ENABLED ) {
        LOG_DEBUG("Activating Enabled BBR object(%s), type = BBR\n",
                  seg->name);
        trgt = EngFncs->dm_allocate_target( DM_TARGET_BBR, 0,
                                               seg->size, 0, 0 );
        if ( !trgt ) {
            LOG_EXIT_INT(rc);
            return rc;
        }

        bbr = trgt->data.bbr;
        if ( !bbr ) {
            LOG_EXIT_INT(rc);
            return rc;
        }

        bbr->device.major = child->dev_major;
        bbr->device.minor = child->dev_minor;
        bbr->device.start = seg->start;
        bbr->table1_lba = pdata->bbr_table_lsn1;
        bbr->table2_lba = pdata->bbr_table_lsn2;
        bbr->replacement_blocks_lba = pdata->replacement_blocks_lsn;
        bbr->table_size =  pdata->bbr_table_size_in_sectors;
        bbr->num_replacement_blocks = pdata->replacement_blocks_needed;
        bbr->block_size = pdata->block_size;

	if (EngFncs->is_2_4_kernel()) {
		trgt->length &= ~1;
	}
    }
    else {
        LOG_DEBUG("Activating Disabled BBR object(%s), type = Linear\n",
                  seg->name);
        trgt = EngFncs->dm_allocate_target( DM_TARGET_LINEAR, 0, seg->size, 0, 0);
        if ( !trgt ) {
            LOG_EXIT_INT(rc);
            return rc;
        }

        lin = trgt->data.linear;
        if ( !lin ) {
            LOG_EXIT_INT(rc);
            return rc;
        }
        lin->major = child->dev_major;
        lin->minor = child->dev_minor;
        lin->start = seg->start;
    }

    rc = EngFncs->dm_activate( seg, trgt );
    EngFncs->dm_deallocate_targets(trgt);

    if (rc) {
	    rc = get_DM_info(seg);  // FAILED ... test if already active
    }

    if ( !rc ) {
        seg->flags &= ~SOFLAG_NEEDS_ACTIVATE;
    }

    LOG_EXIT_INT(rc);
    return rc;
}

static int BBR_DeActivate( storage_object_t * seg )
{
    int rc;
    LOG_ENTRY();
    rc = EngFncs->dm_deactivate( seg );
    LOG_EXIT_INT(rc);
    return rc;
}

static int BBR_GetPluginFunctions( storage_object_t * object,
                                   function_info_array_t ** functions )
{
    int rc = ENOSYS;
    function_info_array_t * fia;
    BBR_Private_Data * pdata;

    LOG_ENTRY();

    *functions = NULL; //in case of early termination

    if ( object == NULL ) {
        LOG_EXIT_INT(rc);
        return rc;
    }

    pdata = (BBR_Private_Data *)object->private_data;
    if (pdata->bbr_state & BBR_CHANGE_STATE) {
        LOG_EXIT_INT(rc);
        return rc;
    }

    if ( pdata->bbr_state & BBR_ENABLED ) {
        sector_count_t bb;

        if ( isa_kernel_bbr_object( object ) )
            bb = get_kernel_bbr_remap_sector_count( object );
        else
            bb = get_engine_remap_sector_count( object );

        if ( bb ) { //remapped sectors? if so no plugin functions avialable
            LOG_DEBUG("There are remapped sectors you cannot disable\n");
            LOG_EXIT_INT(rc);
            return rc;
        }
    }

    rc = ENOMEM;
    fia = EngFncs->engine_alloc( sizeof( function_info_array_t ) );

    if ( ! fia ) {
        LOG_ERROR("Error allocating memory for an action info array.\n");
        LOG_EXIT_INT(rc);
        return rc;
    }

    rc = 0;
    if ( ( pdata->bbr_state & BBR_ENABLED ) == 0 ){
        fia->info[0].function = BBR_FUNCTION_ENABLE;
        SET_STRING(fia->info[0].name, "enable");
        SET_STRING(fia->info[0].title, "Enable");
        SET_STRING(fia->info[0].verb, "Enable");
        SET_STRING(fia->info[0].help, "Bad Block Relocation has been disabled on this device, this will re-enable it.");
    }
    else {
        fia->info[0].function = BBR_FUNCTION_DISABLE;
        SET_STRING(fia->info[0].name, "disable");
        SET_STRING(fia->info[0].title, "Disable");
        SET_STRING(fia->info[0].verb, "Disable");
        SET_STRING(fia->info[0].help, "Stop performing Bad Block Relocation on this device");
    }
    fia->count = 1;
    *functions = fia;
    LOG_EXIT_INT(rc);
    return rc;
}

static int BBR_PluginFunction( storage_object_t * object,
                               task_action_t action,
                               list_anchor_t objects,
                               option_array_t * options )
{
    int rc = EINVAL;
    BBR_Private_Data *pdata = (BBR_Private_Data *)object->private_data;

    LOG_ENTRY();

    switch ( action ) {
    case BBR_FUNCTION_ENABLE:
        object->flags |= SOFLAG_DIRTY + SOFLAG_NEEDS_ACTIVATE;
        pdata->bbr_state |= BBR_ACTIVATE;
        rc = 0;
        break;
    case BBR_FUNCTION_DISABLE:
        object->flags |= SOFLAG_DIRTY + SOFLAG_NEEDS_ACTIVATE;
        /* NOTE: There is a timing hole here if a remap happens
           between now and activate you won't be able to access
           the remapped sector (thus our check for no bad blocks
           doesn't cover everycase.  However,
           you were disabling BBR, so do you care.
        */
        pdata->bbr_state |= BBR_CHANGE_STATE;
        rc = 0;
        break;
    default:
        LOG_ERROR("Action code 0x%x is unknown.\n", action );
        break;
    }

    LOG_EXIT_INT(rc);
    return rc;
}
/*--------------------------------------------------------------------------+
+                                                                           +
+                     PLUGIN FUNCTION TABLE                                 +
+                                                                           +
+---------------------------------------------------------------------------*/
static struct plugin_functions_s fft={

    // located above
    setup_evms_plugin:                      BBR_SetupEVMSPlugin,
    cleanup_evms_plugin:                    BBR_Cleanup,
    can_set_volume:                         BBR_CanSetVolume,
    can_delete:                             BBR_CanDelete,
    can_expand:                             BBR_CanExpand,
    can_expand_by:                          BBR_CanExpandBy,
    can_shrink:                             BBR_CanShrink,
    can_shrink_by:                          BBR_CanShrinkBy,
    can_unassign:                           BBR_CanUnAssign,
    discover:                               BBR_Feature_Discovery,
    create:                                 BBR_Create,
    assign:                                 BBR_Assign,
    unassign:                               BBR_UnAssign,
    delete:                                 BBR_Delete,
    discard:                                BBR_Discard,
    expand:                                 BBR_Expand,
    shrink:                                 BBR_Shrink,
    add_sectors_to_kill_list:               BBR_AddSectorsToKillList,
    commit_changes:                         BBR_CommitChanges,
    read:                                   BBR_Read,
    write:                                  BBR_Write,
    set_volume:                             BBR_SetVolume,
    activate:                               BBR_Activate,
    deactivate:                             BBR_DeActivate,
    get_plugin_functions:                   BBR_GetPluginFunctions,
    plugin_function:                        BBR_PluginFunction,

    // located in bbroptions.c
    get_option_count:                       BBR_GetOptionCount,
    init_task:                              BBR_InitTask,
    set_option:                             BBR_SetOption,
    set_objects:                            BBR_SetObjects,
    get_info:                               BBR_GetInfo,
    get_plugin_info:                        BBR_GetPluginInfo
};


/*-------------------------------------------------------------------------+
+                                                                          +
+                BUILD AND EXPORT AN EVMS PLUGIN RECORD                    +
+                                                                          +
+--------------------------------------------------------------------------*/

static plugin_record_t bbr_plugin_record = {

    id: SetPluginID(EVMS_OEM_IBM, EVMS_SEGMENT_MANAGER, EVMS_BBR_FEATURE_ID ),

    version:                     {MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL},

    required_engine_api_version: {13,0,0},
    required_plugin_api_version: {plugin: {12,0,0} },

    short_name:                  "BBRseg",
    long_name:                   "Bad Block Relocation Segment Manager",
    oem_name:                    "IBM",

    functions:                   {plugin: &fft},

    container_functions:         NULL

};

// Vector of plugin record ptrs that we export for the EVMS Engine.
plugin_record_t *evms_plugin_records[] = {
    &bbr_plugin_record,
    NULL
};
