/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: expand.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include "fullengine.h"
#include "expand.h"
#include "lists.h"
#include "engine.h"
#include "handlemgr.h"
#include "common.h"
#include "commit.h"
#include "discover.h"
#include "object.h"
#include "memman.h"
#include "internalAPI.h"
#include "message.h"
#include "remote.h"
#include "volume.h"


int get_object_expand_points(storage_object_t * obj, u_int64_t * max_delta_size, list_anchor_t expand_points) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = obj->plugin->functions.plugin->can_expand(obj, max_delta_size, expand_points);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int get_volume_expand_points(logical_volume_t * volume, list_anchor_t expand_points) {

	int rc = 0;
	sector_count_t max_delta_size = -1;

	LOG_PROC_ENTRY();

	if (is_kernel_volume_mounted(volume, ERROR)) {
		LOG_PROC_EXIT_INT(EBUSY);
		return EBUSY;
	}

	if (volume->flags & VOLFLAG_MKFS) {
		LOG_ERROR("Volume %s cannot be expanded because it is scheduled to have a file system installed on it.\n", volume->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/*
	 * If the volume currently has an FSIM and the FSIM is not scheduled to be
	 * removed, use the FSIM's limits.
	 */
	if ((volume->original_fsim != NULL) &&
	    !(volume->flags & VOLFLAG_UNMKFS)) {

		/*
		 * Reget the current size and limits because the file system could be
		 * mounted and be active.
		 */
		rc = get_volume_sizes_and_limits(volume);

		if (rc == 0) {
			max_delta_size = volume->max_vol_size - volume->vol_size;
		}
	}

	if (rc == 0) {
		rc = get_object_expand_points(volume->object, &max_delta_size, expand_points);
	}

	if (rc == 0) {
		/*
		 * If the volume has an FSIM and the FSIM is not scheduled to be removed
		 * and there are expand points, check if the FSIM can handle the expand.
		 *
		 */
		if ((volume->original_fsim != NULL) &&
		    !(volume->flags & VOLFLAG_UNMKFS) &&
		    !list_empty(expand_points)) {

			max_delta_size = round_down_to_hard_sector(max_delta_size, volume->object);

			rc = volume->original_fsim->functions.fsim->can_expand_by(volume, &max_delta_size);

			/*
			 * The FSIM cannot handle the expand.  Delete all the expand
			 * points from the list.
			 */
			if (rc != 0) {
				/*
				 * Can't use delete_all_elements(expand_points)
				 * because we need to free the expand points.
				 */
				list_element_t iter1;
				list_element_t iter2;
				void * thing;

				LIST_FOR_EACH_SAFE(expand_points, iter1, iter2, thing) {
					delete_element(iter1);
					engine_free(thing);
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int can_expand_fs_on_volume(logical_volume_t * vol,
				   u_int64_t * max_expand_size) {

	int rc = 0;
	u_int64_t usable_object_size;
	u_int64_t expand_size;

	LOG_PROC_ENTRY();

	if (max_expand_size != NULL) {
		*max_expand_size = 0;
	}

	if (vol == NULL) {
		LOG_DETAILS("No volume given.\n");
		LOG_PROC_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	if (is_kernel_volume_mounted(vol, DETAILS)) {
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (vol->object->feature_header == NULL) {
		usable_object_size = vol->object->size;
	} else {
		usable_object_size = vol->object->size - EVMS_FEATURE_HEADER_SECTORS * 2;
	}

	if (vol->vol_size < usable_object_size) {
		expand_size = usable_object_size - vol->fs_size;
		rc = vol->file_system_manager->functions.fsim->can_expand_by(vol, &expand_size);

	} else {
		expand_size = 0;
		rc = ENOSPC;
	}

	if (max_expand_size != NULL) {
		*max_expand_size = expand_size;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Find out if an object can be expanded.  The "thing" must be either a volume
 * or a top level object.  What we'll do is call the internal
 * get_expand_points() and see if any expand candidates show up.  If we get
 * expand points then return 0 to say that the object can be expanded.  Else,
 * return an error.
 */
int evms_can_expand(object_handle_t thing) {

	int rc = 0;
	void * object;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_expand(thing);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(thing,
			      &object,
			      &type);

	if (rc == HANDLE_MANAGER_NO_ERROR) {

		switch (type) {
			case DISK:
			case SEGMENT:
			case REGION:
			case EVMS_OBJECT:
				{
					storage_object_t * obj = object;

					/*
					 * Only top level objects can be expanded.
					 */
					if (is_top_object(obj)) {
						STATIC_LIST_DECL(expand_points);
						u_int64_t max_delta_size = -1;

						rc = get_object_expand_points(obj, &max_delta_size, &expand_points);

						if (rc == 0) {
							if (list_empty(&expand_points)) {
								rc = ENOENT;
							}
						}

					} else {
						LOG_DETAILS("Object %s is not a top level object.  Only volumes and top level objects can be expanded.\n", obj->name);
						rc = EINVAL;
					}
				}
				break;

			case VOLUME:
				{
					logical_volume_t * volume = (logical_volume_t *) object;

					if (volume->flags & VOLFLAG_COMPATIBILITY) {
						if (is_kernel_volume_mounted(volume, DETAILS)) {
							rc = EBUSY;
						}
					}
					
					if (rc == 0) {
						if (volume->flags & (VOLFLAG_ACTIVE | VOLFLAG_NEW)) {
							STATIC_LIST_DECL(expand_points);

							rc = get_volume_expand_points(volume, &expand_points);

							if (rc == 0) {
								if (list_empty(&expand_points)) {
									rc = can_expand_fs_on_volume(volume, NULL);
								}
							}

						} else {
							LOG_DETAILS("Volume %s cannot be expanded because it is not active.\n", volume->name);
							rc = EINVAL;
						}
					}
				}
				break;

			default:
				LOG_ERROR("Cannot expand object of type %#x.\n", type);
				rc = EINVAL;
				break;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Make an array of expand handles (expand_handle_array_t) for the objects in
 * a list_anchor_t.
 */
static int make_expand_handle_array(list_anchor_t list, expand_handle_array_t * * peha) {

	int rc = 0;
	uint count;
	uint size;
	expand_object_info_t * object_info;
	expand_handle_array_t * eha = NULL;
	logical_volume_t * expand_vol = NULL;
	u_int64_t max_vol_expand_size = 0;

	LOG_PROC_ENTRY();

	count = list_count(list);

	LOG_DEBUG("Number of objects in the list:  %d\n", count);

	object_info = first_thing(list, NULL);
	if (object_info != NULL) {
		expand_vol = object_info->object->volume;
	}

	if (can_expand_fs_on_volume(expand_vol, &max_vol_expand_size) == 0) {
		count++;
	}

	if (count > 1) {
		size = sizeof(expand_handle_array_t) + ((count -1) * sizeof(expand_handle_t));
	} else {
		size = sizeof(expand_handle_array_t);
	}

	eha = alloc_app_struct(size, NULL);
	if (eha != NULL) {
		list_element_t iter;

		if (max_vol_expand_size != 0) {
			LOG_DEBUG("Add entry for volume %s.\n", expand_vol->name);

			rc = ensure_app_handle(expand_vol);
			if (rc == 0) {
				eha->expand_point[eha->count].object = expand_vol->app_handle;
				eha->expand_point[eha->count].max_expand_size = max_vol_expand_size;
				eha->count++;
			}
		}

		LIST_FOR_EACH(list, iter, object_info) {

			LOG_DEBUG("Add entry for storage object %s.\n", object_info->object->name);

			rc = ensure_app_handle(object_info->object);
			if (rc == 0) {
				eha->expand_point[eha->count].object = object_info->object->app_handle;
				eha->expand_point[eha->count].max_expand_size = object_info->max_expand_size;
				eha->count++;
			}
		}

	} else {
		rc = ENOMEM;
	}

	*peha = eha;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Find out which objects in the stack can expand.  evms_get_expand_points()
 * must be targeted at either a volume or a top level object.  Any plug-in
 * that can expand will add an expand_object_info_t to the expand_points
 * list.  The list is then converted into an expand_handle_array_t for return
 * to the user.
 */
int evms_get_expand_points(object_handle_t thing, expand_handle_array_t * * expand_points) {

	int rc = 0;
	void * object;
	object_type_t type;
	STATIC_LIST_DECL(expand_point_list);

	LOG_PROC_ENTRY();

	rc = check_engine_read_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_get_expand_points(thing, expand_points);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(thing,
			      &object,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (expand_points == NULL) {
		LOG_ERROR("The pointer to the expand points list cannot be NULL.\n");
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	switch (type) {
		case VOLUME:
			{
				logical_volume_t * volume = (logical_volume_t *) object;

				if (volume->flags & VOLFLAG_COMPATIBILITY) {
					if (is_kernel_volume_mounted(volume, ERROR)) {
						rc = EBUSY;
                                        }
				}
				
				if (rc == 0) {
					if (volume->flags & (VOLFLAG_ACTIVE | VOLFLAG_NEW)) {
						rc = get_volume_expand_points(volume, &expand_point_list);

					} else {
						LOG_DETAILS("Volume %s cannot be expanded because it is not active.\n", volume->name);
						rc = EINVAL;
					}
				}
			}
			break;

		case EVMS_OBJECT:
		case REGION:
		case SEGMENT:
		case DISK:
			{
				storage_object_t * obj = (storage_object_t *) object;
				/*
				 * Only top level objects can be expanded.
				 */
				if (is_top_object(obj)) {
					u_int64_t max_delta_size = -1;

					rc = get_object_expand_points(obj, &max_delta_size, &expand_point_list);

				} else {
					LOG_ERROR("Object %s is not a top level object.  Only volumes and top level objects can be expanded.\n", obj->name);
					rc = EINVAL;
				}
			}
			break;

		default:
			LOG_ERROR("An object of type %d cannot be expanded.\n", type);
			rc = EINVAL;
			break;
	}

	if (rc == 0) {
		rc = make_expand_handle_array(&expand_point_list, expand_points);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int do_volume_expand(logical_volume_t * vol) {

	int rc = 0;
	u_int64_t usable_object_size;
	u_int64_t max_expand_size;

	LOG_PROC_ENTRY();

	rc = can_expand_fs_on_volume(vol, &max_expand_size);
	if ((rc == 0) && (max_expand_size != 0)) {

		if (vol->object->feature_header == NULL) {
			usable_object_size = vol->object->size;
		} else {
			usable_object_size = vol->object->size - EVMS_FEATURE_HEADER_SECTORS * 2;
		}

		vol->vol_size = usable_object_size;

		if (vol->flags & VOLFLAG_HAS_OWN_DEVICE) {
			vol->flags |= VOLFLAG_NEEDS_ACTIVATE;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * If the object is part of a volume and the volume has no FSIM, warn the
 * user that there is no FSIM to handle the expand.  The user has the
 * choice of continuing or aborting.
 * Returns 0 if the user wants to continue, E_CANCELED if the user wants to abort.
 */
static int warn_if_no_fsim(storage_object_t * obj) {

	int rc = 0;

	LOG_PROC_ENTRY();

	if (obj->volume != NULL) {
		if (obj->volume->original_fsim == NULL) {
			char * choices[] = {"Continue", "Cancel", NULL};
			int answer = 0;	    /* Initialize to "Continue" */

			engine_user_message(&answer, choices,
					    "WARNING: Volume \"%s\" does not have an associated File System Interface Module which could coordinate the expansion of the file system on the volume.  "
					    "The file system will not be expanded.  "
					    "You may need to run a separate utility to expand the file system after this operation completes.  "
					    "Expanding the volume may make the file system unusable.\n",
					    obj->volume->name);

			if (answer != 0) {
				rc = E_CANCELED;
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int isa_valid_expand_input_object(storage_object_t    * obj,
					 storage_container_t * disk_group) {

	LOG_PROC_ENTRY();

	if ((obj->object_type != DISK) &&
	    (obj->object_type != SEGMENT) &&
	    (obj->object_type != REGION) &&
	    (obj->object_type != EVMS_OBJECT)) {
		LOG_ERROR("This is not a storage object.\n");
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* The object must not be corrupt. */
	if (obj->flags & SOFLAG_CORRUPT) {
		LOG_ERROR("Object %s is not a valid input object.  It is corrupt.\n", obj->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* The object must not insist on being the top object. */
	if (obj->flags & SOFLAG_MUST_BE_TOP) {
		LOG_ERROR("Object %s is not a valid input object.  It insists it must be a top level object.\n", obj->name);
		return EINVAL;
	}

	/* The object must be in the specified disk group. */
	if (obj->disk_group != disk_group) {
		LOG_ERROR("Object %s in disk group %s is not in disk group %s.\n", obj->name, (obj->disk_group != NULL) ? obj->disk_group->name : "(local)", (disk_group != NULL) ? disk_group->name : "(local)");
		return EINVAL;
	}

	LOG_DEBUG("Object %s is a valid input object.\n", obj->name);
	LOG_PROC_EXIT_INT(0);
	return 0;
}


static void post_expand_cleanup(storage_object_t * top_object,
				storage_object_t * obj) {
	
	list_element_t iter;
	storage_object_t * child_obj;

	/*
	 * Make sure the children of the expand object are marked as being in
	 * the same volume.
	 */
	set_volume_in_object(obj,
			     obj->volume);

	LIST_FOR_EACH(obj->child_objects, iter, child_obj) {

		/*
		 * Any stop data on the children can be wiped out by the parent.
		 */
		child_obj->flags &= ~SOFLAG_HAS_STOP_DATA;
	}

	if (obj->object_type == EVMS_OBJECT) {
		/*
		 * Mark all the child feature headers dirty to ensure that they
		 * get the correct depth and get written in the right place.
		 */
		LIST_FOR_EACH(obj->child_objects, iter, child_obj) {
			mark_feature_headers_dirty(child_obj);
		}
	}

	/* If the object is part of a volume... */
	if (obj->volume != NULL) {

		/*
		 * Mark it dirty so that the feature headers get written to
		 * their new locations.
		 */
		obj->volume->flags |= VOLFLAG_DIRTY;

		if (!(obj->volume->flags & VOLFLAG_COMPATIBILITY)) {
			obj->volume->flags |= VOLFLAG_NEEDS_ACTIVATE;
		}

		/* Update the volume size. */
		obj->volume->vol_size = top_object->size;

		if (top_object->feature_header != NULL) {
			/*
			 * Object has EVMS volume feature headers.  Subtract
			 * them from the size.
			 */
			obj->volume->vol_size -= EVMS_FEATURE_HEADER_SECTORS * 2;
		}

		obj->volume->vol_size = round_down_to_hard_sector(obj->volume->vol_size, top_object);
	}
}


static int do_object_expand(storage_object_t * obj, handle_array_t * input_objects, option_array_t * options) {

	int rc = 0;
	STATIC_LIST_DECL(input_object_list);

	LOG_PROC_ENTRY();

	if (obj->volume != NULL) {
		if (!(obj->volume->flags & (VOLFLAG_ACTIVE | VOLFLAG_NEW))) {
			LOG_DETAILS("Object %s cannot be expanded because volume %s is not active.\n", obj->name, obj->volume->name);
			LOG_PROC_EXIT_INT(EINVAL);
			return EINVAL;

		} else {
			if (obj->volume->flags & VOLFLAG_MKFS) {
				LOG_ERROR("Volume %s cannot be expanded because it is scheduled to have a file system installed on it.\n", obj->volume->name);
				LOG_PROC_EXIT_INT(EINVAL);
				return EINVAL;
			}
		}
	}

	rc = make_list(input_objects, &input_object_list);

	if (rc == 0) {
		list_element_t iter;
		storage_object_t * input_obj;

		/*
		 * Make sure all of the input objects are storage
		 * objects.  (The caller could have supplied handles
		 * for bogus items such as plug-ins or volumes.)
		 */
		LIST_FOR_EACH(&input_object_list, iter, input_obj) {
			rc = isa_valid_expand_input_object(input_obj, obj->disk_group);
			if (rc != 0) {
				break;
			}
		}

		if (rc == 0) {

			/* Find the top object in the feature stack. */
			storage_object_t * top_object = obj;
			uint parent_count = 0;

			parent_count = list_count(top_object->parent_objects);

			while (parent_count == 1) {
				top_object = first_thing(top_object->parent_objects, NULL);
				parent_count = list_count(top_object->parent_objects);
			}

			if (parent_count <= 1) {

				/*
				 * Warn the user if there is no FSIM to handle
				 * the expand.
				 */
				rc = warn_if_no_fsim(obj);

				if (rc == 0) {
					/*
					 * If the object is not part of a
					 * volume and it has stop data,
					 * remove the stop data from its
					 * current location.  It will be
					 * reapplied during commit.
					 */
					if ((top_object->volume == NULL) &&
					    (top_object->flags & SOFLAG_HAS_STOP_DATA)) {
						top_object->plugin->functions.plugin->add_sectors_to_kill_list(top_object, top_object->size - (EVMS_FEATURE_HEADER_SECTORS * 2), EVMS_FEATURE_HEADER_SECTORS * 2);
						top_object->flags &= ~SOFLAG_HAS_STOP_DATA;
					}

					/*
					 * Start the expand at the top level
					 * object.
					 */
					rc = top_object->plugin->functions.plugin->expand(top_object, obj, &input_object_list, options);

					if (rc == 0) {
						post_expand_cleanup(top_object, obj);
					}
				}

			} else {
				if (parent_count > 1) {
					LOG_ERROR("Object %s cannot be expanded because it has multiple parents.\n", obj->name);
					rc = EINVAL;
				} else {
					LOG_WARNING("Error code %d encountered when trying to find the top level object.\n", rc);
				}
			}

		} else {
			LOG_ERROR("One or more items in the input object list is not a storage object.\n");
			rc = EINVAL;
		}

	} else {
		LOG_ERROR("Error code %d when making a list from the input_objects handle array.\n", rc);
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Expand the size of an object.  The expand gets targeted at the top object of
 * the feature stack.  The plug-ins pass it on down until it hits the object
 * that is to do the real expansion.
 */
int evms_expand(object_handle_t thing, handle_array_t * input_objects, option_array_t * options) {

	int rc = 0;
	void * object;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();

	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(thing,
			      &object,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	switch (type) {
		case VOLUME:
			rc = do_volume_expand((logical_volume_t *) object);
			break;

		case EVMS_OBJECT:
		case REGION:
		case SEGMENT:
		case DISK:
			rc = do_object_expand((storage_object_t *) object, input_objects, options);
			break;

		default:
			LOG_ERROR("Object of type %d cannot be expanded.\n", type);
			rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * engine_can_expand_by() is a service that the Engine provides for the
 * plug-ins.  A plug-in calls engine_can_expand_by() to find out if all of its
 * parent objects, including the volume, would approve of an expansion by the
 * specified amount.  The Engine walks up the object tree calling each
 * plug-in's can_expand_by() function to see if it would approve of the
 * expansion.  Each plug-in can update the "size" if the resulting size of its
 * object would have a different delta size than the delta size of its child
 * object.  If the top of the object stack is reached and the object is part of
 * a volume, the Engine then checks the resulting delta size against any limits
 * that an FSIM may have imposed on the volume.
 */
int engine_can_expand_by(storage_object_t * object, sector_count_t * delta_size) {

	int rc = 0;
	storage_object_t * curr_object = object;
	sector_count_t original_delta_size = *delta_size;

	LOG_PROC_ENTRY();

	while ((rc == 0) && !list_empty(curr_object->parent_objects)) {

		curr_object = first_thing(curr_object->parent_objects, NULL);

		if (curr_object != NULL) {
			rc = curr_object->plugin->functions.plugin->can_expand_by(curr_object, delta_size);
		}
	}

	if (rc == 0) {
		/*
		 * If this object is part of a volume and the volume has an FSIM that is
		 * not scheduled to be removed, check to make sure the final size does
		 * not go over the FSIM's maximum size it can handle for its backing
		 * storage object.
		 */
		if (object->volume != NULL) {

			if (object->volume->flags & VOLFLAG_COMPATIBILITY) {
				if (is_kernel_volume_mounted(object->volume, DETAILS)) {
					rc = EBUSY;
				}
			}
			
			if (rc == 0) {
				if ((object->volume->original_fsim != NULL) &&
				    !(object->volume->flags & VOLFLAG_UNMKFS)) {
					rc = object->volume->original_fsim->functions.fsim->can_expand_by(object->volume, delta_size);
				}
			}
		}
	}

	if (rc == 0) {
		if (*delta_size != original_delta_size) {
			rc = EAGAIN;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

