/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: commit.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <dirent.h>
#include <mntent.h>
#include <sys/ioctl.h>
#include <sys/mount.h>
#include <sys/reboot.h>
#include <sys/time.h>

#include "fullengine.h"
#include "lists.h"
#include "commit.h"
#include "common.h"
#include "engine.h"
#include "discover.h"
#include "memman.h"
#include "message.h"
#include "internalAPI.h"
#include "discover.h"
#include "volume.h"
#include "crc.h"
#include "dm.h"
#include "cluster.h"
#include "remote.h"

boolean commit_in_progress = FALSE;
boolean need_reboot        = FALSE;

STATIC_LIST_DECL(commit_node_order);

static char status_msg[256];

void status_message(char * fmt, ...) {

	va_list args;

	va_start(args, fmt);
	vsprintf(status_msg, fmt, args);
	va_end(args);

	LOG_DEBUG("%s", status_msg);

	if (ui_callbacks != NULL) {
		if (ui_callbacks->status != NULL) {
			ui_callbacks->status(status_msg);
		}
	}
}

/*
 * After a certain point (the deleting of volumes in the kernel) the commit
 * process is not allowed to abort due to errors.  If it were to abort, the
 * disks could be left in an inconsistent state and the system could be
 * unusable.  The commit process must try to get as much to disk as possible.
 * However, that presents a problem for error codes.  The commit process could
 * generate several errors.  Which one is to be returned to the user?
 * The following functions provide services for error reporting.  The logic is:
 * - Errors are recorded by class; the debug levels are used for the classes.
 * - The first error for each class is the one that is kept.
 * - When asked for the final overall error, the non-zero error code of the
 *    most critical class (lowest debug level) is returned as the error.
 */

static int commit_error[WARNING + 1] = {0};


static void clear_commit_errors() {

	int i;

	LOG_PROC_ENTRY();

	for (i = 0; i <= WARNING; i++) {
		commit_error[i] = 0;
	}

	LOG_PROC_EXIT_VOID();
}


static void set_commit_error(debug_level_t level, int error) {

	LOG_PROC_ENTRY();

	LOG_DEBUG("level: %d  error: %d: %s\n", level, error, evms_strerror(error));

	if (level <= WARNING) {
		if (commit_error[level] == 0) {
			commit_error[level] = error;
		}

	} else {
		LOG_DEBUG("Attempt to set error code %d at invalid error level %d.\n", error, level);
	}

	LOG_PROC_EXIT_VOID();
}


static int get_commit_error() {

	int rc = 0;
	int i;

	LOG_PROC_ENTRY();

	for (i = 0; (i <= WARNING) && (rc == 0); i++) {
		rc = commit_error[i];
	}

	LOG_PROC_EXIT_INT(rc);
	return(rc);
}


/*
 * Find any volumes that are marked for unmkfs and run unmkfs on them.  Volumes
 * marked for unmkfs can be in the volumes_list and in the VolumeDeleteList.
 */
static int unmkfs_volumes() {

	int rc = 0;
	list_element_t iter;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(&volumes_list, iter, vol) {
		if (vol->flags & VOLFLAG_UNMKFS) {

			/* Make sure there is an FSIM to do the unmkfs. */
			if (vol->original_fsim != NULL) {

				/*
				 * Replace the private data pointer in the
				 * logical_volume_t with the private data
				 * pointer of the original FSIM in case the FSIM
				 * has any private data it needs to do the
				 * unmkfs.
				 */
				void * private_data;

				private_data = vol->private_data;
				vol->private_data = vol->original_fsim_private_data;

				status_message("Running unmkfs on volume %s...\n", vol->name);
				rc = vol->original_fsim->functions.fsim->unmkfs(vol);

				vol->private_data = private_data;
			}

			if (rc == 0) {
				vol->flags &= ~VOLFLAG_UNMKFS;
				vol->original_fsim = NULL;
				vol->original_fsim_private_data = NULL;

			} else {
				/* Any errors stop the commit process. */
				break;
			}
		}
	}

	if (rc == 0) {
		LIST_FOR_EACH(&volume_delete_list, iter, vol) {
			if (vol->flags & VOLFLAG_UNMKFS) {

				/* Make sure there is an FSIM to do the unmkfs. */
				if (vol->original_fsim != NULL) {

					/*
					 * Replace the private data pointer in
					 * the logical_volume_t with the private
					 * data pointer of the original FSIM in
					 * case the FSIM has any private data it
					 * needs to do the unmkfs.
					 */
					void * private_data;

					private_data = vol->private_data;
					vol->private_data = vol->original_fsim_private_data;

					status_message("Running unmkfs on volume %s...\n", vol->name);
					rc = vol->original_fsim->functions.fsim->unmkfs(vol);

					vol->private_data = private_data;
				}

				if (rc == 0) {
					vol->flags &= ~VOLFLAG_UNMKFS;
					vol->original_fsim = NULL;
					vol->original_fsim_private_data = NULL;

				} else {
					/* Any errors stop the commit process. */
					break;
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return(rc);
}


/*
 * Find any volumes that have shrunk in size and call the FSIM for
 * the volume to do the shrink.
 */
static int shrink_volumes() {

	int rc = 0;
	list_element_t iter;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(&volumes_list, iter, vol) {
		if (vol->shrink_vol_size < vol->original_vol_size) {

			LOG_DEBUG("Shrink volume \"%s\" from %"PRIu64" sectors to %"PRIu64" sectors.\n", vol->name, vol->original_vol_size, vol->shrink_vol_size);
			if (vol->original_fsim != NULL) {

				if (vol->fs_size > vol->shrink_vol_size) {
					u_int64_t new_size;

					status_message("Shrinking volume %s...\n", vol->name);
					LOG_DEBUG("Calling %s FSIM to do the shrink.\n", vol->original_fsim->short_name);
					rc = vol->original_fsim->functions.fsim->shrink(vol, vol->shrink_vol_size, &new_size);

					if (rc == 0) {
						/*
						 * Reget the limits for the
						 * file system.
						 */
						get_volume_sizes_and_limits(vol);

						/* Safety check */
						if (vol->fs_size > vol->shrink_vol_size) {
							engine_user_message(NULL, NULL, "FSIM plug-in %s only shrunk the file system on volume %s to %"PRIu64" sectors which is not small enough for the new volume size of %"PRIu64" sectors.\n", vol->original_fsim->short_name, vol->name, vol->fs_size, vol->shrink_vol_size);
							rc = ENOSPC;
							set_commit_error(WARNING, rc);
						}

					} else {
						engine_user_message(NULL, NULL, "FSIM plug-in %s returned error code %d when called to shrink volume %s to %"PRIu64" sectors.\n", vol->original_fsim->short_name, rc, vol->name, vol->shrink_vol_size);
						set_commit_error(WARNING, rc);
					}

				} else {
					LOG_DEBUG("The file system on volume\"%s\" is %"PRIu64" sectors which is contained in the shrink size of %"PRIu64" sectors.  Shrinking is not necessary.\n", vol->name, vol->fs_size, vol->shrink_vol_size);
				}

			} else {
				LOG_DEBUG("Volume \"%s\" has no FSIM.  The volume's size will be set to %"PRIu64" sectors.\n", vol->name, vol->shrink_vol_size);
			}

			if (rc == 0) {
				/* Set the volume's new original size. */
				vol->original_vol_size = vol->shrink_vol_size;

			} else {
				/* Any errors stop the commit process. */
				break;
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return(rc);
}


static int rename_volumes() {

	int rc = 0;
	list_element_t iter1;
	list_element_t iter2;
	void * dummy;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH_SAFE(&rename_volume_list, iter1, iter2, dummy) {
		int tmp_rc;
		rename_volume_t * rv = (rename_volume_t *) iter1;

		LOG_DEBUG("Rename %s to %s.\n", rv->old_vol_name, rv->new_vol_name);

		tmp_rc = dm_rename_volume(rv->volume, rv->old_vol_name, rv->new_vol_name);

		if (tmp_rc != 0) {
			engine_user_message(NULL, NULL, "Error from device-mapper when renaming volume %s to %s.\n", rv->old_vol_name, rv->new_vol_name);
			if (rc == 0) {
				rc = tmp_rc;
			}
		}

		/*
		 * Since *rv behaves like an element, delete_element() will
		 * free the *rv.
		 */
		delete_element(iter1);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int delete_volumes(void) {
	
	int rc = 0;
	list_element_t iter1;
	list_element_t iter2;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH_SAFE(&volume_delete_list, iter1, iter2, vol) {
		LOG_DEBUG("Request to delete volume %s.\n", vol->name);

		if ((vol->flags & VOLFLAG_HAS_OWN_DEVICE) &&
		    (vol->flags & (VOLFLAG_ACTIVE | VOLFLAG_NEEDS_DEACTIVATE))) {
			dm_deactivate_volume(vol);
		}

		rc = unlink(vol->name);
		if (rc != 0) {
			if (errno == ENOENT) {
				rc = 0;
			} else {
				rc = errno;
				LOG_WARNING("unlink(\"%s\") returned error code %d: %s\n", vol->name, rc, strerror(rc));
			}
		}

		if (rc == 0) {
			delete_element(iter1);
			engine_free(vol->mount_point);
			engine_free(vol->mkfs_options);
			engine_free(vol->fsck_options);
			engine_free(vol->defrag_options);
			engine_free(vol);

		} else {
			/* Any errors stop the commit process. */
			break;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static void deactivate_volumes(void) {
	
	list_element_t iter;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(&volumes_list, iter, vol) {
		if ((vol->flags & VOLFLAG_HAS_OWN_DEVICE) &&
		    (vol->flags & VOLFLAG_NEEDS_DEACTIVATE)) {

			LOG_DEBUG("Deactivate volume %s.\n", vol->name);
			dm_deactivate_volume(vol);
		}
	}

	LOG_PROC_EXIT_VOID();
	return;
}


static char dev_name[EVMS_OBJECT_NODE_PATH_LEN + EVMS_NAME_SIZE + 1];

static int deactivate_object(storage_object_t * obj) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = obj->plugin->functions.plugin->deactivate(obj);
	if (rc == 0) {
		strcpy(dev_name, EVMS_OBJECT_NODE_PATH);
		strcat(dev_name, obj->name);

		if (unlink(dev_name) != 0) {
			LOG_WARNING("unlink(\"%s\") returned error code %d:  %s\n", dev_name, errno, strerror(errno));
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int process_deactivate_list(void) {

	int rc = 0;
	list_element_t iter1;
	list_element_t iter2;
	storage_object_t * obj;

	LOG_PROC_ENTRY();

	/*
	 * The deactivate_list is ordered.  Processing must go from the start of
	 * the list to the end of the list.
	 */
	LIST_FOR_EACH_SAFE(&deactivate_list, iter1, iter2, obj) {
		rc = deactivate_object(obj);

		if (rc == 0) {
			delete_element(iter1);

			/*
			 * The object already had it's lists destroyed and its
			 * name unregistered.  All we have to do is free it.
			 */
			engine_free(obj);

		} else {
			/* Any errors stop the processing. */
			break;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int deactivate_objects(void) {

	int rc = 0;
	list_element_t iter;
	storage_object_t * obj;

	rc = process_deactivate_list();

        if (rc == 0) {

		/* Deactivate any objects that say they need deactivation. */
		list_anchor_t all_objects;

		rc = engine_get_object_list(0,			// all object types
					    DATA_TYPE,
					    NULL,		// any plug-in
					    NULL,		// any nodeid
					    0,			// no filter flags
					    &all_objects);
		if (rc == 0) {
			LIST_FOR_EACH(all_objects, iter, obj) {
				if (obj->flags & SOFLAG_NEEDS_DEACTIVATE) {
					rc = deactivate_object(obj);

					/* Any errors stop the processing. */
					if (rc != 0) {
						break;
					}
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static void remove_unneeded_stop_data(list_anchor_t list) {

	list_element_t iter;
	storage_object_t * obj;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(list, iter, obj) {
		/*
		 * Only check objects that are part of volumes.
		 * Stop data can be left on non-volume objects.
		 */
		if (obj->volume != NULL) {
			if (obj->flags & SOFLAG_HAS_STOP_DATA) {
				LOG_DEBUG("Removing stop data from object %s.\n", obj->name);
				obj->plugin->functions.plugin->add_sectors_to_kill_list(obj,
											obj->size - (EVMS_FEATURE_HEADER_SECTORS * 2),
											EVMS_FEATURE_HEADER_SECTORS * 2);
				obj->flags &= ~SOFLAG_HAS_STOP_DATA;

			} else {
				LOG_DEBUG("Object %s does not have stop data on it.\n", obj->name);
			}

		} else {
			LOG_DEBUG("Object %s is not part of a volume.\n", obj->name);
		}
	}

	LOG_PROC_EXIT_VOID();
	return;
}


static void cleanup_orphaned_stop_data(void) {

	LOG_PROC_ENTRY();

	remove_unneeded_stop_data(&disks_list);
	remove_unneeded_stop_data(&segments_list);
	remove_unneeded_stop_data(&regions_list);

	LOG_PROC_EXIT_VOID();
	return;
}


/*
 * Process the KillSectorList.  The KillSectorList contains a list of
 * kill_sector_record_ts.  Each kill_sector_record_t contains a disk, a starting
 * LBA, and a count of sectors to be zeroed out.
 */
static int kill_sectors() {

	int rc = 0;
	list_element_t iter1;
	list_element_t iter2;
	kill_sector_record_t * ksr;
	char                 * buffer = engine_alloc(4096);
	int                    current_buffer_size = 4096;
	int                    buffer_size_needed = 0;

	LOG_PROC_ENTRY();

	if (buffer == NULL) {
		LOG_CRITICAL("Error allocating memory for a zero filled buffer for killing sectors.\n");
		set_commit_error(CRITICAL, ENOMEM);
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	LIST_FOR_EACH_SAFE(&kill_sectors_list, iter1, iter2, ksr) {
		/* Test validity.  Ignore bad records. */
		if (ksr->logical_disk != NULL) {
			if (ksr->logical_disk->plugin != NULL) {

				/* Grow our zero filled buffer if needed. */
				buffer_size_needed = ksr->sector_count * EVMS_VSECTOR_SIZE;
				if (current_buffer_size < buffer_size_needed) {
					buffer = engine_realloc(buffer, buffer_size_needed);
					if (buffer != NULL) {
						current_buffer_size = buffer_size_needed;

					} else {
						LOG_CRITICAL("Error allocating memory for a zero filled buffer for killing sectors.\n");
						rc = ENOMEM;
						set_commit_error(CRITICAL, rc);
					}
				}

				/* Zap the sectors. */
				if (rc == 0) {
					LOG_DEBUG("Writing %"PRIu64" sector%s of zeros to disk %s at sector %"PRIu64".\n", ksr->sector_count, (ksr->sector_count == 1) ? "" : "s", ksr->logical_disk->name, ksr->sector_offset);
					rc = ksr->logical_disk->plugin->functions.plugin->write(ksr->logical_disk, ksr->sector_offset, ksr->sector_count, buffer);

					if (rc == 0) {
						/*
						 * The sectors were killed.
						 * Remove the ksr from the list
						 * and free it.
						 */
						delete_element(iter1);
						engine_free(ksr);

					} else {
						engine_user_message(NULL, NULL, "Error code %d from call to write zeroes to %"PRIu64" sectors starting at sector %"PRIu64" on disk %s.\n", rc, ksr->sector_count, ksr->sector_offset, ksr->logical_disk->name);
						set_commit_error(WARNING, rc);
					}
				}

			} else {
				LOG_WARNING("Kill sector record does not have a valid disk.  The disk's plug-in pointer is NULL.\n");
				set_commit_error(WARNING, ENXIO);
			}

		} else {
			LOG_WARNING("Kill sector record does not have a valid disk.  The disk pointer is NULL.\n");
			set_commit_error(WARNING, ENXIO);
		}
	}

	engine_free(buffer);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Find any dirty segments and call their plug-ins to commit the changes
 * to disk.
 */
static void commit_segments(commit_phase_t phase) {

	int rc = 0;
	list_element_t iter;
	storage_object_t * segment;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(&segments_list, iter, segment) {
		if ((segment->flags & SOFLAG_DIRTY) &&
		    (segment->plugin != NULL) &&
		    (segment->plugin->functions.plugin != NULL)) {

			status_message("Phase %d:  Committing changes on segment %s...\n", phase, segment->name);
			rc = segment->plugin->functions.plugin->commit_changes(segment, phase);

			if ((rc != 0) && (rc != E_CANCELED)) {
				engine_user_message(NULL, NULL, "Plug-in %s returned error %d when committing changes for segment %s during phase %d.\n", segment->plugin->short_name, rc, segment->name, phase);
				set_commit_error(WARNING, rc);
			}
		}
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * Find any dirty containers and call their plug-ins to commit the changes
 * to disk.
 */
static void commit_containers(commit_phase_t phase) {

	int rc = 0;
	list_element_t iter;
	storage_container_t * con;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(&containers_list, iter, con) {
		if ((con->flags & SCFLAG_DIRTY) &&
		    (con->plugin != NULL) &&
		    (con->plugin->container_functions != NULL)) {

			status_message("Phase %d:  Committing changes on container %s...\n", phase, con->name);
			rc = con->plugin->container_functions->commit_container_changes(con, phase);

			if (rc != 0) {
				engine_user_message(NULL, NULL, "Plug-in %s returned error %d when committing changes for container %s during phase %d.\n", con->plugin->short_name, rc, con->name, phase);
				set_commit_error(WARNING, rc);
			}
		}
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * Find any dirty regions and call their plug-ins to commit the changes
 * to disk.
 */
static void commit_regions(commit_phase_t phase) {

	int rc = 0;
	list_element_t iter;
	storage_object_t * region;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(&regions_list, iter, region) {
		if ((region->flags & SOFLAG_DIRTY) &&
		    (region->plugin != NULL) &&
		    (region->plugin->functions.plugin != NULL)) {

			status_message("Phase %d:  Committing changes on region %s...\n", phase, region->name);
			rc = region->plugin->functions.plugin->commit_changes(region, phase);

			if ((rc != 0) && (rc != E_CANCELED)) {
				engine_user_message(NULL, NULL, "Plug-in %s returned error %d when committing changes for region %s during phase %d.\n", region->plugin->short_name, rc, region->name, phase);
				set_commit_error(WARNING, rc);
			}
		}
	}

	LOG_PROC_EXIT_VOID();
}


typedef struct commit_parms_s {
	u_int32_t      depth;
	commit_phase_t phase;
} commit_parms_t;


/*
 * Write this object's feature header to disk, if the header is marked dirty.
 * "parameters" points to the parent object.
 */
static int write_feature_header(storage_object_t * obj,
				u_int32_t          depth,
				commit_phase_t     phase) {

	int rc = 0;

	LOG_PROC_ENTRY();

	if ((phase == FIRST_METADATA_WRITE) ||
	    (phase == SECOND_METADATA_WRITE)) {

		/* Only write the feature header if it is marked dirty. */
		if (obj->flags & SOFLAG_FEATURE_HEADER_DIRTY) {
			lsn_t  fh_lsn;

			/* Setup the feature header in phase one. */
			if (phase == FIRST_METADATA_WRITE) {
				evms_feature_header_t * fh = obj->feature_header;

				/* Make sure the signature is correct. */

				fh->signature = EVMS_FEATURE_HEADER_SIGNATURE;

				/* Set the feature header version. */
				fh->version.major = EVMS_FEATURE_HEADER_MAJOR;
				fh->version.minor = EVMS_FEATURE_HEADER_MINOR;
				fh->version.patchlevel = EVMS_FEATURE_HEADER_PATCHLEVEL;

				/*
				 * Set the version of the Engine that wrote the feature header.
				 */
				fh->engine_version.major = MAJOR_VERSION;
				fh->engine_version.minor = MINOR_VERSION;
				fh->engine_version.patchlevel = PATCH_LEVEL;

				/* Set the object depth. */
				fh->object_depth = depth;

				/*
				 * If the object is part of a volume, set the volume
				 * information.
				 */
				if (obj->volume != NULL) {
					char * prefix = get_volume_prefix(obj->disk_group);

					fh->flags &= ~EVMS_VOLUME_DATA_OBJECT;
					fh->volume_serial_number = obj->volume->serial_number;
					fh->volume_system_id = obj->volume->dev_minor;
					memset(fh->volume_name, 0, sizeof(fh->volume_name));

					/*
					 * Strip off the volume prefix when copying the volume
					 * name.  Volume names in the feature header are
					 * relative to the prefix.
					 */
					if (prefix != NULL) {
						strcpy(fh->volume_name, obj->volume->name + strlen(prefix));
						engine_free(prefix);
					} else {
						/*
						 * Error getting the prefix.
						 * Skip the EVMS dev node prefix
						 * and hope for the best.
						 */
						strcpy(fh->volume_name, obj->volume->name + strlen(EVMS_DEV_NODE_PATH));
					}
				} else {
					fh->flags |= EVMS_VOLUME_DATA_OBJECT;
					fh->volume_serial_number = 0;
					fh->volume_system_id = 0;
					memset(fh->volume_name, 0, sizeof(fh->volume_name));
				}
			}

			/*
			 * Convert the feature header to disk endian format before
			 * calculating the CRC.
			 */
			feature_header_cpu_to_disk(obj->feature_header);
			obj->feature_header->crc = 0;
			obj->feature_header->crc = CPU_TO_DISK32(calculate_CRC(0xFFFFFFFF, obj->feature_header, sizeof(evms_feature_header_t)));

			if (phase == FIRST_METADATA_WRITE) {
				fh_lsn = obj->size - EVMS_FEATURE_HEADER_SECTORS;
			} else {
				fh_lsn = obj->size - (EVMS_FEATURE_HEADER_SECTORS * 2);
			}

			/* Write the feature header for the object. */
			status_message("Phase %d:  Writing feature header on object %s...\n", phase, obj->name);
			rc = obj->plugin->functions.plugin->write(obj, fh_lsn, EVMS_FEATURE_HEADER_SECTORS, obj->feature_header);

			/* Convert the feature header back to CPU endian format. */
			feature_header_disk_to_cpu(obj->feature_header);

			if (rc != 0) {
				engine_user_message(NULL, NULL, "Error code %d when writing phase %d feature header to object %s.\n", rc, phase, obj->name);
				set_commit_error(WARNING, rc);
			}

			/* Committing of a feature header is finished after phase 2. */
			if (phase >= SECOND_METADATA_WRITE) {

				/* Turn off the SOFLAG_FEATURE_HEADER_DIRTY flag. */
				/* If this object had stop data, it's gone now. */
				obj->flags &= ~(SOFLAG_FEATURE_HEADER_DIRTY | SOFLAG_HAS_STOP_DATA);
			}

		} else {
			LOG_DEBUG("Object %s does not have its feature header marked dirty.\n", obj->name);
		}
	}

	/* Don't let errors stop the commit process. */
	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static void commit_feature_header(storage_object_t * object,
				  storage_object_t * parent_object,
				  u_int32_t          depth,
				  commit_phase_t     phase) {

	LOG_PROC_ENTRY();

	if ((phase == FIRST_METADATA_WRITE) ||
	    (phase == SECOND_METADATA_WRITE)) {

		if (object->flags & SOFLAG_FEATURE_HEADER_DIRTY) {
			list_element_t iter;
			storage_object_t * obj;

			/*
			 * Setup the feature header for this object and its
			 * siblings in phase 1.  Other phases just write the same
			 * feature header in different location(s).
			 */
			if (phase == FIRST_METADATA_WRITE) {
				u_int64_t sequence_number = 0;

				/* Get the highest sequence number of the siblings. */
				LIST_FOR_EACH(parent_object->child_objects, iter, obj) {
					if (obj->feature_header != NULL) {
						if (obj->feature_header->sequence_number < sequence_number) {
							sequence_number = obj->feature_header->sequence_number;
						}
					}
				}

				/* Bump the sequence number. */
				sequence_number ++;

				/* Set the new sequence number in the siblings. */
				LIST_FOR_EACH(parent_object->child_objects, iter, obj) {
					if (obj->feature_header != NULL) {

						/*
						 * Set the feature header sequence number if it is not already at the
						 * specified number.
						 */
						if (obj->feature_header->sequence_number != sequence_number) {
							obj->feature_header->sequence_number = sequence_number;

							/* Mark the feature header dirty so it gets rewritten to disk. */
							obj->flags |= SOFLAG_FEATURE_HEADER_DIRTY;
						}
					}
				}
			}

			/*
			 * Step up to the parent object and commit all of its child
			 * objects' feature headers together.  All the child objects'
			 * feature headers should have the same sequence number and
			 * volume complete flag.
			 */
			LIST_FOR_EACH(parent_object->child_objects, iter, obj) {
				write_feature_header(obj, depth, phase);
			}
		}
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * Given an object, commit the object and its children to disk.  If the object
 * is an EVMS object, first call this function recursively on the child objects
 * to commit any potential dirty children or any potentially dirty feature
 * headers on the children.  The recursion goes down until we hit a System Data
 * object, which is always the bottom object of a feature stack.  Once the
 * recursion is finished, then if the object is dirty, call its plug-in to
 * commit the changes to disk.
 */
static void commit_object(storage_object_t * obj,
			  u_int32_t          depth,
			  commit_phase_t     phase) {
	int rc = 0;
	storage_object_t * child_obj = NULL;

	LOG_PROC_ENTRY();

	/* If this is an EVMS object, first commit the child objects. */
	if (obj->object_type == EVMS_OBJECT) {

		/*
		 * Pick one of the children for examining and writing this object's feature
		 * header(s), which is/are hanging off the child object(s).
		 */
		child_obj = first_thing(obj->child_objects, NULL);

		if (child_obj != NULL) {
			list_element_t iter;
			storage_object_t * child;

			/*
			 * Increase the depth for the child objects if this object has
			 * a feature header hanging off its child and the child is
			 * not a replace object.  Replace objects do not increase the
			 * depth.
			 */
			if ((child_obj->feature_header != NULL) &&
			    (child_obj->plugin != replace_plugin)) {
				depth++;
			}

			/*
			 * Call myself recursively to commit any changes on my child
			 * objects.
			 */
			LIST_FOR_EACH(obj->child_objects, iter, child) {
				commit_object(child, depth, phase);
			}

			/*
			 * Reset the depth for the commit of this object's feature
			 * header(s).
			 */
			if ((child_obj->feature_header != NULL) &&
			    (child_obj->plugin != replace_plugin)) {
				depth--;
			}
		}

		/* Call the feature to commit the object if the object is dirty. */
		if (obj->flags & SOFLAG_DIRTY) {
			status_message("Phase %d:  Committing changes to object %s...\n", phase, obj->name);
			rc = obj->plugin->functions.plugin->commit_changes(obj, phase);

			if ((rc != 0) && (rc != E_CANCELED)) {
				engine_user_message(NULL, NULL, "Plug-in %s returned error %d when committing changes for object %s during phase %d.\n", obj->plugin->short_name, rc, obj->name, phase);
				set_commit_error(WARNING, rc);
			}
		}

		if ((obj->object_type == EVMS_OBJECT) &&
		    (child_obj != NULL)) {
			/* Commit any feature header for this object. */
			commit_feature_header(child_obj, obj, depth, phase);
		}
	}

	/* Don't allow errors to stop the commit process. */
	LOG_PROC_EXIT_VOID();
	return;
}


/*
 * Mark all the feature headers in an object stack dirty so that they will
 * be re-written with updated data.
 */

void mark_feature_headers_dirty(storage_object_t * obj) {

	LOG_PROC_ENTRY();

	if ((obj->object_type == DISK) ||
	    (obj->object_type == SEGMENT) ||
	    (obj->object_type == REGION) ||
	    (obj->object_type == EVMS_OBJECT)) {

		/* If this object has a feature header, mark it dirty. */
		if (obj->feature_header != NULL) {
			LOG_DEBUG("Mark feature header dirty on object %s.\n", obj->name);
			obj->flags |= SOFLAG_FEATURE_HEADER_DIRTY;
		}

		/*
		 * If the object is an EVMS object, call myself recursively on
		 * all my children to set their feature headers dirty.
		 */
		if (obj->object_type == EVMS_OBJECT) {
			list_element_t iter;
			storage_object_t * child;

			LIST_FOR_EACH(obj->child_objects, iter, child) {
				mark_feature_headers_dirty(child);
			}
		}
	}

	LOG_PROC_EXIT_VOID();
	return;
}


/*
 * Find the volumes that are marked dirty and call the plug-in for the
 * volume's object to commit its changes to disk.
 */
static void commit_volumes(commit_phase_t phase) {

	list_element_t iter;
	logical_volume_t * vol;
	u_int32_t          depth = 1;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(&volumes_list, iter, vol) {
		if (vol->flags & VOLFLAG_DIRTY) {

			/*
			 * Mark all the feature headers for all the EVMS objects
			 * in the stack dirty in phase one so that the new volume
			 * information will be recorded in all the feature headers.
			 */
			if (phase == FIRST_METADATA_WRITE) {
				mark_feature_headers_dirty(vol->object);
			}
		}

		/*
		 * If the volume's object has a feature header, that means
		 * the volume has no EVMS features.  It just has a feature
		 * header for setting the EVMS volume information.
		 */
		if (vol->object->feature_header != NULL) {

			write_feature_header(vol->object, depth, phase);
		}

		/*
		 * Use the commit_object() function to start the commit of the
		 * objects under the volume.  The last parameter is used to
		 * pass the phase of the commit.
		 */
		commit_object(vol->object, depth, phase);

		/*
		 * Commit of volumes is finished after phase 2.
		 * VOLFLAG_NEW is left to its current setting so that the
		 * code at the end of commit will probe for file systems
		 * on the new volumes.
		 */
		if (phase >= SECOND_METADATA_WRITE) {
			vol->flags &= ~VOLFLAG_DIRTY;
		}
	}

	LOG_PROC_EXIT_VOID();
	return;
}


static evms_feature_header_t stop_data = {

	signature:      EVMS_FEATURE_HEADER_SIGNATURE,
	version:        {major:      EVMS_FEATURE_HEADER_MAJOR,
		         minor:      EVMS_FEATURE_HEADER_MINOR,
		         patchlevel: EVMS_FEATURE_HEADER_PATCHLEVEL},
	engine_version: {major:      MAJOR_VERSION,
		         minor:      MINOR_VERSION,
		         patchlevel: PATCH_LEVEL},
	flags:          EVMS_VOLUME_DATA_STOP,
	object_depth:   1
};


/*
 * Given an object, check to see if it should have stop data written on it.
 * This function is called only with storage objects that are topmost and
 * writeable.
 */
static int write_stop_data_on_object(storage_object_t * obj,
				     commit_phase_t     phase) {
	int rc = 0;
	sector_count_t  fh_lsn;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Request to write phase %d stop data on object %s.\n", phase, obj->name);

	if (!(obj->flags & (SOFLAG_CORRUPT | SOFLAG_READ_ONLY))) {
		if (!(obj->flags & SOFLAG_HAS_STOP_DATA)) {

			/*
			 * Convert to disk endian format and calculate the stop data CRC
			 * if we haven't done so already.
			 */
			if (stop_data.crc == 0) {
				feature_header_cpu_to_disk(&stop_data);
				stop_data.crc = CPU_TO_DISK32(calculate_CRC(0xFFFFFFFFL, &stop_data, sizeof(evms_feature_header_t)));
			}

			LOG_DEBUG("Write phase %d stop data on object %s.\n", phase, obj->name);

			status_message("Phase %d:  Writing stop data on object %s...\n", phase, obj->name);
			if (phase == FIRST_METADATA_WRITE) {
				fh_lsn = obj->size - EVMS_FEATURE_HEADER_SECTORS;
			} else {
				fh_lsn = obj->size - (EVMS_FEATURE_HEADER_SECTORS * 2);
			}

			rc = obj->plugin->functions.plugin->write(obj, fh_lsn, EVMS_FEATURE_HEADER_SECTORS, &stop_data);

			if (rc != 0) {
				engine_user_message(NULL, NULL, "Error code %d from write of stop data on object %s.\n", rc, obj->name);
				set_commit_error(WARNING, rc);
			}

			if (phase >= SECOND_METADATA_WRITE) {
				obj->flags |= SOFLAG_HAS_STOP_DATA;
			}

		} else {
			LOG_DEBUG("Object %s already has stop data on it.\n", obj->name);
		}

	} else {
		if (obj->flags & SOFLAG_CORRUPT) {
			LOG_DEBUG("Object %s is corrupt.  I'm not writing stop data on it.\n", obj->name);
		}
		if (obj->flags & SOFLAG_READ_ONLY) {
			LOG_DEBUG("Object %s is read only.  I'm not writing stop data on it.\n", obj->name);
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Put stop data on all the top level segments and regions.
 */
static void commit_stop_data(commit_phase_t phase) {

	int rc = 0;
	list_anchor_t top_object_list;
	list_element_t iter;
	storage_object_t * obj;

	LOG_PROC_ENTRY();

	if ((phase == FIRST_METADATA_WRITE) ||
	    (phase == SECOND_METADATA_WRITE)) {

		rc = engine_get_object_list(DISK | SEGMENT | REGION,
					    DATA_TYPE,
					    NULL,
					    NULL,
					    TOPMOST | WRITEABLE,
					    &top_object_list);

		if (rc == 0) {
			LIST_FOR_EACH(top_object_list, iter, obj) {
				write_stop_data_on_object(obj, phase);
			}
		}
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * Find any dirty EVMS objects and call their plug-ins to commit
 * the changes to disk.
 */
static void commit_objects(commit_phase_t phase) {

	int rc = 0;
	list_anchor_t top_object_list;
	u_int32_t      depth = 1;
	list_element_t iter;
	storage_object_t * obj;

	LOG_PROC_ENTRY();

	/* Get a list of EVMS objects that are topmost. */
	rc = engine_get_object_list(EVMS_OBJECT,
				    DATA_TYPE,
				    NULL,
				    NULL,
				    TOPMOST,
				    &top_object_list);

	if (rc == 0) {
		/*
		 * Call commit_object() for each object in the top_object_list.
		 */
		LIST_FOR_EACH(top_object_list, iter, obj) {
			commit_object(obj, depth, phase);
		}

	} else {
		set_commit_error(WARNING, rc);
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * Find any volumes that are marked for mkfs and run mkfs on them.
 */
static void mkfs_volumes() {

	int rc = 0;
	list_element_t iter;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(&volumes_list, iter, vol) {
		if (vol->flags & VOLFLAG_MKFS) {

			/* Make sure there is an FSIM to do the mkfs. */
			if (vol->file_system_manager != NULL) {
				status_message("Running %s mkfs on volume %s...\n", vol->file_system_manager->short_name, vol->name);
				rc = vol->file_system_manager->functions.fsim->mkfs(vol, vol->mkfs_options);

				if (rc == 0) {
					vol->original_fsim = vol->file_system_manager;
					vol->original_fsim_private_data = vol->private_data;

					/*
					 * Turn off VOLFLAG_NEW and VOLFLAG_PROBE_FS, if they
					 * are on, so that the code at the end of commit won't
					 * try to find an FSIM for the new volume.
					 */
					vol->flags &= ~(VOLFLAG_NEW | VOLFLAG_PROBE_FS);

					/* Get the new limits for the file system. */
					get_volume_sizes_and_limits(vol);

				} else {
					set_commit_error(WARNING, rc);
					engine_user_message(NULL, NULL, "FSIM plug-in %s returned error code %d when called to mkfs on volume %s\n",
							    vol->file_system_manager->short_name, rc, vol->name);

					/*
					 * mkfs failed.  Reset the FSIM that manages this
					 * volume.
					 */
					vol->file_system_manager = vol->original_fsim;
					vol->private_data = vol->original_fsim_private_data;
				}
			}
			vol->flags &= ~VOLFLAG_MKFS;

			free_option_array(vol->mkfs_options);
			vol->mkfs_options = NULL;
		}
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * Find any volumes that have expanded in size and call the FSIM for
 * the volume to do the expand.
 */
static void expand_volumes() {

	int rc = 0;
	list_element_t iter;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(&volumes_list, iter, vol) {
		if (vol->vol_size > vol->original_vol_size) {

			if (vol->original_fsim != NULL) {
				u_int64_t new_size = vol->vol_size;

				status_message("Expanding volume %s...\n", vol->name);
				rc = vol->original_fsim->functions.fsim->expand(vol, &new_size);
				if (rc == 0) {
					/* Reget the limits for the file system. */
					get_volume_sizes_and_limits(vol);

				} else {
					set_commit_error(WARNING, rc);
					engine_user_message(NULL, NULL, "FSIM plug-in %s returned error code %d when called to expand volume %s to %"PRIu64" sectors.\n",vol->original_fsim->short_name, rc, vol->name, new_size);
				}
			}

			/* Set the volume's new original size. */
			vol->original_vol_size = vol->vol_size;

			/*
			 * The new volume size is also the new minimum size for any
			 * future shrinks.
			 */
			vol->shrink_vol_size = vol->vol_size;
		}
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * Find any volumes that are marked for fsck and run fsck on them.
 */
static void fsck_volumes() {

	int rc = 0;
	list_element_t iter;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(&volumes_list, iter, vol) {
		if (vol->flags & VOLFLAG_FSCK) {

			/* Make sure there is an FSIM to do the fsck. */
			if (vol->file_system_manager != NULL) {
				status_message("Running fsck on volume %s...\n", vol->name);
				rc = vol->file_system_manager->functions.fsim->fsck(vol, vol->fsck_options);

				if (rc == 0) {
					/* Reget the limits for the file system. */
					get_volume_sizes_and_limits(vol);

				} else {
					set_commit_error(WARNING, rc);
					engine_user_message(NULL, NULL, "FSIM plug-in %s returned error code %d when called to fsck volume %s.\n",vol->file_system_manager->short_name, rc, vol->name);
				}
			}
			vol->flags &= ~VOLFLAG_FSCK;

			free_option_array(vol->fsck_options);
			vol->fsck_options = NULL;
		}
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * Find any volumes that are marked for defrag and run defrag on them.
 */
static void defrag_volumes() {

	int rc = 0;
	list_element_t iter;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(&volumes_list, iter, vol) {
		if (vol->flags & VOLFLAG_DEFRAG) {

			/* Make sure there is an FSIM to do the defrag. */
			if (vol->file_system_manager != NULL) {
				status_message("Running defrag on volume %s...\n", vol->name);
				rc = vol->file_system_manager->functions.fsim->defrag(vol, vol->defrag_options);

				if (rc == 0) {
					/* Reget the limits for the file system. */
					get_volume_sizes_and_limits(vol);

				} else {
					set_commit_error(WARNING, rc);
					engine_user_message(NULL, NULL, "FSIM plug-in %s returned error code %d when called to defrag volume %s.\n",vol->file_system_manager->short_name, rc, vol->name);
				}
				vol->flags &= ~VOLFLAG_DEFRAG;

				free_option_array(vol->defrag_options);
				vol->defrag_options = NULL;
			}
		}
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * Check to make sure all volumes have a dev node.
 */
static void check_volume_dev_nodes(void) {

	int rc;
	list_element_t iter;
	logical_volume_t * vol;

	LIST_FOR_EACH(&volumes_list, iter, vol) {
		rc = hasa_dev_node(vol->name, vol->dev_major, vol->dev_minor);

		if (rc == 0) {
			vol->flags |= VOLFLAG_ACTIVE;
			memcpy(vol->dev_node, vol->name, sizeof(vol->dev_node));
		} else {
			vol->flags &= ~VOLFLAG_ACTIVE;
			memset(vol->dev_node, 0, sizeof(vol->dev_node));
		}
	}
}


/*
 * Find he new volumes and let the FSIMs see if they can find a file system
 * on them.
 */
static void find_fsim_for_new_volumes(void) {

	list_element_t iter;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	LIST_FOR_EACH(&volumes_list, iter, vol) {
		if (vol->flags & (VOLFLAG_NEW     |
				  VOLFLAG_ACTIVE  |
				  VOLFLAG_PROBE_FS)) {
			status_message("Checking for a file system on volume %s...\n", vol->name);
			find_fsim_for_volume(vol);
			vol->flags &= ~(VOLFLAG_NEW | VOLFLAG_PROBE_FS);
		}
	}

	LOG_PROC_EXIT_VOID();
}


#define NUM_ALLOC_CHANGES	4

static inline int add_change_record(char                    * name,
				    object_type_t             type,
				    u_int32_t                 changes,
				    change_record_array_t * * p_record_array) {

	int rc = 0;
        change_record_array_t * record_array = *p_record_array;

	if (record_array != NULL) {
		if (record_array->count % NUM_ALLOC_CHANGES == 0) {
			/* Need room for more entries */
			record_array = engine_realloc(record_array, sizeof(change_record_array_t) + (record_array->count + NUM_ALLOC_CHANGES) * sizeof(change_record_t));

		}

		if (record_array != NULL) {
			record_array->changes_pending[record_array->count].name = name;
			record_array->changes_pending[record_array->count].type = type;
			record_array->changes_pending[record_array->count].changes = changes;
			record_array->count++;

		} else {
			LOG_CRITICAL("Error allocating memory for a change_record_array_t\n");
			rc =  ENOMEM;
		}
	}


	*p_record_array = record_array;
	return rc;
}


/*
 * Check if a given volume has changes pending.
 */
boolean is_volume_change_pending(logical_volume_t        * vol,
				 changes_pending_parms_t * parms) {

	u_int32_t change_flags = 0;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Checking volume %s.\n", vol->name);

	if (vol->flags & (VOLFLAG_DIRTY | VOLFLAG_NEEDS_ACTIVATE | VOLFLAG_MKFS | VOLFLAG_UNMKFS | VOLFLAG_FSCK | VOLFLAG_DEFRAG)) {
		if (debug_level >= DEFAULT) {
			if (vol->flags & VOLFLAG_DIRTY) {
				change_flags |= CHANGE_DIRTY;
				LOG_DEFAULT("Change pending: Volume %s is dirty.\n", vol->name);
			}
			if (vol->flags & VOLFLAG_NEEDS_ACTIVATE) {
				if (vol->flags & VOLFLAG_ACTIVE) {
					change_flags |= CHANGE_REACTIVATE;
					LOG_DEFAULT("Change pending: Volume %s needs to be reactivated.\n", vol->name);
				} else {
					change_flags |= CHANGE_ACTIVATE;
					LOG_DEFAULT("Change pending: Volume %s needs to be activated.\n", vol->name);
				}
			}
			if (vol->flags & VOLFLAG_MKFS) {
				change_flags |= CHANGE_MKFS;
				LOG_DEFAULT("Change pending: Volume %s needs to have the %s file system put on it.\n", vol->name, vol->file_system_manager->short_name);
			}
			if (vol->flags & VOLFLAG_UNMKFS) {
				change_flags |= CHANGE_UNMKFS;
				LOG_DEFAULT("Change pending: Volume %s needs to have the %s file system removed.\n", vol->name, vol->original_fsim->short_name);
			}
			if (vol->flags & VOLFLAG_FSCK) {
				change_flags |= CHANGE_FSCK;
				LOG_DEFAULT("Change pending: Volume %s needs to have fsck run on it.\n", vol->name);
			}
			if (vol->flags & VOLFLAG_DEFRAG) {
				change_flags |= CHANGE_DEFRAG;
				LOG_DEFAULT("Change pending: Volume %s needs to be defragmented.\n", vol->name);
			}
		}

		parms->result = TRUE;
	}

	if (!(vol->flags & VOLFLAG_ACTIVE)) {
		change_flags |= CHANGE_ACTIVATE;
		LOG_DEFAULT("Change pending: Volume %s is not active.\n", vol->name);

		parms->result = TRUE;
	}

	if (vol->shrink_vol_size < vol->original_vol_size) {
		change_flags |= CHANGE_SHRINK;
		LOG_DEFAULT("Change pending: Volume %s needs to be shrunk.\n", vol->name);

		parms->result = TRUE;
	}

	if (vol->vol_size > vol->original_vol_size) {
		change_flags |= CHANGE_EXPAND;
		LOG_DEFAULT("Change pending: Volume %s needs to be expanded.\n", vol->name);

		parms->result = TRUE;
	}

	if ((change_flags != 0) && (parms != NULL)) {
		if (parms->changes != NULL) {
			add_change_record(vol->name, VOLUME, change_flags, &parms->changes);
		}
	}

	LOG_PROC_EXIT_BOOLEAN(parms->result);
	return parms->result;
}


/*
 * Check if a given container is dirty.
 * This is a service function of are_changes_pending() below.
 */
boolean is_container_change_pending(storage_container_t     * con,
				    changes_pending_parms_t * parms) {

	u_int32_t change_flags = 0;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Checking container %s.\n", con->name);

	if (con->flags & SCFLAG_DIRTY) {
		change_flags |= CHANGE_DIRTY;
		LOG_DEFAULT("Change pending: Container %s is dirty.\n", con->name);

		parms->result = TRUE;
	}

	if ((change_flags != 0) && (parms != NULL)) {
		if (parms->changes != NULL) {
			add_change_record(con->name, CONTAINER, change_flags, &parms->changes);
		}
	}

	LOG_PROC_EXIT_BOOLEAN(parms->result);
	return parms->result;
}


/*
 * Check if a given object is dirty or needs to be activated.
 * This is a service function of are_changes_pending() below.
 */
boolean is_object_change_pending(storage_object_t        * obj,
				 changes_pending_parms_t * parms) {

	u_int32_t change_flags = 0;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Checking object %s.\n", obj->name);

	if ((obj->flags & SOFLAG_DIRTY) ||
	    ((obj->data_type == DATA_TYPE) &&
	     (obj->flags & (SOFLAG_NEEDS_ACTIVATE | SOFLAG_NEEDS_DEACTIVATE)))) {
		if (debug_level >= DEFAULT) {
			if (obj->flags & SOFLAG_DIRTY) {
				change_flags |= CHANGE_DIRTY;
				LOG_DEFAULT("Change pending: Object %s is dirty.\n", obj->name);
			}
			if (obj->flags & SOFLAG_NEEDS_ACTIVATE) {
				if (obj->flags & SOFLAG_ACTIVE) {
					change_flags |= CHANGE_REACTIVATE;
					LOG_DEFAULT("Change pending: Object %s needs to be reactivated.\n", obj->name);
				} else {
					change_flags |= CHANGE_ACTIVATE;
					LOG_DEFAULT("Change pending: Object %s needs to be activated.\n", obj->name);
				}
			}
			if (obj->flags & SOFLAG_NEEDS_DEACTIVATE) {
				change_flags |= CHANGE_DEACTIVATE;
				LOG_DEFAULT("Change pending: Object %s needs to be deactivated.\n", obj->name);
			}
		}

		parms->result = TRUE;
	}

	if ((change_flags != 0) && (parms != NULL)) {
		if (parms->changes != NULL) {
			add_change_record(obj->name, obj->object_type, change_flags, &parms->changes);
		}
	}

	LOG_PROC_EXIT_BOOLEAN(parms->result);
	return parms->result;
}


/*
 * Check if anything in the Engine has changes pending.
 */
boolean are_changes_pending(change_record_array_t * * changes) {

	list_element_t iter;
	kill_sector_record_t * ksr;
	logical_volume_t     * vol;
	storage_container_t  * con;
	storage_object_t     * obj;
	changes_pending_parms_t parms = {result:  FALSE,
		                         changes: NULL};

	LOG_PROC_ENTRY();

        if (changes != NULL) {
		parms.changes = engine_alloc(sizeof(change_record_array_t));
	}
							
	LIST_FOR_EACH(&kill_sectors_list, iter, ksr) {
		LOG_DETAILS("Kill sectors are scheduled:\n");
		LOG_DETAILS("  Disk:          %s\n", ksr->logical_disk->name);
		LOG_DETAILS("  Sector offset: %"PRIu64"\n", ksr->sector_offset);
		LOG_DETAILS("  Sector count:  %"PRIu64"\n", ksr->sector_count);
		add_change_record(ksr->logical_disk->name, DISK, CHANGE_KILL_SECTORS, &parms.changes);
		parms.result = TRUE;
	}

	LIST_FOR_EACH(&volume_delete_list, iter, vol) {
		LOG_DETAILS("Volume %s is scheduled to be deleted.\n", vol->name);
		add_change_record(vol->name, VOLUME, CHANGE_DELETE, &parms.changes);
		parms.result = TRUE;
	}

	LIST_FOR_EACH(&deactivate_list, iter, obj) {
		LOG_DETAILS("Object %s is scheduled to be deactivated.\n", obj->name);
		add_change_record(obj->name, obj->object_type, CHANGE_DELETE, &parms.changes);
		parms.result = TRUE;
	}

	LIST_FOR_EACH(&disks_list, iter, obj) {
		is_object_change_pending(obj, &parms);
	}

	LIST_FOR_EACH(&segments_list, iter, obj) {
		is_object_change_pending(obj, &parms);
	}

	LIST_FOR_EACH(&containers_list, iter, con) {
		is_container_change_pending(con, &parms);
        }

	LIST_FOR_EACH(&regions_list, iter, obj) {
		is_object_change_pending(obj, &parms);
	}

	LIST_FOR_EACH(&EVMS_objects_list, iter, obj) {
		is_object_change_pending(obj, &parms);
	}

	LIST_FOR_EACH(&volumes_list, iter, vol) {
		is_volume_change_pending(vol,&parms);
	}

	if (changes != NULL) {
		*changes = parms.changes;
        }

	LOG_PROC_EXIT_BOOLEAN(parms.result);
	return parms.result;
}


static int activate_object(storage_object_t * obj) {

	int rc = 0;
	list_element_t iter;
	storage_object_t * child;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Request to activate object %s.\n", obj->name);

	if (obj->object_type != DISK) {
		int tmp_rc = 0;

		/*
		 * If the object has an associated object, make sure the
		 * associated object is active.
		 */
		if (obj->associated_object != NULL) {
			tmp_rc = activate_object(obj->associated_object);
		}

		if (tmp_rc != 0) {
			LOG_WARNING("Error code %d activating object %s: %s\n", tmp_rc, obj->associated_object->name, evms_strerror(tmp_rc));
			set_commit_error(WARNING, tmp_rc);
			if (rc == 0) {
				rc = tmp_rc;
			}
		}

		/*
		 * Make sure all of this object's children are active.
		 * If the object was produced from a container, then
		 * make sure all the objects consumed by the container
		 * are active.
		 */
		if (obj->producing_container == NULL) {
			LIST_FOR_EACH(obj->child_objects, iter, child) {
				tmp_rc = activate_object(child);
				if (tmp_rc != 0) {
					LOG_WARNING("Error code %d activating object %s: %s\n", tmp_rc, child->name, evms_strerror(tmp_rc));
					set_commit_error(WARNING, tmp_rc);
				}
				if (rc == 0) {
					rc = tmp_rc;
				}
			}
		} else {
			LIST_FOR_EACH(obj->producing_container->objects_consumed, iter, child) {
				tmp_rc = activate_object(child);
				if (tmp_rc != 0) {
					LOG_WARNING("Error code %d activating object %s: %s\n", tmp_rc, child->name, evms_strerror(tmp_rc));
					set_commit_error(WARNING, tmp_rc);
					if (rc == 0) {
						rc = tmp_rc;
					}
				}
			}
		}

		if (rc == 0) {
			if ((obj->data_type == DATA_TYPE) &&
			    (obj->flags & SOFLAG_NEEDS_ACTIVATE)) {

				/* Activate the object. */
				LOG_DEBUG("Activating %s.\n", obj->name);
				rc = obj->plugin->functions.plugin->activate(obj);
				LOG_DEBUG("activate(%s) returned %d: %s\n", obj->name, rc, evms_strerror(rc));

				if (rc == 0) {
					make_object_dev_node(obj);
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int activate_volume(logical_volume_t * vol) {

	int rc = 0;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Request to activate volume %s.\n", vol->name);

	if ((!(vol->flags & VOLFLAG_ACTIVE)) ||
	    (vol->flags & VOLFLAG_NEEDS_ACTIVATE)) {

		rc = activate_object(vol->object);

		if (rc == 0) {
			/*
			 * A featureless EVMS volume needs its own device-mapper
			 * mapping.
			 */
			storage_object_t * working_obj;

			working_obj = get_working_top_object(vol->object);

			if (working_obj != NULL) {
				LOG_DEBUG("working top object is %s\n", working_obj->name);
				if (!(vol->flags & VOLFLAG_COMPATIBILITY) &&
				    (working_obj->object_type != EVMS_OBJECT)) {
					rc = make_dm_map_for_volume(vol);
					vol->flags |= VOLFLAG_HAS_OWN_DEVICE;

				} else {
					vol->dev_major = vol->object->dev_major;
					vol->dev_minor = vol->object->dev_minor;
					vol->flags &= ~VOLFLAG_HAS_OWN_DEVICE;
				}

				if (rc == 0) {
					/*
					 * make_volume_dev_node() checks the volume's
					 * dev node and adjusts it if necessary.
					 */
					rc = make_volume_dev_node(vol);
				}

				if ((rc == 0) && (vol->flags & VOLFLAG_ACTIVE)) {
					/*
					 * Set VOLFLAG_PROBE_FS so that the later code in commit
					 * will scan the volume for an FSIM.
					 */
					vol->flags |= VOLFLAG_PROBE_FS;
					vol->flags &= ~(VOLFLAG_DIRTY | VOLFLAG_NEEDS_ACTIVATE);
				}
			}
		}
	}

	/*
	 * Always return success so that the remaining volumes in the list
	 * are processed.
	 */
	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int activate(void) {

	int rc = 0;
	list_element_t iter;
	logical_volume_t * vol;
	storage_object_t * obj;

	list_anchor_t all_objects;

	LOG_PROC_ENTRY();

	/* Activate volumes. */
	LIST_FOR_EACH(&volumes_list, iter, vol) {
		activate_volume(vol);
	}

	/*
	 * Activate any remaining objects that say they need
	 * activation.  There is no need to start with top level objects.
	 * The activation of each object makes sure that its child
	 * objects are active.
	 */
	rc = engine_get_object_list(0,			// all object types
				    DATA_TYPE,
				    NULL,		// any plug-in
				    NULL,		// any nodeid
				    0,			// no filter flags
				    &all_objects);

	if (rc == 0) {
		LIST_FOR_EACH(all_objects, iter, obj) {
			if (obj->flags & SOFLAG_NEEDS_ACTIVATE) {
				int tmp_rc = 0;

				tmp_rc = activate_object(obj);

				if (tmp_rc != 0) {
					LOG_WARNING("Error code %d activating object %s: %s\n", tmp_rc, obj->name, evms_strerror(tmp_rc));
					if (rc == 0) {
						rc = tmp_rc;
					}
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int mini_commit(storage_object_t * obj) {

	int rc = 0;
	plugin_functions_t * funcs = obj->plugin->functions.plugin;
	u_int32_t      depth;
	commit_phase_t phase;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Mini commit of object %s.\n", obj->name);

	if (!list_empty(obj->parent_objects) && (obj->feature_header != NULL)) {
		depth = obj->feature_header->object_depth + 1;
	} else {
		depth = 1;
	}

	if (obj->flags & SOFLAG_NEEDS_DEACTIVATE) {
		rc = obj->plugin->functions.plugin->deactivate(obj);

		if (rc != 0) {
			LOG_WARNING("Call to plug-in %s to deactivate object %s returned error code %d: %s\n",
				    obj->plugin->short_name, obj->name, rc, evms_strerror(rc));
			set_commit_error(WARNING, rc);
		}
	}

	kill_sectors();

	for (phase = SETUP; phase <= POST_ACTIVATE; phase++) {

		if (obj->object_type == EVMS_OBJECT) {
			commit_object(obj, depth, phase);

		} else {
			if (obj->flags & SOFLAG_DIRTY) {
				rc = funcs->commit_changes(obj, phase);
				if ((rc != 0) && (rc != E_CANCELED)) {
					LOG_WARNING("Call to plug-in %s to commit phase %d for object %s returned error code %d: %s\n",
						    obj->plugin->short_name, phase, obj->name, rc, evms_strerror(rc));
					set_commit_error(WARNING, rc);
				}
			}
		}

		if (obj->flags & SOFLAG_FEATURE_HEADER_DIRTY) {
			rc = write_feature_header(obj, depth, phase);
			if (rc != 0) {
				LOG_WARNING("Writing feature header on object %s returned error code %d: %s\n",
					    obj->name, rc, evms_strerror(rc));
				set_commit_error(WARNING, rc);
			}
		}

		if ((phase == SECOND_METADATA_WRITE) &&
		    (obj->flags & SOFLAG_NEEDS_ACTIVATE)) {
			rc = activate_object(obj);
			if (rc != 0) {
				set_commit_error(WARNING, rc);
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int cleanup_replace_object(storage_object_t * replace_object) {

	int rc = 0;
	list_element_t iter;
	list_anchor_t parents = NULL;
	STATIC_LIST_DECL(children);
	storage_object_t * parent;
	storage_object_t * source;
	storage_object_t * target;
	storage_object_t * new_child;
	storage_object_t * new_top_obj;

	LOG_PROC_ENTRY();

	/*
	 * Source is the first object in the replace_object's
	 * child list.
	 */
	source = first_thing(replace_object->child_objects, NULL);

	/*
	 * Target is the second (last) object in the replace_object's
	 * child list.
	 */
	target = last_thing(replace_object->child_objects, NULL);

	if (!list_empty(replace_object->parent_objects)) {
		
		/* Get a copy of the replace object's parent list so
		 * we can insert the replace object after it
		 * has been created.  When the replace object
		 * is created it changes the source's parent
		 * list to point to the new replace object.
		 */
			parents = copy_list(replace_object->parent_objects);
			if (parents == NULL) {
				LOG_SERIOUS("Error making a copy of the parent list.\n");
				LOG_PROC_EXIT_INT(ENOMEM);
				return ENOMEM;
			}
	}

	if (replace_object->flags & SOFLAG_DIRTY) {
		/*
		 * The copy failed.  Restore things to the original
		 * state, i.e., replace the replace_obj with the
		 * source.
		 */
		new_child = source;
		new_top_obj = target;

	} else {
		new_child = target;
		new_top_obj = source;
	}

	if (parents != NULL) {

		LIST_FOR_EACH(parents, iter, parent) {
			rc = parent->plugin->functions.plugin->replace_child(parent, replace_object, new_child);

			if (rc != 0) {
				LOG_SERIOUS("Plug-in %s failed to replace child object %s of object %s with the original child object %s.  Error code is %d: %s\n",
					    parent->plugin->short_name, replace_object->name, parent->name, source->name, rc, evms_strerror(rc));

				destroy_list(parents);
				LOG_PROC_EXIT_INT(rc);
				return rc;
			}
		}
	}

	/*
	 * If the replace worked the source and target are now members of
	 * new volumes.
	 */
	if (!(replace_object->flags & SOFLAG_DIRTY)) {

		/*
		 * The replace object has no parents.  We must have replaced
		 * the object for a volume.
		 */
		if (parents == NULL) {
			replace_object->volume->object = target;

			/*
			 * If the new volume object is not an EVMS
			 * object it may need a feature header to make
			 * it into an EVMS volume.
			 */
			if (target->object_type != EVMS_OBJECT) {
				if (target->feature_header == NULL) {
					add_volume_feature_header_to_object(target);
				}
				if (source->object_type == EVMS_OBJECT) {
					/*
					 * The source object is an EVMS object,
					 * the target is not.  The volume will
					 * need its own device-mapper device.
					 * Mark the volume not active so that
					 * dm_activate_volume() won't try to
					 * reactivate a volume that doesn't
					 * have a mapping.
					 */
					replace_object->volume->flags &= ~VOLFLAG_ACTIVE;
				}

			} else {
				if (source->object_type != EVMS_OBJECT) {
					/*
					 * The volume is a raw EVMS volume, but
					 * its new object is an EVMS object.
					 * Deactivate the mapping for the
					 * raw volume.
					 */
					dm_deactivate_volume(replace_object->volume);
				}
			}
		}

		set_volume_in_object(target, source->volume);
		mark_feature_headers_dirty(target);
		set_volume_in_object(source, NULL);
	}

	remove_thing(new_child->parent_objects, replace_object);
	mark_feature_headers_dirty(new_child);
	remove_thing(new_top_obj->parent_objects, replace_object);
	remove_feature_headers(new_top_obj);
	mark_feature_headers_dirty(new_top_obj);

	/*
	 * If the replace object had parents recommit the parents.
	 * Else, recommit the new child.
	 */
	if (parents != NULL) {
		LIST_FOR_EACH(parents, iter, parent) {
			mini_commit(parent);
		}
		destroy_list(parents);

	} else {
		mini_commit(new_child);

		new_child->volume->flags |= VOLFLAG_NEEDS_ACTIVATE;
		activate_volume(new_child->volume);
	}

	/*
	 * Apply the appropriate metadata to the top object so that it won't be
	 * discovered as a volume.
	 */
	if (new_top_obj->object_type == EVMS_OBJECT) {
		u_int32_t      depth;
		commit_phase_t phase;

		depth = 1;
		phase = FIRST_METADATA_WRITE;
		commit_object(new_top_obj, depth, phase);

		phase = SECOND_METADATA_WRITE;
		commit_object(new_top_obj, depth, phase);

	} else {
		commit_phase_t phase;

		phase = FIRST_METADATA_WRITE;
		write_stop_data_on_object(new_top_obj, phase);
		phase = SECOND_METADATA_WRITE;
		write_stop_data_on_object(new_top_obj, phase);
	}

	replace_plugin->functions.plugin->deactivate(replace_object);
	replace_plugin->functions.plugin->delete(replace_object, &children);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static void replace_cleanup(void) {

	int rc;
	int i;
	list_element_t iter1;
	list_element_t iter2;
	storage_object_t * obj;
	list_anchor_t list[] = {
		&disks_list,
		&segments_list,
		&regions_list,
		&EVMS_objects_list,
		NULL
	};

	LOG_PROC_ENTRY();

	for (i = 0; list[i] != NULL; i++) {

		LIST_FOR_EACH_SAFE(list[i], iter1, iter2, obj) {
			if (obj->plugin->id == REPLACE_PLUGIN_ID) {
				rc = cleanup_replace_object(obj);
			}
		}
	}

	LOG_PROC_EXIT_VOID();
}


static inline void do_phase(commit_phase_t phase) {

	commit_segments(phase);
	commit_containers(phase);
	commit_regions(phase);
	commit_stop_data(phase);
	commit_volumes(phase);
	commit_objects(phase);
}


/*
 * Commit to disk the configuration changes that are in memory.
 */
int evms_commit_changes(void) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_commit_changes();

		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	commit_in_progress = TRUE;
	need_reboot = FALSE;

	rc = rediscover();
	if (rc == 0) {
		rc = unmkfs_volumes();

		if (rc == 0) {
			rc = shrink_volumes();

			if (rc == 0) {
				rc = rename_volumes();

				if (rc == 0) {
					rc = delete_volumes();

					if (rc == 0) {
						deactivate_volumes();

						rc = deactivate_objects();

						if (rc == 0) {
							/*
							 * This is the point of no
							 * return.  All of the following
							 * code *must* be run.  Errors
							 * can not terminate the commit.
							 * Errors will be collected in
							 * the commit_error[] array.
							 */

							clear_commit_errors();
							cleanup_orphaned_stop_data();

							do_phase(SETUP);

							status_message("Processing the Kill Sectors List...\n");
							kill_sectors();

							do_phase(FIRST_METADATA_WRITE);
							do_phase(SECOND_METADATA_WRITE);

							status_message("Activating volumes...\n");
							activate();

							do_phase(POST_ACTIVATE);
							do_phase(MOVE);

							replace_cleanup();

							evms_rediscover();
							if (cluster_manager != NULL) {
								remote_rediscover();
							}

							/*
							 * Check to see if all volumes
							 * have a dev node.
							 */
							// BUGBUG  Is this needed?
							status_message("Checking to make sure all volumes have a dev node...\n");
							check_volume_dev_nodes();

							expand_volumes();

							mkfs_volumes();

							fsck_volumes();

							defrag_volumes();

							/*
							 * Check if any of the new volumes has a
							 * file system on it.
							 */
							find_fsim_for_new_volumes();

							status_message("Checking if anything in the system is still dirty...\n");

							rc = get_commit_error();
						}
					}
				}
			}
		}
	}

	if (need_reboot) {
		char * choices[] = {"Reboot now", "Reboot later", NULL};
		int answer = 0;	    /* Default is "Reboot now". */

		engine_user_message(&answer, choices,
				    "The system must be rebooted for all the changes to take effect.  "
				    "If you do not reboot now, some of the device nodes for the opened volumes will be in an "
				    "inconsistent state and may affect the operation of the system.  "
				    "Do you want to reboot now or reboot at a later time?\n");
		switch (answer) {
			case 0:
				{
					/* Reboot now */
					status_message("Rebooting the system...\n");
					fsync(log_file_fd);

					/* Switch to runlevel 6. */
					execl("/sbin/init", "init", "6", NULL);
					execl("/etc/init",  "init", "6", NULL);
					execl("/bin/init",  "init", "6", NULL);

					engine_user_message(NULL, NULL,
							    "Unable to switch to runlevel 6.  "
							    "You must reboot the system manually.\n");
				}
				break;

			case 1:
			default:
				break;
		}
	}

	status_message("Finished committing changes.\n");
	commit_in_progress = FALSE;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}
