#include <klocale.h>
#include <qvbox.h>
#include <qlabel.h>
#include <kpushbutton.h>
#include <kmessagebox.h>
#include <kaction.h>
#include <kactionclasses.h>
#include <kapplication.h>
#include <kdebug.h>
#include <qwidgetstack.h>
#include <qsplitter.h>
#include <kparts/part.h>
#include <ktrader.h>
#include <klibloader.h>
#include <kstatusbar.h>

#include <apt-pkg/packagemanager.h>
#include <apt-front/manager.h>
#include <apt-front/init.h>
#include <apt-front/cache/entity/package.h>
#include <apt-front/cache/component/state.h>
#include <apt-front/cache/component/history.h>
#include <apt-front/predicate/factory.h>

#include <libept/acqprogresswidget.h>

#include "app.h"

#include <libept/dpkgpm-gui.h> // EVIL

using namespace aptFront;
using namespace aptFront::cache;
using namespace ept;

void WaitForLister::waiting()
{
    kdDebug() << "WaitForLister::waiting()" << endl;
    if (app->m_list->lister()->busy())
        QTimer::singleShot( 100, this, SLOT( waiting() ) );
    else {
        QTimer::singleShot( 0, app, slot );
        deleteLater();
    }
}

TestApp::TestApp() {
    m_pkgSystem = new PkgSystem();
    aptFront::init();
    try {
        cache::Global::get().open( Cache::OpenDefault | Cache::OpenDebtags );
    } catch (...) {
        try {
            cache::Global::get().open( Cache::OpenDefault
                                       | Cache::OpenDebtags
                                       | Cache::OpenReadOnly );
            KMessageBox::information(
                this, "You need to run this application as root to make "
                "make any use of it.", "Root Required" );
            exit( 1 );
        } catch (...) {
            KMessageBox::sorry(
                this, "The APT Database could not be opened!"
                " This may be caused by incorrect APT configuration"
                " or something similar. Try running apt-setup and"
                " apt-get update in terminal and see if it helps"
                " to resolve the problem.", "Could not open cache" );
            exit( 1 );
        }
    }

    // xxx unhardcode the package name somehow?
    if (cache::Global::get().packages()
        .packageByName( "libqt-perl" ).isInstalled())
        putenv( "DEBIAN_FRONTEND=kde" );

    observeComponent< component::State >();

    cache::Global::get().addComponent(
        m_history = new component::History< component::State >() );

    m_all = new QVBox( this );
    m_stack = new QWidgetStack( m_all );
    m_next = new KPushButton( m_all );

    m_stack->addWidget( m_list = new ept::View( m_stack ) );
    m_stack->addWidget( m_bye = new QLabel( "Update Complete, Good Bye!", m_stack ) );
    m_stack->addWidget( m_start = new QLabel( "Welcome to Adept Updater!", m_stack ) );
    m_bye->setAlignment( Qt::AlignHCenter | Qt::AlignVCenter );
    m_start->setAlignment( Qt::AlignHCenter | Qt::AlignVCenter );

    // set up preview widget
    m_list->filterList()->setHiddenPredicate(
        predicate::adapt< entity::Entity >(
            (not predicate::Package::member( &entity::Package::markedKeep ))
            or predicate::Package::member( &entity::Package::isUpgradable )
            or predicate::Package::member( &entity::Package::isBroken )
            or predicate::Package::member( &entity::Package::willBreak )
            ) );

    QValueList< int > szl;
    szl.append( 0 );
    szl.append( 1 );
    m_list->setSizes( szl );

    m_stack->addWidget( m_progress = new ept::AcqProgressWidget( m_stack ) );

    setStandardToolBarMenuEnabled( false );
    createStandardStatusBarAction();
    setupActions();
    setupGUI( Keys|StatusBar|Save|Create );
    delete toolBar();

    statusBar()->message( "Initializing..." );
    statusBar()->insertItem( "", 0 );
    statusBar()->insertItem( "", 1 );

    QTimer::singleShot(
        0, this,
        SLOT( delayed() ) );

    m_stack->raiseWidget( m_start );
    setCentralWidget( m_all );
}

void TestApp::delayed() {
    loadKonsole();
    statusBar()->clear();
    notifyPostChange(0);
    start();
}

void TestApp::setupActions()
{
    KStdAction::quit( kapp, SLOT( quit() ), actionCollection() );
    m_undo = KStdAction::undo( this, SLOT( undo() ), actionCollection() );
    m_redo = KStdAction::redo( this, SLOT( redo() ), actionCollection() );
    setHistoryEnabled( false );
    createStandardStatusBarAction();
}

void TestApp::checkpoint() {
    setHistoryEnabled( false );
    m_history->checkpoint();
    setHistoryEnabled( true );
}

void TestApp::setHistoryEnabled( bool e ) {
    if (e) {
        m_undo->setEnabled( m_history->canUndo() );
        m_redo->setEnabled( m_history->canRedo() );
    } else {
        m_undo->setEnabled( false );
        m_redo->setEnabled( false );
    }
}

void TestApp::undo() {
    setHistoryEnabled( false );
    m_history->undo();
    setHistoryEnabled( true );
}

void TestApp::redo() {
    setHistoryEnabled( false );
    m_history->redo();
    setHistoryEnabled( true );
}

void TestApp::notifyPreChange( component::Base * )
{
    checkpoint();
}

void TestApp::notifyPostChange( component::Base * )
{
    cache::component::State &s = cache::Global::get().state();
    // cache::component::Packages &p = cache::Global::get().packages();
    kdDebug() << "TestApp::notifyPostChange()" << endl;
    statusBar()->changeItem(
        QString( " Changes: install %1, upgrade %2, remove %3 packages " )
        .arg( s.newInstallCount() ).arg( s.upgradeCount() )
        .arg( s.removeCount() ), 0 );
    statusBar()->changeItem(
        QString( " Currently %1 installed, %2 upgradable, %3 available packages " )
        .arg( s.installedCount() ).arg( s.upgradableCount() )
        .arg( s.availableCount() ), 1 );
    m_list->lister()->scheduleRebuild(); // rebuild on change...
}

void TestApp::setNext( QString s, const char *slot ) {
    m_next->setText( s );
    m_next->setEnabled( true );
    connect( m_next, SIGNAL( clicked() ),
             this, slot );
}

void TestApp::disableNext() {
    m_next->setEnabled( false );
    disconnect( m_next, SIGNAL( clicked() ), 0, 0 );
}

void TestApp::start() {
    disableNext();
    setNext( "Fetch Updates", SLOT( update() ) );
}

void TestApp::update() {
    kdDebug() << "TestApp::update" << endl;
    disableNext();

    aptFront::Manager m;
    m.setProgressCallback( m_progress->callback() );
    m.setUpdateInterval( 100000 );
    try {
        m_stack->raiseWidget( m_progress );
        m.update();
    } catch (...) {
        KMessageBox::sorry(
            this, "There was an error downloading updates. ",
            "Could not fetch updates" );
    }

    cache::Global::get().state().distUpgrade();
    if (cache::Global::get().state().changed()) {
        m_list->lister()->scheduleRebuild();
        m_stack->raiseWidget( m_list );
        setHistoryEnabled( true );
        setNext( "Changes OK, Commit", SLOT( commit() ) );
    } else {
        m_stack->raiseWidget( m_bye );
        setNext( "Nothing to do, Quit", SLOT( close() ) );
    }
}

void TestApp::commit() {
    kdDebug() << "TestApp::commit" << endl;
    disableNext();
    setHistoryEnabled( false );
    if (m_list->lister()->busy()) {
        new WaitForLister( this, SLOT( commit() ) );
        return;
    }

    aptFront::Manager m;
    m.setProgressCallback( m_progress->callback() );
    m.setUpdateInterval( 100000 );
    try {
        m_stack->raiseWidget( m_progress );
        m.download();
        m_stack->raiseWidget( m_konsole->widget() );
        m.commit();
    } catch (...) {
        KMessageBox::sorry(
            this, "There was an error commiting changes. "
            "Possibly there was a problem downloading some "
            "packages or the commit would break packages. ",
            "Could not commit changes" );
    }

    m_stack->raiseWidget( m_bye );
    setNext( "Done: Quit", SLOT( close() ) );
}

void TestApp::loadKonsole() {
    KLibFactory* factory = KLibLoader::self()->factory( "libsanekonsolepart" );
    if (!factory)
        factory = KLibLoader::self()->factory( "libkonsolepart" );
    assert( factory );
    m_konsole = static_cast<KParts::Part*>(
        factory->create( m_stack, "konsolepart", "QObject",
                         "KParts::ReadOnlyPart" ) );
    assert( terminal() );
    terminal()->setAutoDestroy( false );
    terminal()->setAutoStartShell( false );
    m_stack->addWidget( m_konsole->widget() );
    QStrList l; l.append( "echo" ); l.append( "-n" );
    terminal()->startProgram( "/bin/echo", l );
    m_pkgSystem->setTerminal( m_konsole );
}

ExtTerminalInterface *TestApp::terminal() {
    return static_cast<ExtTerminalInterface*>(
        m_konsole->qt_cast( "ExtTerminalInterface" ) );
}

#include "app.moc"
