# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.tests.elisa_test_case import ElisaTestCase
from elisa.core.manager import Manager, AlreadyRegistered, CannotUnregister
from elisa.core.component import Component


from elisa.core import common

from twisted.internet import defer 
from twisted.trial.unittest import SkipTest

class SimpleComponent(Component):
    running = False

    def clean(self):
        self.running = False

class TestManager(ElisaTestCase):

    def setUp(self):
        ElisaTestCase.setUp(self)

        # patch the application
        self.old_application = None
        self.patch_application()

        self._manager = Manager()
        self._provider = SimpleComponent()
        self._manager.register_component(self._provider)

    def tearDown(self):
        ElisaTestCase.tearDown(self)
        # unpatch the application
        self.unpatch_application()

    def test_component_start(self):
        """
        test if start gets passed through correctly
        """
        self._provider.running = False
        self._manager.start()
        self.assertEquals(self._provider.running, False)

    def test_component_stop(self):
        """
        test if stop gets passed through correctly
        """
        self._provider.running = True
        self._manager.stop()
        self.assertEquals(self._provider.running, False)


    def test_load_components_empty_list(self):
        """
        test the load_components method with an empty list
        """
        # even in the case we give an empty list we should get a deferred
        empty_dfr = self._manager.load_components([])
        self.assertTrue(isinstance(empty_dfr, defer.Deferred))

        def nothing_given(result):
            self.assertEquals(result, [])

        empty_dfr.addCallback(nothing_given)

        return empty_dfr

    def test_load_components(self):
        """
        test if the load_components method works as expected in the simple case
        """
        # given providers:
        dfr = self._manager.load_components(['nope', 'none'])

        # check for the order
        def order_check(result):
            self.assertEquals(self._manager._components[1:], ['nope', 'none'])

        dfr.addCallback(order_check)
        return dfr

    def test_load_componentss_with_a_wrong_provider(self):
        """
        test if the chain of loading providers goes on, even one fails
        """
        # for this we have to fake the log_failure system
        common.application.log_failure = lambda x:'y'
        dfr = self._manager.load_components(['one', 'two', 'fail', 'three'])

        def check_failed(result):
            self.assertEquals(self._manager._components[-3:],
                                                        ['one', 'two', 'three'])

        dfr.addCallback(check_failed)
        return dfr

    def patch_application(self):
        """
        Setup common.application, saving the old application object.

        Make common.application a generic object so we can set
        common.application.config and common.application.plugin_registry
        """
        assert self.old_application is None

        self.old_application = common.application

        class PluginRegistry(object):

            def create_component(self, name):
                if name == 'fail':
                    return defer.fail(Exception(name))
                dfr = defer.succeed(name)
                return dfr


        class Application(object):
            config = None
            plugin_registry = None

        common.application = Application()
        common.application.plugin_registry = PluginRegistry()

    def unpatch_application(self):
        """
        Restore the application object saved in patch_application().
        """
        common.application = self.old_application
        self.old_application = None

    def test_register_unregister(self):
        """
        just make sure that registering and unregistering works as expected
        """
        class SimpleComponent2(SimpleComponent):
            pass
        
        comp = SimpleComponent2()

        # register and check if it worked out
        self._manager.register_component(comp)
        self.assertIn(comp, self._manager._components)

        # check that re-registering fails
        self.assertRaises(AlreadyRegistered, 
                            self._manager.register_component,
                            comp)

        # unregister and check if it is not in there anymore
        self._manager.unregister_component(comp)
        self.assertNotIn(comp, self._manager._components)

    def test_unregister_of_noncomponent_fails(self):
        """
        try to unregister a not registered component and make sure that it
        fails
        """
        self.assertRaises(CannotUnregister,
                            self._manager.unregister_component,
                            SimpleComponent())

