# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
MediaFile abstraction support
"""

from elisa.core.utils import classinit

class MediaFile(object):
    """
    This class provides a file-like API to access data at a given URI.

    MediaFiles are created by
    L{elisa.core.components.media_provider.MediaProvider}
    objects. They act as proxies to the latter, thus are bound to
    MediaProviders.

    @ivar descriptor: media descriptor symbolizing the opened media
    @type descriptor: object
    """

    __metaclass__ = classinit.ClassInitMeta
    __classinit__ = classinit.build_properties

    def __init__(self, media_provider, descriptor):
        """

        @param media_provider: the Component which provides media access to the
                               descriptor, hence the media URI
        @type media_provider:  L{elisa.core.components.media_provider.MediaProvider}
        @param descriptor:     media descriptor symbolizing the opened media
        @type descriptor:      object
        """
        self._media_provider = media_provider
        self._descriptor = descriptor

    def close(self):
        """ Close myself via my media_provider
        """
        self._media_provider.close(self)

    def descriptor__get(self):
        """ Descriptor readonly access
        """
        return self._descriptor

    def read(self, size=-1):
        """ Read some data from my descriptor using my media_provider

        @param size:   length of data to retrieve. The media_provider will do
                       its best to fetch that amount of data. size = -1 means
                       retrieving all available data at once
        @type size:    int
        @returns:      string
        """
        return self._media_provider.read(self, size)

    def seek(self, offset, whence=0):
        """ Seek to given offset of my descriptor, via my media_provider

        @param offset:   how much to seek in the file
        @type offset:    int
        @param whence:   from where to seek in the file, default=0 means
                         current position
        @type whence:    int
        @returns:        bool
        """
        return self._media_provider.seek(self, offset, whence)

    def _blocking_read(self, size=-1):
        return self._media_provider._blocking_read(self, size=size)

    def _blocking_seek(self, offset, whence=0):
        return self._media_provider._blocking_seek(self, offset, whence)

    def _blocking_close(self):
        return self._media_provider._blocking_close(self)
