# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors: Eric Casteleijn <eric.casteleijn@canonical.com>

"""Tests for the RecordDict object on which the Contacts API is built."""

from testtools import TestCase
import doctest

# pylint does not like relative imports from containing packages
# pylint: disable-msg=F0401
from desktopcouch.records.record import (Record, RecordDict, MergeableList,
    record_factory, IllegalKeyException, validate)


class TestRecords(TestCase):
    """Test the record functionality"""

    def setUp(self):
        """Test setup."""
        self.dict = {
            "a": "A",
            "b": "B",
            "subfield": {
                    "field11s": "value11s",
                    "field12s": "value12s"},
            "subfield_uuid": {
                    "e47455fb-da05-481e-a2c7-88f14d5cc163": {
                        "field11": "value11",
                        "field12": "value12",},
                    "d6d2c23b-279c-45c8-afb2-ec84ee7c81c3": {
                        "field21": "value21",
                        "field22": "value22"}},
            "application_annotations": {
                "my_awesome_app": {"foo": "bar", "some_setting": "set_to"}},
            "record_type": "http://fnord.org/smorgasbord",
        }
        self.record = Record(self.dict)

    def test_get_item(self):
        "Does a RecordDict basically wrap a dict properly?"
        self.assertEqual(self.dict["a"], self.record["a"])
        self.assertRaises(KeyError,
                          self.record.__getitem__, "application_annotations")

    def test_get(self):
        "Does a RecordDict get() work?"
        self.assertEqual(self.dict["a"], self.record.get("a"))
        self.assertEqual(None, self.record.get("application_annotations"))

    def test_keys(self):
        """Test getting the keys."""
        self.assertEqual(
            ['a', 'b', 'record_type', 'subfield', 'subfield_uuid'],
            sorted(self.record.keys()))

    def test_application_annotations(self):
        """Test getting application specific data."""
        my_app_data = self.record.application_annotations["my_awesome_app"]
        self.assertEqual(
            'bar',
            my_app_data['foo'])
        my_app_data["foo"] = "baz"
        self.assertEqual(
            {'my_awesome_app': {'foo': 'baz', "some_setting": "set_to"}},
            self.record._data["application_annotations"])
        self.assertEqual(
            'baz',
            my_app_data['foo'])

    def test_loads_dict_subdict(self):
        "Are subdicts supported?"
        self.assertEqual(2, len(self.record["subfield"]))
        subfield_single = self.record["subfield"]
        self.assertEqual(
            subfield_single["field11s"],
            self.dict["subfield"]["field11s"])

    def test_loads_dict_multi_subdict(self):
        "Are subdicts with multiple entries supported?"
        self.assertEqual(2, len(self.record["subfield_uuid"]))

    def test_mergeable_list_index(self):
        """Test the subset of list behavior we'll be supporting"""
        self.assertEqual("value21", self.record["subfield_uuid"][0]["field21"])
        self.assertEqual("value22", self.record["subfield_uuid"][0]["field22"])
        self.assertEqual("value11", self.record["subfield_uuid"][-1]["field11"])
        self.assertEqual("value12", self.record["subfield_uuid"][-1]["field12"])

    def test_mergeable_list_del(self):
        """Test deletion of uuid keys."""
        del self.record["subfield_uuid"][0]
        self.assertEqual(1, len(self.record["subfield_uuid"]))
        self.assertEqual("value11", self.record["subfield_uuid"][0]["field11"])
        self.assertEqual("value12", self.record["subfield_uuid"][0]["field12"])

    def test_mergeable_list_set_value_in_list_item(self):
        """Test assignment by index."""
        self.record["subfield_uuid"][0]["field12"] = "new exciting value"
        self.assertEqual(
            "new exciting value", self.record["subfield_uuid"][0]["field12"])

    def test_mergeable_list_append(self):
        """Test append."""
        self.record["subfield_uuid"].append(
            {"field31": "value31", "field32": "value32"})
        self.assertEqual(3, len(self.record["subfield_uuid"]))
        self.assertEqual("value32", self.record["subfield_uuid"][-1]["field32"])

    def test_mergeable_list_append_record_dict(self):
        """Test appending a RecordDict value."""
        value = RecordDict({
            "field31": "value31",
            "field32": "value32"})
        self.record["subfield_uuid"].append(value)
        self.assertEqual(3, len(self.record["subfield_uuid"]))
        self.assertEqual("value32", self.record["subfield_uuid"][-1]["field32"])

    def test_tuple(self):
        """Test assigning tuples to a key results in mergeable lists."""
        rec = Record({'record_type': 'http://fnord.org/smorgasbord'})
        rec['key'] = (1, 2, 3, 4)
        self.assert_(isinstance(rec['key'], MergeableList))
        self.assertEqual([1, 2, 3, 4], [value for value in rec['key']])

    def test_list(self):
        """Test assigning lists to a key results in mergeable lists."""
        rec = Record({'record_type': 'http://fnord.org/smorgasbord'})
        rec['key'] = [1, 2, 3, 4]
        self.assert_(isinstance(rec['key'], MergeableList))
        self.assertEqual([1, 2, 3, 4], [value for value in rec['key']])

    def test_dictionary_access_to_mergeable_list(self):
        """Test that appropriate errors are raised."""
        keys = self.record["subfield_uuid"]._data.keys()
        self.assertRaises(
            TypeError,
            self.record["subfield_uuid"].__getitem__, keys[0])
        self.assertRaises(
            TypeError,
            self.record["subfield_uuid"].__setitem__, keys[0], 'stuff')

    def test_validate(self):
        """Test that validation of mixed keys raises appropriate
        error."""
        self.assertRaises(
            IllegalKeyException,
            validate,
            {'foo': 1, 'bar': 2, 'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3': 3})
        self.assertRaises(
            IllegalKeyException,
            validate,
            {'baz':
             {'foo': 1, 'bar': 2, 'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3': 3},
             'qux': 5})

    def test_uuid_like_keys(self):
        """Test that appropriate errors are raised."""
        keys = self.record["subfield_uuid"]._data.keys()
        self.assertRaises(
            IllegalKeyException,
            self.record["subfield"].__setitem__, keys[0], 'stuff')

    def test_record_type(self):
        """Test the record_type property."""
        self.assertEqual('http://fnord.org/smorgasbord',
          self.record.record_type)

    def test_run_doctests(self):
        results = doctest.testfile('../doc/records.txt')
        self.assertEqual(0, results.failed)


class TestRecordFactory(TestCase):
    """Test Record/Mergeable List factories."""

    def setUp(self):
        """Test setup."""
        self.dict = {
            "a": "A",
            "b": "B",
            "subfield": {
                "field11s": "value11s",
                "field12s": "value12s",},
            "subfield_uuid":
                [
                {"field11": "value11",
                 "field12": "value12"},
                {"field21": "value21",
                 "field22": "value22"}],
            "record_type": "http://fnord.org/smorgasbord",
        }

    def test_build(self):
        """Test RecordDict/MergeableList factory method."""
        record = record_factory(self.dict)
        self.assertEquals('A', record._data['a'])
        self.assertEquals(
            'value12s', record._data['subfield']['field12s'])
