#!/usr/bin/python
# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Author: Stuart Langridge <stuart.langridge@canonical.com>
#         Eric Casteleijn <eric.casteleijn@canonical.com>

"""Specify location of files relevant to desktop CouchDB.

Checks to see whether we're running out of the source tree or not.
"""
from __future__ import with_statement
import os
import xdg.BaseDirectory
import subprocess
import logging
try:
    import ConfigParser as configparser
except ImportError:
    import configparser

def mkpath(rootdir, path):
    "Remove .. from paths"
    return os.path.realpath(os.path.join(rootdir, path))

rootdir = os.path.join(xdg.BaseDirectory.xdg_cache_home, "desktop-couch")
if not os.path.isdir(rootdir):
    os.mkdir(rootdir)

config_dir = xdg.BaseDirectory.save_config_path("desktop-couch")

FILE_INI = os.path.join(config_dir, "desktop-couchdb.ini")
DIR_DB = xdg.BaseDirectory.save_data_path("desktop-couch")

FILE_PID = mkpath(rootdir, "desktop-couchdb.pid")
FILE_LOG = mkpath(rootdir, "desktop-couchdb.log")
FILE_STDOUT = mkpath(rootdir, "desktop-couchdb.stdout")
FILE_STDERR = mkpath(rootdir, "desktop-couchdb.stderr")

COUCH_EXE = os.environ.get('COUCHDB')
if not COUCH_EXE:
    for x in os.environ['PATH'].split(':'):
        if os.path.exists(os.path.join(x, 'couchdb')):
            COUCH_EXE = os.path.join(x, 'couchdb')
if not COUCH_EXE:
    raise ImportError("Could not find couchdb")

def couch_chain_ini_files():
    process = subprocess.Popen([COUCH_EXE, '-V'], shell=False,
        stdout=subprocess.PIPE)
    line = process.stdout.read().split('\n')[0]
    couchversion = line.split()[-1]

    # Explicitly add default ini file
    ini_files = ["/etc/couchdb/default.ini"]

    # find all ini files in the desktopcouch XDG_CONFIG_DIRS and add them to
    # the chain
    xdg_config_dirs = xdg.BaseDirectory.load_config_paths("desktop-couch")
    # Reverse the list because it's in most-important-first order
    for folder in reversed(list(xdg_config_dirs)):
        ini_files.extend([os.path.join(folder, x)
                      for x in sorted(os.listdir(folder))
                      if x.endswith(".ini")])

    if FILE_INI not in ini_files:
        ini_files.append(FILE_INI)

    chain = "-n -a %s " % " -a ".join(ini_files)

    return chain

class NoOAuthTokenException(Exception):
    def __init__(self, file_name):
        super(Exception, self).__init__()
        self.file_name = file_name
    def __str__(self):
        return "OAuth details were not found in the ini file (%s)" % (
            self.file_name)

def get_oauth_tokens(config_file_name=FILE_INI):
    """Return the OAuth tokens from the desktop Couch ini file.
       CouchDB OAuth is two-legged OAuth (not three-legged like most OAuth).
       We have one "consumer", defined by a consumer_key and a secret,
       and an "access token", defined by a token and a secret.
       The OAuth signature is created with reference to these and the requested
       URL.
       (More traditional 3-legged OAuth starts with a "request token" which is
       then used to procure an "access token". We do not require this.)
    """
    c = configparser.ConfigParser()
    # monkeypatch ConfigParser to stop it lower-casing option names
    c.optionxform = lambda s: s
    c.read(config_file_name)
    try:
        oauth_token_secrets = c.items("oauth_token_secrets")
        oauth_consumer_secrets = c.items("oauth_consumer_secrets")
    except configparser.NoSectionError:
        raise NoOAuthTokenException(config_file_name)
    if not oauth_token_secrets or not oauth_consumer_secrets:
        raise NoOAuthTokenException(config_file_name)
    try:
        out = {
            "token": oauth_token_secrets[0][0],
            "token_secret": oauth_token_secrets[0][1],
            "consumer_key": oauth_consumer_secrets[0][0],
            "consumer_secret": oauth_consumer_secrets[0][1]
        }
    except IndexError:
        raise NoOAuthTokenException(config_file_name)
    return out


def get_bind_address(config_file_name=FILE_INI):
    """Retreive a string if it exists, or None if it doesn't."""
    c = configparser.ConfigParser()
    try:
        c.read(config_file_name)
        return c.get("httpd", "bind_address")
    except (configparser.NoOptionError, OSError), e:
        logging.warn("config file %r error. %s", config_file_name, e)
        return None

def set_bind_address(address, config_file_name=FILE_INI):
    c = configparser.SafeConfigParser()
    # monkeypatch ConfigParser to stop it lower-casing option names
    c.optionxform = lambda s: s
    c.read(config_file_name)
    if not c.has_section("httpd"):
        c.add_section("httpd")
    c.set("httpd", "bind_address", address)
    with open(config_file_name, 'wb') as configfile:
        c.write(configfile)


# You will need to add -b or -k on the end of this
COUCH_EXEC_COMMAND = [COUCH_EXE, couch_chain_ini_files(), '-p', FILE_PID,
  '-o', FILE_STDOUT, '-e', FILE_STDERR]


# Set appropriate permissions on relevant files and folders
for fn in [FILE_PID, FILE_STDOUT, FILE_STDERR, FILE_INI, FILE_LOG]:
    if os.path.exists(fn):
        os.chmod(fn, 0600)
for dn in [rootdir, config_dir, DIR_DB]:
    if os.path.isdir(dn):
        os.chmod(dn, 0700)

