/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef DECIBEL_DAEMON_CLIENT_CHANNELHANDLER_H
#define DECIBEL_DAEMON_CLIENT_CHANNELHANDLER_H

#include <QtCore/QObject>
#include <QtCore/QString>

#include <Decibel/Types>

class QDBusObjectPath;

namespace QtTapioca
{
class Connection;
class Channel;
}

namespace Decibel
{

class ChannelHandlerPrivate;

/**
 * @brief A ChannelHandler is called by Decibel whenever it
 * activates a external service to handle a incoming information
 * channel.
 *
 * The ChannelHandler implements an D-Bus interface that is used
 * by the Decibel daemon to activate external clients. It hands over
 * all required information to a client so that it can find the new
 * channel it is requested to handle. A client may decline the request
 * by the decibel server.
 *
 * Override the handleChannel(...) method in your client to customize
 * the behaviour.
 *
 * @author Tobias Hunger <info@basyskom.de>
 */
class Q_DECL_EXPORT ChannelHandler : public QObject
{
    Q_OBJECT
    Q_DECLARE_PRIVATE(ChannelHandler)

public:
    /** @brief Constructor. */
    explicit ChannelHandler(QObject * parent = 0);
    /** @brief Destructor. */
    virtual ~ChannelHandler();

    /**
     * @brief Decide whether this ChannelHandler is interested in
     * the given channel.
     * @param connection The Connection the Channel is part of
     * @param channel A pointer to the Channel proxy object.
     * @param incoming true if this channel is incoming.
     * @return true if this ChannelHandler wants to handle the new
     * Channel and false otherwise.
     *
     * This method provides a way for a ChannelHandler to decline
     * responsibility for a channel. If it does accept the channel
     * then this method should be overridden to initialize the handling
     * of the channel by the client application.
     */
    virtual bool handleChannel(QtTapioca::Connection * connection,
                               QtTapioca::Channel * channel,
                               const bool incoming) = 0;

public Q_SLOTS:
    /**
     * @brief Notification slot for the D-Bus signal.
     * @param channel_info The D-Bus information required to set up
     * a Channel.
     * @return true if this ChannelHandler wants to handle the new
     * Channel and false otherwise.
     *
     * This method is called by D-Bus and sets up QtTapioca Connection
     * and Channel objects before calling the virtual HandleChannel
     * method and returning its result.
     *
     * This slot calls the virtual handleChannel(...) method which does
     * the heavy lifting.
     */
    bool handleChannel(const Decibel::ChannelInfo & channel_info);

private:
    ChannelHandlerPrivate * const d;
};

} // namespace

#endif
