#!/bin/sh

# This script will strip VC dirs from the sources and
# then package up a release tarball of decibel
#
# Usage: release.sh /path/to/top/of/decibel/src/dir [TAG]
#
# Build a tarball of the release.
# When the "TAG" string is passed into this script it
# will tag the current release in SVN in addition to it
# and then update the ChangeLog file for the new version.

SRCDIR="$1"
CURRENTDIR=`pwd`

# test whether this script exists in the given place:
test -x "$SRCDIR/cmake/release.sh" || exit 1

cd "$SRCDIR"
SRCDIR=`pwd`

# Read repository data:
#   TAGDIR: Directory that will contain the tags.
. "$HOME/.decibel-release.conf"
echo "Tags are stored in $TAGDIR..."

TMPDIR=`mktemp -d`
echo Using $TMPDIR...

cleanup()
{
    MESSAGE=$1
    EXITCODE=$2

    echo "$MESSAGE"
    rm -rf "$TMPDIR"

    cd "$CURRENTDIR"
    exit $EXITCODE
}


MAJOR_VERSION=`cat "$SRCDIR/CMakeLists.txt" | \
    egrep "SET\s*\(\s*MAJOR_VERSION" | \
    sed -e "s/^.*MAJOR_VERSION\s*\"\(.*\)\".*$/\1/"`
MINOR_VERSION=`cat "$SRCDIR/CMakeLists.txt" | \
    egrep "SET\s*\(\s*MINOR_VERSION" | \
    sed -e "s/^.*MINOR_VERSION\s*\"\(.*\)\".*$/\1/"`
PATCH_LEVEL=`cat "$SRCDIR/CMakeLists.txt" | \
    egrep "SET\s*\(\s*PATCH_LEVEL" | \
    sed -e "s/^.*PATCH_LEVEL\s*\"\(.*\)\".*$/\1/"`
VERSION="$MAJOR_VERSION.$MINOR_VERSION.$PATCH_LEVEL"

echo "Version: $VERSION."

# check whether that version is already tagged:
svn ls "$TAGDIR/$VERSION" > /dev/null 2>&1 && exit 2

NAME="decibel-$VERSION"
WORKDIR="$TMPDIR/$NAME"

mkdir "$WORKDIR"

# copy sources:
( cd "$SRCDIR" ; tar -cf - . ) | ( cd "$WORKDIR" ; tar -xf -)

# clean up:

# ... stuff not in svn:
SVNCLEAN=`which svn-clean`
test "x$SVNCLEAN"="x" && SVNCLEAN=`which svn-clean-kde`

cd "$WORKDIR"
if test "x$SVNCLEAN" != "x" ; then
  if test `$SVNCLEAN -f | wc -l` -ne 0 ; then \
    cleanup "Additional files found... please remove or commit them." 3
  fi
else
  cleanup "Please install svn-clean utility from kdesupport." 4
fi

# ... uncommited changes:
if test `svn -R revert . | wc -l` -ne 0 ; then
  cleanup "Uncommited changes found... please commit them and retry." 5
fi

if test `svn update | wc -l` -ne 1 ; then
    cleanup "Updates found on server, please merge and retry." 2
fi

# Update ChangeLog and tag version:
sed -i -e "s/^Development version:$/Version $VERSION:/" ChangeLog

if test "x$2" = "xTAG" ; then
  svn commit -m "Update ChangeLog for new release."
  svn copy -m "Tag release $VERSION" . "$TAGDIR/$VERSION"

#########
# Change work environment for new version:
#########

  # Append Development version to changelog:
  cd "$SRCDIR"
  svn update
  if grep "Development version:" ChangeLog ; then
    echo "No need to add development version to ChangeLog."
  else
    mv ChangeLog ChangeLog.new
    echo "Development version:" > ChangeLog
    echo >> ChangeLog
    cat ChangeLog.new >> ChangeLog
    rm ChangeLog.new
    echo "Please increment version number in CMakeLists.txt! Current version is $VERSION"
    echo "COMMIT AFTERWARDS."
  fi
fi

# ... svn dirs:
( cd "$WORKDIR" ; \
  find . -name \.svn -type d -exec rm -rf \{} \; > /dev/null 2>&1 )

( cd "$TMPDIR" ; \
  rm -f "/tmp/$NAME.tar.gz" ; \
  tar -czf "/tmp/$NAME.tar.gz" "$NAME" )

cleanup "Releasefile is /tmp/$NAME.tar.gz..." 0

