# Written by Henrik Nilsen Omma
# (C) Canonical, Ltd. Licensed under the GPL

import urllib, urllib2, cookielib
import re
import os
import string

import utils, MultipartPostHandler

baseUrl = "https://launchpad.net/bugs/"

def sanitize_html(text):
    mapping = [['&nbsp;', ' '],
               ['<wbr></wbr>', ''],
               ['&lt;', '<'],
               ['&gt;', '>'],
               ['&amp;', '&']]
    for m in mapping:
        text = text.replace(m[0], m[1])
    return text

class BugHelperURLopener(urllib.FancyURLopener):
    version = "bughelper/%s (%s)" % (utils.find_version_number(), \
                                     urllib.FancyURLopener.version)

urllib._urlopener = BugHelperURLopener()

class BugInfo(object):
    def __init__(self, nr, status, importance, summary):
        self.nr = nr
        self.status = status
        self.importance = importance
        self.summary = summary
    def __int__(self):
        return int(self.nr)
    def __repr__(self):
        return str(self.nr)

class BugPage(object):
    def __init__(self, URL):
        # grab page content
        self.url = URL
        self.following_page = None

        self.bugs = []

        sock = urllib.urlopen(self.url)
        self.pageBody = sock.read()
        sock.close() 

        bugTagMask = '/([0-9]+)">(.*)</a>[^<]*<[^"]*(Critical|High|Medium|Low|Und\
+ecided|Wishlist)[^"]*(Unconfirmed|Needs Info|Rejected|Confirmed|In Progress|Fix Committed|Fix Released)'
        bugfilter = re.compile(bugTagMask, re.MULTILINE)
        filteredInfos = bugfilter.findall(self.pageBody)
        for bug_nr, bug_sum, bug_imp, bug_stat in filteredInfos:
            self.bugs.append(BugInfo(bug_nr, bug_stat, bug_imp,sanitize_html(bug_sum)))
        
        next_link = 'href="(.*)">Next</a>'
        matches = re.findall(next_link, self.pageBody)
        if matches:
            self.following_page = sanitize_html(matches[0])

    def countComments(self): 
        commentMask = '<div class="boardComment">'
        commentfilter = re.compile(commentMask)
        commentList = commentfilter.findall(self.pageBody)
        self.comments = len(commentList)

class BugList(object):
    def __init__(self, opt):
        self.bugs = set()
        bugexclude = set()
        opt_bug = set()

        bp = BugPage(opt.url)
        self.bugs.update(bp.bugs)

        while(bp.following_page!=None):
            bp = BugPage(bp.following_page)
            self.bugs.update(bp.bugs)

        if opt.upstream and opt.sourcepackage:
            bp = BugPage("http://launchpad.net/%s/+bugs" % opt.sourcepackage)
            self.bugs.update(bp.bugs)
            while(bp.following_page!=None):
                bp = BugPage(bp.following_page)
                self.bugs.update(bp.bugs)

        if opt.minbug:
            for bug in self.bugs:
                if int(bug.nr) < int(opt.minbug):
                    bugexclude.add(bug)

        if opt.filterbug:
            for bug in opt.filterbug.split(','):
                opt_bug.add(bug)
            for bug in self.bugs:
                if bug.nr in opt_bug: 
                    bugexclude.add(bug)
                    
    # run the global importance/status filter
        for attr in ["status","importance"]:
            l = getattr(opt,attr).split(",")
            if l[0] != '':
                for bug in self.bugs:
                    if getattr(bug,attr) not in l:
                        bugexclude.add(bug)
            
        self.bugs = self.bugs.difference(bugexclude)
        
class BugAttachment(object):
    def __init__(self, URL, attachments_path, content_types, 
                 attachments_regex=None):
        self.url = URL
        self.contenttype = ""
        self.text = ""
        self.filename = ""
        if "text/plain" not in content_types:
            content_types.append("text/plain")
        self.filename = \
            os.path.expanduser(os.path.join(attachments_path, 
                         string.join(self.url.split("/")[-2:], "/")))
        if os.path.exists(self.filename):
            attachment_file = open(self.filename, "r")
            self.text = attachment_file.read()
            attachment_file.close()
        else:
            if (not attachments_regex) or \
               (attachments_regex and re.search(attachments_regex, 
                                          self.url.split("/")[-1])):
                    sock = urllib.urlopen(self.url)
                    self.contenttype = sock.info()["Content-type"]
                    for ct in content_types:
                        if self.contenttype.startswith(ct):
                            self.text = sock.read()
                            utils.lazy_makedir(os.path.dirname(self.filename))
                            attachment_file = open(self.filename, "w")
                            attachment_file.write(self.text)
                            attachment_file.close()
                    sock.close()
            else:
                self.filename = ""


class Bug(object):
    def __init__(self, bugnumber, sourcepackage=None, attachments_path=None, \
                 content_types=[], attachments_regex=None, cookie_file=None):
        self.bugnumber = bugnumber
        self.url = baseUrl + str(bugnumber)
        self.sourcepackage = sourcepackage
        attachmentslist = set()
        self.attachments = set()
        self.info = ''
        # the following values are set in get_metadata()
        self.title = None
        self.description = None
        self.tags = None
        self.nickname = None
                
        if cookie_file:
            cj = cookielib.MozillaCookieJar()
            cj.load(cookie_file)
            self.cookie_handler = urllib2.HTTPCookieProcessor(cj)
            opener = urllib2.build_opener(self.cookie_handler)
            sock = opener.open(self.url)
        else:
            self.cookie_handler = None
            sock = urllib.urlopen(self.url)
        self.text = sock.read()
        # Get the rewritten URL so that we have a valid one to attach comments
        # to
        self.url = sock.geturl()
        sock.close() 
        
        if not self.sourcepackage:
            titleFilter = '<title>Bug #[0-9]* in (.*?) .*: .*'
            filteredSourcePackage = re.findall(titleFilter, self.text)
            if filteredSourcePackage:
                self.sourcepackage = filteredSourcePackage[0]
                if self.sourcepackage == 'Ubuntu:':
                    self.sourcepackage = None

        infoFilter = '>([^<]*)\((upstream|Ubuntu)\)[^"]*"status[^>]*>([^<]*)[^"]*"impo\
rtance[^>]*>([^<]*)'
        filteredInfos = re.findall(infoFilter, re.sub("\n", "", self.text))
        for i in filteredInfos:
            self.info += "".join("[%s%s: %s/%s]"%(i[0], i[1], i[2], i[3]));
        
        if attachments_path:
            download_link = '<li class="download">.*\n.*<a href="(.*)">'
            matches = re.findall(download_link, self.text)
            if matches:
                attachmentslist.update(matches)
            for a in attachmentslist:
                self.attachments.add(BugAttachment(a, attachments_path, \
                                                   content_types, \
                                                   attachments_regex))
        self.text = sanitize_html(self.text)

    def add_comment(self, subject, text, attachment=None, description=None, is_patch=False):
        '''Add a new comment (with given subject) to this bug.

        If attachment is not None, it must be a file-like object that gets
        attached to the bug with the given description. Set is_patch to True
        for patches to get the correct MIME type handling.

        Return the resulting urllib2 file object.'''

        assert (subject is not None) and (text is not None)
        assert self.cookie_handler, 'you must supply a cookie file with a Launchpad login cookie in the constructor'
        opener = urllib2.build_opener(self.cookie_handler, MultipartPostHandler.MultipartPostHandler)
        args = { 'FORM_SUBMIT': '1', 'field.comment': text, 
            'field.subject': subject }
        if attachment:
            assert description
            args['field.filecontent'] = attachment
            args['field.attachment_description'] = description
            args['field.patch'] = is_patch and 'on' or 'off'
            args['field.include_attachment'] = 'on'
        else:
            args['field.include_attachment'] = 'off'

        result = opener.open(self.url + '/+addcomment', args)
        return result

    def get_metadata(self):
        '''Load title, description, tags, and nickname.'''

        assert self.cookie_handler, 'you must supply a cookie file with a Launchpad login cookie in the constructor'

        opener = urllib2.build_opener(self.cookie_handler)
        sock = opener.open(self.url + '/+edit')
        text = sock.read()
        sock.close()

        # parse out title
        m = re.search('<input .*id="field.title".*value=["\'](.*)["\']\ */>', text)
        assert m
        self.title = sanitize_html(m.group(1))

        # parse out description
        m = re.search('<textarea .*id="field.description"[^>]*>((.|\n)*)</textarea>', text)
        assert m
        self.description = sanitize_html(m.group(1))

        # parse out tags
        m = re.search('<input .*id="field.tags".*value="([^"]*)"', text)
        assert m
        self.tags = sanitize_html(m.group(1)).split()

        # parse out nickname
        m = re.search('<input .*id="field.name".*value="([^"]*)"', text)
        assert m
        self.nickname = sanitize_html(m.group(1))

    def set_metadata(self):
        '''Write back title, description, tags, and nickname attributes.'''

        assert self.title is not None
        assert self.cookie_handler, 'you must supply a cookie file with a Launchpad login cookie in the constructor'
        opener = urllib2.build_opener(self.cookie_handler, MultipartPostHandler.MultipartPostHandler)
        args = { 'field.actions.change': '1', 
            'field.title': self.title, 
            'field.description': self.description, 
            'field.tags': ' '.join(self.tags),
            'field.name': self.nickname
             }

        result = opener.open(self.url + '/+edit', args)
        return result
