/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2008  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dbus/dbus-glib.h>

#include <glib/gi18n.h>

#include <gtk/gtk.h>

#include "client.h"

#include "dbus-glue.h"

#include "bluetooth-instance.h"
#include "bluetooth-device-selection.h"

static gboolean singleton = FALSE;

static BluetoothClient *client;

static gchar *address = NULL;
static gchar *passkey = "123456";

static GtkWidget *page_search = NULL;
static GtkWidget *page_info = NULL;
static GtkWidget *page_pairing = NULL;
static GtkWidget *page_summary = NULL;

static GtkWidget *label_pairing = NULL;
static GtkWidget *label_passkey = NULL;

static GtkWidget *selector = NULL;

static gboolean passkey_agent_request(GObject *agent,
				const char *path, const char *address,
					DBusGMethodInvocation *context)
{
	const char *value = passkey;

	/* Apple Wireless and Mighty Mouse */
	if (g_str_has_prefix(address, "00:0A:95:") == TRUE ||
				g_str_has_prefix(address, "00:14:51:") == TRUE)
		value = "0000";

	dbus_g_method_return(context, value);

	return TRUE;
}

static gboolean passkey_agent_confirm(GObject *agent,
			const char *path, const char *address,
			const char *value, DBusGMethodInvocation *context)
{
	dbus_g_method_return(context);

	return TRUE;
}

static gboolean passkey_agent_cancel(GObject *agent,
			const char *path, const char *address, GError **error)
{
	return TRUE;
}

static gboolean passkey_agent_release(GObject *agent, GError **error)
{
	return TRUE;
}

#include "passkey-agent-glue.h"

#if 0
static gint page_forward(gint current_page, gpointer data)
{
	return current_page + 1;
}
#endif

static void close_callback(GtkWidget *assistant, gpointer data)
{
	gtk_widget_destroy(assistant);

	gtk_main_quit();
}

static void cancel_callback(GtkWidget *assistant, gpointer data)
{
	gtk_widget_destroy(assistant);

	gtk_main_quit();
}

static void prepare_callback(GtkWidget *assistant,
				GtkWidget *page, gpointer data)
{
	if (page == page_search) {
		bluetooth_device_selection_start_discovery(BLUETOOTH_DEVICE_SELECTION(selector));
		return;
	}

	if (page == page_pairing) {
		bluetooth_client_cancel_discovery(client, NULL);

		bluetooth_client_register_passkey_agent(client,
					"/org/bluez/applet", address,
					&dbus_glib_passkey_agent_object_info);

		bluetooth_client_create_bonding(client, NULL, address);

		gtk_label_set_markup(GTK_LABEL(label_pairing), address);
		gtk_label_set_markup(GTK_LABEL(label_passkey), passkey);
		return;
	}

	gtk_assistant_set_page_complete(GTK_ASSISTANT(assistant), page, TRUE);
}

static GtkWidget *create_vbox(GtkWidget *assistant, GtkAssistantPageType type,
				const gchar *title, const gchar *section)
{
	GtkWidget *vbox;
	GtkWidget *label;
	GdkPixbuf *pixbuf;
	gchar *str;

	vbox = gtk_vbox_new(FALSE, 6);

	gtk_container_set_border_width(GTK_CONTAINER(vbox), 24);

	gtk_assistant_append_page(GTK_ASSISTANT(assistant), vbox);

	gtk_assistant_set_page_type(GTK_ASSISTANT(assistant), vbox, type);

	str = g_strdup_printf(" %s", title);

	gtk_assistant_set_page_title(GTK_ASSISTANT(assistant), vbox, str);

	g_free(str);

	//pixbuf = gtk_widget_render_icon(GTK_WIDGET(assistant),
	//		GTK_STOCK_DIALOG_INFO, GTK_ICON_SIZE_DIALOG, NULL);

	pixbuf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						"bluetooth", 32, 0, NULL);

	gtk_assistant_set_page_header_image(GTK_ASSISTANT(assistant),
								vbox, pixbuf);

	g_object_unref(pixbuf);

	if (section) {
		label = gtk_label_new(NULL);

		str = g_strdup_printf("<b>%s</b>\n", section);

		gtk_label_set_markup(GTK_LABEL(label), str);

		g_free(str);

		gtk_misc_set_alignment(GTK_MISC(label), 0, 0);

		gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	}

	return vbox;
}

static void create_intro(GtkWidget *assistant)
{
	GtkWidget *vbox;
	GtkWidget *label;
	GtkWidget *combo;
	GtkTreeModel *model;
	GtkCellRenderer *renderer;

	vbox = create_vbox(assistant, GTK_ASSISTANT_PAGE_INTRO,
			_("Introduction"),
			_("Welcome to the Bluetooth device setup wizard"));

	label = gtk_label_new(NULL);

	gtk_label_set_markup(GTK_LABEL(label), _("The device wizard will "
				"walk you through the process of configuring "
				"Bluetooth enabled devices for use with this "
				"computer.\n\n"));

	gtk_misc_set_alignment(GTK_MISC(label), 0, 0);

	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);

	gtk_widget_set_size_request(GTK_WIDGET(label), 380, -1);

	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, TRUE, 0);

	combo = gtk_combo_box_new();

	model = bluetooth_client_get_model_adapter_list(client);

	gtk_combo_box_set_model(GTK_COMBO_BOX(combo), model);

	g_object_unref(model);

	gtk_combo_box_set_active(GTK_COMBO_BOX(combo), 0);

	gtk_cell_layout_clear(GTK_CELL_LAYOUT(combo));

	renderer = gtk_cell_renderer_text_new();

	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combo), renderer, TRUE);

	gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(combo), renderer,
						"text", COLUMN_NAME, NULL);

	if (gtk_tree_model_iter_n_children(model, NULL) > 1)
		gtk_box_pack_start(GTK_BOX(vbox), combo, FALSE, FALSE, 0);
}

static void create_type(GtkWidget *assistant)
{
	GtkWidget *vbox;
	GtkWidget *button;
	GSList *group = NULL;

	vbox = create_vbox(assistant, GTK_ASSISTANT_PAGE_CONTENT,
			_("Device type"),
			_("Select the type of device you want to setup"));

	button = gtk_radio_button_new_with_label(group, _("Mouse"));

	group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));

	//gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);

	button = gtk_radio_button_new_with_label(group, _("Keyboard"));

	group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));

	//gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);

	button = gtk_radio_button_new_with_label(group, _("Mobile phone"));

	group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));

	gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);

	button = gtk_radio_button_new_with_label(group, _("Printer"));

	group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));

	gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);

	button = gtk_radio_button_new_with_label(group, _("Headset"));

	group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));

	gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);

	button = gtk_radio_button_new_with_label(group, _("Any device"));

	group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
}

static void select_callback(BluetoothDeviceSelection *sel,
				gchar *selected, gpointer user_data)
{
	GtkAssistant *assistant = user_data;

	if (selected != NULL) {
		g_free(address);

		address = g_strdup(selected);

		gtk_assistant_set_page_complete(assistant, page_search, TRUE);
	} else
		gtk_assistant_set_page_complete(assistant, page_search, FALSE);
}

static void create_search(GtkWidget *assistant)
{
	GtkWidget *vbox;

	vbox = create_vbox(assistant, GTK_ASSISTANT_PAGE_CONTENT,
				_("Device search"),
				_("Select the device you want to setup"));

	selector = bluetooth_device_selection_new(NULL);

	g_object_set(selector, "show-search", FALSE,
				"show-device-category", FALSE, NULL);

	g_signal_connect(selector, "selected-device-changed",
				G_CALLBACK(select_callback), assistant);

	bluetooth_device_selection_start_discovery(BLUETOOTH_DEVICE_SELECTION(selector));

	gtk_container_add(GTK_CONTAINER(vbox), selector);

	page_search = vbox;
}

static void create_info(GtkWidget *assistant)
{
	GtkWidget *vbox;
	GtkWidget *label;

	vbox = create_vbox(assistant, GTK_ASSISTANT_PAGE_CONTENT,
				_("Device information"),
				_("Gathering information about device"));

	label = gtk_label_new(NULL);

	gtk_misc_set_alignment(GTK_MISC(label), 0, 0);

	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, TRUE, 0);

	page_info = vbox;
}

static void create_pairing(GtkWidget *assistant)
{
	GtkWidget *vbox;
	GtkWidget *label;

	vbox = create_vbox(assistant, GTK_ASSISTANT_PAGE_CONTENT,
				_("Device pairing"),
				_("Pairing with new device"));

	label = gtk_label_new(NULL);

	gtk_misc_set_alignment(GTK_MISC(label), 0, 0);

	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, TRUE, 0);

	label_pairing = label;

	label = gtk_label_new(NULL);

	gtk_misc_set_alignment(GTK_MISC(label), 0, 0);

	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, TRUE, 0);

	label_passkey = label;

	page_pairing = vbox;
}

static void create_summary(GtkWidget *assistant)
{
	GtkWidget *vbox;

	vbox = create_vbox(assistant, GTK_ASSISTANT_PAGE_SUMMARY,
				_("Summary"),
				_("Succesfully configured new device"));

	page_summary = vbox;
}

static GtkWidget *create_wizard(void)
{
	GtkWidget *assistant;

	assistant = gtk_assistant_new();

	gtk_window_set_title(GTK_WINDOW(assistant), _("Bluetooth Device Wizard"));

	gtk_window_set_position(GTK_WINDOW(assistant), GTK_WIN_POS_CENTER);

	gtk_window_set_default_size(GTK_WINDOW(assistant), 440, 440);

#if 0
	gtk_assistant_set_forward_page_func(GTK_ASSISTANT(assistant),
						page_forward, NULL, NULL);
#endif

	create_intro(assistant);

	create_type(assistant);

	create_search(assistant);

	create_info(assistant);

	create_pairing(assistant);

	create_summary(assistant);

	g_signal_connect(G_OBJECT(assistant), "close",
					G_CALLBACK(close_callback), NULL);

	g_signal_connect(G_OBJECT(assistant), "cancel",
					G_CALLBACK(cancel_callback), NULL);

	g_signal_connect(G_OBJECT(assistant), "prepare",
					G_CALLBACK(prepare_callback), NULL);

	gtk_widget_show_all(assistant);

	gtk_assistant_update_buttons_state(GTK_ASSISTANT(assistant));

	return assistant;
}

static GOptionEntry options[] = {
	{ "singleton", 0, 0, G_OPTION_ARG_NONE, &singleton,
			"Only allow one instance of this application", NULL },
	{ NULL },
};

int main(int argc, char *argv[])
{
	BluetoothInstance *instance;
	GtkWidget *window;
	GError *error = NULL;

	bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);

	if (gtk_init_with_args(&argc, &argv, NULL,
				options, GETTEXT_PACKAGE, &error) == FALSE) {
		if (error) {
			g_print("%s\n", error->message);
			g_error_free(error);
		} else
			g_print("An unknown error occurred\n");

		gtk_exit(1);
	}

	instance = bluetooth_instance_new("wizard", singleton);
	if (instance == NULL)
		gtk_exit(0);

	gtk_window_set_default_icon_name("bluetooth");

	client = bluetooth_client_new();

	window = create_wizard();

	bluetooth_instance_set_window(instance, GTK_WINDOW(window));

	gtk_main();

	g_object_unref(client);

	g_object_unref(instance);

	return 0;
}
