/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2008  Marcel Holtmann <marcel@holtmann.org>
 *  Copyright (C) 2006-2007  Bastien Nocera <hadess@hadess.net>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dbus/dbus-glib.h>

#ifdef HAVE_HAL
#include <dbus/dbus-glib-lowlevel.h>
#include <hal/libhal.h>
#endif

#include <gconf/gconf-client.h>

#include <glib/gi18n.h>

#include <gtk/gtk.h>

#include "bluetooth-instance.h"
#include "bluetooth-device-selection.h"

#include "notify.h"
#include "agent.h"
#include "obex.h"

static gboolean singleton = FALSE;

static DBusGConnection *conn;

#ifdef HAVE_HAL
static gboolean use_hal = FALSE;
#endif

static GList *adapter_list = NULL;

struct adapter_data {
	char *path;
	int attached;
	char *old_mode;
};

enum {
	ICON_POLICY_NEVER,
	ICON_POLICY_ALWAYS,
	ICON_POLICY_PRESENT,
};

static int icon_policy = ICON_POLICY_PRESENT;

#define PREF_DIR		"/apps/bluetooth-manager"
#define PREF_USE_HAL		PREF_DIR "/use_hal"
#define PREF_ICON_POLICY	PREF_DIR "/icon_policy"
#define PREF_AUTO_AUTHORIZE	PREF_DIR "/auto_authorize"
#define PREF_RECEIVE_ENABLED	PREF_DIR "/receive_enabled"
#define PREF_SHARING_ENABLED	PREF_DIR "/sharing_enabled"

static GConfClient* gconf;

static GtkWidget *menuitem_sendto = NULL;
static GtkWidget *menuitem_browse = NULL;

static void bonding_created(DBusGProxy *object,
				const char *address, gpointer user_data)
{
	const char *adapter = NULL, *name = NULL;
	gchar *device, *text;

	dbus_g_proxy_call(object, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &adapter, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "GetRemoteName", NULL,
				G_TYPE_STRING, address, G_TYPE_INVALID,
				G_TYPE_STRING, &name, G_TYPE_INVALID);

	if (name) {
		if (g_strrstr(name, address))
			device = g_strdup(name);
		else
			device = g_strdup_printf("%s (%s)", name, address);
	} else
		device = g_strdup(address);

	text = g_strdup_printf(_("Created bonding with %s"), device);

	g_free(device);

	show_notification(adapter ? adapter : _("Bluetooth device"),
						text, NULL, 6000, NULL);

	g_free(text);
}

static void bonding_removed(DBusGProxy *object,
				const char *address, gpointer user_data)
{
	const char *adapter = NULL, *name = NULL;
	gchar *device, *text;

	dbus_g_proxy_call(object, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &adapter, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "GetRemoteName", NULL,
				G_TYPE_STRING, address, G_TYPE_INVALID,
				G_TYPE_STRING, &name, G_TYPE_INVALID);

	if (name) {
		if (g_strrstr(name, address))
			device = g_strdup(name);
		else
			device = g_strdup_printf("%s (%s)", name, address);
	} else
		device = g_strdup(address);

	text = g_strdup_printf(_("Removed bonding with %s"), device);

	g_free(device);

	show_notification(adapter ? adapter : _("Bluetooth device"),
						text, NULL, 6000, NULL);

	g_free(text);
}

static void trust_added(DBusGProxy *object,
				const char *address, gpointer user_data)
{
}

static void trust_removed(DBusGProxy *object,
				const char *address, gpointer user_data)
{
}

static void set_new_mode(struct adapter_data *adapter, const char *mode)
{
	g_free(adapter->old_mode);

	adapter->old_mode = g_strdup(mode);
}

static void mode_changed(DBusGProxy *object,
				const char *mode, gpointer user_data)
{
	struct adapter_data *adapter = (struct adapter_data *) user_data;
	const char *adapter_name = NULL;
	const char *text;

	if (icon_policy == ICON_POLICY_PRESENT) {
		if (g_str_equal(mode, "off") == TRUE) {
			set_new_mode(adapter, mode);
			return;
		}
		if (g_str_equal(adapter->old_mode, "off")
				&& g_str_equal(mode, "connectable")) {
			set_new_mode(adapter, mode);
			return;
		}
	}

	if (g_str_equal(mode, "off") != FALSE) {
		text = N_("Device has been switched off");
	} else if (g_str_equal(mode, "connectable") != FALSE
		   && g_str_equal(adapter->old_mode, "discoverable") != FALSE) {
		text = N_("Device has been made non-discoverable");
	} else if (g_str_equal(mode, "connectable") != FALSE) {
		text = N_("Device has been made connectable");
	} else if (g_str_equal (mode, "discoverable") != FALSE) {
		text = N_("Device has been made discoverable");
	} else if (g_str_equal(mode, "limited") != FALSE) {
		text = N_("Device has been made limited discoverable");
	} else if (g_str_equal(mode, "pairing") != FALSE) {
		text = N_("Device has been switched into pairing mode");
	} else {
		set_new_mode(adapter, mode);
		return;
	}

	dbus_g_proxy_call(object, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &adapter_name, G_TYPE_INVALID);

	show_notification(adapter_name ? adapter_name : _("Bluetooth device"),
							_(text), NULL, 3000, NULL);

	set_new_mode(adapter, mode);
}

static void adapter_free(gpointer data, gpointer user_data)
{
	struct adapter_data *adapter = data;

	adapter_list = g_list_remove(adapter_list, adapter);

	g_free(adapter->path);
	g_free(adapter->old_mode);
	g_free(adapter);
}

static void adapter_disable(gpointer data, gpointer user_data)
{
	struct adapter_data *adapter = data;

	adapter->attached = 0;
}

static gint adapter_compare(gconstpointer a, gconstpointer b)
{
	const struct adapter_data *adapter = a;
	const char *path = b;

	return g_ascii_strcasecmp(adapter->path, path);
}

static void adapter_count(gpointer data, gpointer user_data)
{
	struct adapter_data *adapter = data;
	int *count = user_data;

	if (adapter->attached)
		(*count)++;
}

static int attached_adapters(void)
{
	int count = 0;

	g_list_foreach(adapter_list, adapter_count, &count);

	return count;
}

#ifdef HAVE_HAL
static char *get_form_factor(void)
{
	LibHalContext *ctx;
	char *formfactor;

	ctx = libhal_ctx_new();
	if (ctx == NULL)
		return NULL;

	if (libhal_ctx_set_dbus_connection(ctx,
			dbus_g_connection_get_connection(conn)) == FALSE) {
		libhal_ctx_free(ctx);
		return NULL;
	}

	if (libhal_ctx_init(ctx, NULL) == FALSE) {
		g_warning("Unable to init HAL context");
		libhal_ctx_free(ctx);
		return NULL;
	}

	formfactor = libhal_device_get_property_string(ctx,
				"/org/freedesktop/Hal/devices/computer",
						"system.formfactor", NULL);

	libhal_ctx_shutdown(ctx, NULL);
	libhal_ctx_free(ctx);

	if (g_str_equal(formfactor, "laptop") == TRUE &&
			g_str_equal(formfactor, "desktop") == TRUE &&
			g_str_equal(formfactor, "server") == TRUE &&
			g_str_equal(formfactor, "handheld") == TRUE) {
		/* We don't know anything else */
		g_free(formfactor);
		formfactor = g_strdup("uncategorized");
	}

	return formfactor;
}

static void adapter_setup(gpointer data, gpointer user_data)
{
	struct adapter_data *adapter = data;
	const char *formfactor = user_data;
	DBusGProxy *object;

	if (formfactor == NULL)
		return;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
					adapter->path, "org.bluez.Adapter");
	if (!object)
		return;

	dbus_g_proxy_call(object, "SetMajorClass", NULL,
		G_TYPE_STRING, "computer", G_TYPE_INVALID, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "SetMinorClass", NULL,
		G_TYPE_STRING, formfactor, G_TYPE_INVALID, G_TYPE_INVALID);
}
#endif

static void add_adapter(const char *path)
{
	GList *list;
	DBusGProxy *object;
	struct adapter_data *adapter;
	const char *old_mode;

	if (icon_policy != ICON_POLICY_NEVER)
		show_icon();

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;

		adapter->attached = 1;
		return;
	}

	adapter = g_try_malloc0(sizeof(*adapter));
	if (!adapter)
		return;

	adapter->path = g_strdup(path);
	adapter->attached = 1;

	adapter_list = g_list_append(adapter_list, adapter);

#ifdef HAVE_HAL
	if (use_hal == TRUE) {
		char *formfactor;

		formfactor = get_form_factor();
		adapter_setup(adapter, formfactor);
		g_free(formfactor);
	}
#endif

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
						path, "org.bluez.Adapter");

	dbus_g_proxy_add_signal(object, "ModeChanged",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "ModeChanged",
				G_CALLBACK(mode_changed), adapter, NULL);

	dbus_g_proxy_add_signal(object, "BondingCreated",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "BondingCreated",
				G_CALLBACK(bonding_created), NULL, NULL);

	dbus_g_proxy_add_signal(object, "BondingRemoved",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "BondingRemoved",
				G_CALLBACK(bonding_removed), NULL, NULL);

	dbus_g_proxy_add_signal(object, "TrustAdded",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "TrustAdded",
				G_CALLBACK(trust_added), NULL, NULL);

	dbus_g_proxy_add_signal(object, "TrustRemoved",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "TrustRemoved",
				G_CALLBACK(trust_removed), NULL, NULL);

	old_mode = NULL;
	dbus_g_proxy_call(object, "GetMode", NULL,
			  G_TYPE_INVALID, G_TYPE_STRING,
			  &old_mode, G_TYPE_INVALID);
	if (old_mode != NULL)
		set_new_mode(adapter, old_mode);
}

static void adapter_added(DBusGProxy *object,
				const char *path, gpointer user_data)
{
	register_agents();

	add_adapter(path);
}

static void adapter_removed(DBusGProxy *object,
				const char *path, gpointer user_data)
{
	GList *list;

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;

		adapter->attached = 0;
	}

	if (icon_policy != ICON_POLICY_ALWAYS && attached_adapters() == 0)
		hide_icon();
}

static int setup_manager(void)
{
	DBusGProxy *object;
	GError *error = NULL;
	const gchar **array = NULL;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
					"/org/bluez", "org.bluez.Manager");

	dbus_g_proxy_add_signal(object, "AdapterAdded",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "AdapterAdded",
				G_CALLBACK(adapter_added), NULL, NULL);

	dbus_g_proxy_add_signal(object, "AdapterRemoved",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "AdapterRemoved",
				G_CALLBACK(adapter_removed), NULL, NULL);

	dbus_g_proxy_call(object, "ListAdapters", &error,
			G_TYPE_INVALID,	G_TYPE_STRV, &array, G_TYPE_INVALID);

	if (error == NULL) {
		while (*array) {
			add_adapter(*array);
			array++;
		}
	} else
		g_error_free(error);

	return 0;
}

static gboolean program_available(const char *program)
{
	gchar *path;

	path = g_find_program_in_path(program);
	if (path == NULL)
		return FALSE;

	g_free(path);

	return TRUE;
}

static void close_callback(GtkWidget *dialog, gpointer user_data)
{
	gtk_widget_destroy(dialog);
}

static void open_uri(GtkWindow *parent, const char *uri)
{
	GtkWidget *dialog;
	GdkScreen *screen;
	GError *error = NULL;
	gchar *cmdline;

	screen = gtk_window_get_screen(parent);

	cmdline = g_strconcat("xdg-open ", uri, NULL);

	if (gdk_spawn_command_line_on_screen(screen,
						cmdline, &error) == FALSE) {
		dialog = gtk_message_dialog_new(parent,
			GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_ERROR,
					GTK_BUTTONS_CLOSE, error->message);
		gtk_dialog_run(GTK_DIALOG(dialog));
		gtk_widget_destroy(dialog);
		g_error_free(error);
	}

	g_free(cmdline);
}

static void about_url_hook(GtkAboutDialog *dialog,
					const gchar *url, gpointer data)
{
	open_uri(GTK_WINDOW(dialog), url);
}

static void about_email_hook(GtkAboutDialog *dialog,
					const gchar *email, gpointer data)
{
	gchar *uri;

	uri = g_strconcat("mailto:", email, NULL);
	open_uri(GTK_WINDOW(dialog), uri);
	g_free(uri);
}

static void about_callback(GtkWidget *item, gpointer user_data)
{
	const gchar *authors[] = {
		"Marcel Holtmann <marcel@holtmann.org>",
		"Bastien Nocera <hadess@hadess.net>",
		NULL
	};
	const gchar *artists[] = {
		"Andreas Nilsson <nisses.mail@home.se>",
		NULL,
	};

	gtk_about_dialog_set_url_hook(about_url_hook, NULL, NULL);
	gtk_about_dialog_set_email_hook(about_email_hook, NULL, NULL);

	gtk_show_about_dialog(NULL, "version", VERSION,
		"copyright", "Copyright \xc2\xa9 2005-2008 Marcel Holtmann",
		"comments", _("A Bluetooth manager for the GNOME desktop"),
		"authors", authors,
		"artists", artists,
		"translator-credits", _("translator-credits"),
		"website", "http://www.bluez.org",
		"website-label", "www.bluez.org",
		"logo-icon-name", "bluetooth", NULL);
}

static void settings_callback(GObject *widget, gpointer user_data)
{
	const char *command = "bluetooth-properties --singleton";

	if (!g_spawn_command_line_async(command, NULL))
		g_printerr("Couldn't execute command: %s\n", command);
}

static void response_callback(GtkWidget *dialog,
					gint response, gpointer user_data)
{
	GtkWidget *selector = user_data;
	gchar *command, *address = NULL;

	if (response != GTK_RESPONSE_ACCEPT) {
		gtk_widget_destroy(dialog);
		return;
	}

	g_object_get(selector, "device-selected", &address, NULL);

	gtk_widget_destroy(dialog);

	command = g_strdup_printf("%s --no-default-window \"obex://[%s]\"",
							"nautilus", address);

	g_free(address);

	if (!g_spawn_command_line_async(command, NULL))
		g_printerr(_("Couldn't execute command: %s\n"), command);

	g_free(command);
}

static void selected_device_changed(BluetoothDeviceSelection *selector,
					gchar *address, gpointer user_data)
{
	GtkWidget *dialog = user_data;

	gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
				GTK_RESPONSE_ACCEPT, address != NULL);
}

static void browse_callback(GObject *widget, gpointer user_data)
{
	GtkWidget *dialog;
	GtkWidget *selector;

	dialog = gtk_dialog_new_with_buttons(_("Browse Devices"),
				NULL, GTK_DIALOG_NO_SEPARATOR,
				GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
				GTK_STOCK_CONNECT, GTK_RESPONSE_ACCEPT, NULL);

	gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
						GTK_RESPONSE_ACCEPT, FALSE);

	gtk_window_set_default_size(GTK_WINDOW(dialog), 450, 400);

	gtk_container_set_border_width(GTK_CONTAINER (dialog), 5);
	gtk_box_set_spacing(GTK_BOX(GTK_DIALOG(dialog)->vbox), 2);

	selector = bluetooth_device_selection_new(_("Select device to browse"));
	gtk_container_set_border_width(GTK_CONTAINER(selector), 5);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), selector);

	g_signal_connect(selector, "selected-device-changed",
				G_CALLBACK(selected_device_changed), dialog);

	g_signal_connect(dialog, "close",
				G_CALLBACK(close_callback), NULL);

	g_signal_connect(dialog, "response",
				G_CALLBACK(response_callback), selector);

	gtk_widget_show_all(dialog);

	g_object_set(selector, "show-search", FALSE, NULL);

	bluetooth_device_selection_start_discovery(BLUETOOTH_DEVICE_SELECTION(selector));
}

static void sendto_callback(GObject *widget, gpointer user_data)
{
	const char *command = "bluetooth-sendto";

	if (!g_spawn_command_line_async(command, NULL))
		g_printerr("Couldn't execute command: %s\n", command);
}

#if 0
static void wizard_callback(GObject *widget, gpointer user_data)
{
	const char *command = "bluetooth-wizard --singleton";

	if (!g_spawn_command_line_async(command, NULL))
		g_printerr("Couldn't execute command: %s\n", command);
}
#endif

static void activate_callback(GObject *widget, gpointer user_data)
{
	show_agents();
}

static void popup_callback(GObject *widget, guint button,
				guint activate_time, gpointer user_data)
{
	GtkWidget *menu = user_data;

	gtk_widget_set_sensitive(menuitem_sendto,
				program_available("obex-data-server") &&
						attached_adapters() > 0);

	gtk_widget_set_sensitive(menuitem_browse,
					program_available("nautilus") &&
						attached_adapters() > 0);

	gtk_menu_popup(GTK_MENU(menu), NULL, NULL,
			gtk_status_icon_position_menu,
			GTK_STATUS_ICON(widget), button, activate_time);
}

static GtkWidget *create_popupmenu(void)
{
	GtkWidget *menu;
	GtkWidget *item;

	menu = gtk_menu_new();

	item = gtk_image_menu_item_new_from_stock(GTK_STOCK_PREFERENCES, NULL);
	g_signal_connect(item, "activate",
				G_CALLBACK(settings_callback), NULL);
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

	item = gtk_image_menu_item_new_from_stock(GTK_STOCK_ABOUT, NULL);
	g_signal_connect(item, "activate",
				G_CALLBACK(about_callback), NULL);
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

	item = gtk_separator_menu_item_new();
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

	item = gtk_image_menu_item_new_with_label(_("Send File..."));
	g_signal_connect(item, "activate",
				G_CALLBACK(sendto_callback), NULL);
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

	menuitem_sendto = item;

	item = gtk_image_menu_item_new_with_label(_("Browse Device..."));
	g_signal_connect(item, "activate",
				G_CALLBACK(browse_callback), NULL);
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

	menuitem_browse = item;

#if 0
	item = gtk_separator_menu_item_new();
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

	item = gtk_menu_item_new_with_label(_("Setup New Device"));
	g_signal_connect(item, "activate",
				G_CALLBACK(wizard_callback), NULL);
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
#endif

	return menu;
}

static void name_owner_changed(DBusGProxy *object, const char *name,
			const char *prev, const char *new, gpointer user_data)
{
	if (g_str_equal(name, "org.bluez") == TRUE && *new == '\0') {
		unregister_agents();

		g_list_foreach(adapter_list, adapter_disable, NULL);

		if (icon_policy != ICON_POLICY_ALWAYS)
			hide_icon();
	}
}

static int setup_dbus(void)
{
	DBusGProxy *proxy;

	proxy = dbus_g_proxy_new_for_name(conn, DBUS_SERVICE_DBUS,
					DBUS_PATH_DBUS, DBUS_INTERFACE_DBUS);

	dbus_g_proxy_add_signal(proxy, "NameOwnerChanged",
		G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(proxy, "NameOwnerChanged",
				G_CALLBACK(name_owner_changed), NULL, NULL);

	return setup_agents(conn);
}

static GConfEnumStringPair icon_policy_enum_map [] = {
	{ ICON_POLICY_NEVER,	"never"		},
	{ ICON_POLICY_ALWAYS,	"always"	},
	{ ICON_POLICY_PRESENT,	"present"	},
	{ ICON_POLICY_PRESENT,	NULL		},
};

static void gconf_callback(GConfClient *client, guint cnxn_id,
					GConfEntry *entry, gpointer user_data)
{
	GConfValue *value;

	value = gconf_entry_get_value(entry);
	if (value == NULL)
		return;

	if (g_str_equal(entry->key, PREF_USE_HAL) == TRUE) {
#ifdef HAVE_HAL
		use_hal = gconf_value_get_bool(value);

		if (use_hal == TRUE) {
			gchar *formfactor;

			formfactor = get_form_factor();
			g_list_foreach(adapter_list, adapter_setup, formfactor);
			g_free(formfactor);
		}
#endif
		return;
	}

	if (g_str_equal(entry->key, PREF_ICON_POLICY) == TRUE) {
		const char *str;

		str = gconf_value_get_string(value);
		if (!str)
			return;

		gconf_string_to_enum(icon_policy_enum_map, str, &icon_policy);

		if (icon_policy == ICON_POLICY_NEVER)
			hide_icon();
		else if (icon_policy == ICON_POLICY_ALWAYS)
			show_icon();
		else if (icon_policy == ICON_POLICY_PRESENT) {
			if (attached_adapters() > 0)
				show_icon();
			else
				hide_icon();
		}
		return;
	}

	if (g_str_equal(entry->key, PREF_AUTO_AUTHORIZE) == TRUE) {
		set_auto_authorize(gconf_value_get_bool(value));
		return;
	}

	if (g_str_equal(entry->key, PREF_RECEIVE_ENABLED) == TRUE) {
		set_receive_enabled(gconf_value_get_bool(value));
		return;
	}

	if (g_str_equal(entry->key, PREF_SHARING_ENABLED) == TRUE) {
		set_sharing_enabled(gconf_value_get_bool(value));
		return;
	}
}

static GOptionEntry options[] = {
	{ "singleton", 0, 0, G_OPTION_ARG_NONE, &singleton,
			"Only allow one instance of this application", NULL },
	{ NULL },
};

int main(int argc, char *argv[])
{
	BluetoothInstance *instance;
	GtkStatusIcon *statusicon;
	GtkWidget *menu;
	GError *error = NULL;
	char *str;

	bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);

	if (gtk_init_with_args(&argc, &argv, NULL,
				options, GETTEXT_PACKAGE, &error) == FALSE) {
		if (error) {
			g_print("%s\n", error->message);
			g_error_free(error);
		} else
			g_print("An unknown error occurred\n");

		gtk_exit(1);
	}

	instance = bluetooth_instance_new("applet", singleton);
	if (instance == NULL)
		gtk_exit(0);

	g_set_application_name(_("Bluetooth Applet"));

	gtk_window_set_default_icon_name("bluetooth");

	conn = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
	if (error != NULL) {
		g_printerr("Connecting to system bus failed: %s\n",
							error->message);
		g_error_free(error);
		gtk_exit(0);
	}

	gconf = gconf_client_get_default();

#ifdef HAVE_HAL
	use_hal = gconf_client_get_bool(gconf, PREF_USE_HAL, NULL);
#endif

	str = gconf_client_get_string(gconf, PREF_ICON_POLICY, NULL);
	if (str) {
		gconf_string_to_enum(icon_policy_enum_map, str, &icon_policy);
		g_free(str);
	}

	set_auto_authorize(gconf_client_get_bool(gconf,
						PREF_AUTO_AUTHORIZE, NULL));

	set_receive_enabled(gconf_client_get_bool(gconf,
						PREF_RECEIVE_ENABLED, NULL));

	set_sharing_enabled(gconf_client_get_bool(gconf,
						PREF_SHARING_ENABLED, NULL));

	gconf_client_add_dir(gconf, PREF_DIR, GCONF_CLIENT_PRELOAD_NONE, NULL);

	gconf_client_notify_add(gconf, PREF_DIR,
					gconf_callback, NULL, NULL, NULL);

	statusicon = init_notification();

	if (icon_policy != ICON_POLICY_ALWAYS)
		hide_icon();

	g_signal_connect(statusicon, "activate",
				G_CALLBACK(activate_callback), NULL);

	menu = create_popupmenu();

	g_signal_connect(statusicon, "popup-menu",
				G_CALLBACK(popup_callback), menu);

	setup_dbus();

	setup_manager();

	register_agents();

	setup_obex();

	gtk_main();

	gtk_widget_destroy(menu);

	cleanup_obex();

	g_object_unref(gconf);

	cleanup_notification();

	cleanup_agents();

	g_list_foreach(adapter_list, adapter_free, NULL);

	dbus_g_connection_unref(conn);

	g_object_unref(instance);

	return 0;
}
