/* cmd-missing.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/bugs/exception.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/fs/file-names.h"
#include "libfsutils/rmrf.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/patch-logs.h"
#include "libarch/local-cache.h"
#include "libarch/make-changeset.h"
#include "libarch/archive.h"
#include "libarch/missing.h"
#include "libarch/changeset-report.h"
#include "commands/cmdutils.h"
#include "commands/missing.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] [version]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_archive, "A", "archive", 1, \
      N_("Override `my-default-archive'")) \
  OP (opt_dir, "d", "dir DIR", 1, \
      N_("cd to DIR first")) \
  OP (opt_reverse, "r", "reverse", 0, \
      N_("sort from newest to oldest")) \
  OP (opt_summary, "s", "summary", 0, \
      N_("display a summary of each missing patch")) \
  OP (opt_creator, "c", "creator", 0, \
      N_("display the creator of each missing patch")) \
  OP (opt_date, "D", "date", 0, \
      N_("display the date of each missing patch")) \
  OP (opt_unqualified, 0, "unqualified", 0, \
      N_("print unqualified revision names")) \
  OP (opt_skip_present, 0, "skip-present", 0, \
      N_("skip patches that contain 1 or more patch logs already in this tree"))

t_uchar arch_cmd_missing_help[] = N_("print patches missing from a project tree\n"
                                   "Print a list of patches missing in the project tree containing\n"
                                   "DIR (or the current directory) for VERSION (or the default version.\n"
                                   "of the project tree).\n"
                                   "\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_missing (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * default_archive = 0;
  t_uchar * dir = 0;
  int reverse = 0;
  int summarized_headers = 0;
  int full = 1;
  int skip_present = 0;

  dir = str_save (0, ".");

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_missing_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;


        case opt_archive:
          {
            lim_free (0, default_archive);
            default_archive = str_save (0, option->arg_string);
            break;
          }

        case opt_dir:
          {
            lim_free (0, dir);
            dir = str_save (0, option->arg_string);
            break;
          }

        case opt_unqualified:
          {
            full = 0;
            break;
          }

        case opt_reverse:
          {
            reverse = 1;
            break;
          }
        case opt_summary:
          {
            summarized_headers |= arch_include_summary;
            break;
          }
        case opt_creator:
          {
            summarized_headers |= arch_include_creator;
            break;
          }
        case opt_date:
          {
            summarized_headers |= arch_include_date;
            break;
          }
        case opt_skip_present:
          {
            skip_present = 1;
            break;
          }
        }
    }

  if (argc > 2)
    goto usage_error;

  {
    arch_project_tree_t * tree; 
    t_uchar * vsnspec = NULL;
    t_uchar * version= NULL;
    t_uchar * namespace = NULL;
    struct arch_archive * arch = 0;
    rel_table whats_missing = 0;

    if (default_archive && !arch_valid_archive_name (default_archive))
      {
        safe_printfmt (2, _("%s: invalid archive name: %s\n"),
                       argv[0], default_archive);
        exit (1);
      }

    /* This sets up a struct for us that gathers all sorts of information about the tree */
    tree = arch_project_tree_new (talloc_context, dir);

    /* We need the fully qualified version. Either they gave it to us on argv[1], or 
     * we need to grab it from the tree info
     */ 
    if (argc == 2)
      vsnspec = argv[1];
    else
      {
        vsnspec = tree->fqversion;
        if (!vsnspec)
          {
            safe_printfmt (2, _("%s: tree has no default version\n  tree: %s\n"),
                           argv[0], tree->root);
            exit (1);
          }
      }


    /* Given a tree and a versionspec, return an archive
     * connection and the namespace given. Here, we already 
     * know a "what or where", so we give that, to get our archive
     * connection and the namespace description
     */
    arch_project_tree_check_name (tree, &arch, &namespace, vsnspec);
    if (!arch)
      {
        safe_printfmt(2, _("%s: failed to connect to archive for %s\n"), argv[0], vsnspec);
        exit(2);
      }

    /* make sure that the user actually gave us a fqvn */
    if (!arch_valid_package_name (namespace, arch_maybe_archive, arch_req_version, 0))
      {
        safe_printfmt (2, _("%s: illegal version name: %s\n"), argv[0], namespace);
        exit (2);
      }

    version = arch_parse_package_name (arch_ret_non_archive, 0, namespace);

    arch_check_for (arch, arch_req_version, version);
    
    whats_missing = arch_missing (tree->root, arch, version, skip_present);

    if (reverse)
      arch_sort_table_by_patch_level_field (1, whats_missing, 0);

    {
        int x;

        for (x = 0; x < rel_n_records (whats_missing); ++x)
          {
            if (!summarized_headers)
              {
                if (full)
                  safe_printfmt (1, "%s/%s--%s\n", arch->official_name, version, whats_missing[x][0]);
                else
                  safe_printfmt (1, "%s\n", whats_missing[x][0]);
              }
            else
              {
                t_uchar * revision = 0;
                t_uchar * log = 0;
                t_uchar * body = 0;
                assoc_table headers = 0;

                revision = str_alloc_cat_many (0, version, "--", whats_missing[x][0], str_end);
                log = arch_archive_log (arch, revision);

                arch_parse_log (0, &headers, &body, log);
                
                if (full)
                  safe_printfmt (1, "%s/%s--%s\n", arch->official_name, version, whats_missing[x][0]);
                else
                  safe_printfmt (1, "%s\n", whats_missing[x][0]);

                arch_print_headers_summary (1, 4, headers, summarized_headers);

                lim_free (0, revision);
                lim_free (0, log);
                lim_free (0, body);
                free_assoc_table (headers);
              }
          }
      }

    arch_archive_close (arch);

    lim_free (0, namespace);
    lim_free (0, version);
    arch_project_tree_delete (tree);
    rel_free_table (whats_missing);
  }


  lim_free (0, dir);
  lim_free (0, default_archive);

  return 0;
}



/* tag: Tom Lord Sat May 24 23:36:40 2003 (whats-missing.c)
 */
