/* AbiSource Application Framework
 * Copyright (C) 1998 AbiSource, Inc.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "ut_string.h"
#include "ut_assert.h"
#include "ut_debugmsg.h"
#include "ut_Language.h"
#include "xap_Dlg_Language.h"
#include "xap_Strings.h"   


static int s_compareQ(const void * a, const void * b)                           
{                                                                               
	const XML_Char ** A = (const XML_Char **)(a);                                              
	const XML_Char ** B = (const XML_Char **)(b);                                   	
	return UT_strcoll(*A,*B);                                               
}       

/*****************************************************************/

XAP_Dialog_Language::XAP_Dialog_Language(XAP_DialogFactory * pDlgFactory, XAP_Dialog_Id id)
	: XAP_Dialog_NonPersistent(pDlgFactory,id, "interface/dialoglanguage")
{
	UT_uint32 nDontSort = 0, nSort = 0;
	UT_uint32 i;	

	m_answer		   = a_CANCEL;
	m_pLanguage		   = NULL;
	m_pLangProperty	   = NULL;
	m_bChangedLanguage = false;
	m_pLangTable = new UT_Language;
	const XML_Char ** ppLanguagesTemp = new const XML_Char * [m_pLangTable->getCount()];	
	
	UT_ASSERT(m_pLangTable);
	m_iLangCount = m_pLangTable->getCount();
	m_ppLanguages = new const XML_Char * [m_iLangCount];	
	m_ppLanguagesCode = new const XML_Char * [m_iLangCount];
	
	for(i=0; i<m_iLangCount; i++)                                           
	{                                                                       
		if (m_pLangTable->getNthId(i)==XAP_STRING_ID_LANG_0) // Unsorted languages
		{
			m_ppLanguages[nDontSort]=m_pLangTable->getNthLangName(i);                                                    
			nDontSort++;                                             
		}
		else
		{
			ppLanguagesTemp[nSort] = m_pLangTable->getNthLangName(i);                                                                     
			nSort++;
		}
	}                                                                       

	// sort the temporary array                                                                                                                      
	qsort(ppLanguagesTemp, m_iLangCount-nDontSort, sizeof(XML_Char *), s_compareQ);

	  
	// Copy the sorted codes and a ssign each language its code
	for(UT_uint32 nLang = 0; nLang < m_iLangCount; nLang++)
	{
		if (nLang>=nDontSort)
			m_ppLanguages[nLang]=ppLanguagesTemp[nLang-nDontSort];

		for(i = 0; i < m_iLangCount; i++)
		{
			if (strcmp (m_ppLanguages[nLang], m_pLangTable->getNthLangName(i))==0)
			{
				m_ppLanguagesCode[nLang] = m_pLangTable->getNthLangCode(i);
				break;
			}
		}
	}
	delete [] ppLanguagesTemp;

	// TODO: move spell-checking into XAP land and make this dialog
	// TODO: more like the MSWord one (i.e. add):
	// [] do not check spelling or grammar
	m_bSpellCheck = true;
}

XAP_Dialog_Language::~XAP_Dialog_Language(void)
{
	if(m_pLangTable)
		delete m_pLangTable;
	if(m_ppLanguages)
		delete [] m_ppLanguages;
	if(m_ppLanguagesCode)
		delete [] m_ppLanguagesCode;
}

// we will not use the value passed to us, but rather will reference
// ourselves into m_pLangTable; that way we do not have to worry about
// the string disappearing on us, nor do we need to clone it
void XAP_Dialog_Language::setLanguageProperty(const XML_Char * pLangProp)
{
	UT_ASSERT(m_pLangTable);
	UT_uint32 indx = m_pLangTable->getIndxFromCode(
		pLangProp ? pLangProp :"-none-");
	m_pLanguage	    = m_pLangTable->getNthLangName(indx);
	m_pLangProperty = m_pLangTable->getNthLangCode(indx);
}

// in this case we do not need to worry about the lifespan of pLang
// since we call it only internally, always referring back to m_pLangTable
void XAP_Dialog_Language::_setLanguage(const XML_Char * pLang)
{
	UT_ASSERT(m_pLangTable);
	m_pLanguage	    = pLang;
	m_pLangProperty = m_pLangTable->getCodeFromName(pLang);
}


XAP_Dialog_Language::tAnswer XAP_Dialog_Language::getAnswer(void) const
{
	return m_answer;
}

bool XAP_Dialog_Language::getChangedLangProperty(const XML_Char ** pszLangProp) const
{
	UT_ASSERT(pszLangProp);
	*pszLangProp = m_pLangProperty;
	return m_bChangedLanguage;
}

/*
	Creates a vector with a list of support languages for spell checking

	You must to free the allocated memory
*/
UT_Vector* XAP_Dialog_Language::getAvailableDictionaries()
{
	SpellChecker * checker = SpellManager::instance().getInstance();
	UT_Vector& vec= checker->getMapping();
	DictionaryMapping * mapping;
	UT_Vector* vecRslt = new UT_Vector();

	const UT_uint32 nItems = vec.getItemCount();

	for (UT_uint32 iItem = nItems; iItem; --iItem)
	{
		mapping = static_cast<DictionaryMapping*>(vec.getNthItem(iItem - 1));

		if (checker->doesDictionaryExist(mapping->lang.c_str()))
			vecRslt->addItem( strdup(mapping->lang.c_str()));
	}

	return vecRslt;
}


